/*******************************************************************************
 *                                                                             *
 *                  Copyright (C) 2011 Altera Corporation                      *
 *                                                                             *
 * ALTERA, ARRIA, CYCLONE, HARDCOPY, MAX, MEGACORE, NIOS, QUARTUS & STRATIX    *
 * are Reg. U.S. Pat. & Tm. Off. and Altera marks in and outside the U.S.      *
 *                                                                             *
 * All information provided herein is provided on an "as is" basis,            *
 * without warranty of any kind.                                               *
 *                                                                             *
 * Module Name: video_transform       	 File Name: video_transform.v          *
 *                                                                             *
 * Module Function: This file is the top level file for a custom peripheral    *
 *                  in the Qsys verification lab. It contains:                 *
 *                  Avalon slave port                                          *
 *                  Avalon ST ports (source sink) for connection to BFMs       *
 *                  for testing in this lab.                                   *
 *                                                                             *
 * REVISION HISTORY:                                                           *
 *  Revision 1.0    03/30/2011 - Initial Revision                              *
 ******************************************************************************/
 
// synopsys translate_off
`timescale 1 ps / 1 ps
// synopsys translate_on

module video_transform(

  // clocks and resets
  input             csi_clk,
  input             rsi_reset_n,
   
  // Avalon-MM slave port (for Nios II or state machine control)
  input             avs_s0_write,
  input             avs_s0_read,
  input       [3:0] avs_s0_address,
  input      [31:0] avs_s0_writedata,
  output reg [31:0] avs_s0_readdata,

  // Avalon-ST data input port
  output            asi_rgbin_ready,
  input             asi_rgbin_valid, 
  input      [23:0] asi_rgbin_data,

  // Avalon-ST data output port
  input             aso_xyzout_ready,
  output            aso_xyzout_valid,
  output     [31:0] aso_xyzout_data
)
;
  
  // address definitions (ranges of 32 total bytes)
  `define a1_address 0
  `define a2_address 1
  `define a3_address 2
  `define b1_address 3
  `define b2_address 4
  `define b3_address 5
  `define c1_address 6
  `define c2_address 7
  `define c3_address 8

  // parameter for controlling the shift (default=15)
  parameter         shift_val = 15;
  
  // register defines
  reg               valid_out;
  
  // pipeline registers
  reg               rgbin_valid_s1;              // data valid in input capture registers
	
  // multiply-on period
  wire              multiply_on, multiply_on_delayed;
  
  // rgb data input registers                    // Input Vector:
  reg     [7:0]     red_data;                    //   R
  reg     [7:0]     green_data;                  //   G
  reg     [7:0]     blue_data;                   //   B
   
  // coefficient registers                       //  Multiplication Matrix:
  reg signed [31:0] a1, a2, a3;                  //   a1 a2 a3
  reg signed [31:0] b1, b2, b3;                  //   b1 b2 b3
  reg signed [31:0] c1, c2, c3;                  //   c1 c2 c3
   
  // xyz output registers                        // Product Vector (default 32 bits wide)
  reg signed [shift_val+17:0] x_data;            //   X
  reg signed [shift_val+17:0] y_data;            //   Y  (scalable to any fixed point shift)
  reg signed [shift_val+17:0] z_data;            //   Z
   
  // FIFO signals
  wire                        fifo_empty;
  wire                        fifo_full;
  wire    [shift_val+17:0]    shifted_x_data;    // (default = 17+15 = 32 bits wide)
  wire    [shift_val+17:0]    shifted_y_data;
  wire    [shift_val+17:0]    shifted_z_data;    // (scalable to any fixed point shift)
  wire               [7:0]    shifted_x_data_8;
  wire               [7:0]    shifted_y_data_8;
  wire               [7:0]    shifted_z_data_8;
  wire              [23:0]    buffer_data;
  wire              [23:0]    buffer_out;
  reg                         wr_fifo, write_requested;
  reg                         rd_fifo;
 
  // fifo write state machine
  reg fifo_write_curstate, fifo_write_nxtstate;
 
  // declare fifo write states
  parameter chk_fifo_full = 1'b0,  write_to_fifo = 1'b1; 

  // fifo read state machine
  reg fifo_read_curstate, fifo_read_nxtstate;
 
  // declare fifo read states
  parameter chk_fifo_empty = 1'b0, read_from_fifo = 1'b1;
	  

  //------------------------------------------------------------------------
  // code starts here
  //------------------------------------------------------------------------
  // load matrix multiplication coefficients - default is the Identify Matrix
  // with unity coeffs up-shifted by "shift_val" (down-shift done at output)
  always @ (posedge csi_clk or negedge rsi_reset_n) begin
    if (!rsi_reset_n) begin
	   // load Identity Matrix at boot-up:
      a1 <= (1 << shift_val);
		a2 <= 0;
		a3 <= 0;
		b1 <= 0;
		b2 <= (1 << shift_val);
		b3 <= 0;
      c1 <= 0;
		c2 <= 0;
		c3 <= (1 << shift_val);
    end
    else if (avs_s0_write)
      case (avs_s0_address)
        `a1_address : a1 <= avs_s0_writedata;
        `a2_address : a2 <= avs_s0_writedata;
        `a3_address : a3 <= avs_s0_writedata;
        `b1_address : b1 <= avs_s0_writedata;
        `b2_address : b2 <= avs_s0_writedata;
        `b3_address : b3 <= avs_s0_writedata;
        `c1_address : c1 <= avs_s0_writedata;
        `c2_address : c2 <= avs_s0_writedata;
        `c3_address : c3 <= avs_s0_writedata;
      endcase
  end
       
  // read back coefficients
  always @ * begin
    if (avs_s0_read)
      case (avs_s0_address)
        `a1_address : avs_s0_readdata <= a1;
        `a2_address : avs_s0_readdata <= a2;
        `a3_address : avs_s0_readdata <= a3;
        `b1_address : avs_s0_readdata <= b1;
        `b2_address : avs_s0_readdata <= b2;
        `b3_address : avs_s0_readdata <= b3;
        `c1_address : avs_s0_readdata <= c1;
        `c2_address : avs_s0_readdata <= c2;
        `c3_address : avs_s0_readdata <= c3;
        default : avs_s0_readdata <= 32'bx;
      endcase
    else
      avs_s0_readdata <= 32'b0;
  end  
         
  assign asi_rgbin_ready  = ~fifo_full && ~wr_fifo;

  // register RGB data
  always @ (posedge csi_clk or negedge rsi_reset_n) begin
    if (!rsi_reset_n) begin
      red_data       <= 0;
      green_data     <= 0;
      blue_data      <= 0;
    end
    else if (asi_rgbin_ready) begin
      red_data       <= asi_rgbin_data[7:0];
      green_data     <= asi_rgbin_data[15:8];
      blue_data      <= asi_rgbin_data[23:16];
    end
  end 

  // register and pipeline data-valid signals
  always @ (posedge csi_clk or negedge rsi_reset_n) begin
    if (!rsi_reset_n) begin
      rgbin_valid_s1 <= 0;
    end
    else if (asi_rgbin_ready) begin
      rgbin_valid_s1 <= asi_rgbin_valid;
    end
  end 
     
  // multiplication enable  
  assign multiply_on = rgbin_valid_s1 && ~fifo_full;

  // matrix multiply
  always @ (posedge csi_clk or negedge rsi_reset_n) begin
    if (!rsi_reset_n) begin
      x_data <= 0;
      y_data <= 0;
      z_data <= 0;
    end
    else if (multiply_on) begin
      x_data <= (a1 * red_data) + (a2 * green_data) + (a3 * blue_data);
      y_data <= (b1 * red_data) + (b2 * green_data) + (b3 * blue_data);
      z_data <= (c1 * red_data) + (c2 * green_data) + (c3 * blue_data);
    end
  end
    
  // Down-shifting the bits based on the shift_val parameter
  assign shifted_z_data = (z_data >> shift_val);
  assign shifted_y_data = (y_data >> shift_val);
  assign shifted_x_data = (x_data >> shift_val);

  // bit slice result to get rid of warnings in tool
  assign shifted_z_data_8 = shifted_z_data[7:0];
  assign shifted_y_data_8 = shifted_y_data[7:0];
  assign shifted_x_data_8 = shifted_x_data[7:0];
  
  // concatenate XYZ values to write into internal buffer
  assign buffer_data = {shifted_z_data_8, shifted_y_data_8, shifted_x_data_8};


  //-----------------------------------------------------------
  // fifo write state machine
  //-----------------------------------------------------------      
  // state transition logic (sequential)
  always @ (posedge csi_clk or negedge rsi_reset_n)
    if (!rsi_reset_n) 
      fifo_write_curstate <= chk_fifo_full;
    else
      fifo_write_curstate <= fifo_write_nxtstate;
  
  // next state logic (combinatorial)
  always @ * begin
    fifo_write_nxtstate <= fifo_write_curstate;
    case (fifo_write_curstate)
	   chk_fifo_full :
		  if (!fifo_full && rgbin_valid_s1)
          fifo_write_nxtstate <= write_to_fifo;
        else
          fifo_write_nxtstate <= chk_fifo_full;	 
      write_to_fifo :
		  // generate "wr_fifo" output (below)
		  fifo_write_nxtstate <= chk_fifo_full;		  
      default :   
		  fifo_write_nxtstate <= chk_fifo_full;
    endcase
  end
  
  // output logic
  always @ (posedge csi_clk or negedge rsi_reset_n) begin
    if (!rsi_reset_n) begin
      wr_fifo   <= 0;
    end
    else if (fifo_write_nxtstate == write_to_fifo) begin
      wr_fifo   <= 1;
    end
    else begin
      wr_fifo   <= 0;
    end
  end

  // buffer data in FIFO
  FIFO_24  U1 (
    .aclr(!rsi_reset_n),
    .clock(csi_clk),
    .data(buffer_data),
    .rdreq(rd_fifo),
    .wrreq(wr_fifo),
    .empty(fifo_empty),      
    .full(fifo_full),
    .q(buffer_out));

  //-----------------------------------------------------------
  // fifo read state machine
  //-----------------------------------------------------------
  // state transition logic (sequential)
  always @ (posedge csi_clk or negedge rsi_reset_n)
    if (!rsi_reset_n) 
      fifo_read_curstate <= chk_fifo_empty;
    else
      fifo_read_curstate <= fifo_read_nxtstate;
  
  // next state logic (combinatorial)
  always @ * begin
    fifo_read_nxtstate <= fifo_read_curstate;
    case (fifo_read_curstate)
	   chk_fifo_empty :
		  if (!fifo_empty && aso_xyzout_ready)
          fifo_read_nxtstate <= read_from_fifo;
        else
          fifo_read_nxtstate <= chk_fifo_empty;	 
      read_from_fifo :
		  // generate "rd_fifo" and "valid_out" outputs (below)
		  fifo_read_nxtstate <= chk_fifo_empty;		  
      default :   
		  fifo_read_nxtstate <= chk_fifo_empty;
    endcase
  end
  
  // output logic
  always @ (posedge csi_clk or negedge rsi_reset_n) begin
    if (!rsi_reset_n) begin
      rd_fifo   <= 0;
      valid_out <= 0;
    end
    else if (fifo_read_nxtstate == read_from_fifo) begin
      rd_fifo   <= 1;
      valid_out <= 1;
    end
    else begin
      rd_fifo   <= 0;
      valid_out <= 0;
    end
  end
  
  // pad output data with 8 zeros to conform with Qsys FIFO's 32 bit 
  // requirement and to avoid needing a streaming data format adapter
  assign aso_xyzout_data  = {buffer_out, 8'b0};
  
  // send output valid signal to downstream component
  assign aso_xyzout_valid = valid_out;

  
endmodule