-------------------------------------------------------------------------------
-- Title         : PCI Express BFM Root Port Driver 
-- Project       : PCI Express MegaCore function
-------------------------------------------------------------------------------
-- File          : altpcietb_bfm_driver.vhd
-- Author        : Altera Corporation
-------------------------------------------------------------------------------
-- Description :
-- This entity is driver for the Root Port BFM. It processes the list of
-- functions to perform and passes them off to the VC specific interfaces
-------------------------------------------------------------------------------
-- Copyright (c) 2005 Altera Corporation. All rights reserved.  Altera products are
-- protected under numerous U.S. and foreign patents, maskwork rights, copyrights and
-- other intellectual property laws.  
--
-- This reference design file, and your use thereof, is subject to and governed by
-- the terms and conditions of the applicable Altera Reference Design License Agreement.
-- By using this reference design file, you indicate your acceptance of such terms and
-- conditions between you and Altera Corporation.  In the event that you do not agree with
-- such terms and conditions, you may not use the reference design file. Please promptly
-- destroy any copies you have made.
--
-- This reference design file being provided on an "as-is" basis and as an accommodation 
-- and therefore all warranties, representations or guarantees of any kind 
-- (whether express, implied or statutory) including, without limitation, warranties of 
-- merchantability, non-infringement, or fitness for a particular purpose, are 
-- specifically disclaimed.  By making this reference design file available, Altera
-- expressly does not recommend, suggest or require that this reference design file be
-- used in combination with any other product not provided by Altera.
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use std.textio.all;
use work.altpcietb_bfm_constants.all;
use work.altpcietb_bfm_log.all;
use work.altpcietb_bfm_shmem.all;
use work.altpcietb_bfm_rdwr.all;
use work.altpcietb_bfm_configure.all;

entity altpcietb_bfm_driver is

  generic (
    -- TEST_LEVEL is a parameter passed in from the top level test bench that
    -- could control the amount of testing done. It is not currently used. 
    TEST_LEVEL : natural := 1);
  port (
    -- The clk_in and rstn signals are provided for possible use in controlling
    -- the transactions issued, they are not currently used.
    clk_in            : in  std_logic;
    rstn              : in  std_logic;
    INTA              : in  std_logic;
    INTB              : in  std_logic;
    INTC              : in  std_logic;
    INTD              : in  std_logic;
    dummy_out         : out  std_logic    
    );

  -- purpose: Use Reads and Writes to test the target memory
  --          The starting offset in the target memory and the
  --          length can be specified
  procedure target_mem_test (
    constant bar_table : in natural;          -- Pointer to the BAR sizing and
                                              -- address information set up by
                                              -- the configuration routine
    constant tgt_bar : in natural := 0;       -- BAR to use to access the target
                                              -- memory
    constant start_offset : in natural := 0;  -- Starting offset in the target
                                              -- memory to use
    constant tgt_data_len : in natural := 512 -- Length of data to test
    ) is
    constant TGT_WR_DATA_ADDR : natural := 1*(2**16) ;

    variable tgt_rd_data_addr : natural ;
    variable err_addr : integer;

  begin  -- target_mem_test

    ebfm_display(EBFM_MSG_INFO,
                 "Starting Target Write/Read Test.") ;
    ebfm_display(EBFM_MSG_INFO,
                 "  Target BAR = " &
                 integer'image(tgt_bar) ) ;
    ebfm_display(EBFM_MSG_INFO,
                 "  Length = " & integer'image(tgt_data_len) &
                 ", Start Offset = " & integer'image(start_offset) ) ;

    -- Setup some data to write to the Target
    shmem_fill(TGT_WR_DATA_ADDR,SHMEM_FILL_DWORD_INC,tgt_data_len) ;

    -- Setup an address for the data to read back from the Target
    tgt_rd_data_addr := TGT_WR_DATA_ADDR + (2*tgt_data_len) ;
    -- Clear the target data area
    shmem_fill(tgt_rd_data_addr,SHMEM_FILL_ZERO,tgt_data_len) ;

    --
    -- Now write the data to the target with this BFM call
    --
    ebfm_barwr(bar_table,tgt_bar,start_offset,TGT_WR_DATA_ADDR,tgt_data_len) ;
    
    --
    -- Read the data back from the target in one burst, wait for the read to
    -- be complete
    -- 
    ebfm_barrd_wait(bar_table,tgt_bar,start_offset,tgt_rd_data_addr,tgt_data_len) ;

    -- Check the data
    if (shmem_chk_ok(tgt_rd_data_addr,SHMEM_FILL_DWORD_INC,tgt_data_len)) then
      ebfm_display(EBFM_MSG_INFO,"  Target Write and Read compared okay!") ;
    else
      ebfm_display(EBFM_MSG_ERROR_FATAL,"  Stopping simulation due to miscompare") ;
    end if;
    
  end target_mem_test;

  -- purpose: This procedure polls the DMA engine until it is done
  procedure dma_wait_done (
    constant bar_table : in natural;
    constant setup_bar : in natural := 4;
    constant scr_mem : in natural)
  is
  begin  -- dma_wait_done
    shmem_fill(scr_mem,SHMEM_FILL_ONE,4) ;
    while (shmem_read(scr_mem,4)(31) = '1') loop
      ebfm_barrd_wait(bar_table,setup_bar,12,scr_mem,4) ;
    end loop;    
  end dma_wait_done;

  -- purpose: Use the reference design's DMA engine to move data from the BFM's
  -- shared memory to the reference design's master memory and then back
  procedure dma_mem_test (
    constant bar_table : in natural;  -- Pointer to the BAR sizing and
                                            -- address information set up by
                                            -- the configuration routine
    constant setup_bar : in natural := 4;   -- BAR to be used for setting up
                                            -- the DMA operation and checking
                                            -- the status 
    constant start_offset : in natural := 0;  -- Starting offset in the master
                                              -- memory 
    constant dma_data_len : in natural := 512  -- Length of DMA operations 
    ) is

    constant SCR_MEM : natural := (2**17)-4;
    variable dma_rd_data_addr : natural := SCR_MEM + 4 ;
    variable dma_wr_data_addr : natural ;
    variable err_addr : integer ;


  begin

    ebfm_display(EBFM_MSG_INFO,
                 "Starting DMA Read/Write Test.");
    ebfm_display(EBFM_MSG_INFO,
                 "  Setup BAR = " &
                 integer'image(setup_bar) ) ;
    ebfm_display(EBFM_MSG_INFO,
                 "  Length = " & integer'image(dma_data_len) &
                 ", Start Offset = " & integer'image(start_offset) ) ;
    
    dma_rd_data_addr := dma_rd_data_addr + start_offset ; 
    -- Setup some data for the DMA to read
    shmem_fill(dma_rd_data_addr,SHMEM_FILL_DWORD_INC,dma_data_len) ;

    -- Program the DMA to Read Data from Shared Memory
    ebfm_barwr_imm(bar_table,setup_bar, 0,std_logic_vector(to_unsigned(dma_rd_data_addr,32))) ;
    ebfm_barwr_imm(bar_table,setup_bar, 4,X"00000000");
    ebfm_barwr_imm(bar_table,setup_bar, 8,std_logic_vector(to_unsigned(dma_data_len,32)));
    ebfm_barwr_imm(bar_table,setup_bar,12,X"00040000");
    
    -- Wait Until the DMA is done
    dma_wait_done(bar_table,setup_bar,SCR_MEM) ;

    -- Setup an area for DMA to write back to
    -- Currently DMA Engine Uses smae lower address bits for it's MRAM and PCIE
    -- Addresses. So use the same address we started with
    dma_wr_data_addr := dma_rd_data_addr ; 
    shmem_fill(dma_wr_data_addr,SHMEM_FILL_ZERO,dma_data_len) ;

    -- Program the DMA to Write Data Back to Shared Memory
    ebfm_barwr_imm(bar_table,setup_bar, 0,std_logic_vector(to_unsigned(dma_wr_data_addr,32))) ;
    ebfm_barwr_imm(bar_table,setup_bar, 4,X"00000000");
    ebfm_barwr_imm(bar_table,setup_bar, 8,std_logic_vector(to_unsigned(dma_data_len,32)));
    ebfm_barwr_imm(bar_table,setup_bar,12,X"00040040");

    -- Wait Until the DMA is done
    dma_wait_done(bar_table,setup_bar,SCR_MEM) ;

    -- Check the data
    if (shmem_chk_ok(dma_rd_data_addr,SHMEM_FILL_DWORD_INC,dma_data_len)) then
      ebfm_display(EBFM_MSG_INFO,"  DMA Read and Write compared okay!") ;
    else
      ebfm_display(EBFM_MSG_ERROR_FATAL,"  Stopping simulation due to miscompare") ;
    end if;

  end procedure dma_mem_test ;

  -- purpose: Examine the DUT's BAR setup and pick a reasonable BAR to use
  impure function find_mem_bar (
    constant bar_table : natural ;
    constant allowed_bars    : std_logic_vector(5 downto 0);
    constant min_log2_size   : natural 
    ) return natural is
    variable cur_bar : natural := 0;
    variable bar32 : std_logic_vector(31 downto 0) ;
    variable log2_size : natural;
    variable is_mem : std_logic ;
    variable is_pref : std_logic ;
    variable is_64b : std_logic ;
  begin  -- find_mem_bar
    chk_loop : while (cur_bar < 6) loop
      ebfm_cfg_decode_bar(bar_table,cur_bar,log2_size,is_mem,is_pref,is_64b) ;
      if ( (is_mem = '1') and
           (log2_size >= min_log2_size) and
           (allowed_bars(cur_bar) = '1') ) then
        return cur_bar;
      end if;
      if (is_64b = '1') then
        cur_bar := cur_bar + 2 ;
      else
        cur_bar := cur_bar + 1 ;        
      end if;
    end loop chk_loop;
    return natural'high; -- Invalid BAR if we get this far...
  end find_mem_bar;

  constant interrupt_pin_reg : natural := 60;
  constant msi_capabilities  : natural := 16#50#;

  procedure legacy_interrupt_test (
    constant bus_num   : in natural := 0;
    constant dev_num   : in natural := 0;
    constant fnc_num   : in natural := 0;
    constant bar_table : in natural := 0;
    constant bar_num   : in natural := 0 
    ) is
    variable intx                  : std_logic_vector(7 downto 0);
    variable shared_memory_address : integer; 
    variable msi_control_register  : std_logic_vector(15 downto 0);
    variable msi_enable            : std_logic;
    variable restore_msi_reg       : std_logic;
    variable compl_status          : std_logic_vector(2 downto 0);
    variable int_dcba              : std_logic_vector(3 downto 0);
  begin
     shared_memory_address := 0;
     restore_msi_reg := '0';
     -- Query the endpoint to see if it supports Legacy Interrupts
     ebfm_cfgrd_wait(bus_num, dev_num, fnc_num, interrupt_pin_reg, 4, shared_memory_address, compl_status);
     intx := shmem_read(shared_memory_address + 1, 1);
     ebfm_display(EBFM_MSG_INFO,"Starting Legacy Interrupt Test.");
     case intx is
        when "00000000" => ebfm_display(EBFM_MSG_INFO,"  There is no legacy interrupt support for this endpoint.");
                           ebfm_display(EBFM_MSG_INFO,"  Skipping Legacy Interrupt Test.");
                           return;
        when "00000001" => ebfm_display(EBFM_MSG_INFO,"  Endpoint uses INTA#");
        when "00000010" => ebfm_display(EBFM_MSG_INFO,"  Endpoint uses INTB#");
        when "00000011" => ebfm_display(EBFM_MSG_INFO,"  Endpoint uses INTC#");
        when "00000100" => ebfm_display(EBFM_MSG_INFO,"  Endpoint uses INTD#");
        when others  => ebfm_display(EBFM_MSG_ERROR_FATAL,"  Interrupt Pin: Interrupt Pin Registers has Illegal Value ");
     end case;

     shared_memory_address := shared_memory_address + 4;
     -- Read the MSI capabilities register
     ebfm_display(EBFM_MSG_INFO,"  Checking status of MSI register");
     ebfm_cfgrd_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, shared_memory_address, compl_status);
     msi_control_register := shmem_read(shared_memory_address + 2, 2);
     msi_enable := msi_control_register(0);
     -- If the Message Signaled Interrupt (MSI) is enabled, disable it to run the Legacy Interrupt test
     if (msi_enable = '1') then
        ebfm_display(EBFM_MSG_INFO,"  Disabling MSI enable bit to perform legacy interrupt test");
        ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, msi_control_register(15 downto 1) & '0' & "0000000000000000", compl_status);
        ebfm_cfgrd_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, shared_memory_address, compl_status);
        msi_control_register := shmem_read(shared_memory_address + 2, 2);
        msi_enable := msi_control_register(0);
        restore_msi_reg := '1';
     end if;

     -- Write to the example design interrupt register to generate the interrupt
     ebfm_display(EBFM_MSG_INFO,"  Writing Interrupt Register in Endpoint Example Design to Trigger a Legacy Interrupt");
     ebfm_barwr_imm(bar_table, bar_num, 16#0010#, "00000000000000000000000000000001", 4, 0);
     -- Check to see if the INTx was received
     ebfm_display(EBFM_MSG_INFO,"  Root Complex Checking for Received Interrupt");

     wait until (INTD='1' or INTC='1' or INTB='1' or INTA='1') for 30000 ns;
     -- The time for this delay should probably be adjusted...
     -- The time should be 3000 clock cycles
     if (INTD='1' or INTC='1' or INTB='1' or INTA='1') then
     else
        ebfm_display(EBFM_MSG_INFO,"  Error: The legacy interrupt receive-timer timed out");
     end if;

     int_dcba := INTD & INTC & INTB & INTA;
     case int_dcba is
        when "0000" => ebfm_display(EBFM_MSG_INFO,"  No interrupt detected");
                       ebfm_display(EBFM_MSG_ERROR_FATAL,"  Legacy interrupt test failed");
                       return;
        when "0001" => ebfm_display(EBFM_MSG_INFO,"  Detected INTA#, Legacy interrupt test okay!");
        when "0010" => ebfm_display(EBFM_MSG_INFO,"  Detected INTB#, Legacy interrupt test okay!");
        when "0011" => ebfm_display(EBFM_MSG_INFO,"  Detected INTC#, Legacy interrupt test okay!");
        when "0100" => ebfm_display(EBFM_MSG_INFO,"  Detected INTD#, Legacy interrupt test okay!");
        when others => ebfm_display(EBFM_MSG_ERROR_FATAL,"  Illegal value detected on INTx lines, Legacy interrupt test failed");
     end case;

     -- Re-enable the MSI if it was disabled
     if (restore_msi_reg = '1') then
        ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, msi_control_register(15 downto 1) & '1' & "0000000000000000", compl_status);
     end if;

     -- Clear interrupt from example design interrupt register
     ebfm_barwr_imm(bar_table, bar_num, 16#0010#, "00000000000000000000000000000000", 4, 0);
  end legacy_interrupt_test;

      procedure message_signaled_interrupt_test (
         bus_num                 : in natural;   
         dev_num                 : in natural;   
         fnc_num                 : in natural;   
         bar_table               : in natural;   
         bar_num                 : in natural 
         ) is
         variable msi_address            :  std_logic_vector(31 downto 0) := X"0000_0000";    --  The Root Complex BFM has 2MB of address space
         variable msi_upper_address      :  std_logic_vector(31 downto 0) := X"0000_0000";    
         variable msi_data               :  std_logic_vector(15 downto 0) := X"abc0";    
         variable msi_max                :  integer;   
         variable msi_number             :  integer;   
         variable msi_expected           :  std_logic_vector(15 downto 0);   
         variable msi_received           :  std_logic_vector(15 downto 0);   
         variable msi_traffic_class      :  integer;   
         variable dummy                  :  integer;   
         variable shared_memory_address  :  integer;   
         variable msi_control_register   :  std_logic_vector(15 downto 0);   
         variable msi_64b_capable        :  std_logic;   
         variable multi_message_enable   :  std_logic_vector(2 downto 0);   
         variable multi_message_capable  :  std_logic_vector(2 downto 0);   
         variable msi_enable             :  std_logic;   
         variable compl_status           :  std_logic_vector(2 downto 0);   
         variable Y                      :  integer;
         variable msi_number_bit_vector  :  std_logic_vector(4 downto 0);
         variable loop_ctr               :  integer;
      begin
         -- The MSI test checks the MSI Control register for the requested number of MSI
         -- Then initializes the MSI registers and performs the test
         msi_traffic_class := 0;
         shared_memory_address := 0;    
         ebfm_display(EBFM_MSG_INFO, "Starting Message Signaled Interrupt Test.");    
         -- Read the contents of the MSI Control register
         ebfm_cfgrd_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, shared_memory_address, compl_status);   
         msi_control_register := shmem_read(shared_memory_address + 2, 2);    
         msi_64b_capable      := msi_control_register(7);    
         -- Enable the MSI with Maximum Number of Supported Messages
         multi_message_capable := msi_control_register(3 downto 1);    
         multi_message_enable  := multi_message_capable;    
         msi_enable            := '1';    
         -- Program the MSI Message Control register for testing
         ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, X"00" & msi_64b_capable & multi_message_enable & multi_message_capable & msi_enable & X"0000", compl_status);
         -- Write the rest of the MSI Capabilities Structure: Address and Data Fields
         if (msi_64b_capable = '1') then -- 64-bit Addressing
            ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, (msi_capabilities +  4), 4, msi_address,          compl_status);   
            ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, (msi_capabilities +  8), 4, msi_upper_address,    compl_status);   
            ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, (msi_capabilities + 12), 4, (X"0000" & msi_data), compl_status);   
         else -- 32-bit Addressing
            ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, (msi_capabilities +  4), 4, msi_address,          compl_status);   
            ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, (msi_capabilities +  8), 4, (X"0000" & msi_data), compl_status);   
         end if;
         -- Calculate number of MSIs to test from the Message Control register field
         if (multi_message_enable = "000") then 
            msi_max := 1;
         else
            msi_max := 1;
            loop_ctr := to_integer(unsigned(multi_message_enable(2 downto 0)));
            while (loop_ctr > 0) loop
              msi_max := 2 * msi_max;
              loop_ctr := loop_ctr - 1;
            end loop;
         end if;

         -- Loop to check all possible MSIs in the endpoint
         msi_number := 0;
         while (msi_number < msi_max) loop
            -- Write to the example design to trigger the MSI
            ebfm_display(EBFM_MSG_INFO,"  Writing Interrupt Register in Endpoint Example Design to Trigger MSI " & integer'image(msi_number) ) ;
            ebfm_barwr_imm(bar_table, bar_num, 16#0010#, X"0000" & '0' & std_logic_vector(to_unsigned(msi_traffic_class,3)) & "000" & std_logic_vector(to_unsigned(msi_number,5)) & X"1", 4, 0);   
            
               -- Polling memory for expected MSI data value at the assigned MSI address location
               
               -- Wait to receive the MSI from the endpoint
               -- Set timeout failure if expected MSI is not received
               Y := 0;
               while (TRUE) loop
                  for X in 1 to 50 loop
                     wait until (clk_in'EVENT AND clk_in = '1');
                     Y := Y + 1;
                  end loop;
                  ebfm_display(EBFM_MSG_INFO,"  Polling MSI Address");    
                  msi_received := shmem_read(to_integer(unsigned(msi_address)), 2);
                  ebfm_display(EBFM_MSG_INFO,"    Received data " & himage(msi_received));    
                  -- ebfm_display(EBFM_MSG_INFO,"    Received data " & integer'image(to_integer(unsigned(msi_received(15 downto 0)))));    
                  -- MSI data value written at the MSI address contains the MSI data register value
                  -- and the number of the MSI reported.  This combination changes with the amount
                  -- of MSI enabled in the endpoint.
                  msi_number_bit_vector := std_logic_vector(to_unsigned(msi_number, 5));

                  case multi_message_enable is
                     when "000" => msi_expected := msi_data(15 downto 0);    
                     when "001" => msi_expected := msi_data(15 downto 1) & msi_number_bit_vector(0);    
                     when "010" => msi_expected := msi_data(15 downto 2) & msi_number_bit_vector(1 downto 0);    
                     when "011" => msi_expected := msi_data(15 downto 3) & msi_number_bit_vector(2 downto 0);    
                     when "100" => msi_expected := msi_data(15 downto 4) & msi_number_bit_vector(3 downto 0);    
                     when "101" => msi_expected := msi_data(15 downto 5) & msi_number_bit_vector(4 downto 0);    
                     when others => ebfm_display(EBFM_MSG_ERROR_FATAL, "Illegal multi_message_enable value detected. MSI test fails.");    
                  end case;
                  if (msi_received = msi_expected) then
                     ebfm_display(EBFM_MSG_INFO,"    Received Expected MSI : " & himage(msi_received));
                     --ebfm_display(EBFM_MSG_INFO,"    Received Expected MSI : " & integer'image(to_integer(unsigned(msi_received(15 downto 0)))));
                     shmem_write(shared_memory_address, X"0000000000000000", 2);   
                     exit;
                  end if;
                  if (Y >= 3000) then
                     ebfm_display(EBFM_MSG_ERROR_FATAL, "MSI timeout occured, MSI never received, Test Fails");    
                     exit;
                  end if;
               end loop;
            
            -- Reset the interrupt register in the endpoint example design
            ebfm_barwr_imm(bar_table, bar_num, 16#0010#, "00000000000000000000000000000000", 4, 0);   
            msi_number := msi_number + 1;
         end loop;
         
         -- Return the MSI control register to its original state
         ebfm_display(EBFM_MSG_INFO, "  Returning the MSI Message Control Register to pre-test state.");    
         ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, msi_control_register & "0000000000000000", compl_status);   
         ebfm_display(EBFM_MSG_INFO,"  Message Signaled Interrupt test okay!");    
      end message_signaled_interrupt_test;

end altpcietb_bfm_driver;

architecture behavioral of altpcietb_bfm_driver is

  signal activity_toggle : std_logic := '0';
  
begin  -- behavioral

  main: process

    -- This constant defines where we save the sizes and programmed addresses
    -- of the Endpoint Device Under Test BARs 
    constant bar_table : natural := BAR_TABLE_POINTER;  -- 64 bytes

    -- tgt_bar indicates which bar to use for testing the target memory of the
    -- reference design.
    variable tgt_bar : natural := 0;
    variable dma_bar : natural := 4;
    variable addr_map_4GB_limit : natural := 0;
    
  begin  -- process main

     ebfm_display(EBFM_MSG_INFO,"Starting ebfm_cfg_rp_ep");
     -- Setup the Root Port and Endpoint Configuration Spaces
     ebfm_cfg_rp_ep(
       bar_table => bar_table,                  -- BAR Size/Address info for Endpoint
       ep_bus_num => 1,                         -- Bus Number for Endpoint Under Test
       ep_dev_num => 1,                         -- Device Number for Endpoint Under Test
       rp_max_rd_req_size => 512,               -- Maximum Read Request Size for Root Port
       display_ep_config => 1,                  -- Display EP Config Space after setup
       addr_map_4GB_limit => addr_map_4GB_limit -- Limit the BAR assignments to 4GB address map
       ) ;

     ebfm_display(EBFM_MSG_INFO,"Finished ebfm_cfg_rp_ep");
     activity_toggle <= not activity_toggle ;

     -- Find a memory BAR to use to test the target memory
     -- The reference design implements the target memory on BARs 0,1, 4 or 5
     -- We need one at least 4 KB big
     tgt_bar := find_mem_bar(bar_table,"110011",12) ;
     
     -- Test the reference design's target memory
     if (tgt_bar < 6)  then
       target_mem_test(
         bar_table => bar_table,        -- BAR Size/Address info for Endpoint
         tgt_bar => tgt_bar,            -- BAR to access target memory with
         start_offset => 0,             -- Starting offset from BAR
         tgt_data_len => 4096) ;        -- Length of memory to test
     else
       ebfm_display(EBFM_MSG_WARNING,"Unable to find a 4 KB BAR to test Target Memory, skipping target test.");
     end if;

     activity_toggle <= not activity_toggle ;
     
     -- Find a memory BAR to use to setup the DMA channel
     -- The reference design implements the DMA channel registers on BAR 2 or 3
     -- We need one at least 128 B big
     dma_bar := find_mem_bar(bar_table,"001100",7) ;
     
     -- Test the reference design's DMA channel and master memory
     if (dma_bar < 6) then
       dma_mem_test(
         bar_table => bar_table,        -- BAR Size/Address info for Endpoint
         setup_bar => dma_bar,          -- BAR to access DMA control registers
         start_offset => 0,             -- Starting offset of DMA memory
         dma_data_len => 4096) ;        -- Length of memory to test
     else
       ebfm_display(EBFM_MSG_WARNING,"Unable to find a 128B BAR to test setup DMA channel, skipping DMA test.");
     end if;

      -- The DMA BAR also contains the Example Design registers for triggering interrupts
      -- Use the same BAR as the DMA test
      -- Test both MSI and Legacy Interrupts if available
      if (dma_bar < 6) then
         legacy_interrupt_test(1, 1, 0, bar_table, dma_bar);
         message_signaled_interrupt_test(1, 1, 0, bar_table, dma_bar);
      else
         ebfm_display(EBFM_MSG_WARNING,"Unable to find a 128B BAR to test interrupts, skipping Interrupt tests.");
      end if;

     -- Stop the simulator and indicate successful completion
     ebfm_log_stop_sim(1) ;

     wait;
  end process main;

  -- purpose: this is a watchdog timer, if it sees no activity on the activity
  -- toggle signal for 200 us it ends the simulation
  watchdog: process 
  begin  -- process watchdog
    wait on activity_toggle for 200 us ;
    if (not activity_toggle'event) then
      ebfm_display(EBFM_MSG_ERROR_FATAL,"Simulation stopped due to inactivity!") ;
    end if;
  end process watchdog;

end behavioral;
