LIBRARY ieee;                          
use IEEE.std_logic_1164.all;           
use IEEE.std_logic_arith.all;          
use IEEE.std_logic_unsigned.all;       
                                       
LIBRARY altera_mf;                     
USE altera_mf.altera_mf_components.all;
                                       
--
-- /**
-- *
-- *  This Verilog HDL File is used for simulation and synthesis in the simple DMA design example.  
-- *  This contains the DMA engine configuration registers and buffer memory for target transactions.
-- *
-- */
-------------------------------------------------------------------------------
-- Title         : PCI Express Reference Design Slave Entity
-- Project       : PCI Express MegaCore function
-------------------------------------------------------------------------------
-- File          : altpcierd_slave.v
-- Author        : Altera Corporation
-------------------------------------------------------------------------------
-- Description :
-- This is an entity that handles target transactions from the PCI Express interface
-------------------------------------------------------------------------------
-- Copyright (c) 2005-2006 Altera Corporation. All rights reserved.  Altera products are
-- protected under numerous U.S. and foreign patents, maskwork rights, copyrights and
-- other intellectual property laws.  
--
-- This reference design file, and your use thereof, is subject to and governed by
-- the terms and conditions of the applicable Altera Reference Design License Agreement.
-- By using this reference design file, you indicate your acceptance of such terms and
-- conditions between you and Altera Corporation.  In the event that you do not agree with
-- such terms and conditions, you may not use the reference design file. Please promptly
-- destroy any copies you have made.
--
-- This reference design file being provided on an "as-is" basis and as an accommodation 
-- and therefore all warranties, representations or guarantees of any kind 
-- (whether express, implied or statutory) including, without limitation, warranties of 
-- merchantability, non-infringement, or fitness for a particular purpose, are 
-- specifically disclaimed.  By making this reference design file available, Altera
-- expressly does not recommend, suggest or require that this reference design file be
-- used in combination with any other product not provided by Altera. 
-------------------------------------------------------------------------------
ENTITY altpcierd_slave IS
   GENERIC (
      RAM_SIZE                       :  integer := 8);    
   PORT (
      hclk                    : IN std_logic;   
      hrstn                   : IN std_logic;   
      cfg_busdev              : IN std_logic_vector(12 DOWNTO 0);   
      cfg_maxpload            : IN std_logic_vector(2 DOWNTO 0);   
      cfg_tcvcmap             : IN std_logic_vector(23 DOWNTO 0);   
      m_req                   : OUT std_logic;   
      m_size                  : OUT std_logic_vector(31 DOWNTO 0);   
      m_addr                  : OUT std_logic_vector(63 DOWNTO 0);   
      m_ctrl                  : OUT std_logic_vector(31 DOWNTO 0);   
      m_tgt_adr               : OUT std_logic_vector(12 DOWNTO 0);   
      m_ack                   : IN std_logic;   
      tx_intreq               : OUT std_logic;   
      pm_event                : OUT std_logic;   
      err_cplabort            : OUT std_logic;   
      pm_event_pb             : IN std_logic;   --  PM event push button
      m_pending               : IN std_logic;   --  Indicates that a DMA is pending
      vc0_out                 : OUT std_logic;   
      vc1_out                 : OUT std_logic;   
      rx_busy                 : OUT std_logic;   
      tx_busy                 : OUT std_logic;   
      rx_req                  : IN std_logic;   
      rx_ack                  : OUT std_logic;   
      rx_desc                 : IN std_logic_vector(135 DOWNTO 0);   
      rx_ws                   : OUT std_logic;   
      rx_data                 : IN std_logic_vector(63 DOWNTO 0);   
      rx_be                   : IN std_logic_vector(7 DOWNTO 0);   
      rx_dv                   : IN std_logic;   
      rx_dfr                  : IN std_logic;   
      tx_req                  : OUT std_logic;   
      tx_desc                 : OUT std_logic_vector(127 DOWNTO 0);   
      tx_ack                  : IN std_logic;   
      tx_dfr                  : OUT std_logic;   
      tx_data                 : OUT std_logic_vector(63 DOWNTO 0);   
      tx_dv                   : OUT std_logic;   
      tx_err                  : OUT std_logic;   
      tx_ws                   : IN std_logic;   
      app_msi_num             : OUT std_logic_vector(4 DOWNTO 0);   
      app_msi_tc              : OUT std_logic_vector(2 DOWNTO 0);   
      fetch_tx_data           : IN std_logic;   --  external control to force a data read for tx
      nextcyc_is_last         : OUT std_logic;   --  indicates that next data cycle is last for tx pkt (last dfr cycle)
      last_datacyc            : OUT std_logic;   --  indicates that this data cycle is last for tx pkt (last dv cycle)
      -- streaming interface 
      avalon_tx_data          : OUT std_logic_vector(63 DOWNTO 0);   
      rx_st_eop               : IN std_logic);   
END ENTITY altpcierd_slave;
ARCHITECTURE translated OF altpcierd_slave IS
   COMPONENT altpcierd_dprambe
      GENERIC (
          ADDR_WIDTH                     :  integer := 8;    
          DATA_WIDTH                     :  integer := 32;    
          BYTE_SIZE                      :  integer := 8);    
      PORT (
         wrclk                   : IN  std_logic;
         wraddr                  : IN  std_logic_vector(ADDR_WIDTH - 1 DOWNTO 0);
         wrdata                  : IN  std_logic_vector(DATA_WIDTH - 1 DOWNTO 0);
         wrbe                    : IN  std_logic_vector(DATA_WIDTH / BYTE_SIZE - 1 DOWNTO 0);
         rdclk                   : IN  std_logic;
         rdclken                 : IN  std_logic;
         rdaddr                  : IN  std_logic_vector(ADDR_WIDTH - 1 DOWNTO 0);
         rddata                  : OUT std_logic_vector(DATA_WIDTH - 1 DOWNTO 0));
   END COMPONENT;
   CONSTANT  bk_idl                :  std_logic_vector(1 DOWNTO 0) := "00";    
   CONSTANT  bk_tgtack             :  std_logic_vector(1 DOWNTO 0) := "01";    
   CONSTANT  bk_tgtdata            :  std_logic_vector(1 DOWNTO 0) := "10";    
   SIGNAL bk_sm                    :  std_logic_vector(1 DOWNTO 0);   
   SIGNAL bk_tgtack_r              :  std_logic;   
   SIGNAL bk_tgtack_rr             :  std_logic;   
   SIGNAL bk_tgtack_rrr            :  std_logic;   
   SIGNAL bk_tgtack_rrrr           :  std_logic;   
   CONSTANT  tr_idle               :  std_logic_vector(3 DOWNTO 0) := "0000";    
   CONSTANT  tr_reload0            :  std_logic_vector(3 DOWNTO 0) := "0001";    
   CONSTANT  tr_reqcpl             :  std_logic_vector(3 DOWNTO 0) := "0010";    
   CONSTANT  tr_reqdata            :  std_logic_vector(3 DOWNTO 0) := "0011";    
   CONSTANT  tr_busy               :  std_logic_vector(3 DOWNTO 0) := "0100";    
   CONSTANT  tr_reload1            :  std_logic_vector(3 DOWNTO 0) := "0101";    
   CONSTANT  tr_reload2            :  std_logic_vector(3 DOWNTO 0) := "0110";    
   CONSTANT  tr_idle0              :  std_logic_vector(3 DOWNTO 0) := "0111";    
   CONSTANT  tr_idle1              :  std_logic_vector(3 DOWNTO 0) := "1000";    
   SIGNAL tgtrd_sm                 :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL is_tgt                   :  std_logic;   
   SIGNAL is_wr                    :  std_logic;   
   SIGNAL is_dval                  :  std_logic;   
   SIGNAL r_len                    :  std_logic_vector(9 DOWNTO 0);   
   SIGNAL r_ad32                   :  std_logic_vector(31 DOWNTO 0);   
   SIGNAL rx_wrad                  :  std_logic_vector(11 DOWNTO 0);   
   SIGNAL rx_wr                    :  std_logic;   
   SIGNAL wr_allow                 :  std_logic;   
   SIGNAL datain                   :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL dataout                  :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL tgt_adwr                 :  std_logic_vector(11 DOWNTO 0);   
   SIGNAL tgt_adrd                 :  std_logic_vector(11 DOWNTO 0);   
   SIGNAL wren                     :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL rx_bv                    :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL r_fbe                    :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL r_lbe                    :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL adr_mem                  :  std_logic_vector(31 DOWNTO 0);   
   SIGNAL m_size_r                 :  std_logic_vector(31 DOWNTO 0);   
   SIGNAL m_ctrl_r                 :  std_logic_vector(31 DOWNTO 0);   
   SIGNAL m_addr_r                 :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL m_tgt_adr_r              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL tgt_adr_eq_0             :  std_logic;   
   SIGNAL tgt_adr_eq_1             :  std_logic;   
   SIGNAL tgt_adr_eq_2             :  std_logic;   
   SIGNAL tx_intreq_r              :  std_logic;   
   SIGNAL cpl_abort_r              :  std_logic;   
   SIGNAL pm_event_pb_r            :  std_logic_vector(1 DOWNTO 0);   
   SIGNAL phasecnt                 :  std_logic_vector(9 DOWNTO 0);   
   SIGNAL tgtrdaddr                :  std_logic_vector(11 DOWNTO 0);   
   SIGNAL fetch_cnt                :  std_logic_vector(1 DOWNTO 0);   
   SIGNAL need_to_fetch            :  std_logic;   
   SIGNAL rdclken                  :  std_logic;   
   SIGNAL desc_mem                 :  std_logic_vector(127 DOWNTO 64);   
   SIGNAL fetch_first              :  std_logic;   
   SIGNAL mem_first                :  std_logic;   
   SIGNAL mem_first_del            :  std_logic;   
   SIGNAL tx_dv_r                  :  std_logic;   
   SIGNAL rd_allow                 :  std_logic;   
   SIGNAL rd_out                   :  std_logic;   
   SIGNAL tx_dfr_r                 :  std_logic;   
   SIGNAL adr_lsb                  :  std_logic_vector(1 DOWNTO 0);   
   SIGNAL tx_data_r                :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL tgt_adrd9_r              :  std_logic_vector(8 DOWNTO 0);   
   SIGNAL t_len                    :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL b_lbe                    :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL b_fbe                    :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL t_len_r                  :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL b_lbe_r                  :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL b_fbe_r                  :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL t_len_rr                 :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL bytecount                :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL dpram_sel                :  std_logic;   
   SIGNAL reg_sel                  :  std_logic;   
   SIGNAL m_req_r                  :  std_logic;   
   SIGNAL cpl_abort                :  std_logic;   
   SIGNAL pm_event_mem             :  std_logic;   
   SIGNAL pm_cnt                   :  std_logic_vector(9 DOWNTO 0);   
   SIGNAL cpl_vc                   :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL adb_mask                 :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL bc2adb_n                 :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL next_bytecount_r         :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL next_bytecount           :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL next_dwcount_r           :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL lastoffset               :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL remain                   :  std_logic;   
   SIGNAL adr_msk                  :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL adr_msk_r                :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL rx_be_i                  :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL rx_be_1st                :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL rx_1st_dataphase         :  std_logic;   
   SIGNAL rx_ws_i                  :  std_logic;   
   SIGNAL rx_req_del               :  std_logic;   
   -- support for streaming interface
   SIGNAL desc_phase2              :  std_logic;   
   SIGNAL temp_xhdl26              :  std_logic;   
   SIGNAL temp_xhdl27              :  std_logic;   
   SIGNAL temp_xhdl28              :  std_logic;   
   ----------------------------------------------------
   -- Figure out Byte enable
   ----------------------------------------------------
   SIGNAL temp_xhdl29              :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL temp_xhdl30              :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL temp_xhdl31              :  std_logic;   -- xhdl
   SIGNAL temp_xhdl32              :  std_logic;   
   ---------------------------------------------------------------
   -- target write control for DPRAM interface                                        
   ---------------------------------------------------------------
   SIGNAL temp_xhdl33              :  std_logic_vector(31 DOWNTO 0);   
   SIGNAL temp_xhdl34              :  std_logic;   -- xhdl
   SIGNAL temp_xhdl35              :  std_logic;   -- xhdl
   SIGNAL temp_xhdl36              :  std_logic;   -- xhdl
   SIGNAL temp_xhdl37              :  std_logic;   --  brought rx_ws up one cycle in app module
   SIGNAL temp_xhdl38              :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL port_xhdl40              :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL port_xhdl41              :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL temp_xhdl42              :  std_logic_vector(20 DOWNTO 18);   
   SIGNAL temp_xhdl43              :  std_logic_vector(17 DOWNTO 15);   
   SIGNAL temp_xhdl44              :  std_logic_vector(14 DOWNTO 12);   
   SIGNAL temp_xhdl45              :  std_logic_vector(11 DOWNTO 9);   
   SIGNAL temp_xhdl46              :  std_logic_vector(8 DOWNTO 6);   
   SIGNAL temp_xhdl47              :  std_logic_vector(5 DOWNTO 3);   
   SIGNAL temp_xhdl48              :  std_logic_vector(2 DOWNTO 0);   
   ------------------------------------------------------------------
   -- target completion generation                                   
   ------------------------------------------------------------------
   -- Compute byte count to reach next allowable boundary
   -- 128 byte
   -- 256 byte
   -- 512 byte
   -- 1K byte
   -- 2K byte
   SIGNAL temp_xhdl49              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl50              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl51              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl52              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl53              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl54              :  std_logic;   
   SIGNAL temp_xhdl55              :  std_logic_vector(12 DOWNTO 0);   
   -- flags for external use
   SIGNAL temp_xhdl56              :  std_logic;   -- xhdl
   SIGNAL temp_xhdl57              :  std_logic;   -- xhdl
   SIGNAL temp_xhdl58              :  std_logic;   
   -- read clock enable is used to stall the datapath pipeline
   --  assign rdclken = (need_to_fetch | (~tx_ws & tx_dv_r) | fetch_tx_data) ? 1'b1 : 1'b0 ;  
   SIGNAL temp_xhdl59              :  std_logic;   
   -- generate
   SIGNAL temp_xhdl60              :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL temp_xhdl61              :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL temp_xhdl62              :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL temp_xhdl63              :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL temp_xhdl64              :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL temp_xhdl65              :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL temp_xhdl66              :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL temp_xhdl67              :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL temp_xhdl68              :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL temp_xhdl69              :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL temp_xhdl70              :  std_logic;   
   SIGNAL temp_xhdl71              :  std_logic_vector(1 DOWNTO 0);   
   SIGNAL temp_xhdl72              :  std_logic_vector(1 DOWNTO 0);   
   SIGNAL temp_xhdl73              :  std_logic_vector(1 DOWNTO 0);   
   SIGNAL temp_xhdl74              :  std_logic_vector(1 DOWNTO 0);   
   SIGNAL m_req_xhdl1              :  std_logic;   
   SIGNAL m_size_xhdl2             :  std_logic_vector(31 DOWNTO 0);   
   SIGNAL m_addr_xhdl3             :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL m_ctrl_xhdl4             :  std_logic_vector(31 DOWNTO 0);   
   SIGNAL m_tgt_adr_xhdl5          :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL tx_intreq_xhdl6          :  std_logic;   
   SIGNAL pm_event_xhdl7           :  std_logic;   
   SIGNAL err_cplabort_xhdl8       :  std_logic;   
   SIGNAL vc0_out_xhdl9            :  std_logic;   
   SIGNAL vc1_out_xhdl10           :  std_logic;   
   SIGNAL rx_busy_xhdl11           :  std_logic;   
   SIGNAL tx_busy_xhdl12           :  std_logic;   
   SIGNAL rx_ack_xhdl13            :  std_logic;   
   SIGNAL rx_ws_xhdl14             :  std_logic;   
   SIGNAL tx_req_xhdl15            :  std_logic;   
   SIGNAL tx_desc_xhdl16           :  std_logic_vector(127 DOWNTO 0);   
   SIGNAL tx_dfr_xhdl17            :  std_logic;   
   SIGNAL tx_data_xhdl18           :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL tx_dv_xhdl19             :  std_logic;   
   SIGNAL tx_err_xhdl20            :  std_logic;   
   SIGNAL app_msi_num_xhdl21       :  std_logic_vector(4 DOWNTO 0);   
   SIGNAL app_msi_tc_xhdl22        :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL nextcyc_is_last_xhdl23   :  std_logic;   
   SIGNAL last_datacyc_xhdl24      :  std_logic;   
   SIGNAL avalon_tx_data_xhdl25    :  std_logic_vector(63 DOWNTO 0);   
BEGIN
   m_req <= m_req_xhdl1;
   m_size <= m_size_xhdl2;
   m_addr <= m_addr_xhdl3;
   m_ctrl <= m_ctrl_xhdl4;
   m_tgt_adr <= m_tgt_adr_xhdl5;
   tx_intreq <= tx_intreq_xhdl6;
   pm_event <= pm_event_xhdl7;
   err_cplabort <= err_cplabort_xhdl8;
   vc0_out <= vc0_out_xhdl9;
   vc1_out <= vc1_out_xhdl10;
   rx_busy <= rx_busy_xhdl11;
   tx_busy <= tx_busy_xhdl12;
   rx_ack <= rx_ack_xhdl13;
   rx_ws <= rx_ws_xhdl14;
   tx_req <= tx_req_xhdl15;
   tx_desc <= tx_desc_xhdl16;
   tx_dfr <= tx_dfr_xhdl17;
   tx_data <= tx_data_xhdl18;
   tx_dv <= tx_dv_xhdl19;
   tx_err <= tx_err_xhdl20;
   app_msi_num <= app_msi_num_xhdl21;
   app_msi_tc <= app_msi_tc_xhdl22;
   nextcyc_is_last <= nextcyc_is_last_xhdl23;
   last_datacyc <= last_datacyc_xhdl24;
   avalon_tx_data <= avalon_tx_data_xhdl25;
   avalon_tx_data_xhdl25 <= tx_data_xhdl18 ;
   temp_xhdl26 <= '1' WHEN ((rx_desc(126)) = '1') ELSE '0';
   is_wr <= temp_xhdl26 ;
   temp_xhdl27 <= '1' WHEN (rx_desc(124 DOWNTO 122) = "000") ELSE '0';
   is_tgt <= temp_xhdl27 ;
   temp_xhdl28 <= '1' WHEN (rx_desc(127 DOWNTO 126) = "01") ELSE '0';
   is_dval <= temp_xhdl28 ;
   rx_ws_xhdl14 <= rx_ws_i ;
   temp_xhdl29 <= rx_be_1st WHEN (rx_1st_dataphase) = '1' ELSE "11111111";
   temp_xhdl30 <= "00000000" WHEN (NOT rx_wr) = '1' ELSE temp_xhdl29;
   rx_be_i <= temp_xhdl30 ;
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         rx_1st_dataphase <= '0';    
         rx_be_1st <= "00000000";    
         rx_req_del <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         rx_req_del <= rx_req;    
         IF (wr_allow = '1') THEN
            rx_1st_dataphase <= '0';    
         ELSE
            IF (rx_req = '1') THEN
               rx_1st_dataphase <= '1';    
            END IF;
         END IF;
         IF (rx_req = '1') THEN
            IF (((rx_desc(125) = '0') AND (rx_desc(34) = '1')) OR ((rx_desc(125) = '1') AND (rx_desc(2) = '1'))) THEN
               -- 3DW
               -- 4DW
               
               rx_be_1st <= "11110000";    
            ELSE
               IF (rx_desc(105 DOWNTO 96) = "0000000001") THEN
                  -- 1DW only
                  
                  rx_be_1st <= "00001111";    
               ELSE
                  rx_be_1st <= "11111111";    
               END IF;
            END IF;
         END IF;
      END IF;
   END PROCESS;
   PROCESS (hclk)
   BEGIN
      IF (hclk'EVENT AND hclk = '1') THEN
         -- 
         
         bk_tgtack_rr <= bk_tgtack_r;    
         bk_tgtack_rrr <= bk_tgtack_rr;    
         bk_tgtack_rrrr <= bk_tgtack_rrr;    
      END IF;
   END PROCESS;
   temp_xhdl31 <= '1' WHEN (bk_sm = bk_tgtack) ELSE '0';
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         bk_sm <= bk_idl;    
         rx_ack_xhdl13 <= '0';    
         rx_ws_i <= '0';    
         dpram_sel <= '0';    
         reg_sel <= '0';    
         bk_tgtack_r <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         CASE bk_sm IS
            WHEN bk_idl =>
                     IF (rx_req = '1') THEN
                        IF (is_tgt = '1' AND (is_wr = '1' OR tgtrd_sm = tr_idle)) THEN
                           bk_sm <= bk_tgtack;    
                           rx_ack_xhdl13 <= '1';    
                           rx_ws_i <= '1';    
                           IF ((rx_desc(133 DOWNTO 132) /= "00") OR (rx_desc(129 DOWNTO 128) /= "00")) THEN
                              dpram_sel <= '1';    
                              reg_sel <= '0';    
                           ELSE
                              IF (rx_desc(131 DOWNTO 130) /= "00") THEN
                                 dpram_sel <= '0';    
                                 reg_sel <= '1';    
                              ELSE
                                 dpram_sel <= '0';    
                                 reg_sel <= '0';    
                              END IF;
                           END IF;
                        END IF;
                     END IF;
            WHEN bk_tgtack =>
                     rx_ack_xhdl13 <= '0';    
                     rx_ws_i <= '0';    
                     IF (is_dval = '1') THEN
                        bk_sm <= bk_tgtdata;    
                     ELSE
                        bk_sm <= bk_idl;    
                     END IF;
            WHEN bk_tgtdata =>
                     -- if (rx_dfr == 1'b0 & rx_dv == 1'b1)
                     
                     IF (rx_st_eop = '1') THEN
                        rx_ws_i <= '0';    
                        bk_sm <= bk_idl;    
                     END IF;
            WHEN OTHERS =>
                     NULL;
            
         END CASE;
         bk_tgtack_r <= temp_xhdl31;    
      END IF;
   END PROCESS;
   temp_xhdl32 <= '0' WHEN (bk_sm = bk_idl) ELSE '1';
   rx_busy_xhdl11 <= temp_xhdl32 ;
   temp_xhdl33 <= rx_desc(31 DOWNTO 0) WHEN ((rx_desc(125)) = '1') ELSE rx_desc(63 DOWNTO 32);
   r_ad32 <= temp_xhdl33 ;
   r_fbe <= rx_desc(67 DOWNTO 64) ;
   r_lbe <= rx_desc(71 DOWNTO 68) ;
   r_len <= rx_desc(105 DOWNTO 96) ;
   temp_xhdl34 <= '1' WHEN (r_ad32(6 DOWNTO 3) = "0000") ELSE '0';
   temp_xhdl35 <= '1' WHEN (r_ad32(6 DOWNTO 3) = "0001") ELSE '0';
   temp_xhdl36 <= '1' WHEN (r_ad32(6 DOWNTO 3) = "0010") ELSE '0';
   PROCESS (hclk)
   BEGIN
      IF (hclk'EVENT AND hclk = '1') THEN
         -- 
         
         IF (bk_sm = bk_idl) THEN
            tgt_adr_eq_0 <= temp_xhdl34;    
            tgt_adr_eq_1 <= temp_xhdl35;    
            tgt_adr_eq_2 <= temp_xhdl36;    
         END IF;
      END IF;
   END PROCESS;
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         rx_wrad <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         rx_wr <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF (bk_sm = bk_idl) THEN
            rx_wr <= is_wr;    
         END IF;
         IF (bk_sm = bk_idl) THEN
            rx_wrad <= r_ad32(14 DOWNTO 3);    
         ELSE
            IF (wr_allow = '1') THEN
               rx_wrad <= rx_wrad + "000000000001";    
            END IF;
         END IF;
      END IF;
   END PROCESS;
   tgt_adwr <= rx_wrad ;
   temp_xhdl37 <= '1' WHEN ((rx_wr = '1' AND bk_sm = bk_tgtdata) AND rx_dv = '1') ELSE '0';
   wr_allow <= temp_xhdl37 ;
   --   assign wr_allow = (rx_wr == 1'b1 & !bk_sm == bk_idl & rx_dv == 1'b1) ? 1'b1 : 1'b0 ;
   datain <= rx_data ;
   rx_bv <= rx_be_i ;
   temp_xhdl38 <= rx_bv WHEN (wr_allow = '1' AND dpram_sel = '1') ELSE "00000000";
   wren <= temp_xhdl38 ;
   port_xhdl40 <= tgt_adwr(RAM_SIZE - 1 DOWNTO 0);
   port_xhdl41 <= tgt_adrd(RAM_SIZE - 1 DOWNTO 0);
   dprambe : altpcierd_dprambe 
      GENERIC MAP (
         ADDR_WIDTH => RAM_SIZE,
         DATA_WIDTH => 64,
         BYTE_SIZE => 8)
      PORT MAP (
         wrclk => hclk,
         wraddr => port_xhdl40,
         wrdata => datain,
         wrbe => wren,
         rdclk => hclk,
         rdclken => rdclken,
         rdaddr => port_xhdl41,
         rddata => dataout);   
   
   xhdl_23 : PROCESS (hrstn, hclk)
      VARIABLE i                      :  integer;   
      VARIABLE j                      :  integer;   
      VARIABLE k                      :  integer;   
   BEGIN
      IF (hrstn = '0') THEN
         m_req_r <= '0';    
         m_addr_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         m_size_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         m_ctrl_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         m_tgt_adr_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF ((m_req_r = '0' AND reg_sel = '1') AND wr_allow = '1') THEN
            IF (tgt_adr_eq_0 = '1') THEN
               IF (rx_be_1st(0) = '1') THEN
                  m_addr_r(31 DOWNTO 0) <= datain(31 DOWNTO 0);    
               END IF;
               IF (rx_be_1st(4) = '1') THEN
                  m_addr_r(63 DOWNTO 32) <= datain(63 DOWNTO 32);    
               END IF;
            ELSE
               IF (tgt_adr_eq_1 = '1') THEN
                  IF (rx_be_1st(0) = '1') THEN
                     m_size_r <= datain(31 DOWNTO 0);    
                  END IF;
                  IF (rx_be_1st(4) = '1') THEN
                     m_ctrl_r <= datain(63 DOWNTO 32);    
                  END IF;
               ELSE
                  IF (tgt_adr_eq_2 = '1') THEN
                     IF (rx_be_1st(4) = '1') THEN
                        m_tgt_adr_r <= datain(47 DOWNTO 35);    
                     END IF;
                  END IF;
               END IF;
            END IF;
         END IF;
         IF ((((m_req_r = '0' AND reg_sel = '1') AND wr_allow = '1') AND tgt_adr_eq_1 = '1') AND rx_be_i(7 DOWNTO 4) /= "0000") THEN
            m_req_r <= '1';    
         ELSE
            IF (m_ack = '1') THEN
               m_req_r <= '0';    
            END IF;
         END IF;
      END IF;
   END PROCESS xhdl_23;
   m_req_xhdl1 <= m_req_r ;
   m_size_xhdl2 <= m_size_r ;
   m_addr_xhdl3 <= m_addr_r ;
   m_ctrl_xhdl4 <= m_ctrl_r ;
   m_tgt_adr_xhdl5 <= m_tgt_adr_r ;
   -------------------------------------------------------------------------
   -- Interrupt, PME and Abort Register
   
   PROCESS (hclk)
   BEGIN
      IF (hclk'EVENT AND hclk = '1') THEN
         --  
         
         IF (((reg_sel = '1' AND wr_allow = '1') AND tgt_adwr(3 DOWNTO 0) = "0010") AND (rx_be_i(0)) = '1') THEN
            app_msi_num_xhdl21 <= datain(8 DOWNTO 4);    
            app_msi_tc_xhdl22 <= datain(14 DOWNTO 12);    
         END IF;
      END IF;
   END PROCESS;
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         tx_intreq_r <= '0';    
         pm_event_pb_r <= "00";    
         pm_event_mem <= '0';    
         pm_cnt <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         pm_event_xhdl7 <= '0';    
         cpl_abort_r <= '0';    
         err_cplabort_xhdl8 <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF (((reg_sel = '1' AND wr_allow = '1') AND tgt_adwr(3 DOWNTO 0) = "0010") AND (rx_be_i(0)) = '1') THEN
            tx_intreq_r <= datain(0);    
         END IF;
         -- Detect push-button rising edge
         
         pm_event_pb_r <= pm_event_pb_r(0) & pm_event_pb;    
         IF (((reg_sel = '1' AND wr_allow = '1') AND tgt_adwr(3 DOWNTO 0) = "0011") AND (rx_be_i(0)) = '1') THEN
            pm_event_mem <= datain(0);    
         ELSE
            IF (pm_cnt = "0000000000") THEN
               pm_event_mem <= '0';    
            END IF;
         END IF;
         IF ((pm_event_mem = '1' OR pm_event_pb_r = "01") AND pm_cnt = "0000000000") THEN
            pm_event_xhdl7 <= '1';    
         ELSE
            pm_event_xhdl7 <= '0';    
         END IF;
         IF (((reg_sel = '1' AND wr_allow = '1') AND tgt_adwr(3 DOWNTO 0) = "0011") AND (rx_be_i(0)) = '1') THEN
            pm_cnt <= "1111111111";    
         ELSE
            IF (pm_cnt /= "0000000000") THEN
               pm_cnt <= pm_cnt - "0000000001";    
            END IF;
         END IF;
         IF (((reg_sel = '1' AND wr_allow = '1') AND tgt_adwr(3 DOWNTO 0) = "0100") AND (rx_be_i(0)) = '1') THEN
            cpl_abort_r <= datain(0);    
         ELSE
            IF (tgtrd_sm = tr_reqcpl AND tx_ack = '1') THEN
               cpl_abort_r <= '0';    
            END IF;
         END IF;
         IF ((cpl_abort = '1' AND tgtrd_sm = tr_reqcpl) AND tx_ack = '1') THEN
            err_cplabort_xhdl8 <= '1';    
         ELSE
            err_cplabort_xhdl8 <= '0';    
         END IF;
      END IF;
   END PROCESS;
   tx_intreq_xhdl6 <= tx_intreq_r ;
   cpl_abort <= cpl_abort_r ;
   temp_xhdl42 <= cfg_tcvcmap(20 DOWNTO 18) WHEN (rx_desc(118 DOWNTO 116) = "110") ELSE cfg_tcvcmap(23 DOWNTO 21);
   temp_xhdl43 <= cfg_tcvcmap(17 DOWNTO 15) WHEN (rx_desc(118 DOWNTO 116) = "101") ELSE temp_xhdl42;
   temp_xhdl44 <= cfg_tcvcmap(14 DOWNTO 12) WHEN (rx_desc(118 DOWNTO 116) = "100") ELSE temp_xhdl43;
   temp_xhdl45 <= cfg_tcvcmap(11 DOWNTO 9) WHEN (rx_desc(118 DOWNTO 116) = "011") ELSE temp_xhdl44;
   temp_xhdl46 <= cfg_tcvcmap(8 DOWNTO 6) WHEN (rx_desc(118 DOWNTO 116) = "010") ELSE temp_xhdl45;
   temp_xhdl47 <= cfg_tcvcmap(5 DOWNTO 3) WHEN (rx_desc(118 DOWNTO 116) = "001") ELSE temp_xhdl46;
   temp_xhdl48 <= cfg_tcvcmap(2 DOWNTO 0) WHEN (rx_desc(118 DOWNTO 116) = "000") ELSE temp_xhdl47;
   cpl_vc <= temp_xhdl48 ;
   temp_xhdl49 <= "0011111111111" WHEN (cfg_maxpload = "100") ELSE "0111111111111";
   temp_xhdl50 <= "0001111111111" WHEN (cfg_maxpload = "011") ELSE temp_xhdl49;
   temp_xhdl51 <= "0000111111111" WHEN (cfg_maxpload = "010") ELSE temp_xhdl50;
   temp_xhdl52 <= "0000011111111" WHEN (cfg_maxpload = "001") ELSE temp_xhdl51;
   temp_xhdl53 <= "0000001111111" WHEN (cfg_maxpload = "000") ELSE temp_xhdl52;
   adb_mask <= temp_xhdl53 ;
   adr_msk <= adb_mask AND NOT adr_mem(12 DOWNTO 0) ;
   bc2adb_n <= adr_msk_r + "0000000000001" ;
   temp_xhdl54 <= '1' WHEN (lastoffset(1 DOWNTO 0) /= "00") ELSE '0';
   remain <= temp_xhdl54 ;
   temp_xhdl55 <= bc2adb_n WHEN (bytecount > bc2adb_n) ELSE bytecount;
   next_bytecount <= temp_xhdl55 ;
   lastoffset <= next_bytecount_r + adr_mem(1 DOWNTO 0) ;
   PROCESS (hclk)
   BEGIN
      IF (hclk'EVENT AND hclk = '1') THEN
         -- 
         
         t_len_rr <= t_len_r & "00" - b_fbe_r - b_lbe_r;    --  xhdl
      END IF;
   END PROCESS;
   temp_xhdl56 <= '1' WHEN (((rd_allow = '1') AND (phasecnt = "0000000010")) OR ((rd_allow = '0') AND (phasecnt = "0000000001"))) ELSE '0';
   temp_xhdl57 <= '1' WHEN (((rd_allow = '1') AND (phasecnt = "0000000001")) OR (phasecnt = "0000000000")) ELSE '0';
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         tgtrd_sm <= tr_idle;    
         tx_req_xhdl15 <= '0';    
         tx_dfr_r <= '0';    
         tx_dv_r <= '0';    
         desc_mem <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         adr_mem <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         tgtrdaddr <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         phasecnt <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         tgt_adrd9_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         tx_data_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         need_to_fetch <= '0';    
         fetch_cnt <= '0' & '0';    
         fetch_first <= '0';    
         mem_first <= '0';    
         mem_first_del <= '0';    
         bytecount <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         vc0_out_xhdl9 <= '0';    
         vc1_out_xhdl10 <= '0';    
         next_bytecount_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         next_dwcount_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         t_len_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         b_fbe_r <= "000";    
         b_lbe_r <= "000";    
         adr_msk_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         nextcyc_is_last_xhdl23 <= '0';    
         last_datacyc_xhdl24 <= '0';    
         desc_phase2 <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         mem_first_del <= mem_first;    
         CASE tgtrd_sm IS
            WHEN tr_idle =>
                     IF (((bk_sm = bk_idl AND rx_req = '1') AND is_tgt = '1') AND is_wr = '0') THEN
                        tgtrd_sm <= tr_idle0;    
                     END IF;
            WHEN tr_idle0 =>
                     tgtrd_sm <= tr_idle1;    
            WHEN tr_idle1 =>
                     tgtrd_sm <= tr_busy;    
            WHEN tr_reload0 =>
                     tgtrd_sm <= tr_reload1;    
            WHEN tr_reload1 =>
                     tgtrd_sm <= tr_reload2;    
            WHEN tr_reload2 =>
                     tgtrd_sm <= tr_reqcpl;    
            WHEN tr_reqcpl =>
                     IF (tx_ack = '1') THEN
                        IF (cpl_abort = '1' OR (rd_out = '1' AND phasecnt = "0000000000")) THEN
                           tgtrd_sm <= tr_busy;    
                        ELSE
                           tgtrd_sm <= tr_reqdata;    
                        END IF;
                     END IF;
            WHEN tr_reqdata =>
                     IF ((tx_ws = '0') AND (phasecnt = "0000000000")) THEN
                        tgtrd_sm <= tr_busy;    
                     END IF;
            WHEN tr_busy =>
                     IF (bytecount = "0000000000000") THEN
                        tgtrd_sm <= tr_idle;    
                     ELSE
                        tgtrd_sm <= tr_reload0;    
                     END IF;
            WHEN OTHERS =>
                     NULL;
            
         END CASE;
         t_len_r <= t_len;    
         b_fbe_r <= b_fbe;    
         b_lbe_r <= b_lbe;    
         adr_msk_r <= adr_msk;    
         IF (tgtrd_sm = tr_idle) THEN
            desc_mem <= rx_desc(127 DOWNTO 64);    
            adr_mem(31 DOWNTO 2) <= r_ad32(31 DOWNTO 2);    
            adr_mem(1 DOWNTO 0) <= adr_lsb;    
            -- Memorize read request parameter
            
            IF (cpl_vc = "000") THEN
               vc0_out_xhdl9 <= '1';    
            ELSE
               vc0_out_xhdl9 <= '0';    
            END IF;
            IF (cpl_vc = "001") THEN
               vc1_out_xhdl10 <= '1';    
            ELSE
               vc1_out_xhdl10 <= '0';    
            END IF;
         ELSE
            IF (tgtrd_sm = tr_idle1) THEN
               bytecount <= t_len_rr;    
            ELSE
               IF (((tgtrd_sm = tr_reqdata OR (tgtrd_sm = tr_reqcpl AND tx_ack = '1')) AND (tx_ws = '0')) AND (phasecnt = "0000000000")) THEN
                  -- Update address & bytecount
                  
                  adr_mem <= adr_mem + next_bytecount_r;    
                  bytecount <= bytecount - next_bytecount_r;    
               END IF;
            END IF;
         END IF;
         -- Compute next bytecount & dword count
         
         IF (bytecount > bc2adb_n) THEN
            next_bytecount_r <= bc2adb_n;    
         ELSE
            next_bytecount_r <= bytecount;    
         END IF;
         IF (remain = '1') THEN
            next_dwcount_r <= lastoffset(12 DOWNTO 2) + "00000000001";    
         ELSE
            next_dwcount_r <= lastoffset(12 DOWNTO 2);    
         END IF;
         IF (bk_tgtack_rrrr = '1') THEN
            tgtrdaddr <= adr_mem(14 DOWNTO 3);    
            need_to_fetch <= NOT reg_sel;    --  prefetch only when not targeting register
            fetch_cnt <= "00";    
         ELSE
            IF (need_to_fetch = '1' AND fetch_cnt /= "10") THEN
               tgtrdaddr <= tgtrdaddr + "000000000001";    
               fetch_cnt <= fetch_cnt + "01";    
            ELSE
               IF (need_to_fetch = '1') THEN
                  -- pipe is filled
                  
                  tgtrdaddr <= tgtrdaddr + "000000000001";    
                  need_to_fetch <= '0';    
               ELSE
                  IF (((NOT tx_ws AND tx_dv_r) OR (fetch_tx_data AND NOT reg_sel)) = '1') THEN
                     -- increment address when data is consumed
                     
                     tgtrdaddr <= tgtrdaddr + "000000000001";    
                  END IF;
               END IF;
            END IF;
         END IF;
         IF (tgtrd_sm = tr_reload2) THEN
            tx_req_xhdl15 <= '1';    
         ELSE
            IF (tx_ack = '1') THEN
               tx_req_xhdl15 <= '0';    
            END IF;
         END IF;
         -------------------------------
         -- streaming interface support
         -------------------------------
         --		 
         -- 		 if (tx_ack) begin
         -- 		     desc_phase2 <= 1'b1;
         -- 	     end
         -- 		 else if (tx_avalon_ready) begin
         -- 		     desc_phase2 <= 1'b0;
         -- 		 end
         -- 
         -------------------------------
         
         IF (tgtrd_sm = tr_reload2) THEN
            fetch_first <= '1';    
         ELSE
            fetch_first <= '0';    
         END IF;
         mem_first <= fetch_first;    
         nextcyc_is_last_xhdl23 <= temp_xhdl56;    
         last_datacyc_xhdl24 <= temp_xhdl57;    
         IF (tgtrd_sm = tr_reload2) THEN
            -- phasecnt includes prefetch from memory
            
            IF ((next_dwcount_r(0)) = '1' OR (adr_mem(2)) = '1') THEN
               phasecnt <= next_dwcount_r(10 DOWNTO 1) + "0000000001";    
            ELSE
               phasecnt <= next_dwcount_r(10 DOWNTO 1);    
            END IF;
         ELSE
            IF (rd_allow = '1') THEN
               phasecnt <= phasecnt - "0000000001";    
            END IF;
         END IF;
         IF (tgtrd_sm = tr_reqcpl) THEN
            -- tx_dfr is deasserted before the last data phase
            
            IF (cpl_abort = '1') THEN
               tx_dfr_r <= '0';    
            ELSE
               IF ((phasecnt = "0000000001" AND (mem_first = '1' OR rd_out = '1')) OR phasecnt = "0000000000") THEN
                  tx_dfr_r <= '0';    
               ELSE
                  tx_dfr_r <= '1';    
               END IF;
            END IF;
         ELSE
            IF (rd_out = '1' AND (phasecnt = "0000000001" OR phasecnt = "0000000000")) THEN
               tx_dfr_r <= '0';    
            END IF;
         END IF;
         IF (tgtrd_sm = tr_reqcpl) THEN
            -- direct path
            
            IF (((tx_ws = '0') OR (fetch_tx_data = '1')) AND (phasecnt = "0000000000")) THEN
               -- xhdl
               
               tx_dv_r <= '0';    
            ELSE
               IF (tx_dfr_r = '1') THEN
                  tx_dv_r <= '1';    
               END IF;
            END IF;
         ELSE
            IF (tgtrd_sm = tr_reqdata) THEN
               IF (((tx_ws = '0') OR (fetch_tx_data = '1')) AND (phasecnt = "0000000000")) THEN
                  -- xhdl
                  
                  tx_dv_r <= '0';    
               ELSE
                  --   tx_dv_r <= 1'b1 ; 
                  
                  tx_dv_r <= tx_dv_r;    
               END IF;
            ELSE
               IF (tgtrd_sm = tr_idle) THEN
                  tx_dv_r <= '0';    
               END IF;
            END IF;
         END IF;
         IF (tgtrd_sm = tr_idle) THEN
            tx_data_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         ELSE
            IF (rd_allow = '1') THEN
               IF (reg_sel = '0') THEN
                  IF (rdclken = '1') THEN
                     tx_data_r <= dataout;    
                  END IF;
                  --
                  -- 			   else if (desc_phase2) begin
                  -- 			      tx_data_r <= tx_desc[63:0];          //  descriptor should be ready ahead of time
                  -- 			   end
                  -- 			   else if ((tgtrd_sm == tr_reload2) begin // this is assertion of tx_req
                  -- 			      tx_data_r <= tx_desc[127:64];        //  descriptor should be ready ahead of time
                  -- 			   end
                  -- 			 
                  
                  
               ELSE
                  CASE tgt_adrd9_r(3 DOWNTO 0) IS
                     WHEN "0000" =>
                              tx_data_r <= m_addr_r;    
                     WHEN "0001" =>
                              tx_data_r <= m_pending & m_ctrl_r(30 DOWNTO 0) & m_size_r;    
                     WHEN "0010" =>
                              tx_data_r <= "0000000000000000" & m_tgt_adr_r & "000" & "0000000000000000000000000000000" & tx_intreq_r;    
                     WHEN "0011" =>
                              tx_data_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
                     WHEN "0100" =>
                              tx_data_r <= "00000000000000000000000000000000" & "0000000000000000000000000000000" & cpl_abort_r;    
                     WHEN "0101" =>
                              tx_data_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
                     WHEN OTHERS  =>
                              tx_data_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
                     
                  END CASE;
               END IF;
            END IF;
         END IF;
         tgt_adrd9_r <= tgt_adrd(8 DOWNTO 0);    
      END IF;
   END PROCESS;
   tx_dv_xhdl19 <= tx_dv_r ;
   temp_xhdl58 <= '0' WHEN (tgtrd_sm = tr_idle) ELSE '1';
   tx_busy_xhdl12 <= temp_xhdl58 ;
   -- for DPRAM uninitialized value
   
   xhdl_44 : PROCESS (tx_data_r)
      VARIABLE i                      :  integer;   
   BEGIN
      --  integer i;   //xhdl
      
      FOR i IN 0 TO 63 LOOP
         IF ((tx_data_r(i)) = '1') THEN
            tx_data_xhdl18(i) <= '1';    
         ELSE
            tx_data_xhdl18(i) <= '0';    
         END IF;
      END LOOP;
   END PROCESS xhdl_44;
   -- data phase acknowledge
   rd_out <= tx_dv_r AND NOT tx_ws ;
   -- read operation on DPRAM (including fetch)
   -- assign rd_allow = mem_first | rd_out ;
   rd_allow <= mem_first OR rd_out OR fetch_tx_data ;
   temp_xhdl59 <= '1' WHEN (need_to_fetch OR (NOT tx_ws AND tx_dv_r) OR fetch_tx_data) = '1' ELSE '0';
   rdclken <= temp_xhdl59 ;
   temp_xhdl60 <= "01001010" WHEN (cpl_abort = '0') ELSE "00001010";
   tx_desc_xhdl16(127 DOWNTO 120) <= temp_xhdl60 ;
   tx_desc_xhdl16(119 DOWNTO 106) <= desc_mem(119 DOWNTO 112) & '0' & desc_mem(110 DOWNTO 106) ;
   tx_desc_xhdl16(105 DOWNTO 96) <= next_dwcount_r(9 DOWNTO 0) ;
   -- assign tx_desc[95:80]   = {cfg_busdev, 3'b000}; 
   tx_desc_xhdl16(95 DOWNTO 83) <= cfg_busdev ;
   tx_desc_xhdl16(82 DOWNTO 80) <= "000" ;
   temp_xhdl61 <= "000" WHEN (cpl_abort = '0') ELSE "100";
   tx_desc_xhdl16(79 DOWNTO 77) <= temp_xhdl61 ;
   tx_desc_xhdl16(76 DOWNTO 64) <= '0' & bytecount(11 DOWNTO 0) ;
   tx_desc_xhdl16(63 DOWNTO 32) <= desc_mem(95 DOWNTO 72) & '0' & adr_mem(6 DOWNTO 0) ;
   tx_desc_xhdl16(31 DOWNTO 0) <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' ;
   -- Dual-Port RAM read address
   tgt_adrd <= tgtrdaddr ;
   tx_dfr_xhdl17 <= tx_dfr_r ;
   tx_err_xhdl20 <= '0' ;
   temp_xhdl62 <= "011" WHEN (r_fbe = "1000" OR r_fbe = "0100" OR r_fbe = "0010" OR r_fbe = "0001") ELSE "010";
   temp_xhdl63 <= "001" WHEN (r_fbe = "1110" OR r_fbe = "1101" OR r_fbe = "1011" OR r_fbe = "0111") ELSE temp_xhdl62;
   temp_xhdl64 <= "011" WHEN (r_fbe = "0000") ELSE temp_xhdl63;
   temp_xhdl65 <= "100" WHEN ((r_fbe = "0000") AND (r_len /= "0000000001")) ELSE temp_xhdl64;
   temp_xhdl66 <= "000" WHEN (r_fbe = "1111") ELSE temp_xhdl65;
   b_fbe <= temp_xhdl66 ;
   temp_xhdl67 <= "011" WHEN (r_lbe = "1000" OR r_lbe = "0100" OR r_lbe = "0010" OR r_lbe = "0001") ELSE "010";
   temp_xhdl68 <= "001" WHEN (r_lbe = "1110" OR r_lbe = "1101" OR r_lbe = "1011" OR r_lbe = "0111") ELSE temp_xhdl67;
   temp_xhdl69 <= "000" WHEN (r_lbe = "1111" OR r_lbe = "0000") ELSE temp_xhdl68;
   b_lbe <= temp_xhdl69 ;
   temp_xhdl70 <= '1' WHEN (r_len = "0000000000") ELSE '0';
   t_len(10) <= temp_xhdl70 ;
   t_len(9 DOWNTO 0) <= r_len ;
   temp_xhdl71 <= "11" WHEN ((r_fbe(3)) = '1') ELSE "00";
   temp_xhdl72 <= "10" WHEN ((r_fbe(2)) = '1') ELSE temp_xhdl71;
   temp_xhdl73 <= "01" WHEN ((r_fbe(1)) = '1') ELSE temp_xhdl72;
   temp_xhdl74 <= "00" WHEN ((r_fbe(0)) = '1') ELSE temp_xhdl73;
   adr_lsb <= temp_xhdl74 ;
END ARCHITECTURE translated;
