LIBRARY ieee;                          
use IEEE.std_logic_1164.all;           
use IEEE.std_logic_arith.all;          
use IEEE.std_logic_unsigned.all;       
                                       
LIBRARY altera_mf;                     
USE altera_mf.altera_mf_components.all;
                                       
--
-- /**
-- *
-- *  This Verilog HDL File is used for simulation and synthesis in the simple DMA design example.  
-- *  This contains the DMA engine and memory buffer.
-- *
-- */
-------------------------------------------------------------------------------
-- Title         : PCI Express Reference Design Master Entity
-- Project       : PCI Express MegaCore function
-------------------------------------------------------------------------------
-- File          : altpcierd_master.v
-- Author        : Altera Corporation
-------------------------------------------------------------------------------
-- Description :
-- This is an entity that initiates transactions to the PCI Express interface
-------------------------------------------------------------------------------
-- Copyright (c) 2005-2006 Altera Corporation. All rights reserved.  Altera products are
-- protected under numerous U.S. and foreign patents, maskwork rights, copyrights and
-- other intellectual property laws.  
--
-- This reference design file, and your use thereof, is subject to and governed by
-- the terms and conditions of the applicable Altera Reference Design License Agreement.
-- By using this reference design file, you indicate your acceptance of such terms and
-- conditions between you and Altera Corporation.  In the event that you do not agree with
-- such terms and conditions, you may not use the reference design file. Please promptly
-- destroy any copies you have made.
--
-- This reference design file being provided on an "as-is" basis and as an accommodation 
-- and therefore all warranties, representations or guarantees of any kind 
-- (whether express, implied or statutory) including, without limitation, warranties of 
-- merchantability, non-infringement, or fitness for a particular purpose, are 
-- specifically disclaimed.  By making this reference design file available, Altera
-- expressly does not recommend, suggest or require that this reference design file be
-- used in combination with any other product not provided by Altera. 
-------------------------------------------------------------------------------
LIBRARY ieee;
USE ieee.numeric_std.all;
ENTITY altpcierd_master IS
   GENERIC (
      RAM_SIZE                       :  integer := 10);    
   PORT (
      hclk                    : IN std_logic;   
      hrstn                   : IN std_logic;   
      testtim                 : IN std_logic;   
      ko_cpl_spc_vc0          : IN std_logic_vector(19 DOWNTO 0);   
      ko_cpl_spc_vc1          : IN std_logic_vector(19 DOWNTO 0);   
      cfg_busdev              : IN std_logic_vector(12 DOWNTO 0);   
      cfg_maxpload            : IN std_logic_vector(2 DOWNTO 0);   
      cfg_maxrdreq            : IN std_logic_vector(2 DOWNTO 0);   
      cfg_rcb                 : IN std_logic;   
      cfg_tcvcmap             : IN std_logic_vector(23 DOWNTO 0);   
      trans_pending           : OUT std_logic;   
      err_cpltim              : OUT std_logic;   
      err_cplunexp            : OUT std_logic;   
      m_req                   : IN std_logic;   
      m_size                  : IN std_logic_vector(31 DOWNTO 0);   
      m_addr                  : IN std_logic_vector(63 DOWNTO 0);   
      m_ctrl                  : IN std_logic_vector(31 DOWNTO 0);   
      m_tgt_adr               : IN std_logic_vector(12 DOWNTO 0);   
      m_ack                   : OUT std_logic;   
      m_pending               : OUT std_logic;   
      rx_busy                 : OUT std_logic;   
      tx_busy                 : OUT std_logic;   
      vc0_out                 : OUT std_logic;   
      vc1_out                 : OUT std_logic;   
      rx_req                  : IN std_logic;   
      rx_ack                  : OUT std_logic;   
      rx_desc                 : IN std_logic_vector(135 DOWNTO 0);   
      rx_ws                   : OUT std_logic;   
      rx_data                 : IN std_logic_vector(63 DOWNTO 0);   
      rx_be                   : IN std_logic_vector(7 DOWNTO 0);   
      rx_dv                   : IN std_logic;   
      rx_dfr                  : IN std_logic;   
      tx_req                  : OUT std_logic;   
      tx_desc                 : OUT std_logic_vector(127 DOWNTO 0);   
      tx_ack                  : IN std_logic;   
      tx_dfr                  : OUT std_logic;   
      tx_data                 : OUT std_logic_vector(63 DOWNTO 0);   
      tx_dv                   : OUT std_logic;   
      tx_err                  : OUT std_logic;   
      tx_ws                   : IN std_logic;   
      fetch_tx_data           : IN std_logic;   --  external control to force a data read for tx
      nextcyc_is_last         : OUT std_logic;   --  indicates that next data cycle is last for tx pkt (last dfr cycle)
      last_datacyc            : OUT std_logic;   --  indicates that this data cycle is last for tx pkt (last dv cycle)
      -- streaming interface
      tx_avalon_data          : OUT std_logic_vector(63 DOWNTO 0);   
      rx_st_eop               : IN std_logic);   
END ENTITY altpcierd_master;
ARCHITECTURE translated OF altpcierd_master IS


    FUNCTION or_br (                                        
         val : std_logic_vector) RETURN std_logic IS        
                                                            
         VARIABLE rtn : std_logic := '0';                   
      BEGIN                                                 
         FOR index IN val'RANGE LOOP                        
            rtn := rtn OR val(index);                       
         END LOOP;                                          
         RETURN(rtn);                                       
      END or_br;                                            
                                                            
   COMPONENT altpcierd_dprambe
      GENERIC (
          ADDR_WIDTH                     :  integer := 8;    
          DATA_WIDTH                     :  integer := 32;    
          BYTE_SIZE                      :  integer := 8);    
      PORT (
         wrclk                   : IN  std_logic;
         wraddr                  : IN  std_logic_vector(ADDR_WIDTH - 1 DOWNTO 0);
         wrdata                  : IN  std_logic_vector(DATA_WIDTH - 1 DOWNTO 0);
         wrbe                    : IN  std_logic_vector(DATA_WIDTH / BYTE_SIZE - 1 DOWNTO 0);
         rdclk                   : IN  std_logic;
         rdclken                 : IN  std_logic;
         rdaddr                  : IN  std_logic_vector(ADDR_WIDTH - 1 DOWNTO 0);
         rddata                  : OUT std_logic_vector(DATA_WIDTH - 1 DOWNTO 0));
   END COMPONENT;
   CONSTANT  cpl_idl               :  std_logic_vector(1 DOWNTO 0) := "00";    
   CONSTANT  cpl_rxdesc            :  std_logic_vector(1 DOWNTO 0) := "01";    
   CONSTANT  cpl_rxdata            :  std_logic_vector(1 DOWNTO 0) := "10";    
   SIGNAL cpl_sm                   :  std_logic_vector(1 DOWNTO 0);   
   CONSTANT  req_idle              :  std_logic_vector(1 DOWNTO 0) := "00";    
   CONSTANT  req_txdesc            :  std_logic_vector(1 DOWNTO 0) := "01";    
   CONSTANT  req_txdata            :  std_logic_vector(1 DOWNTO 0) := "10";    
   SIGNAL req_sm                   :  std_logic_vector(1 DOWNTO 0);   
   SIGNAL is_rxmst                 :  std_logic;   
   SIGNAL is_rxwr                  :  std_logic;   
   SIGNAL is_rxdval                :  std_logic;   
   SIGNAL rx_addr                  :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL rx_bytecnt               :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL rx_wr                    :  std_logic;   
   SIGNAL wr_allow                 :  std_logic;   
   SIGNAL datain                   :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL dataout                  :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL rx_adwr                  :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   -- xhdl
   SIGNAL tx_adrd                  :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL wren                     :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL phasecnt                 :  std_logic_vector(9 DOWNTO 0);   
   SIGNAL txaddr                   :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL fetch_cnt                :  std_logic_vector(1 DOWNTO 0);   
   SIGNAL desc_setup_cnt           :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL need_to_fetch            :  std_logic;   
   SIGNAL pipe_fill                :  std_logic;   
   SIGNAL rdclken                  :  std_logic;   
   SIGNAL txaddr_inc               :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL fetch_first              :  std_logic;   
   SIGNAL mem_first                :  std_logic;   
   SIGNAL mem_first_del            :  std_logic;   
   SIGNAL tx_dv_r                  :  std_logic;   
   SIGNAL rd_allow                 :  std_logic;   
   SIGNAL rd_out                   :  std_logic;   
   SIGNAL tx_dfr_r                 :  std_logic;   
   SIGNAL tx_data_r                :  std_logic_vector(63 DOWNTO 0);   
   -- master signals
   SIGNAL mem_desc                 :  std_logic;   
   SIGNAL mem_new                  :  std_logic;   
   SIGNAL m_ack_r                  :  std_logic;   
   SIGNAL mem_addr64               :  std_logic_vector(31 DOWNTO 0);   
   SIGNAL mem_ctrl                 :  std_logic_vector(15 DOWNTO 0);   
   SIGNAL mem_tgt_adr              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL mem_tgt_adr_lsb          :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL mem_pload                :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL mem_addr32               :  std_logic_vector(31 DOWNTO 0);   
   SIGNAL mem_size                 :  std_logic_vector(31 DOWNTO 0);   
   SIGNAL maxreq_cfg               :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL cfg_maxrdreq_i           :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL kod_cfg_maxrdreq         :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL koh_cfg_maxrdreq         :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL ko_cfg_maxrdreq          :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL maxreq                   :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL addr_13lsb               :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL bytecnt_max              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL bytecnt_sub_r            :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL bytecnt_sub              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL bytecnt_sat              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL tx_bytecnt               :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL tx_lastoffset            :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL ack_req                  :  std_logic;   
   SIGNAL tx_remain                :  std_logic;   
   SIGNAL tx_dwcnt_r               :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL tx_dwcnt_aft64B          :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL addr_aft64B              :  std_logic_vector(4 DOWNTO 0);   
   SIGNAL tx_lbe_n                 :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL tx_fbe_n                 :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL tx_lbe_d                 :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL tx_fbe_d                 :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL tx_lbe                   :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL tx_fbe                   :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL rcb                      :  std_logic;   
   SIGNAL rem_cpld                 :  std_logic;   
   SIGNAL rem_cplh                 :  std_logic;   
   SIGNAL xtr_cplh                 :  std_logic;   
   SIGNAL bas_cplh                 :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL cred_cplh_n              :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL cred_cpld_n              :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL cred_cplh                :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL cred_cpld                :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL vc_sel                   :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL cur_cplh                 :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL cur_cpld                 :  std_logic_vector(13 DOWNTO 0);   
   SIGNAL cur_cplh0                :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL cur_cpld0                :  std_logic_vector(13 DOWNTO 0);   
   SIGNAL cur_cplh1                :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL cur_cpld1                :  std_logic_vector(13 DOWNTO 0);   
   SIGNAL en_rdreq                 :  std_logic;   
   SIGNAL en_npwr                  :  std_logic;   
   SIGNAL en_rdreq_r               :  std_logic;   
   SIGNAL en_npwr_r                :  std_logic;   
   SIGNAL en_req                   :  std_logic;   
   SIGNAL en_req_r                 :  std_logic;   
   SIGNAL mem_req                  :  std_logic;   
   SIGNAL mem_req_r                :  std_logic;   
   SIGNAL tx_ack_r                 :  std_logic;   
   SIGNAL mst_tag_vld              :  std_logic;   
   SIGNAL tx_desc_mem              :  std_logic_vector(127 DOWNTO 0);   
   SIGNAL cpl_busy_n               :  std_logic;   
   SIGNAL cpl_busy                 :  std_logic;   
   SIGNAL mst_tag                  :  std_logic_vector(1 DOWNTO 0);   
   SIGNAL vc_r                     :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL cplh_r                   :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL cpld_r                   :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL init                     :  std_logic;   
   SIGNAL cpl_ack                  :  std_logic;   
   SIGNAL cpl_ack_r                :  std_logic;   
   SIGNAL cpl_hdrcred_rls          :  std_logic;   
   SIGNAL dwcnt_msb                :  std_logic;   
   SIGNAL cpl_vc                   :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL cpl_vc_n                 :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL cpl_hdrcred              :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL cpl_datacred             :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL req_cpl_chnl             :  std_logic;   
   SIGNAL tim1ms                   :  std_logic_vector(16 DOWNTO 0);   
   SIGNAL ev1ms                    :  std_logic;   
   SIGNAL cpl_addr_n               :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL cpl_busy0                :  std_logic;   
   SIGNAL cpl_addr0                :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL timer0                   :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL cpl_dwcnt0               :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL hdrcred0                 :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL datacred0                :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL cpl_busy1                :  std_logic;   
   SIGNAL timer1                   :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL cpl_addr1                :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL cpl_dwcnt1               :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL hdrcred1                 :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL datacred1                :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL cpl_busy2                :  std_logic;   
   SIGNAL timer2                   :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL cpl_addr2                :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL cpl_dwcnt2               :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL hdrcred2                 :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL datacred2                :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL cpl_busy3                :  std_logic;   
   SIGNAL timer3                   :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL cpl_addr3                :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL cpl_dwcnt3               :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL hdrcred3                 :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL datacred3                :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL rx_ack_r                 :  std_logic;   
   SIGNAL rx_ack_rr                :  std_logic;   
   SIGNAL rx_tag                   :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL rx_tag0_hit              :  std_logic;   
   SIGNAL rx_tag1_hit              :  std_logic;   
   SIGNAL rx_tag2_hit              :  std_logic;   
   SIGNAL rx_tag3_hit              :  std_logic;   
   SIGNAL rx_addr7                 :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL rx_dwcnt                 :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL rem_rxcpld               :  std_logic;   
   SIGNAL rem_rxcplh               :  std_logic;   
   SIGNAL bas_rxcplh               :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL rx_cplh_n                :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL rx_cpld_n                :  std_logic_vector(8 DOWNTO 0);   
   SIGNAL rx_cplh                  :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL rx_cpld                  :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL rx_tc                    :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL is_txwr                  :  std_logic;   
   SIGNAL cpl_sel                  :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL vc_sel_n                 :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL err_tim0                 :  std_logic;   
   SIGNAL err_tim1                 :  std_logic;   
   SIGNAL err_tim2                 :  std_logic;   
   SIGNAL err_tim3                 :  std_logic;   
   SIGNAL mem_zero                 :  std_logic;   
   SIGNAL mem_ramp1                :  std_logic;   
   SIGNAL mem_ramp4                :  std_logic;   
   SIGNAL data_ramp                :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL data_ramp15              :  std_logic_vector(14 DOWNTO 0);   
   SIGNAL maxreq_n                 :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL rx_addr_dw               :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL rx_addrend_n             :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL rx_addrend               :  std_logic_vector(5 DOWNTO 0);   
   SIGNAL rx_desc_r                :  std_logic_vector(135 DOWNTO 0);   
   SIGNAL rx_req_r                 :  std_logic;   
   -- pipeline stages for dprambe 
   -- write port input.   
   -- for performance.
   SIGNAL rx_adwr_del              :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL datain_del               :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL wren_del                 :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL rx_adwr_del2             :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL datain_del2              :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL wren_del2                :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL rx_adwr_del3             :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL datain_del3              :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL wren_del3                :  std_logic_vector(7 DOWNTO 0);   
   -- pipe stages
   SIGNAL memsize_minus_bytecnt    :  std_logic_vector(31 DOWNTO 0);   
   SIGNAL memaddr32_plus_bytecnt   :  std_logic_vector(31 DOWNTO 0);   
   -- wires for concatenations
   -- xhdl
   SIGNAL mem_addr32_7lsb          :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL mem_addr32_8lsb          :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL mem_addr32_9lsb          :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL mem_addr32_10lsb         :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL mem_addr32_11lsb         :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL mem_addr32_12lsb         :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL mem_addr32_6to0          :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL mem_addr32_7to0          :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL mem_addr32_8to0          :  std_logic_vector(8 DOWNTO 0);   
   SIGNAL mem_addr32_9to0          :  std_logic_vector(9 DOWNTO 0);   
   SIGNAL mem_addr32_10to0         :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL mem_addr32_11to0         :  std_logic_vector(11 DOWNTO 0);   
   SIGNAL rx_desc_r_38to34         :  std_logic_vector(4 DOWNTO 0);   
   SIGNAL data_ramp15_4to0         :  std_logic_vector(4 DOWNTO 0);   
   SIGNAL data_ramp_memramp1       :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL data_ramp_memramp2       :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL ko_cpl_spc_vc0_19to8     :  std_logic_vector(11 DOWNTO 0);   
   SIGNAL ko_cpl_spc_vc1_19to8     :  std_logic_vector(11 DOWNTO 0);   
   -- wire       maxreqis_zero;   
   SIGNAL rx_dwcnt_div2            :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   -- assign    rx_desc_r = rx_desc;
   --  assign     rx_req_r = rx_req;
   SIGNAL cpl_addr_n_add_mem       :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL zero_qword               :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL temp_xhdl21              :  std_logic;   
   -- calculate parameters of next transfer according to PCI-Express rules
   -- 1. max payload size differs from read to write
   SIGNAL temp_xhdl22              :  std_logic_vector(2 DOWNTO 0);   
   -- 2. saturate payload size if small transfer requested
   SIGNAL temp_xhdl23              :  std_logic_vector(2 DOWNTO 0);   
   -- 3. address used to calculate next transfer (force aligned on 128 bytes boundary)
   --128 KB
   --256 KB
   --512 KB
   --1 KB
   --2 KB
   -- assign maxreqis_zero = (maxreq == 3'b000);  // unused.  xhdl
   SIGNAL temp_xhdl24              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl25              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl26              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl27              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl28              :  std_logic_vector(12 DOWNTO 0);   
   -- 4. byte count max of the transfer
   -- 128 KB
   -- 256 KB
   -- 512 KB
   -- 1 KB
   -- 2 KB
   SIGNAL temp_xhdl29              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl30              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl31              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl32              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl33              :  std_logic_vector(12 DOWNTO 0);   
   -- 6. saturation of bytecount at the end of transfer
   SIGNAL temp_xhdl34              :  std_logic_vector(12 DOWNTO 0);   
   SIGNAL temp_xhdl35              :  std_logic;   
   SIGNAL temp_xhdl36              :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL temp_xhdl37              :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL temp_xhdl38              :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL temp_xhdl39              :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL temp_xhdl40              :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL temp_xhdl41              :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL temp_xhdl42              :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL temp_xhdl43              :  std_logic_vector(3 DOWNTO 0);   
   SIGNAL temp_xhdl44              :  std_logic;   
   SIGNAL temp_xhdl45              :  std_logic;   
   SIGNAL temp_xhdl46              :  std_logic;   
   SIGNAL temp_xhdl47              :  std_logic;   
   SIGNAL temp_xhdl48              :  std_logic;   
   SIGNAL temp_xhdl49              :  std_logic_vector(10 DOWNTO 4);   
   -- 10. Select the VC associated to the TC and the associated CPL credits
   SIGNAL temp_xhdl50              :  std_logic_vector(20 DOWNTO 18);   
   SIGNAL temp_xhdl51              :  std_logic_vector(17 DOWNTO 15);   
   SIGNAL temp_xhdl52              :  std_logic_vector(14 DOWNTO 12);   
   SIGNAL temp_xhdl53              :  std_logic_vector(11 DOWNTO 9);   
   SIGNAL temp_xhdl54              :  std_logic_vector(8 DOWNTO 6);   
   SIGNAL temp_xhdl55              :  std_logic_vector(5 DOWNTO 3);   
   SIGNAL temp_xhdl56              :  std_logic_vector(2 DOWNTO 0);   
   -- only two VC are implemented in the design
   SIGNAL temp_xhdl57              :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL temp_xhdl58              :  std_logic_vector(13 DOWNTO 0);   
   -- 11. Enable the read request only enough completion buffer available
   --  required by PCI-Express for deadlock prevention in endpoint and root
   SIGNAL temp_xhdl59              :  std_logic;   
   SIGNAL temp_xhdl60              :  std_logic;   
   SIGNAL temp_xhdl61              :  std_logic;   
   SIGNAL temp_xhdl62              :  std_logic;   
   SIGNAL temp_xhdl63              :  std_logic_vector(2 DOWNTO 0);   
   -- Set the max read request size depending on the cpl buffer allocated.
   SIGNAL temp_xhdl64              :  std_logic_vector(2 DOWNTO 0);   
   -- 14. Monitor completion channel
   -- select one completion channel
   SIGNAL temp_xhdl65              :  std_logic;   
   SIGNAL temp_xhdl66              :  std_logic;   
   SIGNAL temp_xhdl67              :  std_logic;   
   SIGNAL temp_xhdl68              :  std_logic;   
   SIGNAL temp_xhdl69              :  std_logic;   
   SIGNAL temp_xhdl70              :  std_logic;   
   SIGNAL temp_xhdl71              :  std_logic_vector(10 DOWNTO 4);   
   SIGNAL temp_xhdl72              :  std_logic_vector(20 DOWNTO 18);   
   SIGNAL temp_xhdl73              :  std_logic_vector(17 DOWNTO 15);   
   SIGNAL temp_xhdl74              :  std_logic_vector(14 DOWNTO 12);   
   SIGNAL temp_xhdl75              :  std_logic_vector(11 DOWNTO 9);   
   SIGNAL temp_xhdl76              :  std_logic_vector(8 DOWNTO 6);   
   SIGNAL temp_xhdl77              :  std_logic_vector(5 DOWNTO 3);   
   SIGNAL temp_xhdl78              :  std_logic_vector(2 DOWNTO 0);   
   SIGNAL temp_xhdl79              :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL temp_xhdl80              :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL temp_xhdl81              :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL temp_xhdl82              :  std_logic_vector(6 DOWNTO 0);   
   SIGNAL temp_xhdl83              :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL temp_xhdl84              :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL temp_xhdl85              :  std_logic_vector(10 DOWNTO 0);   
   SIGNAL temp_xhdl86              :  std_logic_vector(10 DOWNTO 0);   
   -----------------------------------------------------------------------
   -- Transmit state machine
   SIGNAL temp_xhdl87              :  std_logic;   
   --flags for external use
   SIGNAL temp_xhdl88              :  std_logic;   --  xhdl
   SIGNAL temp_xhdl89              :  std_logic;   --  xhdl
   SIGNAL temp_xhdl90              :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL temp_xhdl91              :  std_logic;   
   -- read clock enable is used to stall the datapath pipeline
   --  assign rdclken = (need_to_fetch == 1'b1 | (tx_dv_r == 1'b1 & tx_ws == 
   --  1'b0)) ? 1'b1 : 1'b0 ; 
   SIGNAL temp_xhdl92              :  std_logic;   
   ---------------------------------------------------------------------
   -- 17. receive state machine
   SIGNAL temp_xhdl93              :  std_logic;   
   SIGNAL temp_xhdl94              :  std_logic;   
   SIGNAL temp_xhdl95              :  std_logic;   
   SIGNAL temp_xhdl96              :  std_logic;   
   -- brought rx_ws up one cycle
   SIGNAL temp_xhdl97              :  std_logic;   
   SIGNAL temp_xhdl98              :  std_logic_vector(7 DOWNTO 0);   
   SIGNAL port_xhdl100             :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   SIGNAL port_xhdl101             :  std_logic_vector(RAM_SIZE - 1 DOWNTO 0);   
   -- error check
   SIGNAL temp_xhdl102             :  std_logic;   
   SIGNAL temp_xhdl103             :  std_logic;   
   SIGNAL temp_xhdl104             :  std_logic;   
   SIGNAL temp_xhdl105             :  std_logic;   
   SIGNAL trans_pending_xhdl1      :  std_logic;   
   SIGNAL err_cpltim_xhdl2         :  std_logic;   
   SIGNAL err_cplunexp_xhdl3       :  std_logic;   
   SIGNAL m_ack_xhdl4              :  std_logic;   
   SIGNAL m_pending_xhdl5          :  std_logic;   
   SIGNAL rx_busy_xhdl6            :  std_logic;   
   SIGNAL tx_busy_xhdl7            :  std_logic;   
   SIGNAL vc0_out_xhdl8            :  std_logic;   
   SIGNAL vc1_out_xhdl9            :  std_logic;   
   SIGNAL rx_ack_xhdl10            :  std_logic;   
   SIGNAL rx_ws_xhdl11             :  std_logic;   
   SIGNAL tx_req_xhdl12            :  std_logic;   
   SIGNAL tx_desc_xhdl13           :  std_logic_vector(127 DOWNTO 0);   
   SIGNAL tx_dfr_xhdl14            :  std_logic;   
   SIGNAL tx_data_xhdl15           :  std_logic_vector(63 DOWNTO 0);   
   SIGNAL tx_dv_xhdl16             :  std_logic;   
   SIGNAL tx_err_xhdl17            :  std_logic;   
   SIGNAL nextcyc_is_last_xhdl18   :  std_logic;   
   SIGNAL last_datacyc_xhdl19      :  std_logic;   
   SIGNAL tx_avalon_data_xhdl20    :  std_logic_vector(63 DOWNTO 0);   
BEGIN
   trans_pending <= trans_pending_xhdl1;
   err_cpltim <= err_cpltim_xhdl2;
   err_cplunexp <= err_cplunexp_xhdl3;
   m_ack <= m_ack_xhdl4;
   m_pending <= m_pending_xhdl5;
   rx_busy <= rx_busy_xhdl6;
   tx_busy <= tx_busy_xhdl7;
   vc0_out <= vc0_out_xhdl8;
   vc1_out <= vc1_out_xhdl9;
   rx_ack <= rx_ack_xhdl10;
   rx_ws <= rx_ws_xhdl11;
   tx_req <= tx_req_xhdl12;
   tx_desc <= tx_desc_xhdl13;
   tx_dfr <= tx_dfr_xhdl14;
   tx_data <= tx_data_xhdl15;
   tx_dv <= tx_dv_xhdl16;
   tx_err <= tx_err_xhdl17;
   nextcyc_is_last <= nextcyc_is_last_xhdl18;
   last_datacyc <= last_datacyc_xhdl19;
   tx_avalon_data <= tx_avalon_data_xhdl20;
   rx_dwcnt_div2 <= '0' & rx_dwcnt(RAM_SIZE - 1 DOWNTO 1) ;
   -- xhdl
   mem_addr32_6to0 <= mem_addr32(6 DOWNTO 0) ;
   mem_addr32_7to0 <= mem_addr32(7 DOWNTO 0) ;
   mem_addr32_8to0 <= mem_addr32(8 DOWNTO 0) ;
   mem_addr32_9to0 <= mem_addr32(9 DOWNTO 0) ;
   mem_addr32_10to0 <= mem_addr32(10 DOWNTO 0) ;
   mem_addr32_11to0 <= mem_addr32(11 DOWNTO 0) ;
   mem_addr32_7lsb <= "000000" & mem_addr32_6to0 ;
   mem_addr32_8lsb <= "00000" & mem_addr32_7to0 ;
   mem_addr32_9lsb <= "0000" & mem_addr32_8to0 ;
   mem_addr32_10lsb <= "000" & mem_addr32_9to0 ;
   mem_addr32_11lsb <= "00" & mem_addr32_10to0 ;
   mem_addr32_12lsb <= '0' & mem_addr32_11to0 ;
   ------------------------------------
   -- streaming interface
   ------------------------------------
   tx_avalon_data_xhdl20 <= tx_data_xhdl15 ;
   zero_qword <= "0000000000000000000000000000000000000000000000000000000000000000" ;
   cpl_addr_n_add_mem(0) <= mem_addr32(2) AND tx_dwcnt_r(0) ;
   cpl_addr_n_add_mem(RAM_SIZE - 1 DOWNTO 1) <= zero_qword(RAM_SIZE - 1 DOWNTO 1) ;
   -- master interface control
   
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         mem_desc <= '0';    
         mem_new <= '0';    
         mem_addr64 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         mem_ctrl <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         mem_tgt_adr <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         mem_tgt_adr_lsb <= '0' & '0' & '0' & '0';    
         mem_pload <= '0' & '0' & '0';    
         mem_addr32 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cpl_addr_n <= (OTHERS => '0');    
         mem_size <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         m_ack_r <= '0';    
         mem_zero <= '0';    
         mem_ramp1 <= '0';    
         mem_ramp4 <= '0';    
         desc_setup_cnt <= "000";    
         rx_desc_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         rx_req_r <= '0';    
         memsize_minus_bytecnt <= "00000000000000000000000000000000";    
         memaddr32_plus_bytecnt <= "00000000000000000000000000000000";    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF (mem_desc = '0' AND m_req = '1') THEN
            mem_desc <= '1';    
         ELSE
            IF (ack_req = '1' AND mem_size = tx_bytecnt) THEN
               mem_desc <= '0';    
            END IF;
         END IF;
         IF (mem_desc = '0' AND m_req = '1') THEN
            mem_new <= '1';    
         ELSE
            IF (desc_setup_cnt = "011") THEN
               mem_new <= '0';    
            END IF;
         END IF;
         -- if (mem_desc == 1'b0 & m_req == 1'b1)
         
         IF ((mem_desc = '0' AND m_req = '1') AND (desc_setup_cnt = "100")) THEN
            -- gaurantee timing between m_ack_r and ack_req
            
            m_ack_r <= '1';    
         ELSE
            m_ack_r <= '0';    
         END IF;
         IF (desc_setup_cnt /= "100") THEN
            desc_setup_cnt <= desc_setup_cnt + "001";    
         ELSE
            IF ((m_ack_r = '1') OR (ack_req = '1')) THEN
               desc_setup_cnt <= "000";    
            END IF;
         END IF;
         rx_desc_r <= rx_desc;    
         rx_req_r <= rx_req;    
         IF (m_ack_r = '1') THEN
            mem_addr64 <= m_addr(63 DOWNTO 32);    
         END IF;
         mem_tgt_adr_lsb <= '0' & tx_bytecnt(2 DOWNTO 0) + mem_addr32(2 DOWNTO 0);    
         IF (m_ack_r = '1') THEN
            mem_tgt_adr <= m_tgt_adr;    
         ELSE
            IF (ack_req = '1') THEN
               mem_tgt_adr <= mem_tgt_adr + tx_bytecnt(12 DOWNTO 3) + mem_tgt_adr_lsb;    
            END IF;
         END IF;
         IF (m_ack_r = '1') THEN
            mem_ctrl <= m_ctrl(15 DOWNTO 0);    
            mem_pload <= m_ctrl(18 DOWNTO 16);    
            mem_zero <= m_ctrl(20);    
            mem_ramp1 <= m_ctrl(21);    
            mem_ramp4 <= m_ctrl(22);    
         END IF;
         memaddr32_plus_bytecnt <= mem_addr32 + tx_bytecnt;    --  pipe for performance
         IF (m_ack_r = '1') THEN
            mem_addr32 <= m_addr(31 DOWNTO 0);    
         ELSE
            IF (ack_req = '1') THEN
               --  mem_addr32 <= mem_addr32 + tx_bytecnt ;  
               
               mem_addr32 <= memaddr32_plus_bytecnt;    
            END IF;
         END IF;
         memsize_minus_bytecnt <= mem_size - tx_bytecnt;    --  pipe for performance
         IF (m_ack_r = '1') THEN
            mem_size <= m_size;    
         ELSE
            IF (ack_req = '1') THEN
               --  mem_size <= mem_size - tx_bytecnt ; 
               
               mem_size <= memsize_minus_bytecnt;    
            END IF;
         END IF;
         IF (m_ack_r = '1') THEN
            cpl_addr_n <= m_tgt_adr(RAM_SIZE - 1 DOWNTO 0);    --  xhdl
         ELSE
            IF (ack_req = '1') THEN
               cpl_addr_n <= cpl_addr_n + tx_dwcnt_r(10 DOWNTO 1) + cpl_addr_n_add_mem;    
            END IF;
         END IF;
      END IF;
   END PROCESS;
   m_ack_xhdl4 <= m_ack_r ;
   temp_xhdl21 <= '1' WHEN (mem_desc = '1' OR tx_busy_xhdl7 = '1' OR tx_dfr_r = '1' OR cpl_busy0 = '1' OR cpl_busy1 = '1' OR cpl_busy2 = '1' OR cpl_busy3 = '1') ELSE '0';
   m_pending_xhdl5 <= temp_xhdl21 ;
   temp_xhdl22 <= cfg_maxpload WHEN ((mem_ctrl(6)) = '1') ELSE cfg_maxrdreq_i;
   maxreq_cfg <= temp_xhdl22 ;
   temp_xhdl23 <= mem_pload WHEN (mem_pload<=maxreq_cfg) ELSE maxreq_cfg;
   maxreq_n <= temp_xhdl23 ;
   temp_xhdl24 <= mem_addr32_11lsb WHEN (maxreq = "100") ELSE mem_addr32_12lsb;
   temp_xhdl25 <= mem_addr32_10lsb WHEN (maxreq = "011") ELSE (temp_xhdl24);
   temp_xhdl26 <= mem_addr32_9lsb WHEN (maxreq = "010") ELSE (temp_xhdl25);
   temp_xhdl27 <= mem_addr32_8lsb WHEN (maxreq = "001") ELSE (temp_xhdl26);
   temp_xhdl28 <= mem_addr32_7lsb WHEN (maxreq = "000") ELSE (temp_xhdl27);
   addr_13lsb <= temp_xhdl28 ;
   temp_xhdl29 <= "0100000000000" WHEN (maxreq = "100") ELSE "1000000000000";
   temp_xhdl30 <= "0010000000000" WHEN (maxreq = "011") ELSE temp_xhdl29;
   temp_xhdl31 <= "0001000000000" WHEN (maxreq = "010") ELSE temp_xhdl30;
   temp_xhdl32 <= "0000100000000" WHEN (maxreq = "001") ELSE temp_xhdl31;
   temp_xhdl33 <= "0000010000000" WHEN (maxreq = "000") ELSE temp_xhdl32;
   bytecnt_max <= temp_xhdl33 ;
   -- 5. substract for non-aligned addresses
   bytecnt_sub <= bytecnt_max - addr_13lsb ;
   temp_xhdl34 <= bytecnt_sub_r WHEN (mem_size >= bytecnt_sub_r) ELSE mem_size(12 DOWNTO 0);
   bytecnt_sat <= temp_xhdl34 ;
   -- 7. pipeline for performance
   
   PROCESS (hclk)
   BEGIN
      IF (hclk'EVENT AND hclk = '1') THEN
         -- 
         
         bytecnt_sub_r <= bytecnt_sub;    
      END IF;
   END PROCESS;
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         maxreq <= "000";    
         tx_bytecnt <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         tx_dwcnt_aft64B <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         tx_dwcnt_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         maxreq <= maxreq_n;    
         tx_bytecnt <= bytecnt_sat;    
         IF (mem_addr32(5 DOWNTO 2) = "0000") THEN
            tx_dwcnt_aft64B <= tx_dwcnt_r;    
         ELSE
            IF ((addr_aft64B(4)) = '1') THEN
               tx_dwcnt_aft64B <= tx_dwcnt_r + mem_addr32(5 DOWNTO 2) - "00000010000";    
            ELSE
               tx_dwcnt_aft64B <= tx_dwcnt_r;    
            END IF;
         END IF;
         tx_dwcnt_r <= tx_lastoffset(12 DOWNTO 2) + tx_remain;    
      END IF;
   END PROCESS;
   -- 8. calculate the DW count and the First BE and Last BE
   tx_lastoffset <= tx_bytecnt + mem_addr32(1 DOWNTO 0) ;
   temp_xhdl35 <= '1' WHEN (tx_lastoffset(1 DOWNTO 0) /= "00") ELSE '0';
   tx_remain <= temp_xhdl35 ;
   addr_aft64B <= ('0' & tx_dwcnt_r(3 DOWNTO 0)) + ('0' & mem_addr32(5 DOWNTO 2)) ;
   temp_xhdl36 <= "1100" WHEN (mem_addr32(1 DOWNTO 0) = "10") ELSE "1000";
   temp_xhdl37 <= "1110" WHEN (mem_addr32(1 DOWNTO 0) = "01") ELSE temp_xhdl36;
   temp_xhdl38 <= "1111" WHEN (mem_addr32(1 DOWNTO 0) = "00") ELSE temp_xhdl37;
   tx_fbe_n <= temp_xhdl38 ;
   temp_xhdl39 <= "0011" WHEN (tx_lastoffset(1 DOWNTO 0) = "10") ELSE "0111";
   temp_xhdl40 <= "0001" WHEN (tx_lastoffset(1 DOWNTO 0) = "01") ELSE temp_xhdl39;
   temp_xhdl41 <= "1111" WHEN (tx_lastoffset(1 DOWNTO 0) = "00") ELSE temp_xhdl40;
   tx_lbe_n <= temp_xhdl41 ;
   temp_xhdl42 <= (tx_fbe AND tx_lbe) WHEN (tx_dwcnt_r = "00000000001") ELSE tx_fbe;
   tx_fbe_d <= temp_xhdl42 ;
   temp_xhdl43 <= "0000" WHEN (tx_dwcnt_r = "00000000001") ELSE tx_lbe;
   tx_lbe_d <= temp_xhdl43 ;
   rcb <= cfg_rcb ;
   -- 9. check the completion buffer availability
   -- calculate completion header credits and data credits for request
   -- The Worst case for header credit requirment is one big transfer is broken
   -- into smaller transfer based on Read Completion Boundary
   -- 
   -- An extra header is allocated in case the address and the length is such
   -- that it crosses 64 Byte Cacheline boundary. When this happens, the completion
   -- will also be broken up.
   cred_cpld_n <= tx_dwcnt_r(10 DOWNTO 0) ;
   temp_xhdl44 <= '1' WHEN (tx_dwcnt_r(10 DOWNTO 4) /= "0000000") ELSE '0';
   temp_xhdl45 <= '1' WHEN ((addr_aft64B(4)) = '1') ELSE temp_xhdl44;
   temp_xhdl46 <= '0' WHEN (mem_addr32(5 DOWNTO 2) = "0000") ELSE temp_xhdl45;
   xtr_cplh <= temp_xhdl46 ;
   temp_xhdl47 <= '1' WHEN (tx_dwcnt_aft64B(4 DOWNTO 0) /= "00000" AND rcb = '1') ELSE '0';
   temp_xhdl48 <= '1' WHEN (tx_dwcnt_aft64B(3 DOWNTO 0) /= "0000" AND rcb = '0') ELSE temp_xhdl47;
   rem_cplh <= temp_xhdl48 ;
   temp_xhdl49 <= tx_dwcnt_aft64B(10 DOWNTO 4) WHEN (rcb = '0') ELSE '0' & tx_dwcnt_aft64B(10 DOWNTO 5);
   bas_cplh <= temp_xhdl49 ;
   cred_cplh_n <= bas_cplh + rem_cplh + xtr_cplh ;
   ----------------------------------------------------------------------
   -- Pipelining stage for performance purpose
   
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         cred_cpld <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cred_cplh <= '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         tx_fbe <= '0' & '0' & '0' & '0';    
         tx_lbe <= '0' & '0' & '0' & '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         cred_cpld <= cred_cpld_n;    
         cred_cplh <= cred_cplh_n;    
         tx_fbe <= tx_fbe_n;    
         tx_lbe <= tx_lbe_n;    
      END IF;
   END PROCESS;
   temp_xhdl50 <= cfg_tcvcmap(20 DOWNTO 18) WHEN (mem_ctrl(10 DOWNTO 8) = "110") ELSE cfg_tcvcmap(23 DOWNTO 21);
   temp_xhdl51 <= cfg_tcvcmap(17 DOWNTO 15) WHEN (mem_ctrl(10 DOWNTO 8) = "101") ELSE temp_xhdl50;
   temp_xhdl52 <= cfg_tcvcmap(14 DOWNTO 12) WHEN (mem_ctrl(10 DOWNTO 8) = "100") ELSE temp_xhdl51;
   temp_xhdl53 <= cfg_tcvcmap(11 DOWNTO 9) WHEN (mem_ctrl(10 DOWNTO 8) = "011") ELSE temp_xhdl52;
   temp_xhdl54 <= cfg_tcvcmap(8 DOWNTO 6) WHEN (mem_ctrl(10 DOWNTO 8) = "010") ELSE temp_xhdl53;
   temp_xhdl55 <= cfg_tcvcmap(5 DOWNTO 3) WHEN (mem_ctrl(10 DOWNTO 8) = "001") ELSE temp_xhdl54;
   temp_xhdl56 <= cfg_tcvcmap(2 DOWNTO 0) WHEN (mem_ctrl(10 DOWNTO 8) = "000") ELSE temp_xhdl55;
   vc_sel_n <= temp_xhdl56 ;
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         vc_sel <= "000";    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         vc_sel <= vc_sel_n;    
      END IF;
   END PROCESS;
   temp_xhdl57 <= cur_cplh0 WHEN (vc_sel = "000") ELSE cur_cplh1;
   cur_cplh <= temp_xhdl57 ;
   temp_xhdl58 <= cur_cpld0 WHEN (vc_sel = "000") ELSE cur_cpld1;
   cur_cpld <= temp_xhdl58 ;
   temp_xhdl59 <= '1' WHEN (cur_cplh >= cred_cplh AND cur_cpld >= cred_cpld) ELSE '0';
   en_rdreq <= temp_xhdl59 ;
   temp_xhdl60 <= '1' WHEN (cur_cplh >= cred_cplh) ELSE '0';
   en_npwr <= temp_xhdl60 ;
   temp_xhdl61 <= en_npwr_r WHEN (mem_ctrl(6 DOWNTO 0) = "1000010" OR mem_ctrl(6 DOWNTO 1) = "100010") ELSE '1';
   temp_xhdl62 <= en_rdreq_r WHEN ((mem_ctrl(6)) = '0') ELSE temp_xhdl61;
   en_req <= temp_xhdl62 ;
   -- 12. generate the descriptor and the request
   
   PROCESS (hclk)
   BEGIN
      IF (hclk'EVENT AND hclk = '1') THEN
         -- 
         
         en_npwr_r <= en_npwr;    
         en_rdreq_r <= en_rdreq;    
      END IF;
   END PROCESS;
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         mem_req <= '0';    
         mem_req_r <= '0';    
         tx_ack_r <= '0';    
         mst_tag_vld <= '0';    
         ack_req <= '0';    
         tx_desc_mem <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         vc_r <= "000";    
         cplh_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cpld_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         mst_tag <= '0' & '0';    
         dwcnt_msb <= '0';    
         req_cpl_chnl <= '0';    
         vc0_out_xhdl8 <= '0';    
         vc1_out_xhdl9 <= '0';    
         en_req_r <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         en_req_r <= en_req;    
         mem_req_r <= mem_req;    
         IF (((((((req_sm = req_idle AND mem_req = '0') AND mem_req_r = '0') AND en_req_r = '1') AND mem_desc = '1') AND mem_new = '0') AND cpl_busy = '0') AND desc_setup_cnt = "100") THEN
            mem_req <= '1';    
         ELSE
            IF (tx_ack = '1') THEN
               mem_req <= '0';    
            END IF;
         END IF;
         ---------------------------------
         
         IF (((((((req_sm = req_idle AND mem_req = '0') AND mem_req_r = '0') AND en_req_r = '1') AND mem_desc = '1') AND mem_new = '0') AND cpl_busy = '0') AND desc_setup_cnt = "100") THEN
            ack_req <= '1';    
         ELSE
            ack_req <= '0';    
         END IF;
         IF (mem_req = '0') THEN
            -- tx_desc_mem[127:96] <= {mem_ctrl[7:0], 1'b0, mem_ctrl[10:8], 4'b0000, mem_ctrl[15:12], 2'b00, tx_dwcnt_r[9:0]} ;    // xhdl
            -- tx_desc_mem[95:64] <= {cfg_busdev, 3'b000, 6'b000000, mst_tag, tx_lbe_d, tx_fbe_d };                                // xhdl
            
            tx_desc_mem(127 DOWNTO 120) <= mem_ctrl(7 DOWNTO 0);    
            tx_desc_mem(119) <= '0';    
            tx_desc_mem(118 DOWNTO 116) <= mem_ctrl(10 DOWNTO 8);    
            tx_desc_mem(115 DOWNTO 112) <= "0000";    
            tx_desc_mem(111 DOWNTO 108) <= mem_ctrl(15 DOWNTO 12);    
            tx_desc_mem(107 DOWNTO 106) <= "00";    
            tx_desc_mem(105 DOWNTO 96) <= tx_dwcnt_r(9 DOWNTO 0);    
            tx_desc_mem(95 DOWNTO 83) <= cfg_busdev;    
            tx_desc_mem(82 DOWNTO 80) <= "000";    
            tx_desc_mem(79 DOWNTO 74) <= "000000";    
            tx_desc_mem(73 DOWNTO 72) <= mst_tag;    
            tx_desc_mem(71 DOWNTO 68) <= tx_lbe_d;    
            tx_desc_mem(67 DOWNTO 64) <= tx_fbe_d;    
            IF ((mem_ctrl(5)) = '0') THEN
               --  tx_desc_mem[63:32] <= {mem_addr32[31:2], 2'b00} ;  //xhdl
               
               tx_desc_mem(63 DOWNTO 34) <= mem_addr32(31 DOWNTO 2);    
               tx_desc_mem(33 DOWNTO 32) <= "00";    
            ELSE
               tx_desc_mem(63 DOWNTO 32) <= mem_addr64;    
            END IF;
            --  tx_desc_mem[31:0] <= {mem_addr32[31:2], 2'b00} ;   //xhdl 
            
            tx_desc_mem(31 DOWNTO 2) <= mem_addr32(31 DOWNTO 2);    
            tx_desc_mem(1 DOWNTO 0) <= "00";    
         END IF;
         IF (mem_req = '0') THEN
            vc_r <= vc_sel;    
            cplh_r <= cred_cplh;    
            cpld_r <= cred_cpld;    
            dwcnt_msb <= tx_dwcnt_r(10);    
         END IF;
         tx_ack_r <= tx_ack;    
         IF (req_sm = req_idle AND (tx_ack_r = '1' OR mst_tag_vld = '0')) THEN
            IF (cpl_busy0 = '0') THEN
               mst_tag <= "00";    
            ELSE
               IF (cpl_busy1 = '0') THEN
                  mst_tag <= "01";    
               ELSE
                  IF (cpl_busy2 = '0') THEN
                     mst_tag <= "10";    
                  ELSE
                     IF (cpl_busy3 = '0') THEN
                        mst_tag <= "11";    
                     END IF;
                  END IF;
               END IF;
            END IF;
         END IF;
         IF (tx_ack_r = '1') THEN
            mst_tag_vld <= NOT cpl_busy_n;    
         END IF;
         IF (mem_req = '0') THEN
            IF ((mem_ctrl(6)) = '0' OR mem_ctrl(6 DOWNTO 0) = "1000010" OR mem_ctrl(6 DOWNTO 1) = "100010") THEN
               req_cpl_chnl <= '1';    
            ELSE
               req_cpl_chnl <= '0';    
            END IF;
         END IF;
         IF ((mem_req = '0' AND tx_dfr_r = '0') AND tx_dv_r = '0') THEN
            IF (vc_sel = "000") THEN
               vc0_out_xhdl8 <= '1';    
            ELSE
               vc0_out_xhdl8 <= '0';    
            END IF;
            IF (vc_sel = "001") THEN
               vc1_out_xhdl9 <= '1';    
            ELSE
               vc1_out_xhdl9 <= '0';    
            END IF;
         END IF;
      END IF;
   END PROCESS;
   tx_req_xhdl12 <= mem_req ;
   tx_desc_xhdl13 <= tx_desc_mem ;
   -- 13. Set the allocated completion credits in the RX buffer
   -----------------------------------------------------------------------------
   -- Header credit is allocated based on the maximum number of completions
   -- that the Completer can possibly break up the Read Requset into. Once the
   -- entire completion is done, all the credits are released. Each unit is 16 bytes
   -- as in PCI Express spec
   --
   -- Data credit is allocated based on the length in the Read request and
   -- is calculated in units of DW. As completions are returned regardless how
   -- it is broken up, the data credits are released based each completion's length
   -----------------------------------------------------------------------------
   ko_cpl_spc_vc0_19to8 <= ko_cpl_spc_vc0(19 DOWNTO 8) ;
   ko_cpl_spc_vc1_19to8 <= ko_cpl_spc_vc1(19 DOWNTO 8) ;
   temp_xhdl63 <= kod_cfg_maxrdreq WHEN (koh_cfg_maxrdreq > kod_cfg_maxrdreq) ELSE koh_cfg_maxrdreq;
   temp_xhdl64 <= cfg_maxrdreq WHEN (ko_cfg_maxrdreq > cfg_maxrdreq) ELSE ko_cfg_maxrdreq;
   PROCESS (hclk)
   BEGIN
      IF (hclk'EVENT AND hclk = '1') THEN
         -- 
         -- Set the max rd req size based on buffer allocation
         
         IF (std_match(ko_cpl_spc_vc0(7 DOWNTO 0), "1-------")) THEN 
                  koh_cfg_maxrdreq <= "101";    
         ELSIF (std_match(ko_cpl_spc_vc0(7 DOWNTO 0), "01------")) THEN 
                  koh_cfg_maxrdreq <= "101";    
         ELSIF (std_match(ko_cpl_spc_vc0(7 DOWNTO 0), "001-----")) THEN 
                  koh_cfg_maxrdreq <= "100";    
         ELSIF (std_match(ko_cpl_spc_vc0(7 DOWNTO 0), "0001----")) THEN 
                  koh_cfg_maxrdreq <= "011";    
         ELSIF (std_match(ko_cpl_spc_vc0(7 DOWNTO 0), "00001---")) THEN 
                  koh_cfg_maxrdreq <= "010";    
         ELSIF (std_match(ko_cpl_spc_vc0(7 DOWNTO 0), "000001--")) THEN 
                  koh_cfg_maxrdreq <= "001";    
         ELSE
                  koh_cfg_maxrdreq <= "000";    
         
         END IF;
         -- Set the max rd req size based on data buffer allocation
         
         IF (std_match(ko_cpl_spc_vc0(19 DOWNTO 12), "1-------")) THEN 
                  kod_cfg_maxrdreq <= "101";    
         ELSIF (std_match(ko_cpl_spc_vc0(19 DOWNTO 12), "01------")) THEN 
                  kod_cfg_maxrdreq <= "101";    
         ELSIF (std_match(ko_cpl_spc_vc0(19 DOWNTO 12), "001-----")) THEN 
                  kod_cfg_maxrdreq <= "101";    
         ELSIF (std_match(ko_cpl_spc_vc0(19 DOWNTO 12), "0001----")) THEN 
                  kod_cfg_maxrdreq <= "101";    
         ELSIF (std_match(ko_cpl_spc_vc0(19 DOWNTO 12), "00001---")) THEN 
                  kod_cfg_maxrdreq <= "100";    
         ELSIF (std_match(ko_cpl_spc_vc0(19 DOWNTO 12), "000001--")) THEN 
                  kod_cfg_maxrdreq <= "011";    
         ELSIF (std_match(ko_cpl_spc_vc0(19 DOWNTO 12), "0000001-")) THEN 
                  kod_cfg_maxrdreq <= "010";    
         ELSIF (std_match(ko_cpl_spc_vc0(19 DOWNTO 12), "00000001")) THEN 
                  kod_cfg_maxrdreq <= "001";    
         ELSE
                  kod_cfg_maxrdreq <= "000";    
         
         END IF;
         ko_cfg_maxrdreq <= temp_xhdl63;    
         cfg_maxrdreq_i <= temp_xhdl64;    
      END IF;
   END PROCESS;
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         init <= '0';    
         cur_cplh0 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cur_cpld0 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cur_cplh1 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cur_cpld1 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         init <= '1';    
         IF (init = '0') THEN
            cur_cplh0 <= ko_cpl_spc_vc0(7 DOWNTO 0);    
            --  cur_cpld0 <= {ko_cpl_spc_vc0[19:8], 2'b00} ;    // xhdl
            
            cur_cpld0 <= ko_cpl_spc_vc0_19to8 & "00";    
         ELSE
            IF ((tx_ack = '1' AND vc_r = "000") AND req_cpl_chnl = '1') THEN
               cur_cplh0 <= cur_cplh0 - cplh_r;    
               cur_cpld0 <= cur_cpld0 - cpld_r;    
            ELSE
               IF (cpl_hdrcred_rls = '1' AND cpl_vc = "000") THEN
                  cur_cpld0 <= cur_cpld0 + cpl_datacred;    
                  cur_cplh0 <= cur_cplh0 + cpl_hdrcred;    
               END IF;
            END IF;
         END IF;
         IF (init = '0') THEN
            cur_cplh1 <= ko_cpl_spc_vc1(7 DOWNTO 0);    
            --  cur_cpld1 <= {ko_cpl_spc_vc1[19:8], 2'b00} ; 
            
            cur_cpld1 <= ko_cpl_spc_vc1_19to8 & "00";    
         ELSE
            IF ((tx_ack = '1' AND vc_r = "001") AND req_cpl_chnl = '1') THEN
               cur_cplh1 <= cur_cplh1 - cplh_r;    
               cur_cpld1 <= cur_cpld1 - cpld_r;    
            ELSE
               IF (cpl_hdrcred_rls = '1' AND cpl_vc = "001") THEN
                  cur_cpld1 <= cur_cpld1 + cpl_datacred;    
                  cur_cplh1 <= cur_cplh1 + cpl_hdrcred;    
               END IF;
            END IF;
         END IF;
      END IF;
   END PROCESS;
   temp_xhdl65 <= '1' WHEN (((cpl_busy0 = '1' AND cpl_busy1 = '1') AND cpl_busy2 = '1') AND cpl_busy3 = '1') ELSE '0';
   cpl_busy_n <= temp_xhdl65 ;
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         cpl_busy <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         cpl_busy <= cpl_busy_n;    
      END IF;
   END PROCESS;
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         tim1ms <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         ev1ms <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF (testtim = '1' AND tim1ms = "00000000001000111") THEN
            -- test mode
            
            tim1ms <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         ELSE
            IF (tim1ms = "11110100001000111") THEN
               -- 125000 clock cycle at 8 ns clock period
               
               tim1ms <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
            ELSE
               tim1ms <= tim1ms + "00000000000000001";    
            END IF;
         END IF;
         IF (testtim = '1' AND tim1ms = "00000000001000111") THEN
            -- test mode
            
            tim1ms <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         ELSE
            IF (tim1ms = "11110100001000111") THEN
               -- 125000 clock cycle at 8 ns clock period
               
               ev1ms <= '1';    
            ELSE
               ev1ms <= '0';    
            END IF;
         END IF;
      END IF;
   END PROCESS;
   -- 15. Completion logic per channel for pipelined request
   -- Completion channel 0
   
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         cpl_busy0 <= '0';    
         hdrcred0 <= '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         datacred0 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cpl_dwcnt0 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cpl_addr0 <= (OTHERS => '0');    
         timer0 <= '0' & '0' & '0' & '0';    
         rx_tag0_hit <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF (rx_tag(1 DOWNTO 0) = "00") THEN
            rx_tag0_hit <= '1';    
         ELSE
            rx_tag0_hit <= '0';    
         END IF;
         IF ((mst_tag = "00" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            cpl_busy0 <= '1';    
         ELSE
            IF (cpl_dwcnt0 = "00000000000" OR timer0 = "0000") THEN
               cpl_busy0 <= '0';    
            END IF;
         END IF;
         IF (cpl_busy0 = '0') THEN
            timer0 <= "1010";    --  10 ms
         ELSE
            IF (ev1ms = '1') THEN
               timer0 <= timer0 - "0001";    
            END IF;
         END IF;
         IF ((mst_tag = "00" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            cpl_dwcnt0 <= dwcnt_msb & tx_desc_mem(105 DOWNTO 96);    
         ELSE
            IF (rx_ack_r = '1' AND rx_tag0_hit = '1') THEN
               cpl_dwcnt0 <= cpl_dwcnt0 - rx_dwcnt;    
            END IF;
         END IF;
         IF ((mst_tag = "00" AND ack_req = '1') AND req_cpl_chnl = '1') THEN
            cpl_addr0 <= cpl_addr_n;    
         ELSE
            IF (rx_ack_r = '1' AND rx_tag0_hit = '1') THEN
               -- cpl_addr0 <= cpl_addr0 + {1'b0,rx_dwcnt[RAM_SIZE-1:1]} ; 
               
               cpl_addr0 <= cpl_addr0 + rx_dwcnt_div2;    -- xhdl
            END IF;
         END IF;
         IF ((mst_tag = "00" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            hdrcred0 <= cplh_r;    
            datacred0 <= cpld_r;    
         END IF;
      END IF;
   END PROCESS;
   -- Completion channel 1
   
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         cpl_busy1 <= '0';    
         timer1 <= '0' & '0' & '0' & '0';    
         hdrcred1 <= '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         datacred1 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cpl_dwcnt1 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cpl_addr1 <= (OTHERS => '0');    
         rx_tag1_hit <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF (rx_tag(1 DOWNTO 0) = "01") THEN
            rx_tag1_hit <= '1';    
         ELSE
            rx_tag1_hit <= '0';    
         END IF;
         IF ((mst_tag = "01" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            cpl_busy1 <= '1';    
         ELSE
            IF (cpl_dwcnt1 = "00000000000" OR timer1 = "0000") THEN
               cpl_busy1 <= '0';    
            END IF;
         END IF;
         IF (cpl_busy1 = '0') THEN
            timer1 <= "1010";    --  10 ms
         ELSE
            IF (ev1ms = '1') THEN
               timer1 <= timer1 - "0001";    
            END IF;
         END IF;
         IF ((mst_tag = "01" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            cpl_dwcnt1 <= dwcnt_msb & tx_desc_mem(105 DOWNTO 96);    
         ELSE
            IF (rx_ack_r = '1' AND rx_tag1_hit = '1') THEN
               cpl_dwcnt1 <= cpl_dwcnt1 - rx_dwcnt;    
            END IF;
         END IF;
         IF ((mst_tag = "01" AND ack_req = '1') AND req_cpl_chnl = '1') THEN
            cpl_addr1 <= cpl_addr_n;    
         ELSE
            IF (rx_ack_r = '1' AND rx_tag1_hit = '1') THEN
               --  cpl_addr1 <= cpl_addr1 + {1'b0,rx_dwcnt[RAM_SIZE-1:1]} ; 
               
               cpl_addr1 <= cpl_addr1 + rx_dwcnt_div2;    -- xhdl
            END IF;
         END IF;
         IF ((mst_tag = "01" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            hdrcred1 <= cplh_r;    
            datacred1 <= cpld_r;    
         END IF;
      END IF;
   END PROCESS;
   -- Completion channel 2
   
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         cpl_busy2 <= '0';    
         timer2 <= '0' & '0' & '0' & '0';    
         hdrcred2 <= '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         datacred2 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cpl_dwcnt2 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cpl_addr2 <= (OTHERS => '0');    
         rx_tag2_hit <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF (rx_tag(1 DOWNTO 0) = "10") THEN
            rx_tag2_hit <= '1';    
         ELSE
            rx_tag2_hit <= '0';    
         END IF;
         IF ((mst_tag = "10" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            cpl_busy2 <= '1';    
         ELSE
            IF (cpl_dwcnt2 = "00000000000" OR timer2 = "0000") THEN
               cpl_busy2 <= '0';    
            END IF;
         END IF;
         IF (cpl_busy2 = '0') THEN
            timer2 <= "1010";    --  10 ms
         ELSE
            IF (ev1ms = '1') THEN
               timer2 <= timer2 - "0001";    
            END IF;
         END IF;
         IF ((mst_tag = "10" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            cpl_dwcnt2 <= dwcnt_msb & tx_desc_mem(105 DOWNTO 96);    
         ELSE
            IF (rx_ack_r = '1' AND rx_tag2_hit = '1') THEN
               cpl_dwcnt2 <= cpl_dwcnt2 - rx_dwcnt;    
            END IF;
         END IF;
         IF ((mst_tag = "10" AND ack_req = '1') AND req_cpl_chnl = '1') THEN
            cpl_addr2 <= cpl_addr_n;    
         ELSE
            IF (rx_ack_r = '1' AND rx_tag2_hit = '1') THEN
               -- cpl_addr2 <= cpl_addr2 + {1'b0,rx_dwcnt[RAM_SIZE-1:1]} ; 
               
               cpl_addr2 <= cpl_addr2 + rx_dwcnt_div2;    --  xhdl
            END IF;
         END IF;
         IF ((mst_tag = "10" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            hdrcred2 <= cplh_r;    
            datacred2 <= cpld_r;    
         END IF;
      END IF;
   END PROCESS;
   -- Completion channel 3
   
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         cpl_busy3 <= '0';    
         timer3 <= '0' & '0' & '0' & '0';    
         hdrcred3 <= '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         datacred3 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cpl_dwcnt3 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         cpl_addr3 <= (OTHERS => '0');    
         rx_tag3_hit <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF (rx_tag(1 DOWNTO 0) = "11") THEN
            rx_tag3_hit <= '1';    
         ELSE
            rx_tag3_hit <= '0';    
         END IF;
         IF ((mst_tag = "11" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            cpl_busy3 <= '1';    
         ELSE
            IF (cpl_dwcnt3 = "00000000000" OR timer3 = "0000") THEN
               cpl_busy3 <= '0';    
            END IF;
         END IF;
         IF (cpl_busy3 = '0') THEN
            timer3 <= "1010";    --  10 ms
         ELSE
            IF (ev1ms = '1') THEN
               timer3 <= timer3 - "0001";    
            END IF;
         END IF;
         IF ((mst_tag = "11" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            cpl_dwcnt3 <= dwcnt_msb & tx_desc_mem(105 DOWNTO 96);    
         ELSE
            IF (rx_ack_r = '1' AND rx_tag3_hit = '1') THEN
               cpl_dwcnt3 <= cpl_dwcnt3 - rx_dwcnt;    
            END IF;
         END IF;
         IF ((mst_tag = "11" AND ack_req = '1') AND req_cpl_chnl = '1') THEN
            cpl_addr3 <= cpl_addr_n;    
         ELSE
            IF (rx_ack_r = '1' AND rx_tag3_hit = '1') THEN
               -- cpl_addr3 <= cpl_addr3 + {1'b0,rx_dwcnt[RAM_SIZE-1:1]} ; 
               
               cpl_addr3 <= cpl_addr3 + rx_dwcnt_div2;    --  xhdl
            END IF;
         END IF;
         IF ((mst_tag = "11" AND tx_ack = '1') AND req_cpl_chnl = '1') THEN
            hdrcred3 <= cplh_r;    
            datacred3 <= cpld_r;    
         END IF;
      END IF;
   END PROCESS;
   -- 16. calculate the 'rx_cpld' and 'rx_cplh' of received completion     
   rx_tag <= rx_desc_r(47 DOWNTO 40) ;
   rx_addr7 <= rx_desc_r(38 DOWNTO 32) ;
   temp_xhdl66 <= '1' WHEN (rx_desc_r(105 DOWNTO 96) = "0000000000" AND (rx_desc_r(126)) = '1') ELSE '0';
   rx_dwcnt(10) <= temp_xhdl66 ;
   rx_dwcnt(9 DOWNTO 0) <= rx_desc_r(105 DOWNTO 96) ;
   rx_tc <= rx_desc_r(118 DOWNTO 116) ;
   temp_xhdl67 <= '1' WHEN (rx_desc_r(75 DOWNTO 64) = "000000000000") ELSE '0';
   rx_bytecnt(12) <= temp_xhdl67 ;
   rx_bytecnt(11 DOWNTO 0) <= rx_desc_r(75 DOWNTO 64) ;
   temp_xhdl68 <= '1' WHEN (rx_dwcnt(1 DOWNTO 0) /= "00") ELSE '0';
   rem_rxcpld <= temp_xhdl68 ;
   rx_cpld_n <= rx_dwcnt(10 DOWNTO 2) + rem_rxcpld ;
   temp_xhdl69 <= '1' WHEN (rx_dwcnt(4 DOWNTO 0) /= "00000" AND rcb = '1') ELSE '0';
   temp_xhdl70 <= '1' WHEN (rx_dwcnt(3 DOWNTO 0) /= "0000" AND rcb = '0') ELSE temp_xhdl69;
   rem_rxcplh <= temp_xhdl70 ;
   temp_xhdl71 <= rx_dwcnt(10 DOWNTO 4) WHEN (rcb = '0') ELSE '0' & rx_dwcnt(10 DOWNTO 5);
   bas_rxcplh <= temp_xhdl71 ;
   rx_cplh_n <= bas_rxcplh + rem_rxcplh ;
   -- assign rx_addr_dw = ({6'b000000, rx_desc_r[38:34]}) ; // xhdl
   rx_desc_r_38to34 <= rx_desc_r(38 DOWNTO 34) ;
   rx_addr_dw <= "000000" & rx_desc_r_38to34 ;
   rx_addrend_n <= rx_dwcnt + rx_addr_dw ;
   -- pipelining stage
   
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         rx_cplh <= '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         rx_cpld <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         rx_addrend <= '0' & '0' & '0' & '0' & '0' & '0';    
         cpl_vc <= "000";    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         rx_cplh <= rx_cplh_n;    
         rx_cpld <= rx_dwcnt(10 DOWNTO 0);    
         cpl_vc <= cpl_vc_n;    
         rx_addrend <= rx_addrend_n(10 DOWNTO 5);    
      END IF;
   END PROCESS;
   temp_xhdl72 <= cfg_tcvcmap(20 DOWNTO 18) WHEN (rx_tc = "110") ELSE cfg_tcvcmap(23 DOWNTO 21);
   temp_xhdl73 <= cfg_tcvcmap(17 DOWNTO 15) WHEN (rx_tc = "101") ELSE temp_xhdl72;
   temp_xhdl74 <= cfg_tcvcmap(14 DOWNTO 12) WHEN (rx_tc = "100") ELSE temp_xhdl73;
   temp_xhdl75 <= cfg_tcvcmap(11 DOWNTO 9) WHEN (rx_tc = "011") ELSE temp_xhdl74;
   temp_xhdl76 <= cfg_tcvcmap(8 DOWNTO 6) WHEN (rx_tc = "010") ELSE temp_xhdl75;
   temp_xhdl77 <= cfg_tcvcmap(5 DOWNTO 3) WHEN (rx_tc = "001") ELSE temp_xhdl76;
   temp_xhdl78 <= cfg_tcvcmap(2 DOWNTO 0) WHEN (rx_tc = "000") ELSE temp_xhdl77;
   cpl_vc_n <= temp_xhdl78 ;
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         cpl_sel <= "0000";    
         cpl_ack <= '0';    
         cpl_ack_r <= '0';    
         cpl_hdrcred_rls <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF (cpl_sm = cpl_rxdesc) THEN
            IF (rx_tag0_hit = '1') THEN
               cpl_sel <= "0001";    
            ELSE
               IF (rx_tag1_hit = '1') THEN
                  cpl_sel <= "0010";    
               ELSE
                  IF (rx_tag2_hit = '1') THEN
                     cpl_sel <= "0100";    
                  ELSE
                     IF (rx_tag3_hit = '1') THEN
                        cpl_sel <= "1000";    
                     ELSE
                        cpl_sel <= "0000";    
                     END IF;
                  END IF;
               END IF;
            END IF;
         END IF;
         IF (rx_ack_r = '1' AND is_rxmst = '1') THEN
            cpl_ack <= '1';    
         ELSE
            cpl_ack <= '0';    
            cpl_ack_r <= cpl_ack;    
            -- release header credit when all completions are received
            --   if (cpl_ack_r & ((cpl_sel & ~({cpl_busy3, cpl_busy2, cpl_busy1, cpl_busy0})) != 4'b0000)) 
            
            IF ((cpl_ack_r AND or_br(cpl_sel AND NOT (cpl_busy3 & cpl_busy2 & cpl_busy1 & cpl_busy0))) = '1') THEN
               -- XHDL
               
               cpl_hdrcred_rls <= '1';    
            ELSE
               cpl_hdrcred_rls <= '0';    
            END IF;
         END IF;
      END IF;
   END PROCESS;
   temp_xhdl79 <= hdrcred3 WHEN ((cpl_sel(3)) = '1') ELSE '0' & '0' & '0' & '0' & '0' & '0' & '0';
   temp_xhdl80 <= hdrcred2 WHEN ((cpl_sel(2)) = '1') ELSE temp_xhdl79;
   temp_xhdl81 <= hdrcred1 WHEN ((cpl_sel(1)) = '1') ELSE temp_xhdl80;
   temp_xhdl82 <= hdrcred0 WHEN ((cpl_sel(0)) = '1') ELSE temp_xhdl81;
   cpl_hdrcred <= temp_xhdl82 ;
   temp_xhdl83 <= datacred3 WHEN ((cpl_sel(3)) = '1') ELSE '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';
   temp_xhdl84 <= datacred2 WHEN ((cpl_sel(2)) = '1') ELSE temp_xhdl83;
   temp_xhdl85 <= datacred1 WHEN ((cpl_sel(1)) = '1') ELSE temp_xhdl84;
   temp_xhdl86 <= datacred0 WHEN ((cpl_sel(0)) = '1') ELSE temp_xhdl85;
   cpl_datacred <= temp_xhdl86 ;
   temp_xhdl87 <= '1' WHEN ((tx_desc_mem(126)) = '1') ELSE '0';
   is_txwr <= temp_xhdl87 ;
   temp_xhdl88 <= '1' WHEN (((rd_allow = '1') AND (phasecnt = "0000000010")) OR ((rd_allow = '0') AND (phasecnt = "0000000001"))) ELSE '0';
   temp_xhdl89 <= '1' WHEN (((rd_allow = '1') AND (phasecnt = "0000000001")) OR (phasecnt = "0000000000")) ELSE '0';
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         req_sm <= req_idle;    
         tx_dfr_r <= '0';    
         tx_dv_r <= '0';    
         txaddr <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         need_to_fetch <= '0';    
         pipe_fill <= '0';    
         fetch_cnt <= '0' & '0';    
         phasecnt <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         tx_data_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         fetch_first <= '0';    
         mem_first <= '0';    
         mem_first_del <= '0';    
         data_ramp15 <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         nextcyc_is_last_xhdl18 <= '0';    
         last_datacyc_xhdl19 <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         mem_first_del <= mem_first;    
         CASE req_sm IS
            WHEN req_idle =>
                     IF ((((((mem_req = '0' AND mem_req_r = '0') AND mem_new = '0') AND en_req_r = '1') AND mem_desc = '1') AND cpl_busy = '0') AND desc_setup_cnt = "100") THEN
                        req_sm <= req_txdesc;    
                     END IF;
            WHEN req_txdesc =>
                     IF (tx_ack = '1') THEN
                        IF (is_txwr = '0' OR (rd_out = '1' AND phasecnt = "0000000000")) THEN
                           req_sm <= req_idle;    
                        ELSE
                           req_sm <= req_txdata;    
                        END IF;
                     END IF;
            WHEN req_txdata =>
                     IF ((tx_ws = '0') AND (phasecnt = "0000000000")) THEN
                        -- xhdl
                        
                        req_sm <= req_idle;    
                     END IF;
            WHEN OTHERS =>
                     NULL;
            
         END CASE;
         -- Memorize read request parameter
         -- When DMA is setup, target address is latch
         -- The address is incremented for the rest of the DMA
         
         IF (mem_new = '1' AND desc_setup_cnt = "010") THEN
            txaddr <= mem_tgt_adr;    
            need_to_fetch <= '1';    
            fetch_cnt <= "00";    
         ELSE
            IF (need_to_fetch = '1' AND fetch_cnt /= "10") THEN
               txaddr <= txaddr + "0000000000001";    
               fetch_cnt <= fetch_cnt + "01";    
            ELSE
               IF (need_to_fetch = '1') THEN
                  -- pipe is filled
                  
                  txaddr <= txaddr + "0000000000001";    
                  need_to_fetch <= '0';    
               ELSE
                  IF (((NOT tx_ws AND tx_dv_r) OR fetch_tx_data) = '1') THEN
                     -- increment address when data is consumed
                     
                     txaddr <= txaddr + "0000000000001";    
                  END IF;
               END IF;
            END IF;
         END IF;
         IF ((((((((req_sm = req_idle AND mem_req = '0') AND mem_req_r = '0') AND mem_new = '0') AND en_req_r = '1') AND mem_desc = '1') AND cpl_busy = '0') AND desc_setup_cnt = "100") AND is_txwr = '1') THEN
            fetch_first <= '1';    
         ELSE
            fetch_first <= '0';    
         END IF;
         mem_first <= fetch_first;    
         nextcyc_is_last_xhdl18 <= temp_xhdl88;    
         last_datacyc_xhdl19 <= temp_xhdl89;    
         IF ((((((((req_sm = req_idle AND mem_req = '0') AND mem_req_r = '0') AND mem_new = '0') AND en_req_r = '1') AND mem_desc = '1') AND desc_setup_cnt = "100") AND cpl_busy = '0') AND is_txwr = '1') THEN
            -- phasecnt includes prefetch from memory
            -- Pop this many times per completion
            
            IF ((tx_dwcnt_r(0)) = '1' OR (mem_addr32(2)) = '1') THEN
               phasecnt <= tx_dwcnt_r(10 DOWNTO 1) + "0000000001";    
            ELSE
               phasecnt <= tx_dwcnt_r(10 DOWNTO 1);    
            END IF;
         ELSE
            IF (rd_allow = '1') THEN
               phasecnt <= phasecnt - "0000000001";    
            END IF;
         END IF;
         IF (req_sm = req_txdesc AND (tx_desc_mem(126)) = '1') THEN
            --  tx_dfr is deasserted before the last data phase
            
            IF ((phasecnt = "0000000001" AND (mem_first = '1' OR rd_out = '1')) OR phasecnt = "0000000000") THEN
               tx_dfr_r <= '0';    
            ELSE
               tx_dfr_r <= '1';    
            END IF;
         ELSE
            IF (rd_out = '1' AND (phasecnt = "0000000001" OR phasecnt = "0000000000")) THEN
               tx_dfr_r <= '0';    
            END IF;
         END IF;
         IF (req_sm = req_txdesc) THEN
            IF (((tx_ws = '0') OR (fetch_tx_data = '1')) AND (phasecnt = "0000000000")) THEN
               --xhdl
               
               tx_dv_r <= '0';    
            ELSE
               IF (tx_dfr_r = '1') THEN
                  tx_dv_r <= '1';    
               END IF;
            END IF;
         ELSE
            IF (req_sm = req_txdata) THEN
               IF (((tx_ws = '0') OR (fetch_tx_data = '1')) AND (phasecnt = "0000000000")) THEN
                  --xhdl
                  
                  tx_dv_r <= '0';    
               ELSE
                  tx_dv_r <= tx_dv_r;    
               END IF;
            ELSE
               IF (req_sm = req_idle) THEN
                  tx_dv_r <= '0';    
               END IF;
            END IF;
         END IF;
         IF (mem_desc = '0' AND tx_dfr_r = '0') THEN
            data_ramp15 <= "000000000000000";    
         ELSE
            IF ((rd_allow = '1') AND tx_dfr_r = '1') THEN
               data_ramp15 <= data_ramp15 + "000000000000001";    
            END IF;
         END IF;
         IF (mem_zero = '1') THEN
            tx_data_r <= '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0' & '0';    
         ELSE
            IF (mem_ramp1 = '1' OR mem_ramp4 = '1') THEN
               tx_data_r <= data_ramp;    
            ELSE
               IF (rdclken = '1') THEN
                  tx_data_r <= dataout;    
               END IF;
            END IF;
         END IF;
      END IF;
   END PROCESS;
   data_ramp15_4to0 <= data_ramp15(4 DOWNTO 0) ;
   -- xhdl
   data_ramp_memramp1 <= data_ramp15_4to0 & "111" & data_ramp15_4to0 & "110" & data_ramp15_4to0 & "101" & data_ramp15_4to0 & "100" & data_ramp15_4to0 & "011" & data_ramp15_4to0 & "010" & data_ramp15_4to0 & "001" & data_ramp15_4to0 & "000" ;
   data_ramp_memramp2 <= "0000000000000000" & data_ramp15 & '1' & "0000000000000000" & data_ramp15 & '0' ;
   temp_xhdl90 <= data_ramp_memramp1 WHEN (mem_ramp1 = '1') ELSE data_ramp_memramp2;
   data_ramp <= temp_xhdl90 ;
   temp_xhdl91 <= '0' WHEN (req_sm = req_idle) ELSE '1';
   tx_busy_xhdl7 <= temp_xhdl91 ;
   tx_dv_xhdl16 <= tx_dv_r ;
   tx_dfr_xhdl14 <= tx_dfr_r ;
   tx_err_xhdl17 <= '0' ;
   tx_adrd <= txaddr ;
   -- for DPRAM uninitialized value
   
   xhdl_73 : PROCESS (tx_data_r)
      VARIABLE i                      :  integer;   
   BEGIN
      --   integer i;   //xhdl
      
      FOR i IN 0 TO 63 LOOP
         IF ((tx_data_r(i)) = '1') THEN
            tx_data_xhdl15(i) <= '1';    
         ELSE
            tx_data_xhdl15(i) <= '0';    
         END IF;
      END LOOP;
   END PROCESS xhdl_73;
   -- data phase acknowledge
   rd_out <= tx_dv_r AND NOT tx_ws ;
   -- read operation on DPRAM (including fetch)
   -- assign rd_allow = mem_first | rd_out ;
   rd_allow <= mem_first OR rd_out OR fetch_tx_data ;
   temp_xhdl92 <= '1' WHEN (need_to_fetch OR (NOT tx_ws AND tx_dv_r) OR fetch_tx_data) = '1' ELSE '0';
   rdclken <= temp_xhdl92 ;
   temp_xhdl93 <= '1' WHEN ((rx_desc_r(126)) = '1') ELSE '0';
   is_rxwr <= temp_xhdl93 ;
   temp_xhdl94 <= '1' WHEN (rx_desc_r(125 DOWNTO 121) = "00101") ELSE '0';
   is_rxmst <= temp_xhdl94 ;
   temp_xhdl95 <= '1' WHEN (rx_desc_r(127 DOWNTO 126) = "01") ELSE '0';
   is_rxdval <= temp_xhdl95 ;
   PROCESS (hclk)
   BEGIN
      IF (hclk'EVENT AND hclk = '1') THEN
         -- 
         
         rx_ack_rr <= rx_ack_r;    
      END IF;
   END PROCESS;
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         cpl_sm <= cpl_idl;    
         rx_ack_r <= '0';    
         rx_ws_xhdl11 <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         CASE cpl_sm IS
            WHEN cpl_idl =>
                     IF (rx_req_r = '1') THEN
                        IF (is_rxmst = '1') THEN
                           cpl_sm <= cpl_rxdesc;    
                           rx_ack_r <= '1';    
                           rx_ws_xhdl11 <= '1';    
                        END IF;
                     END IF;
            WHEN cpl_rxdesc =>
                     -- app receives rx_ack_r and deasserts rx_ws
                     -- in this state for one clock
                     
                     rx_ack_r <= '0';    
                     rx_ws_xhdl11 <= '0';    
                     IF (is_rxdval = '1') THEN
                        cpl_sm <= cpl_rxdata;    
                     ELSE
                        cpl_sm <= cpl_idl;    
                     END IF;
            WHEN cpl_rxdata =>
                     -- if (rx_dfr == 1'b0 & rx_dv == 1'b1)
                     
                     IF (rx_st_eop = '1') THEN
                        cpl_sm <= cpl_idl;    
                        rx_ws_xhdl11 <= '0';    
                     END IF;
            WHEN OTHERS =>
                     NULL;
            
         END CASE;
      END IF;
   END PROCESS;
   rx_ack_xhdl10 <= rx_ack_r ;
   temp_xhdl96 <= '0' WHEN (cpl_sm = cpl_idl) ELSE '1';
   rx_busy_xhdl6 <= temp_xhdl96 ;
   ---------------------------------------------------------------
   -- target write control for DPRAM interface                      
   ---------------------------------------------------------------
   
   PROCESS (hrstn, hclk)
   BEGIN
      IF (hrstn = '0') THEN
         rx_addr <= (OTHERS => '0');    
         rx_wr <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF (cpl_sm = cpl_idl) THEN
            rx_wr <= is_rxwr;    
         END IF;
         -- latch target memory address on DMA setup
         
         IF (cpl_sm = cpl_rxdesc) THEN
            -- rx_tag_hit are one-hot
            
            IF (rx_tag0_hit = '1') THEN
               rx_addr <= cpl_addr0;    
            END IF;
            IF (rx_tag1_hit = '1') THEN
               rx_addr <= cpl_addr1;    
            END IF;
            IF (rx_tag2_hit = '1') THEN
               rx_addr <= cpl_addr2;    
            END IF;
            IF (rx_tag3_hit = '1') THEN
               rx_addr <= cpl_addr3;    
            END IF;
         ELSE
            IF (wr_allow = '1') THEN
               rx_addr <= rx_addr + "0000000001";    
            END IF;
         END IF;
      END IF;
   END PROCESS;
   rx_adwr <= rx_addr ;
   temp_xhdl97 <= '1' WHEN ((rx_wr = '1' AND cpl_sm = cpl_rxdata) AND rx_dv = '1') ELSE '0';
   wr_allow <= temp_xhdl97 ;
   --   assign wr_allow = (rx_wr == 1'b1 & !cpl_sm == cpl_idl & rx_dv == 
   --/   1'b1) ? 1'b1 : 1'b0 ;                                       
   datain <= rx_data ;
   temp_xhdl98 <= rx_be WHEN (wr_allow = '1') ELSE "00000000";
   wren <= temp_xhdl98 ;
   -- dprambe write port pipeline stages
   -- for performance.
   
   PROCESS (hclk, hrstn)
   BEGIN
      IF (NOT hrstn = '1') THEN
         rx_adwr_del <= (OTHERS => '0');    
         datain_del <= "0000000000000000000000000000000000000000000000000000000000000000";    
         wren_del <= "00000000";    
         rx_adwr_del2 <= (OTHERS => '0');    
         datain_del2 <= "0000000000000000000000000000000000000000000000000000000000000000";    
         wren_del2 <= "00000000";    
         rx_adwr_del3 <= (OTHERS => '0');    
         datain_del3 <= "0000000000000000000000000000000000000000000000000000000000000000";    
         wren_del3 <= "00000000";    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         rx_adwr_del2 <= rx_adwr;    
         datain_del2 <= datain;    
         wren_del2 <= wren;    
         rx_adwr_del <= rx_adwr_del3;    
         datain_del <= datain_del3;    
         wren_del <= wren_del3;    
         rx_adwr_del3 <= rx_adwr_del2;    
         datain_del3 <= datain_del2;    
         wren_del3 <= wren_del2;    
      END IF;
   END PROCESS;
   port_xhdl100 <= rx_adwr_del(RAM_SIZE - 1 DOWNTO 0);
   port_xhdl101 <= tx_adrd(RAM_SIZE - 1 DOWNTO 0);
   dprambe : altpcierd_dprambe 
      GENERIC MAP (
         ADDR_WIDTH => RAM_SIZE,
         DATA_WIDTH => 64,
         BYTE_SIZE => 8)
      PORT MAP (
         wrclk => hclk,
         wraddr => port_xhdl100,
         wrdata => datain_del,
         wrbe => wren_del,
         rdclk => hclk,
         rdclken => rdclken,
         rdaddr => port_xhdl101,
         rddata => dataout);   
   
   -- pending transaction
   
   PROCESS (hclk, hrstn)
   BEGIN
      IF (hrstn = '0') THEN
         trans_pending_xhdl1 <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         IF (cpl_busy0 = '1' OR cpl_busy1 = '1' OR cpl_busy2 = '1' OR cpl_busy3 = '1') THEN
            trans_pending_xhdl1 <= '1';    
         ELSE
            trans_pending_xhdl1 <= '0';    
         END IF;
      END IF;
   END PROCESS;
   temp_xhdl102 <= '1' WHEN (cpl_busy0 = '1' AND timer0 = "0000") ELSE '0';
   err_tim0 <= temp_xhdl102 ;
   temp_xhdl103 <= '1' WHEN (cpl_busy1 = '1' AND timer1 = "0000") ELSE '0';
   err_tim1 <= temp_xhdl103 ;
   temp_xhdl104 <= '1' WHEN (cpl_busy2 = '1' AND timer2 = "0000") ELSE '0';
   err_tim2 <= temp_xhdl104 ;
   temp_xhdl105 <= '1' WHEN (cpl_busy3 = '1' AND timer3 = "0000") ELSE '0';
   err_tim3 <= temp_xhdl105 ;
   PROCESS (hclk, hrstn)
   BEGIN
      IF (hrstn = '0') THEN
         err_cpltim_xhdl2 <= '0';    
         err_cplunexp_xhdl3 <= '0';    
      ELSIF (hclk'EVENT AND hclk = '1') THEN
         err_cpltim_xhdl2 <= err_tim0 OR err_tim1 OR err_tim2 OR err_tim3;    
         IF (rx_ack_rr = '1' AND cpl_sel = "0000") THEN
            err_cplunexp_xhdl3 <= '1';    
         ELSE
            err_cplunexp_xhdl3 <= '0';    
         END IF;
      END IF;
   END PROCESS;
END ARCHITECTURE translated;
