LIBRARY ieee;                          
use IEEE.std_logic_1164.all;           
use IEEE.std_logic_arith.all;          
use IEEE.std_logic_unsigned.all;       
                                       
LIBRARY altera_mf;                     
USE altera_mf.altera_mf_components.all;
                                       
LIBRARY ieee;
   USE ieee.std_logic_1164.all;
-- /**
-- *
-- *  This Verilog HDL File is used for simulation and synthesis in the simple DMA design example.  
-- *  It provides master and slave arbitration on the transmit interface for each VC.
-- *
-- */
-------------------------------------------------------------------------------
-- Title         : PCI Express Reference Design apptxchan Entity
-- Project       : PCI Express MegaCore function
-------------------------------------------------------------------------------
-- File          : altpcierd_apptx_streaming_intf.v
-- Author        : Altera Corporation
-------------------------------------------------------------------------------
-- Description :
-- This module arbitrates master and slave transmission requests for the 
-- streaming interface.  It also generates signals for the streaming interface.
-------------------------------------------------------------------------------
-- Copyright (c) 2006 Altera Corporation. All rights reserved.  Altera products are
-- protected under numerous U.S. and foreign patents, maskwork rights, copyrights and
-- other intellectual property laws.  
--
-- This reference design file, and your use thereof, is subject to and governed by
-- the terms and conditions of the applicable Altera Reference Design License Agreement.
-- By using this reference design file, you indicate your acceptance of such terms and
-- conditions between you and Altera Corporation.  In the event that you do not agree with
-- such terms and conditions, you may not use the reference design file. Please promptly
-- destroy any copies you have made.
--
-- This reference design file being provided on an "as-is" basis and as an accommodation 
-- and therefore all warranties, representations or guarantees of any kind 
-- (whether express, implied or statutory) including, without limitation, warranties of 
-- merchantability, non-infringement, or fitness for a particular purpose, are 
-- specifically disclaimed.  By making this reference design file available, Altera
-- expressly does not recommend, suggest or require that this reference design file be
-- used in combination with any other product not provided by Altera. 
-------------------------------------------------------------------------------
ENTITY altpcierd_apptx_streaming_intf IS
   GENERIC (
      TL_SELECTION    : INTEGER := 6        -- streaming interface format selection.
   );
   PORT (
      -- 0 = ICM format.  6 = 7.2 format.
      
      clk_in          : IN STD_LOGIC;
      rstn            : IN STD_LOGIC;
      m_req           : IN STD_LOGIC;       -- tx request from master 
      -- indicates that current cycle is last cycle of packet
      -- master descriptor
      -- master payload data
      -- master tx_err bit 
      -- tx request from slave  
      -- indicates that current cycle is last cycle of packet
      -- slave descriptor 
      -- slave payload data
      -- slave tx_err bit 
      -- completion pending status flag
      -- streaming i/f throttles data fetching.  when set, allows another data fetch.
      
      m_ack           : OUT STD_LOGIC;      -- grant access to master request
      -- grant access to slave request   
      m_data_ack      : OUT STD_LOGIC;      -- fetch next desc/data from master
      m_tx_desc       : IN STD_LOGIC_VECTOR(127 DOWNTO 0);
      m_tx_data       : IN STD_LOGIC_VECTOR(63 DOWNTO 0);
      m_last_datacyc  : IN STD_LOGIC;
      s_req           : IN STD_LOGIC;
      s_ack           : OUT STD_LOGIC;
      s_data_ack      : OUT STD_LOGIC;      -- fetch next desc/data from slave  
      s_tx_desc       : IN STD_LOGIC_VECTOR(127 DOWNTO 0);
      s_tx_data       : IN STD_LOGIC_VECTOR(63 DOWNTO 0);
      s_last_datacyc  : IN STD_LOGIC;
      stream_ready    : IN STD_LOGIC;
      stream_wrdata   : OUT STD_LOGIC_VECTOR(107 DOWNTO 0);     -- data written to streaming i/f
      stream_wr       : OUT STD_LOGIC;      -- writes stream_wrdata to streaming i/f 
      cpl_pending     : IN STD_LOGIC;
      m_tx_err        : IN STD_LOGIC;
      s_tx_err        : IN STD_LOGIC
   );
END ENTITY altpcierd_apptx_streaming_intf;
ARCHITECTURE altpcie OF altpcierd_apptx_streaming_intf IS


                                                
   FUNCTION to_stdlogic (                        
   val      : IN integer) RETURN std_logic IS    
   BEGIN                                         
      IF (val=1) THEN                            
         RETURN('1');                            
      ELSE                                       
         RETURN('0');                            
      END IF;                                    
   END to_stdlogic;                              
   FUNCTION to_stdlogic (                        
   val      : IN boolean) RETURN std_logic IS    
   BEGIN                                         
      IF (val) THEN                              
         RETURN('1');                            
      ELSE                                       
         RETURN('0');                            
      END IF;                                    
   END to_stdlogic;                              
                                                
                                              
   FUNCTION TO_STDLOGICVECTOR (                                        
      val      : IN integer;                                           
      len      : IN integer) RETURN std_logic_vector IS                
                                                                       
      VARIABLE rtn : std_logic_vector(len-1 DOWNTO 0):=(OTHERS => '0');
      VARIABLE num : integer := val;                                   
      VARIABLE r   : integer;                                          
   BEGIN                                                               
      FOR index IN 0 TO len-1 LOOP                                     
         r := num rem 2;                                               
         num := num/2;                                                 
         IF (r = 1) THEN                                               
            rtn(index) := '1';                                         
         ELSE                                                          
            rtn(index) := '0';                                         
         END IF;                                                       
      END LOOP;                                                        
      RETURN(rtn);                                                     
   END TO_STDLOGICVECTOR;                                              
                                                                       
                                                                       
   
   -- tx_state SM states
   CONSTANT TX_WAIT_REQ           : INTEGER := 0;       -- arbitrate between master and slave, write desc_hi
   CONSTANT TX_SEND_DESC2         : INTEGER := 1;       -- write desc_lo, fetch data
   CONSTANT TX_NULL_CYCLE         : INTEGER := 2;       -- data-less pkt.  send ack only.
   CONSTANT TX_SEND_DATA          : INTEGER := 3;       -- write data, fetch data
   

   SIGNAL stream_wrdata_sop       : STD_LOGIC;      -- part of stream_wrdata vector.  Start of Pkt flag.   
   SIGNAL stream_wrdata_eop       : STD_LOGIC;      -- part of stream_wrdata vector.  End of Pkt flag. 
   SIGNAL stream_wrdata_tx_err    : STD_LOGIC;      -- part of stream_wrdata vector.  tx_err bit. 
   SIGNAL stream_wrdata_txdata    : STD_LOGIC_VECTOR(63 DOWNTO 0);      -- part of stream_wrdata vector.  tx_desc/data.  
   
   SIGNAL tx_state                : STD_LOGIC_VECTOR(1 DOWNTO 0);       -- state machine
   SIGNAL stream_ready_del        : STD_LOGIC;
   SIGNAL sel_master              : STD_LOGIC;      -- indicate that master has access
   SIGNAL sel_slave               : STD_LOGIC;      -- indicate that slave has access
   SIGNAL m_tx_data_hold_r        : STD_LOGIC_VECTOR(63 DOWNTO 0);      -- last value of m_tx_data
   SIGNAL s_tx_data_hold_r        : STD_LOGIC_VECTOR(63 DOWNTO 0);      -- last value of s_tx_data
   SIGNAL m_stream_pkt_end_hold_r : STD_LOGIC;      -- last value of m_last_datacyc
   SIGNAL s_stream_pkt_end_hold_r : STD_LOGIC;      -- last value of s_last_datacyc 
   SIGNAL has_payload             : STD_LOGIC;      -- indicates that the pkt has payload
   SIGNAL data_ack_cycle          : STD_LOGIC;      -- indicates that fetched data is ready at input.  
   SIGNAL pkt_ended               : STD_LOGIC;      -- indicates that the last data cycle has been received.  finish transporting the data, and do not fetch more until ready to service a new pkt.
   SIGNAL tx_desc_hold_lo         : STD_LOGIC_VECTOR(63 DOWNTO 0);      -- memorize the low 8 bytes of the descriptor
   SIGNAL last_was_slave          : STD_LOGIC;      -- means that the last tx_req service was the slave (- master gets priority)
   
   SIGNAL pack_dw                 : STD_LOGIC;      -- pack desc & data when 3DW header, with payload, and address not aligned.
   
   SIGNAL zero_33                 : STD_LOGIC_VECTOR(32 DOWNTO 0);
   SIGNAL zero_8                  : STD_LOGIC_VECTOR(7 DOWNTO 0);
   
   -- Declare intermediate signals for referenced outputs
   SIGNAL m_data_ack_xhdl0        : STD_LOGIC;
   SIGNAL s_data_ack_xhdl1        : STD_LOGIC;
BEGIN
   -- Drive referenced outputs
   m_data_ack <= m_data_ack_xhdl0;
   s_data_ack <= s_data_ack_xhdl1;
   zero_33 <= "000000000000000000000000000000000";
   zero_8 <= "00000000";
   
   stream_wrdata <= (zero_33 & stream_wrdata_tx_err & stream_wrdata_sop & stream_wrdata_eop & zero_8 & stream_wrdata_txdata);
   
   --------------------------------------------------------------
   --    Streaming Interface - Input register boundary
   --------------------------------------------------------------
   
   PROCESS (rstn, clk_in)
   BEGIN
      IF (rstn = '0') THEN
         stream_ready_del <= '0';
      ELSIF (clk_in'EVENT AND clk_in = '1') THEN
         stream_ready_del <= stream_ready;
      END IF;
   END PROCESS;
   
   --------------------------------------------------------------
   --    Arbitration between master and target for transmission.
   --    Multiplex Descriptor and Data onto Streaming data bus.
   --    Generate Streaming interface SOP/EOP flags.
   --------------------------------------------------------------
   
   PROCESS (rstn, clk_in)
   BEGIN
      IF (rstn = '0') THEN
         tx_state <= to_stdlogicvector(TX_WAIT_REQ, 2);
         m_ack <= '0';
         s_ack <= '0';
         sel_master <= '0';
         sel_slave <= '0';
         m_data_ack_xhdl0 <= '0';
         s_data_ack_xhdl1 <= '0';
         stream_wrdata_sop <= '0';
         stream_wrdata_eop <= '0';
         stream_wrdata_tx_err <= '0';
         stream_wrdata_txdata <= "0000000000000000000000000000000000000000000000000000000000000000";
         stream_wr <= '0';
         data_ack_cycle <= '0';
         m_tx_data_hold_r <= "0000000000000000000000000000000000000000000000000000000000000000";
         s_tx_data_hold_r <= "0000000000000000000000000000000000000000000000000000000000000000";
         m_stream_pkt_end_hold_r <= '0';
         s_stream_pkt_end_hold_r <= '0';
         has_payload <= '0';
         pkt_ended <= '0';
         tx_desc_hold_lo <= "0000000000000000000000000000000000000000000000000000000000000000";
         last_was_slave <= '0';
         pack_dw <= '0';
      ELSIF (clk_in'EVENT AND clk_in = '1') THEN
         --defaults 
         m_ack <= '0';
         s_ack <= '0';
         m_data_ack_xhdl0 <= '0';
         s_data_ack_xhdl1 <= '0';
         stream_wr <= '0';
         stream_wrdata_sop <= '0';
         stream_wrdata_eop <= '0';      -- when dataphase present, m_data_ack and s_data_ack assert on same cycle as data_ack_cycle
         data_ack_cycle <= stream_ready_del;
         
         pkt_ended <= '0';
         IF (sel_master = '1') THEN
            stream_wrdata_tx_err <= m_tx_err;
         ELSE
            
            -- hold last value of master/slave data 
            stream_wrdata_tx_err <= s_tx_err;
         END IF;
         IF (data_ack_cycle = '1') THEN
            m_tx_data_hold_r <= m_tx_data;
            s_tx_data_hold_r <= s_tx_data;
            m_stream_pkt_end_hold_r <= m_last_datacyc;
            s_stream_pkt_end_hold_r <= s_last_datacyc;
         ELSE
            m_tx_data_hold_r <= m_tx_data_hold_r;
            s_tx_data_hold_r <= s_tx_data_hold_r;
            m_stream_pkt_end_hold_r <= m_stream_pkt_end_hold_r;
            s_stream_pkt_end_hold_r <= s_stream_pkt_end_hold_r;
         END IF;
         
         -- issue ack (tx request grant) and ready (data throttling) on stream_ready_del.
         -- data is valid one clock cycle later (data_ack_cycle) -- do all data evaluations on this cycle. 
         CASE (tx_state) IS
            -- arbitrate tx request from master and slave
            -- when streaming i/f can accept more data (stream_ready_del):
            --    - write desc_hi to streaming i/f data bus
            --    - fetch desc_lo 
            -- grant master if last req was from slave or 
            -- if slave is not requesting access.
            -- ADAPTOR DWORD SWAP 
            -- has payld, 3DW hder, nonaligned address
            -- grant slave if master is not requesting access or 
            -- if last request serviced was from master.
            -- ADAPTOR DWORD SWAP
            -- has payld, 3DW hder, nonaligned address
            
            -- write sop flag
            -- write data and flag to streaming i/f
            -- TX_WAIT_REQ
            WHEN "00" =>
               IF ((m_req AND stream_ready_del AND (last_was_slave OR NOT(s_req))) = '1') THEN
                  sel_master <= '1';
                  m_ack <= '1';
                  m_data_ack_xhdl0 <= '1';
                  has_payload <= m_tx_desc(126);
                  IF (TL_SELECTION = 0) THEN
                     stream_wrdata_txdata <= m_tx_desc(127 DOWNTO 64);
                     tx_desc_hold_lo <= m_tx_desc(63 DOWNTO 0);
                     pack_dw <= '0';
                  ELSE
                     tx_desc_hold_lo <= m_tx_desc(63 DOWNTO 0);
                     stream_wrdata_txdata <= (m_tx_desc(95 DOWNTO 64) & m_tx_desc(127 DOWNTO 96));
                     pack_dw <= m_tx_desc(126) AND NOT(m_tx_desc(125)) AND m_tx_desc(34);
                  END IF;
               ELSIF ((s_req AND stream_ready_del) = '1') THEN
                  sel_slave <= '1';
                  s_ack <= '1';
                  s_data_ack_xhdl1 <= '1';
                  has_payload <= s_tx_desc(126);
                  IF (TL_SELECTION = 0) THEN
                     stream_wrdata_txdata <= s_tx_desc(127 DOWNTO 64);
                     tx_desc_hold_lo <= s_tx_desc(63 DOWNTO 0);
                     pack_dw <= '0';
                  ELSE
                     stream_wrdata_txdata <= (s_tx_desc(95 DOWNTO 64) & s_tx_desc(127 DOWNTO 96));
                     tx_desc_hold_lo <= s_tx_desc(63 DOWNTO 0);
                     pack_dw <= s_tx_desc(126) AND NOT(s_tx_desc(125)) AND s_tx_desc(34);
                  END IF;
               END IF;
               IF (((m_req OR s_req) AND stream_ready_del) = '1') THEN
                  tx_state <= to_stdlogicvector(TX_SEND_DESC2, 2);
               END IF;
               stream_wrdata_sop <= (m_req OR s_req);
               stream_wr <= ((m_req OR s_req) AND stream_ready_del);
            
            -- if streaming i/f can accept more data (stream_ready_del):
            --    - fetch data if streaming i/f can accept more data 
            -- write desc_lo to streaming i/f  
            -- any data that is fetched, is accepted by 
            -- streaming i/f (data is only fetched if 
            -- streaming i/f can accept it).
            
            -- no desc/data packing
            -- TX_SEND_DESC2
            WHEN "01" =>
               last_was_slave <= sel_slave;
               m_data_ack_xhdl0 <= sel_master AND stream_ready_del;
               s_data_ack_xhdl1 <= sel_slave AND stream_ready_del;
               IF (TL_SELECTION = 0) THEN
                  IF (data_ack_cycle = '1') THEN
                     IF (has_payload = '1') THEN
                        tx_state <= to_stdlogicvector(TX_SEND_DATA, 2);
                     ELSE
                        tx_state <= to_stdlogicvector(TX_NULL_CYCLE, 2);
                     END IF;
                     stream_wrdata_txdata <= tx_desc_hold_lo;
                     stream_wrdata_eop <= NOT(has_payload);
                     stream_wr <= '1';
                  END IF;
               ELSE
                  IF (pack_dw = '1') THEN
                     IF (data_ack_cycle = '1') THEN
                        IF ((has_payload AND ((sel_master AND NOT(m_last_datacyc)) OR (sel_slave AND NOT(s_last_datacyc)))) = '1') THEN
                           tx_state <= to_stdlogicvector(TX_SEND_DATA, 2);
                        ELSE
                           tx_state <= to_stdlogicvector(TX_NULL_CYCLE, 2);
                        END IF;
                        IF (sel_master = '1') THEN
                           stream_wrdata_txdata <= (m_tx_data(63 DOWNTO 32) & tx_desc_hold_lo(63 DOWNTO 32));
                        ELSE
                           stream_wrdata_txdata <= (s_tx_data(63 DOWNTO 32) & tx_desc_hold_lo(63 DOWNTO 32));
                        END IF;
                        stream_wrdata_eop <= NOT(has_payload) OR (sel_slave AND s_last_datacyc) OR (sel_master AND m_last_datacyc);
                        stream_wr <= '1';
                     END IF;
                  ELSE
                     IF (data_ack_cycle = '1') THEN
                        IF (has_payload = '1') THEN
                           tx_state <= to_stdlogicvector(TX_SEND_DATA, 2);
                        ELSE
                           tx_state <= to_stdlogicvector(TX_NULL_CYCLE, 2);
                        END IF;
                        stream_wrdata_txdata <= (tx_desc_hold_lo(31 DOWNTO 0) & tx_desc_hold_lo(63 DOWNTO 32));
                        stream_wrdata_eop <= NOT(has_payload);
                        stream_wr <= '1';
                     END IF;
                  END IF;
               END IF;
            -- whenever streaming i/f can accept more data, do fetch. 
            -- fetch until end of pkt is detected. 
            
            -- when packing data, data cycles are not deferred
            
            -- every time input data is advanced, write the previous data to streaming i/f.
            -- because they are multiplexed onto the same bus, data coinciding with the desc 
            -- phase is held and written on the next cycle.  
            -- all data thereafter is also held and written one cycle later.
            -- when end of pkt is written, return to start  
            -- TX_SEND_DATA
            WHEN "11"  =>
               m_data_ack_xhdl0 <= sel_master AND stream_ready_del AND NOT(pkt_ended);
               s_data_ack_xhdl1 <= sel_slave AND stream_ready_del AND NOT(pkt_ended);
               IF (((sel_master AND m_data_ack_xhdl0 AND m_last_datacyc) OR (sel_slave AND s_data_ack_xhdl1 AND s_last_datacyc)) = '1') THEN
                  pkt_ended <= '1';
               ELSE
                  pkt_ended <= pkt_ended;
               END IF;
               IF (pack_dw = '1') THEN
                  IF (data_ack_cycle = '1') THEN
                     IF (((sel_master AND m_last_datacyc) OR (sel_slave AND s_last_datacyc)) = '1') THEN
                        tx_state <= to_stdlogicvector(TX_WAIT_REQ, 2);
                     END IF;
                     IF (sel_master = '1') THEN
                        stream_wrdata_txdata <= m_tx_data;
                        stream_wrdata_eop <= m_last_datacyc;
                     ELSE
                        stream_wrdata_txdata <= s_tx_data;
                        stream_wrdata_eop <= s_last_datacyc;
                     END IF;
                     stream_wr <= '1';
                     IF (m_last_datacyc = '1') THEN
                        sel_master <= '0';
                     ELSE
                        sel_master <= sel_master;
                     END IF;
                     IF (s_last_datacyc = '1') THEN
                        sel_slave <= '0';
                     ELSE
                        sel_slave <= sel_slave;
                     END IF;
                  END IF;
               ELSE
                  IF (data_ack_cycle = '1') THEN
                     IF (((sel_master AND m_stream_pkt_end_hold_r) OR (sel_slave AND s_stream_pkt_end_hold_r)) = '1') THEN
                        tx_state <= to_stdlogicvector(TX_WAIT_REQ, 2);
                     END IF;
                     IF (sel_master = '1') THEN
                        stream_wrdata_txdata <= m_tx_data_hold_r;
                        stream_wrdata_eop <= m_stream_pkt_end_hold_r;
                     ELSE
                        stream_wrdata_txdata <= s_tx_data_hold_r;
                        stream_wrdata_eop <= s_stream_pkt_end_hold_r;
                     END IF;
                     stream_wr <= '1';
                     IF (m_stream_pkt_end_hold_r = '1') THEN
                        sel_master <= '0';
                     ELSE
                        sel_master <= sel_master;
                     END IF;
                     IF (s_stream_pkt_end_hold_r = '1') THEN
                        sel_slave <= '0';
                     ELSE
                        sel_slave <= sel_slave;
                     END IF;
                  END IF;
               END IF;
            -- no data phase.  ack being issued on this cycle
            -- TX_NULL_CYCLE
            WHEN "10" => 
               tx_state <= to_stdlogicvector(TX_WAIT_REQ, 2);
               sel_master <= '0';
               sel_slave <= '0';
            WHEN OTHERS =>
               NULL;
         END CASE;
      END IF;
   END PROCESS;
   
END ARCHITECTURE altpcie;
