LIBRARY ieee;                          
use IEEE.std_logic_1164.all;           
use IEEE.std_logic_arith.all;          
use IEEE.std_logic_unsigned.all;       
                                       
LIBRARY altera_mf;                     
USE altera_mf.altera_mf_components.all;
                                       
LIBRARY ieee;
   USE ieee.std_logic_1164.all;
-- synthesis verilog_version verilog_2001
-------------------------------------------------------------------------------
-- Title         : PCI Express Reference Design Example Application 
-- Project       : PCI Express MegaCore function
-------------------------------------------------------------------------------
-- File          : altpcierd_apprxstream_formatter.v
-- Author        : Altera Corporation
-------------------------------------------------------------------------------
-- Description :
-- This module extracts packet information from the streaming interface and
-- routes the packets to the master or slave of the "simple dma" design example.
-------------------------------------------------------------------------------
-- Copyright (c) 2006 Altera Corporation. All rights reserved.  Altera products are
-- protected under numerous U.S. and foreign patents, maskwork rights, copyrights and
-- other intellectual property laws.  
--
-- This reference design file, and your use thereof, is subject to and governed by
-- the terms and conditions of the applicable Altera Reference Design License Agreement.
-- By using this reference design file, you indicate your acceptance of such terms and
-- conditions between you and Altera Corporation.  In the event that you do not agree with
-- such terms and conditions, you may not use the reference design file. Please promptly
-- destroy any copies you have made.
--
-- This reference design file being provided on an "as-is" basis and as an accommodation 
-- and therefore all warranties, representations or guarantees of any kind 
-- (whether express, implied or statutory) including, without limitation, warranties of 
-- merchantability, non-infringement, or fitness for a particular purpose, are 
-- specifically disclaimed.  By making this reference design file available, Altera
-- expressly does not recommend, suggest or require that this reference design file be
-- used in combination with any other product not provided by Altera. 
------------------------------------------------------------------------------- 
ENTITY altpcierd_apprxstream_formatter IS
   GENERIC (
      TL_SELECTION            : INTEGER := 6        -- streaming interface format selection.
      --  0 = ICM format.  6 = 7.2 format.
      
      -- rx_stream_state 
   );
   PORT (
      
      clk                     : IN STD_LOGIC;
      rstn                    : IN STD_LOGIC;
      fifo_valid              : IN STD_LOGIC;       -- from streaming intf.  means streaming data (fifo_data) is valid
      fifo_data               : IN STD_LOGIC_VECTOR(107 DOWNTO 0);      -- from streaming intf.  data from streaming interface
      -- from application.  streaming intf throttle control.
      
      rx_st_data              : OUT STD_LOGIC_VECTOR(107 DOWNTO 0);     -- to application.  reformatted streaming data.
      rx_st_valid             : OUT STD_LOGIC;      -- to application.  means rx_st_data is valid.
      rx_st_ready             : IN STD_LOGIC;
      fifo_ready              : OUT STD_LOGIC       -- to streaming interface.  streaming intf throttle control.  
   );
END ENTITY altpcierd_apprxstream_formatter;
ARCHITECTURE altpcie OF altpcierd_apprxstream_formatter IS


                                                
   FUNCTION to_stdlogic (                        
   val      : IN integer) RETURN std_logic IS    
   BEGIN                                         
      IF (val=1) THEN                            
         RETURN('1');                            
      ELSE                                       
         RETURN('0');                            
      END IF;                                    
   END to_stdlogic;                              
   FUNCTION to_stdlogic (                        
   val      : IN boolean) RETURN std_logic IS    
   BEGIN                                         
      IF (val) THEN                              
         RETURN('1');                            
      ELSE                                       
         RETURN('0');                            
      END IF;                                    
   END to_stdlogic;                              
                                                
                                              
   FUNCTION TO_STDLOGICVECTOR (                                        
      val      : IN integer;                                           
      len      : IN integer) RETURN std_logic_vector IS                
                                                                       
      VARIABLE rtn : std_logic_vector(len-1 DOWNTO 0):=(OTHERS => '0');
      VARIABLE num : integer := val;                                   
      VARIABLE r   : integer;                                          
   BEGIN                                                               
      FOR index IN 0 TO len-1 LOOP                                     
         r := num rem 2;                                               
         num := num/2;                                                 
         IF (r = 1) THEN                                               
            rtn(index) := '1';                                         
         ELSE                                                          
            rtn(index) := '0';                                         
         END IF;                                                       
      END LOOP;                                                        
      RETURN(rtn);                                                     
   END TO_STDLOGICVECTOR;                                              
                                                                       
                                                                       
   
   CONSTANT       SOP_CYCLE               : INTEGER := 0;       -- wait for 1st desc cycle of any pkt
   CONSTANT       H3DW_PAYLD_DESC2_CYCLE  : INTEGER := 1;       -- wait for 2nd desc cycle of 3DW Payload Pkt
   CONSTANT       SEND_DELAYED_DATA       : INTEGER := 2;       -- receiving data from 3DW non-aligned pkt (desc/data unpacking required)
   CONSTANT       SEND_NORMAL_DATA_EOP    : INTEGER := 3;       -- receiving data from aligned or 4DW pkt (no desc/data unpacking required)
   CONSTANT       SEND_DELAYED_EOP        : INTEGER := 4;       -- sends last 'delayed data' for 3DW non-aligned pkts;

   SIGNAL fifo_data_last     : STD_LOGIC_VECTOR(107 DOWNTO 0);      -- previous fifo_data
   SIGNAL fifo_be_data_last  : STD_LOGIC_VECTOR(7 DOWNTO 0);        -- previous byte enable field of fifo_data
   SIGNAL fifo_be_data       : STD_LOGIC_VECTOR(7 DOWNTO 0);        -- byte enable field of fifo_data 
   SIGNAL rx_st_ready_del    : STD_LOGIC;
   
   SIGNAL rx_stream_state    : STD_LOGIC_VECTOR(2 DOWNTO 0);        -- state machine - mux controls for reformatting the output streaming data 
   
   SIGNAL rx_sop             : STD_LOGIC;
   SIGNAL rx_eop             : STD_LOGIC;
   SIGNAL rx_data            : STD_LOGIC_VECTOR(63 DOWNTO 0);
   SIGNAL rx_be              : STD_LOGIC_VECTOR(7 DOWNTO 0);
   SIGNAL rx_bar             : STD_LOGIC_VECTOR(7 DOWNTO 0);
   SIGNAL rx_bar_r           : STD_LOGIC_VECTOR(7 DOWNTO 0);
   
   SIGNAL fifo_data_eop      : STD_LOGIC;
   SIGNAL fifo_data_sop      : STD_LOGIC;
   SIGNAL fifo_data_sop_last : STD_LOGIC;
   
   SIGNAL fifo_data_swapped  : STD_LOGIC_VECTOR(63 DOWNTO 0);
   SIGNAL fifo_data_shifted  : STD_LOGIC_VECTOR(63 DOWNTO 0);
   
   -- parse byte enable bits    
   
   -- read from streaming FIFO as long as application can accept 
   -- more data.  except when inserting Delayed EOP --
   -- last cycle data is sourced from this module rather than from fifo.
   
   -- allow to read ahead on fifo_data_eop even tho we need to insert a cycle.  
   -- defer hold-off of read to next cycle.
   
   --//////////////
   -- REGISTERS  
   --//////////////
   
   --////////////////////////////////////////////////////////////
   -- 7.2 state machine controls output data mux.
   -- reformats streaming data input from 72 format to 71 format
   --////////////////////////////////////////////////////////////
   -- wait for new pkt.
   -- if pkt has payload and is 3DW, wait for 2nd desc phase to decode address,
   -- else, there is no desc/data packing - pass data straight thru   
   -- in 7.2, data is dword swapped.  payload bit = 30.
   -- header is 3DW and there is payload
   -- 1DW packed payld
   -- multiple DW packed payload (unaligned address)
   -- non-packed payload 
   -- payload cycles for packed 3DW pkts
   -- for non-packed data, pass data thru as-is
   -- insert eop cycle when application requests Fifo read
   -- on 2nd to last data cycle
   
   --//////////////////////////////////////////////////////
   -- MUXED OUTPUT TO APP
   -- REFORMAT STREAMING DATA TO LOOK LIKE 71_ICM FORMAT
   --////////////////////////////////////////////////////// 
   
   SIGNAL zero32             : STD_LOGIC_VECTOR(31 DOWNTO 0);
   SIGNAL zero26             : STD_LOGIC_VECTOR(25 DOWNTO 0);
BEGIN
   fifo_data_eop <= fifo_data(72);
   fifo_data_sop <= fifo_data(73);
   fifo_be_data <= fifo_data(81 DOWNTO 74);
   fifo_ready <= to_stdlogic((rx_st_ready = '1') AND (rx_stream_state /= to_stdlogicvector(SEND_DELAYED_EOP, 3)));
   PROCESS (clk, rstn)
   BEGIN
      IF ((NOT(rstn)) = '1') THEN
         fifo_data_last <= "000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
         fifo_data_sop_last <= '0';
         fifo_be_data_last <= "00000000";
         rx_stream_state <= to_stdlogicvector(SOP_CYCLE, 3);
         rx_st_ready_del <= '0';
         rx_bar_r <= "00000000";
      ELSIF (clk'EVENT AND clk = '1') THEN
         IF (rx_st_ready = '1') THEN
            fifo_data_last <= fifo_data;
            fifo_data_sop_last <= fifo_data_sop;
            fifo_be_data_last <= fifo_be_data;
         ELSE
            fifo_data_last <= fifo_data_last;
            fifo_data_sop_last <= fifo_data_sop_last;
            fifo_be_data_last <= fifo_be_data_last;
         END IF;
         rx_st_ready_del <= rx_st_ready;
         rx_bar_r <= rx_bar;
         CASE (rx_stream_state) IS
            -- SOP_CYCLE
            WHEN "000" =>
               IF ((rx_st_ready AND fifo_data_sop) = '1') THEN
                  IF ((fifo_data(30) AND NOT(fifo_data(29))) = '1') THEN
                     rx_stream_state <= to_stdlogicvector(H3DW_PAYLD_DESC2_CYCLE, 3);
                  ELSE
                     rx_stream_state <= to_stdlogicvector(SEND_NORMAL_DATA_EOP, 3);
                  END IF;
               END IF;
            -- H3DW_PAYLD_DESC2_CYCLE
            WHEN "001" =>
               IF (rx_st_ready = '1') THEN
                  IF ((fifo_data(2) AND fifo_data_eop) = '1') THEN
                     rx_stream_state <= to_stdlogicvector(SEND_DELAYED_EOP, 3);
                  ELSIF ((fifo_data(2)) = '1') THEN
                     rx_stream_state <= to_stdlogicvector(SEND_DELAYED_DATA, 3);
                  ELSE
                     rx_stream_state <= to_stdlogicvector(SEND_NORMAL_DATA_EOP, 3);
                  END IF;
               END IF;
               -- SEND_DELAYED_DATA,
            WHEN "010" =>
               IF ((rx_st_ready AND fifo_data_eop) = '1') THEN
                  rx_stream_state <= to_stdlogicvector(SEND_DELAYED_EOP, 3);
               END IF;
               -- SEND_NORMAL_DATA_EOP
            WHEN "011" =>
               IF ((rx_st_ready AND fifo_data_eop) = '1') THEN
                  rx_stream_state <= to_stdlogicvector(SOP_CYCLE, 3);
               END IF;
               -- SEND_DELAYED_EOP, 
            WHEN "100" =>
               IF (rx_st_ready = '1') THEN
                  rx_stream_state <= to_stdlogicvector(SOP_CYCLE, 3);
               END IF;
            WHEN OTHERS =>
               NULL;
         END CASE;
      END IF;
   END PROCESS;
   zero32 <= "00000000000000000000000000000000";
   zero26 <= "00000000000000000000000000";
   
   fifo_data_swapped <= (fifo_data(31 DOWNTO 0) & fifo_data(63 DOWNTO 32));
   fifo_data_shifted <= (fifo_data(31 DOWNTO 0) & zero32);
   
   PROCESS (fifo_valid, fifo_data, rx_be, rx_sop, rx_eop, rx_bar, rx_data, rx_st_ready_del, rx_stream_state, fifo_data_swapped, fifo_data_sop, fifo_data_shifted, fifo_data_last, fifo_be_data_last, fifo_data_sop_last, fifo_be_data, fifo_data_eop, rx_bar_r)
   BEGIN
      
      IF (TL_SELECTION = 0) THEN
         -- streaming data is per ICM MODE
         -- pass thru as-is 
         rx_st_valid <= fifo_valid;
         rx_st_data <= fifo_data;
         rx_sop <= '0';
         rx_eop <= '0';
         rx_bar <= "00000000";
         rx_data <= "0000000000000000000000000000000000000000000000000000000000000000";
         rx_be <= "00000000";
      ELSE
         -- streaming data is 7.2 MODE
         -- convert to 7.1 MODE
         
         -- pass valid signal thru.  
         -- add 1 cycle at end if data was packed -- generate valid 1 cycle after
         -- application requests Fifo read & there is only EOP left to send.
         rx_st_data <= (zero26 & rx_be & rx_sop & rx_eop & rx_bar & rx_data);
         
         -- reformat streaming data to match 7.1 ICM data  
         rx_st_valid <= fifo_valid OR to_stdlogic(((rx_stream_state = to_stdlogicvector(SEND_DELAYED_EOP, 3)) AND (rx_st_ready_del = '1')));
         CASE (rx_stream_state) IS
            -- DWord swap the descriptor
            -- SOP_CYCLE
            WHEN "000" =>
               rx_data <= fifo_data_swapped;
               rx_be <= "00000000";
               rx_eop <= '0';
               rx_sop <= fifo_data_sop;
               rx_bar <= fifo_data(71 DOWNTO 64);
            -- Shift the descriptor
            -- H3DW_PAYLD_DESC2_CYCLE
            WHEN "001" =>
               rx_data <= fifo_data_shifted;
               rx_be <= "00000000";
               rx_eop <= '0';
               rx_sop <= fifo_data_sop;
               rx_bar <= fifo_data(71 DOWNTO 64);
            -- SEND_DELAYED_DATA
            WHEN "010" =>
               rx_data <= fifo_data_last(63 DOWNTO 0);
               rx_be <= fifo_be_data_last;
               rx_eop <= '0';
               rx_sop <= fifo_data_sop;
               rx_bar <= fifo_data(71 DOWNTO 64);
            -- swap on descriptor cycle
            -- no swap on data cycle
            -- SEND_NORMAL_DATA_EOP
            WHEN "011"  =>
               IF (fifo_data_sop_last = '1') THEN
                  rx_data <= fifo_data_swapped;
               ELSE
                  rx_data <= fifo_data(63 DOWNTO 0);
               END IF;
               rx_be <= fifo_be_data;
               rx_eop <= fifo_data_eop;
               rx_sop <= fifo_data_sop;
               rx_bar <= fifo_data(71 DOWNTO 64);
            -- SEND_DELAYED_EOP
            WHEN "100"  =>      -- Should never reach this
               rx_data <= fifo_data_last(63 DOWNTO 0);
               rx_be <= fifo_be_data_last;
               rx_eop <= '1';
               rx_sop <= '0';
               rx_bar <= rx_bar_r;
            WHEN OTHERS =>
               rx_data <= fifo_data_swapped;
               rx_be <= "00000000";
               rx_eop <= '0';
               rx_sop <= fifo_data_sop;
               rx_bar <= fifo_data(71 DOWNTO 64);
         END CASE;
      END IF;
   END PROCESS;
   
END ARCHITECTURE altpcie;
