LIBRARY ieee;                          
use IEEE.std_logic_1164.all;           
use IEEE.std_logic_arith.all;          
use IEEE.std_logic_unsigned.all;       
                                       
LIBRARY altera_mf;                     
USE altera_mf.altera_mf_components.all;
                                       
LIBRARY ieee;
   USE ieee.std_logic_1164.all;
-- synthesis verilog_version verilog_2001
-------------------------------------------------------------------------------
-- Title         : PCI Express Reference Design Example Application 
-- Project       : PCI Express MegaCore function
-------------------------------------------------------------------------------
-- File          : altpcierd_apprxstream_decode.v
-- Author        : Altera Corporation
-------------------------------------------------------------------------------
-- Description :
-- This module extracts packet information from the streaming interface and
-- routes the packets to the master or slave of the "simple dma" design example.
-------------------------------------------------------------------------------
-- Copyright (c) 2006 Altera Corporation. All rights reserved.  Altera products are
-- protected under numerous U.S. and foreign patents, maskwork rights, copyrights and
-- other intellectual property laws.  
--
-- This reference design file, and your use thereof, is subject to and governed by
-- the terms and conditions of the applicable Altera Reference Design License Agreement.
-- By using this reference design file, you indicate your acceptance of such terms and
-- conditions between you and Altera Corporation.  In the event that you do not agree with
-- such terms and conditions, you may not use the reference design file. Please promptly
-- destroy any copies you have made.
--
-- This reference design file being provided on an "as-is" basis and as an accommodation 
-- and therefore all warranties, representations or guarantees of any kind 
-- (whether express, implied or statutory) including, without limitation, warranties of 
-- merchantability, non-infringement, or fitness for a particular purpose, are 
-- specifically disclaimed.  By making this reference design file available, Altera
-- expressly does not recommend, suggest or require that this reference design file be
-- used in combination with any other product not provided by Altera. 
------------------------------------------------------------------------------- 
ENTITY altpcierd_apprxstream_decode IS
   PORT (
      
      clk          : IN STD_LOGIC;
      rstn         : IN STD_LOGIC;
      
      rx_st_valid  : IN STD_LOGIC;		-- means streaming data (rx_st_data) is valid
      rx_st_data   : IN STD_LOGIC_VECTOR(135 DOWNTO 0);		-- data from streaming interface
      -- means master is busy and cannot accept new rx request 
      -- means slave is busy and cannot accept new rx request 
      -- master accepts rx request (new pkt)
      -- slave accepts rx request (new pkt)
      
      rx_st_ready  : OUT STD_LOGIC;		-- throttles streaming data (rx_st_data)
      m_busy       : IN STD_LOGIC;
      s_busy       : IN STD_LOGIC;
      m_ack        : IN STD_LOGIC;
      s_ack        : IN STD_LOGIC;
      m_rx_sop     : OUT STD_LOGIC;		-- indicates that there is a new pkt available for the master
      s_rx_sop     : OUT STD_LOGIC;		-- indicates that there is a new pkt available for the slave
      -- packet payload data extracted from streaming interface
      -- packet descriptor extracted from streaming interface
      -- packet byte enable mask extracted from streaming interface
      -- means rx_data is valid for master
      -- means rx_data is valid for slave
      rx_eop       : OUT STD_LOGIC;		-- means current cycle is the last of the pkt
      rx_data      : OUT STD_LOGIC_VECTOR(63 DOWNTO 0);
      rx_desc      : OUT STD_LOGIC_VECTOR(135 DOWNTO 0);
      rx_be        : OUT STD_LOGIC_VECTOR(7 DOWNTO 0);
      m_rx_dv      : OUT STD_LOGIC;
      s_rx_dv      : OUT STD_LOGIC
   );
END ENTITY altpcierd_apprxstream_decode;
ARCHITECTURE altpcie OF altpcierd_apprxstream_decode IS
   

   SIGNAL stream_sop_del    : STD_LOGIC;
   SIGNAL rx_desc_hi_r      : STD_LOGIC_VECTOR(63 DOWNTO 0);		-- registered version of descriptor high bytes (rx_desc[127:64])
   SIGNAL rx_desc_lo_r      : STD_LOGIC_VECTOR(63 DOWNTO 0);		-- registered version of descriptor low boytes (rx_desc[63:0]) 
   SIGNAL stream_sop_del2   : STD_LOGIC;
   SIGNAL rx_bar_r          : STD_LOGIC_VECTOR(7 DOWNTO 0);		-- registered veresion of bar field (rx_desc[135:128])
   
   SIGNAL is_mst            : STD_LOGIC;		-- indicates that current pkt is for master
   SIGNAL is_tgt            : STD_LOGIC;		-- indicates that current pkt is for slave (target)
   
   SIGNAL is_mst_r          : STD_LOGIC;		-- registered version of is_mst
   SIGNAL is_tgt_r          : STD_LOGIC;		-- registered version of is_tgt
   SIGNAL has_payload       : STD_LOGIC;		-- indicates that the current pkt has payload
   SIGNAL has_payload_r     : STD_LOGIC;		-- registered version of has_payload
   SIGNAL rx_st_ready_r     : STD_LOGIC;		-- registered version of rx_st_ready
   SIGNAL m_rx_sop_r        : STD_LOGIC;		-- registered version of m_rx_sop
   SIGNAL s_rx_sop_r        : STD_LOGIC;		-- registered version of s_rx_sop
   
   SIGNAL assert_m_rx_sop   : STD_LOGIC;
   SIGNAL assert_s_rx_sop   : STD_LOGIC;
   SIGNAL is_mst_or_tgt_r   : STD_LOGIC;
   SIGNAL assert_ms_rx_sop  : STD_LOGIC;
   SIGNAL ms_rx_sop         : STD_LOGIC;
   SIGNAL ms_rx_sop_r       : STD_LOGIC;
   SIGNAL is_mst_inter      : STD_LOGIC;
   SIGNAL is_tgt_inter      : STD_LOGIC;
   
   SIGNAL stream_sop        : STD_LOGIC;
   SIGNAL stream_eop        : STD_LOGIC;
   SIGNAL stream_data_bits  : STD_LOGIC_VECTOR(63 DOWNTO 0);
   SIGNAL stream_bar_bits   : STD_LOGIC_VECTOR(7 DOWNTO 0);
   SIGNAL stream_be_bits    : STD_LOGIC_VECTOR(7 DOWNTO 0);
   
   -- Declare intermediate signals for referenced outputs
   SIGNAL rx_st_ready_xhdl4 : STD_LOGIC;
   SIGNAL m_rx_sop_xhdl1    : STD_LOGIC;
   SIGNAL s_rx_sop_xhdl6    : STD_LOGIC;
   SIGNAL rx_eop_xhdl3      : STD_LOGIC;
   SIGNAL rx_desc_xhdl2     : STD_LOGIC_VECTOR(135 DOWNTO 0);
   SIGNAL m_rx_dv_xhdl0     : STD_LOGIC;
   SIGNAL s_rx_dv_xhdl5     : STD_LOGIC;
BEGIN
   -- Drive referenced outputs
   rx_st_ready <= rx_st_ready_xhdl4;
   m_rx_sop <= m_rx_sop_xhdl1;
   s_rx_sop <= s_rx_sop_xhdl6;
   rx_eop <= rx_eop_xhdl3;
   rx_desc <= rx_desc_xhdl2;
   m_rx_dv <= m_rx_dv_xhdl0;
   s_rx_dv <= s_rx_dv_xhdl5;
   
   stream_be_bits <= rx_st_data(81 DOWNTO 74);
   stream_sop <= rx_st_data(73);
   stream_eop <= rx_st_data(72);
   stream_bar_bits <= rx_st_data(71 DOWNTO 64);
   stream_data_bits <= rx_st_data(63 DOWNTO 0);
   
   ---------------------------------------------------------------- 
   --  Generate app-side signals.
   --    - Demux the Descriptor/Data from the streaming interface
   --      and present on non-multiplexed application interface.
   --    - Generate rx request to master/slave when a new pkt is 
   --      received.
   ----------------------------------------------------------------
   
   -- decode descriptor phase.
   -- determine if current pkt is for master or for slave.
   
   is_mst_inter <= '1' WHEN rx_st_data(61 DOWNTO 57) = "00101" ELSE
                   '0';
   is_mst <= is_mst_inter WHEN stream_sop = '1' ELSE
             is_mst_r;
   
   is_tgt_inter <= '1' WHEN rx_st_data(60 DOWNTO 58) = "000" ELSE
                   '0';
   is_tgt <= is_tgt_inter WHEN stream_sop = '1' ELSE
             is_tgt_r;
   
   -- decode whether packet has payload 
   has_payload <= rx_st_data(62) WHEN stream_sop = '1' ELSE
                  has_payload_r;
   
   -- extract packet fields (data, byte enable bits, 
   -- descriptor) from the multiplexed streaming interface bus
   -- to desc/data interface to the application
   -- descriptor is presented 
   rx_data <= stream_data_bits;
   rx_be <= stream_be_bits;
   rx_desc_xhdl2(135 DOWNTO 128) <= stream_bar_bits WHEN ((stream_sop_del AND NOT(stream_sop))) = '1' ELSE		-- valid on 2nd stream cycle
                                    rx_bar_r;
   rx_desc_xhdl2(127 DOWNTO 64) <= rx_desc_hi_r;		-- valid on sop cycle, but presented on next cycle
   rx_desc_xhdl2(63 DOWNTO 0) <= stream_data_bits WHEN ((stream_sop_del AND NOT(stream_sop))) = '1' ELSE		-- valid on 2nd stream cycle  
                                 rx_desc_lo_r;
   
   -- assert a request for the master or slave when
   -- on 2nd cycle (desc2 phase) of new packet.  deassert when acknowledged.  
   -- full descriptor (rx_desc) is available to the application when the request is asserted.
   
   assert_m_rx_sop <= '1' WHEN (stream_sop_del AND NOT(stream_sop) AND is_mst_r) = '1' ELSE
                      '0';
   m_rx_sop_xhdl1 <= '0' WHEN m_ack = '1' ELSE
                     '1' WHEN assert_m_rx_sop = '1' ELSE
                     m_rx_sop_r;
   
   assert_s_rx_sop <= '1' WHEN (stream_sop_del AND NOT(stream_sop) AND is_tgt_r) = '1' ELSE
                      '0';
   s_rx_sop_xhdl6 <= '0' WHEN s_ack = '1' ELSE
                     '1' WHEN assert_s_rx_sop = '1' ELSE
                     s_rx_sop_r;
   
   assert_ms_rx_sop <= '1' WHEN (stream_sop_del AND NOT(stream_sop) AND is_mst_or_tgt_r) = '1' ELSE
                       '0';
   ms_rx_sop <= '0' WHEN ((m_ack OR s_ack)) = '1' ELSE
                '1' WHEN assert_ms_rx_sop = '1' ELSE
                ms_rx_sop_r;
   
   -- detect the end of packet
   rx_eop_xhdl3 <= stream_eop;
   
   PROCESS (clk, rstn)
   BEGIN
      IF ((NOT(rstn)) = '1') THEN
         rx_desc_hi_r <= "0000000000000000000000000000000000000000000000000000000000000000";
         rx_desc_lo_r <= "0000000000000000000000000000000000000000000000000000000000000000";
         rx_bar_r <= "00000000";
         stream_sop_del2 <= '0';
         is_mst_r <= '0';
         is_tgt_r <= '0';
         is_mst_or_tgt_r <= '0';
         has_payload_r <= '0';
         m_rx_dv_xhdl0 <= '0';
         s_rx_dv_xhdl5 <= '0';
         stream_sop_del <= '0';
         m_rx_sop_r <= '0';
         s_rx_sop_r <= '0';
         ms_rx_sop_r <= '0';
      ELSIF (clk'EVENT AND clk = '1') THEN
         IF (stream_sop = '1') THEN
            rx_desc_hi_r <= stream_data_bits;
         ELSE
            
            rx_desc_hi_r <= rx_desc_hi_r;
         END IF;
         rx_desc_lo_r <= rx_desc_xhdl2(63 DOWNTO 0);
         rx_bar_r <= rx_desc_xhdl2(135 DOWNTO 128);
         stream_sop_del2 <= stream_sop_del;
         is_mst_r <= is_mst;
         is_tgt_r <= is_tgt;
         is_mst_or_tgt_r <= is_mst OR is_tgt;
         has_payload_r <= has_payload;		-- indicates a new pkt is received.                
         stream_sop_del <= stream_sop AND rx_st_valid;
         m_rx_sop_r <= m_rx_sop_xhdl1;
         s_rx_sop_r <= s_rx_sop_xhdl6;
         
         ms_rx_sop_r <= ms_rx_sop;		-- assert data valid flag if there is payload.  
         IF ((stream_sop_del2 AND is_mst_r AND has_payload) = '1') THEN
            m_rx_dv_xhdl0 <= rx_st_valid;		-- deassert after the last data cycle.
         ELSIF (rx_eop_xhdl3 = '1') THEN
            m_rx_dv_xhdl0 <= '0';
         ELSE
            
            m_rx_dv_xhdl0 <= m_rx_dv_xhdl0;
         END IF;
         IF ((stream_sop_del2 AND is_tgt_r AND has_payload) = '1') THEN
            s_rx_dv_xhdl5 <= rx_st_valid;
         ELSIF (rx_eop_xhdl3 = '1') THEN
            s_rx_dv_xhdl5 <= '0';
         ELSE
            
            s_rx_dv_xhdl5 <= s_rx_dv_xhdl5;
         END IF;
      END IF;
   END PROCESS;
   
   --------------------------------------------------
   -- Generate streaming interface side 
   -- 'ready' signal to throttle data based
   -- on master/slave throttling.
   --------------------------------------------------
   
   -- when a pkt is received for either the master or slave,
   -- throttle the data stream until acked by master/slave.
   -- simple dma does not throttle dataphase.
   
   rx_st_ready_xhdl4 <= '1' WHEN ((m_ack OR s_ack)) = '1' ELSE
                        '0' WHEN ((ms_rx_sop)) = '1' ELSE
                        rx_st_ready_r;
   
   PROCESS (clk, rstn)
   BEGIN
      IF ((NOT(rstn)) = '1') THEN
         rx_st_ready_r <= '1';
      ELSIF (clk'EVENT AND clk = '1') THEN
         rx_st_ready_r <= rx_st_ready_xhdl4;
      END IF;
   END PROCESS;
   
END ARCHITECTURE altpcie;
