`timescale 1 ps / 1 ps 
//-----------------------------------------------------------------------------
// Title         : PCI Express BFM Root Port Driver 
// Project       : PCI Express MegaCore function
//-----------------------------------------------------------------------------
// File          : altpcietb_bfm_driver.v
// Author        : Altera Corporation
//-----------------------------------------------------------------------------
// Description :
// This module is driver for the Root Port BFM. It processes the list of
// functions to perform and passes them off to the VC specific interfaces
//-----------------------------------------------------------------------------
// Copyright (c) 2005 Altera Corporation. All rights reserved.  Altera products are
// protected under numerous U.S. and foreign patents, maskwork rights, copyrights and
// other intellectual property laws.  
//
// This reference design file, and your use thereof, is subject to and governed by
// the terms and conditions of the applicable Altera Reference Design License Agreement.
// By using this reference design file, you indicate your acceptance of such terms and
// conditions between you and Altera Corporation.  In the event that you do not agree with
// such terms and conditions, you may not use the reference design file. Please promptly
// destroy any copies you have made.
//
// This reference design file being provided on an "as-is" basis and as an accommodation 
// and therefore all warranties, representations or guarantees of any kind 
// (whether express, implied or statutory) including, without limitation, warranties of 
// merchantability, non-infringement, or fitness for a particular purpose, are 
// specifically disclaimed.  By making this reference design file available, Altera
// expressly does not recommend, suggest or require that this reference design file be
// used in combination with any other product not provided by Altera.
//-----------------------------------------------------------------------------
module altpcietb_bfm_driver (clk_in, rstn, INTA, INTB, INTC, INTD, dummy_out);

   // TEST_LEVEL is a parameter passed in from the top level test bench that
   // could control the amount of testing done. It is not currently used. 
   parameter TEST_LEVEL  = 1;

   `include "altpcietb_bfm_constants.v"
   `include "altpcietb_bfm_log.v"
   `include "altpcietb_bfm_shmem.v"
   `include "altpcietb_bfm_rdwr.v"
   `include "altpcietb_bfm_configure.v"

   // The clk_in and rstn signals are provided for possible use in controlling
   // the transactions issued, they are not currently used.
   input clk_in; 
   input rstn;
   input INTA;
   input INTB;
   input INTC;
   input INTD;
   output dummy_out; 

   // purpose: Use Reads and Writes to test the target memory
   //          The starting offset in the target memory and the
   //          length can be specified
   task target_mem_test;
      input bar_table;      // Pointer to the BAR sizing and
      integer bar_table;    // address information set up by
                            // the configuration routine
      input tgt_bar;        // BAR to use to access the target
      integer tgt_bar;      // memory
      input start_offset;   // Starting offset in the target
      integer start_offset; // memory to use
      input tgt_data_len;   // Length of data to test
      integer tgt_data_len;

      parameter TGT_WR_DATA_ADDR = 1 * (2 ** 16); 
      integer tgt_rd_data_addr; 
      integer err_addr; 

      reg dummy ;
      
      begin  // target_mem_test
         dummy = ebfm_display(EBFM_MSG_INFO, "Starting Target Write/Read Test."); 
         dummy = ebfm_display(EBFM_MSG_INFO, {"  Target BAR = ", dimage1(tgt_bar)}); 
         dummy = ebfm_display(EBFM_MSG_INFO, {"  Length = ", dimage6(tgt_data_len), ", Start Offset = ", dimage6(start_offset)}); 
         // Setup some data to write to the Target
         shmem_fill(TGT_WR_DATA_ADDR, SHMEM_FILL_DWORD_INC, tgt_data_len, {64{1'b0}}); 
         // Setup an address for the data to read back from the Target
         tgt_rd_data_addr = TGT_WR_DATA_ADDR + (2 * tgt_data_len); 
         // Clear the target data area
         shmem_fill(tgt_rd_data_addr, SHMEM_FILL_ZERO, tgt_data_len, {64{1'b0}}); 
         //
         // Now write the data to the target with this BFM call
         //
         ebfm_barwr(bar_table, tgt_bar, start_offset, TGT_WR_DATA_ADDR, tgt_data_len, 0); 
         //
         // Read the data back from the target in one burst, wait for the read to
         // be complete
         // 
         ebfm_barrd_wait(bar_table, tgt_bar, start_offset, tgt_rd_data_addr, tgt_data_len, 0); 
         // Check the data
         if (shmem_chk_ok(tgt_rd_data_addr, SHMEM_FILL_DWORD_INC, tgt_data_len, {64{1'b0}}, 1'b1))
         begin
            dummy = ebfm_display(EBFM_MSG_INFO, "  Target Write and Read compared okay!"); 
         end
         else
         begin
            dummy = ebfm_display(EBFM_MSG_ERROR_FATAL, "  Stopping simulation due to miscompare"); 
         end 
      end
   endtask

   // purpose: This procedure polls the DMA engine until it is done
   task dma_wait_done;
      input bar_table; 
      integer bar_table;
      input setup_bar; 
      integer setup_bar;
      input scr_mem; 
      integer scr_mem;

      reg [31:0] dma_sts ;
      
      begin  // dma_wait_done
         shmem_fill(scr_mem, SHMEM_FILL_ONE, 4, {64{1'b0}});
         dma_sts = 32'hffffffff ;
         while ((dma_sts[31]) == 1'b1)
         begin
            ebfm_barrd_wait(bar_table, setup_bar, 12, scr_mem, 4, 0);
            dma_sts = shmem_read(scr_mem,4) ;
         end 
      end
   endtask

   // purpose: Use the reference design's DMA engine to move data from the BFM's
   // shared memory to the reference design's master memory and then back
   task dma_mem_test;
      input bar_table;      // Pointer to the BAR sizing and
      integer bar_table;    // address information set up by
                            // the configuration routine
      input setup_bar;      // BAR to be used for setting up
      integer setup_bar;    // the DMA operation and checking
                            // the status 
      input start_offset;   // Starting offset in the master
      integer start_offset; // memory 
      input dma_data_len;   // Length of DMA operations 
      integer dma_data_len;

      parameter SCR_MEM = (2 ** 17) - 4; 
      integer dma_rd_data_addr; 
      integer dma_wr_data_addr; 
      integer err_addr; 

      reg dummy ;
      
      begin
         dummy = ebfm_display(EBFM_MSG_INFO, "Starting DMA Read/Write Test."); 
         dummy = ebfm_display(EBFM_MSG_INFO, {"  Setup BAR = ", dimage1(setup_bar)}); 
         dummy = ebfm_display(EBFM_MSG_INFO, {"  Length = ", dimage6(dma_data_len), 
                                      ", Start Offset = ", dimage6(start_offset)}); 
         dma_rd_data_addr = SCR_MEM + 4 + start_offset; 
         // Setup some data for the DMA to read
         shmem_fill(dma_rd_data_addr, SHMEM_FILL_DWORD_INC, dma_data_len, {64{1'b0}}); 
         // Program the DMA to Read Data from Shared Memory
         ebfm_barwr_imm(bar_table, setup_bar, 0, dma_rd_data_addr, 4, 0); 
         ebfm_barwr_imm(bar_table, setup_bar, 4, 32'h00000000, 4, 0); 
         ebfm_barwr_imm(bar_table, setup_bar, 8, dma_data_len, 4, 0); 
         ebfm_barwr_imm(bar_table, setup_bar, 12, 32'h00040000, 4, 0); 
         // Wait Until the DMA is done
         dma_wait_done(bar_table, setup_bar, SCR_MEM); 
         // Setup an area for DMA to write back to
         // Currently DMA Engine Uses smae lower address bits for it's MRAM and PCIE
         // Addresses. So use the same address we started with
         dma_wr_data_addr = dma_rd_data_addr; 
         shmem_fill(dma_wr_data_addr, SHMEM_FILL_ZERO, dma_data_len, {64{1'b0}}); 
         // Program the DMA to Write Data Back to Shared Memory
         ebfm_barwr_imm(bar_table, setup_bar, 0, dma_wr_data_addr, 4, 0); 
         ebfm_barwr_imm(bar_table, setup_bar, 4, 32'h00000000, 4, 0); 
         ebfm_barwr_imm(bar_table, setup_bar, 8, dma_data_len, 4, 0); 
         ebfm_barwr_imm(bar_table, setup_bar, 12, 32'h00040040, 4, 0); 
         // Wait Until the DMA is done
         dma_wait_done(bar_table, setup_bar, SCR_MEM); 
         // Check the data
         if (shmem_chk_ok(dma_rd_data_addr, SHMEM_FILL_DWORD_INC, dma_data_len, {64{1'b0}}, 1'b1))
         begin
            dummy = ebfm_display(EBFM_MSG_INFO, "  DMA Read and Write compared okay!"); 
         end
         else
         begin
            dummy = ebfm_display(EBFM_MSG_ERROR_FATAL, "  Stopping simulation due to miscompare"); 
         end 
      end
   endtask

   // purpose: Examine the DUT's BAR setup and pick a reasonable BAR to use
   task find_mem_bar;
      input bar_table; 
      integer bar_table;
      input[5:0] allowed_bars; 
      input min_log2_size; 
      integer min_log2_size;
      output sel_bar;
      integer sel_bar;
       
      integer cur_bar; 
      reg[31:0] bar32; 
      integer log2_size; 
      reg is_mem; 
      reg is_pref; 
      reg is_64b; 

      begin  // find_mem_bar
         cur_bar = 0;
         begin : sel_bar_loop
            while (cur_bar < 6)
            begin
               ebfm_cfg_decode_bar(bar_table, cur_bar, log2_size, is_mem, is_pref, is_64b); 
               if ((is_mem == 1'b1) & (log2_size >= min_log2_size) & ((allowed_bars[cur_bar]) == 1'b1))
               begin
                  sel_bar = cur_bar;
                  disable sel_bar_loop ;
               end 
               if (is_64b == 1'b1)
               begin
                  cur_bar = cur_bar + 2; 
               end
               else
               begin
                  cur_bar = cur_bar + 1; 
               end 
            end 
            sel_bar = 7 ; // Invalid BAR if we get this far...
         end
      end
   endtask

localparam interrupt_pin_reg = 60;
localparam msi_capabilities = 32'h50;

   task legacy_interrupt_test;
      input bus_num; 
      integer bus_num;
      input dev_num; 
      integer dev_num;
      input fnc_num; 
      integer fnc_num;
      input bar_table; 
      integer bar_table;
      input bar_num; 
      integer bar_num;
      reg [7:0] intx;
      integer shared_memory_address;
      reg [15:0] msi_control_register;
      reg msi_enable;
      reg restore_msi_reg;
      reg [2:0] compl_status;
      integer dummy;
      begin 
         shared_memory_address = 0;
         restore_msi_reg = 0;
         // Query the endpoint to see if it supports Legacy Interrupts
         ebfm_cfgrd_wait(bus_num, dev_num, fnc_num, interrupt_pin_reg, 4, shared_memory_address, compl_status); 
         intx = shmem_read(shared_memory_address + 1, 1);
         dummy = ebfm_display(EBFM_MSG_INFO,"Starting Legacy Interrupt Test.");
          case (intx)
            8'h00:   begin
                        dummy = ebfm_display(EBFM_MSG_INFO,"  There is no legacy interrupt support for this endpoint.");
                        dummy = ebfm_display(EBFM_MSG_INFO,"  Skipping Legacy Interrupt Test.");
                        disable legacy_interrupt_test;
                     end
            8'h01:   dummy = ebfm_display(EBFM_MSG_INFO,"  Endpoint uses INTA#");
            8'h02:   dummy = ebfm_display(EBFM_MSG_INFO,"  Endpoint uses INTB#");
            8'h03:   dummy = ebfm_display(EBFM_MSG_INFO,"  Endpoint uses INTC#");
            8'h04:   dummy = ebfm_display(EBFM_MSG_INFO,"  Endpoint uses INTD#");
            default: dummy = ebfm_display(EBFM_MSG_ERROR_FATAL,{"  Interrupt Pin: Interrupt Pin Registers has Illegal Value ", himage2(intx)});
         endcase

         shared_memory_address = shared_memory_address + 4;
         // Read the MSI capabilities register
         dummy = ebfm_display(EBFM_MSG_INFO,"  Checking status of MSI register");
         ebfm_cfgrd_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, shared_memory_address, compl_status); 
         msi_control_register = shmem_read(shared_memory_address+2, 2);
         msi_enable = msi_control_register[0];
         // If the Message Signaled Interrupt (MSI) is enabled, disable it to run the Legacy Interrupt test
         if (msi_enable)
            begin
               dummy = ebfm_display(EBFM_MSG_INFO,"  Disabling MSI enable bit to perform legacy interrupt test");
               ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, {msi_control_register[15:1], 1'b0, 16'h0000}, compl_status);
               ebfm_cfgrd_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, shared_memory_address, compl_status); 
               msi_control_register = shmem_read(shared_memory_address+2, 2);
               msi_enable = msi_control_register[0];
               restore_msi_reg = 1'b1;
            end

         // Write to the example design interrupt register to generate the interrupt
         dummy = ebfm_display(EBFM_MSG_INFO,"  Writing Interrupt Register in Endpoint Example Design to Trigger a Legacy Interrupt");
         ebfm_barwr_imm(bar_table, bar_num, 8'h10, 32'h0000_0001, 4, 0); 
         // Check to see if the INTx was received
         dummy = ebfm_display(EBFM_MSG_INFO,"  Root Complex Checking for Received Interrupt");
         fork
            begin:timeout_intx
               repeat (3000) @(posedge clk_in);
               dummy = ebfm_display(EBFM_MSG_INFO,"  Error: The legacy interrupt receive-timer timed out");
               disable wait_for_intx;
            end
            begin:wait_for_intx
               wait (|{INTD, INTC, INTB, INTA});
               disable timeout_intx;
            end
         join
         case ({INTD, INTC, INTB, INTA})
            8'h00:   begin
                         dummy = ebfm_display(EBFM_MSG_INFO,"  No interrupt detected");
                         dummy = ebfm_display(EBFM_MSG_ERROR_FATAL,"  Legacy interrupt test failed");
                         disable legacy_interrupt_test;
                      end
            8'h01:   dummy = ebfm_display(EBFM_MSG_INFO,"  Detected INTA#, Legacy interrupt test okay!");
            8'h02:   dummy = ebfm_display(EBFM_MSG_INFO,"  Detected INTB#, Legacy interrupt test okay!");
            8'h03:   dummy = ebfm_display(EBFM_MSG_INFO,"  Detected INTC#, Legacy interrupt test okay!");
            8'h04:   dummy = ebfm_display(EBFM_MSG_INFO,"  Detected INTD#, Legacy interrupt test okay!");
            default: dummy = ebfm_display(EBFM_MSG_ERROR_FATAL,"  Illegal value detected on INTx lines, Legacy interrupt test failed");
         endcase
         // Re-enable the MSI if it was disabled
         if (restore_msi_reg)
            begin
               ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, {msi_control_register[15:1], 1'b1, 16'h0000}, compl_status);
            end
         // Clear interrupt from example design interrupt register
         ebfm_barwr_imm(bar_table, bar_num, 8'h10, 32'h0000_0000, 4, 0); 
      end
   endtask

   task message_signaled_interrupt_test;
      localparam msi_address = 32'h0000_0000;           // The Root Complex BFM has 2MB of address space
      localparam msi_upper_address = 32'h0000_0000;
      localparam msi_data = 16'habc0;
      input bus_num; 
      integer bus_num;
      input dev_num; 
      integer dev_num;
      input fnc_num; 
      integer fnc_num;
      input bar_table;
      integer bar_table;
      input bar_num;
      integer bar_num;
      integer msi_max;
      integer msi_expected;
      integer msi_received;
      integer dummy;
      integer shared_memory_address;
      reg [ 5:0] msi_number;
      reg [ 2:0] msi_traffic_class;
      reg [15:0] msi_control_register;
      reg        msi_64b_capable;
      reg [2:0]  multi_message_enable;
      reg [2:0]  multi_message_capable;
      reg        msi_enable;
      reg [2:0] compl_status;
      begin
         // The MSI test checks the MSI Control register for the requested number of MSI
         // Then initializes the MSI registers and performs the test
         shared_memory_address = 0;
         msi_traffic_class     = 0;
         dummy = ebfm_display(EBFM_MSG_INFO,"Starting Message Signaled Interrupt Test.");
         // Read the contents of the MSI Control register
         ebfm_cfgrd_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, shared_memory_address, compl_status); 
         msi_control_register  = shmem_read(shared_memory_address+2, 2);
         msi_64b_capable       = msi_control_register[7];
         // Enable the MSI with Maximum Number of Supported Messages
         multi_message_capable = msi_control_register[3:1];
         multi_message_enable  = multi_message_capable;
         msi_enable            = 1'b1;
         // Program the MSI Message Control register for testing
         ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, {8'h00, msi_64b_capable, multi_message_enable, multi_message_capable, msi_enable, 16'h0000}, compl_status);
         // Write the rest of the MSI Capabilities Structure: Address and Data Fields
         if (msi_64b_capable) // 64-bit Addressing
         begin
            ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities + 4'h4, 4, msi_address,       compl_status);
            ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities + 4'h8, 4, msi_upper_address, compl_status);
            ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities + 4'hC, 4, msi_data,          compl_status);
         end
         else // 32-bit Addressing
         begin
            ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities + 4'h4, 4, msi_address, compl_status);
            ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities + 4'h8, 4, msi_data,    compl_status);
         end
         // Calculate number of MSIs to test from the Message Control register field
         msi_max = (multi_message_enable == 0)? 1: (2 << multi_message_enable - 1);
         // Loop to check all possible MSIs in the endpoint
         for (msi_number = 5'b0_0000; msi_number < msi_max; msi_number = msi_number + 5'b0_0001)
         begin
            // Write to the example design to trigger the MSI
            dummy = ebfm_display(EBFM_MSG_INFO,{"  Writing Interrupt Register in Endpoint Example Design to Trigger MSI ", dimage2(msi_number)});
            ebfm_barwr_imm(bar_table, bar_num, 8'h10, {16'h0000, 1'b0, msi_traffic_class, 3'b000, msi_number[4:0], 4'h1}, 4, 0); 
            // Wait to receive the MSI from the endpoint
            fork
            // Set timeout failure if expected MSI is not received
            begin:timeout_msi
               repeat (3000) @(posedge clk_in);
               dummy = ebfm_display(EBFM_MSG_ERROR_FATAL, "MSI timeout occured, MSI never received, Test Fails");
               disable wait_for_msi;
            end
            // Polling memory for expected MSI data value at the assigned MSI address location
            begin:wait_for_msi
               forever
               begin
                  repeat (50) @(posedge clk_in);
                  dummy = ebfm_display(EBFM_MSG_INFO, "  Polling MSI Address");
                  msi_received = shmem_read (msi_address, 2); 
                  dummy = ebfm_display(EBFM_MSG_INFO, {"    Received data ", himage4(msi_received)});
                  // MSI data value written at the MSI address contains the MSI data register value
                  // and the number of the MSI reported.  This combination changes with the amount
                  // of MSI enabled in the endpoint.
                  case (multi_message_enable)
                  3'b000:  msi_expected =  msi_data[15:0];
                  3'b001:  msi_expected = {msi_data[15:1], msi_number[0]  };
                  3'b010:  msi_expected = {msi_data[15:2], msi_number[1:0]};
                  3'b011:  msi_expected = {msi_data[15:3], msi_number[2:0]};
                  3'b100:  msi_expected = {msi_data[15:4], msi_number[3:0]};
                  3'b101:  msi_expected = {msi_data[15:5], msi_number[4:0]};
                  default: dummy = ebfm_display(EBFM_MSG_ERROR_FATAL, "Illegal multi_message_enable value detected. MSI test fails.");
                  endcase
                  if (msi_received == msi_expected)
                  begin
                     dummy = ebfm_display(EBFM_MSG_INFO, {"    Received Expected MSI : ", himage4(msi_received)});
                     shmem_write( shared_memory_address, 16'h0000, 2);
                     disable timeout_msi;
                     disable wait_for_msi;
                  end
               end
            end
            join
            // Reset the interrupt register in the endpoint example design
            ebfm_barwr_imm(bar_table, bar_num, 8'h10, 32'h0000_0000, 4, 0); 
         end
         // Return the MSI control register to its original state
         dummy = ebfm_display(EBFM_MSG_INFO, "  Returning the MSI Message Control Register to pre-test state.");
         ebfm_cfgwr_imm_wait(bus_num, dev_num, fnc_num, msi_capabilities, 4, {msi_control_register[15:0], 16'h0000}, compl_status);
         dummy = ebfm_display(EBFM_MSG_INFO, "  Message Signaled Interrupt test okay!");
      end
   endtask

   reg activity_toggle; 
   reg timer_toggle ;
   time time_stamp ;
   localparam TIMEOUT = 200000000 ; // 200 us

   initial
     begin
        time_stamp = $time ;
        activity_toggle = 1'b0;
        timer_toggle    = 1'b0;
   end

   always 
   begin : main  // behavioral

      // If you want to relocate the bar_table, modify the BAR_TABLE_POINTER in altpcietb_bfm_shmem.
      // Directly modifying the bar_table at this location may disable overwrite protection for the bar_table
      // If the bar_table is overwritten incorrectly, this will break the testbench functionality.

      // This constant defines where we save the sizes and programmed addresses
      // of the Endpoint Device Under Test BARs 
      parameter bar_table = BAR_TABLE_POINTER; // Default BAR_TABLE_SIZE is 64 bytes

      // tgt_bar indicates which bar to use for testing the target memory of the
      // reference design.
      integer tgt_bar; 
      integer dma_bar;
      reg     addr_map_4GB_limit;
      reg     dummy ;
      

      // Setup the Root Port and Endpoint Configuration Spaces
      addr_map_4GB_limit = 1'b0;
      ebfm_cfg_rp_ep(
                     bar_table,         // BAR Size/Address info for Endpoint       
                     1,                 // Bus Number for Endpoint Under Test
                     1,                 // Device Number for Endpoint Under Test
                     512,               // Maximum Read Request Size for Root Port
                     1,                 // Display EP Config Space after setup
                     addr_map_4GB_limit // Limit the BAR assignments to 4GB address map
                     ); 
      activity_toggle <= ~activity_toggle ; 

      // Find a memory BAR to use to test the target memory
      // The reference design implements the target memory on BARs 0,1, 4 or 5
      // We need one at least 4 KB big
      find_mem_bar(bar_table, 6'b110011, 12, tgt_bar); 
      // Test the reference design's target memory
      if (tgt_bar < 6)
      begin
         target_mem_test(
                         bar_table, // BAR Size/Address info for Endpoint
                         tgt_bar,   // BAR to access target memory with
                         0,         // Starting offset from BAR
                         4096       // Length of memory to test
                         );
      end
      else
      begin
         dummy = ebfm_display(EBFM_MSG_WARNING, "Unable to find a 4 KB BAR to test Target Memory, skipping target test."); 
      end 
      activity_toggle <= ~activity_toggle ; 
      // Find a memory BAR to use to setup the DMA channel
      // The reference design implements the DMA channel registers on BAR 2 or 3
      // We need one at least 128 B big
      find_mem_bar(bar_table, 6'b001100, 7, dma_bar); 
      // Test the reference design's DMA channel and master memory
      if (dma_bar < 6)
      begin
         dma_mem_test(
                      bar_table, // BAR Size/Address info for Endpoint
                      dma_bar,   // BAR to access DMA control registers
                      0,         // Starting offset of DMA memory
                      4096       // Length of memory to test
                      );
      end
      else
      begin
         dummy = ebfm_display(EBFM_MSG_WARNING, "Unable to find a 128B BAR to test setup DMA channel, skipping DMA test."); 
      end 
      // The DMA BAR also contains the Example Design registers for triggering interrupts
      // Use the same BAR as the DMA test
      // Test both MSI and Legacy Interrupts if available
      if (dma_bar < 6)
      begin
         legacy_interrupt_test(1, 1, 0, bar_table, dma_bar);
         message_signaled_interrupt_test(1, 1, 0, bar_table, dma_bar);
      end
      else
      begin
         dummy = ebfm_display(EBFM_MSG_WARNING, "Unable to find a 128B BAR to test interrupts, skipping Interrupt tests."); 
      end 

      // Stop the simulator and indicate successful completion
      dummy = ebfm_log_stop_sim(1); 
      forever #100000; 
   end 

   always
     begin
        #(TIMEOUT)
          timer_toggle <= ! timer_toggle ;
     end
   
   reg dummy ;
   // purpose: this is a watchdog timer, if it sees no activity on the activity
   // toggle signal for 200 us it ends the simulation
   always @(activity_toggle or timer_toggle)
     begin : watchdog
        
        if ( ($time - time_stamp) >= TIMEOUT)
          begin
             dummy = ebfm_display(EBFM_MSG_ERROR_FATAL, "Simulation stopped due to inactivity!"); 
          end
        time_stamp <= $time ;
     end

   always @(INTA)
      begin
         if (INTA)
           dummy = ebfm_display(EBFM_MSG_INFO, "Interrupt Monitor: Interrupt INTA Asserted");
           else
           dummy = ebfm_display(EBFM_MSG_INFO, "Interrupt Monitor: Interrupt INTA Deasserted");
      end

   always @(INTB)
      begin
         if (INTB)
           dummy = ebfm_display(EBFM_MSG_INFO, "Interrupt Monitor: Interrupt INTB Asserted");
           else
           dummy = ebfm_display(EBFM_MSG_INFO, "Interrupt Monitor: Interrupt INTB Deasserted");
      end
   always @(INTC)
      begin
         if (INTC)
           dummy = ebfm_display(EBFM_MSG_INFO, "Interrupt Monitor: Interrupt INTC Asserted");
           else
           dummy = ebfm_display(EBFM_MSG_INFO, "Interrupt Monitor: Interrupt INTC Deasserted");
      end

   always @(INTD)
      begin
         if (INTD)
           dummy = ebfm_display(EBFM_MSG_INFO, "Interrupt Monitor: Interrupt INTD Asserted");
           else
           dummy = ebfm_display(EBFM_MSG_INFO, "Interrupt Monitor: Interrupt INTD Deasserted");
      end

endmodule
