
// /**
// *
// *  This Verilog HDL File is used for simulation and synthesis in the simple DMA design example.  
// *  It provides master and slave arbitration on the transmit interface for each VC.
// *
// */
// synthesis verilog_input_version verilog_2001
`timescale 1 ps / 1 ps 
//-----------------------------------------------------------------------------
// Title         : PCI Express Reference Design apptxchan Entity
// Project       : PCI Express MegaCore function
//-----------------------------------------------------------------------------
// File          : altpcierd_apptx_streaming_intf.v
// Author        : Altera Corporation
//-----------------------------------------------------------------------------
// Description :
// This module arbitrates master and slave transmission requests for the 
// streaming interface.  It also generates signals for the streaming interface.
//-----------------------------------------------------------------------------
// Copyright (c) 2006 Altera Corporation. All rights reserved.  Altera products are
// protected under numerous U.S. and foreign patents, maskwork rights, copyrights and
// other intellectual property laws.  
//
// This reference design file, and your use thereof, is subject to and governed by
// the terms and conditions of the applicable Altera Reference Design License Agreement.
// By using this reference design file, you indicate your acceptance of such terms and
// conditions between you and Altera Corporation.  In the event that you do not agree with
// such terms and conditions, you may not use the reference design file. Please promptly
// destroy any copies you have made.
//
// This reference design file being provided on an "as-is" basis and as an accommodation 
// and therefore all warranties, representations or guarantees of any kind 
// (whether express, implied or statutory) including, without limitation, warranties of 
// merchantability, non-infringement, or fitness for a particular purpose, are 
// specifically disclaimed.  By making this reference design file available, Altera
// expressly does not recommend, suggest or require that this reference design file be
// used in combination with any other product not provided by Altera. 
//-----------------------------------------------------------------------------

 
 
module altpcierd_apptx_streaming_intf #(
                                          parameter TL_SELECTION = 6    // streaming interface format selection.
                                                                        // 0 = ICM format.  6 = 7.2 format.
                                         )  ( clk_in, rstn,  
                                              m_req, m_ack, m_data_ack, m_tx_desc,
                                              m_tx_data,  m_last_datacyc, 
                                              s_req, s_ack, s_data_ack, s_tx_desc,
                                              s_tx_data,   s_last_datacyc,  
                                              stream_ready, stream_wrdata, stream_wr,
                                              cpl_pending, m_tx_err, s_tx_err);  
          

   
   input         clk_in; 
   input         rstn;  
   input         m_req;                    // tx request from master 
   input         m_last_datacyc;           // indicates that current cycle is last cycle of packet
   input[127:0]  m_tx_desc;                // master descriptor
   input[63:0]   m_tx_data;                // master payload data
   input         m_tx_err;                 // master tx_err bit 
   input         s_req;                    // tx request from slave  
   input         s_last_datacyc;           // indicates that current cycle is last cycle of packet
   input[127:0]  s_tx_desc;                // slave descriptor 
   input[63:0]   s_tx_data;                // slave payload data
   input         s_tx_err;                 // slave tx_err bit 
   input         cpl_pending;              // completion pending status flag
   input         stream_ready;             // streaming i/f throttles data fetching.  when set, allows another data fetch.

   output        m_ack;                    // grant access to master request
   output        s_ack;                    // grant access to slave request   
   output        m_data_ack;               // fetch next desc/data from master
   output        s_data_ack;               // fetch next desc/data from slave  
   output[107:0] stream_wrdata;            // data written to streaming i/f
   output        stream_wr;                // writes stream_wrdata to streaming i/f 


   // tx_state SM states
   parameter  TX_WAIT_REQ   = 2'h0;   // arbitrate between master and slave, write desc_hi
   parameter  TX_SEND_DESC2 = 2'h1;   // write desc_lo, fetch data
   parameter  TX_NULL_CYCLE = 2'h2;   // data-less pkt.  send ack only.
   parameter  TX_SEND_DATA  = 2'h3;   // write data, fetch data
   
   wire[107:0]   stream_wrdata;            
   reg           stream_wrdata_sop;        // part of stream_wrdata vector.  Start of Pkt flag.   
   reg           stream_wrdata_eop;        // part of stream_wrdata vector.  End of Pkt flag. 
   reg           stream_wrdata_tx_err;     // part of stream_wrdata vector.  tx_err bit. 
   reg[63:0]     stream_wrdata_txdata;     // part of stream_wrdata vector.  tx_desc/data.  
   
   reg           stream_wr;             
   reg[1:0]      tx_state;                 // state machine
   reg           m_ack;                    
   reg           s_ack;                    
   reg           stream_ready_del;         
   reg           sel_master;               // indicate that master has access
   reg           sel_slave;                // indicate that slave has access
   reg           m_data_ack;                  
   reg           s_data_ack;                            
   reg[63:0]     m_tx_data_hold_r;         // last value of m_tx_data
   reg[63:0]     s_tx_data_hold_r;         // last value of s_tx_data
   reg           m_stream_pkt_end_hold_r;  // last value of m_last_datacyc
   reg           s_stream_pkt_end_hold_r;  // last value of s_last_datacyc 
   reg           has_payload;              // indicates that the pkt has payload
   reg           data_ack_cycle;           // indicates that fetched data is ready at input.  
   reg           pkt_ended;                // indicates that the last data cycle has been received.  finish transporting the data, and do not fetch more until ready to service a new pkt.
   reg[63:0]     tx_desc_hold_lo;          // memorize the low 8 bytes of the descriptor
   reg           last_was_slave;           // means that the last tx_req service was the slave (- master gets priority)


   reg           pack_dw;                  // pack desc & data when 3DW header, with payload, and address not aligned.

   
   wire[32:0]   zero_33;  assign zero_33 = 33'h0;
   wire[7:0]    zero_8;   assign zero_8  = 8'h0;
   
   assign stream_wrdata = {zero_33, stream_wrdata_tx_err, stream_wrdata_sop, stream_wrdata_eop, zero_8, stream_wrdata_txdata};
   
   //------------------------------------------------------------
   //    Streaming Interface - Input register boundary
   //------------------------------------------------------------

   always @(negedge rstn or posedge clk_in) begin
      if (rstn == 1'b0) begin
          stream_ready_del <= 1'b0;
      end
      else begin
          stream_ready_del <= stream_ready;
      end
   end
   
   //------------------------------------------------------------
   //    Arbitration between master and target for transmission.
   //    Multiplex Descriptor and Data onto Streaming data bus.
   //    Generate Streaming interface SOP/EOP flags.
   //------------------------------------------------------------
   

 
  
   always @(negedge rstn or posedge clk_in) begin
      if (rstn == 1'b0) begin
          tx_state                <= TX_WAIT_REQ; 
          m_ack                   <= 1'b0;
          s_ack                   <= 1'b0;
          sel_master              <= 1'b0;
          sel_slave               <= 1'b0;
          m_data_ack              <= 1'b0;
          s_data_ack              <= 1'b0;
          stream_wrdata_sop       <= 1'b0;     
          stream_wrdata_eop       <= 1'b0;
          stream_wrdata_tx_err    <= 1'b0;
          stream_wrdata_txdata    <= 64'h0;
          stream_wr               <= 1'b0;
          data_ack_cycle          <= 1'b0;
          m_tx_data_hold_r        <= 64'h0;
          s_tx_data_hold_r        <= 64'h0;
          m_stream_pkt_end_hold_r <= 1'b0;
          s_stream_pkt_end_hold_r <= 1'b0; 
          has_payload             <= 1'b0;
          pkt_ended               <= 1'b0;
          tx_desc_hold_lo         <= 64'h0; 
          last_was_slave          <= 1'b0;
          pack_dw                 <= 1'b0;
      end
      else begin  
         //defaults 
         m_ack                    <= 1'b0;
         s_ack                    <= 1'b0;  
         m_data_ack               <= 1'b0;
         s_data_ack               <= 1'b0;      
         stream_wr                <= 1'b0;
         stream_wrdata_sop        <= 1'b0;
         stream_wrdata_eop        <= 1'b0;
         data_ack_cycle           <= stream_ready_del;   // when dataphase present, m_data_ack and s_data_ack assert on same cycle as data_ack_cycle
         pkt_ended                <= 1'b0; 
         
         if (sel_master)
             stream_wrdata_tx_err <= m_tx_err;
         else
             stream_wrdata_tx_err <= s_tx_err;  
         
         // hold last value of master/slave data 
         if (data_ack_cycle) begin
             m_tx_data_hold_r         <= m_tx_data;  
             s_tx_data_hold_r         <= s_tx_data;    
             m_stream_pkt_end_hold_r  <= m_last_datacyc;
             s_stream_pkt_end_hold_r  <= s_last_datacyc;  
         end
         else begin
             m_tx_data_hold_r         <= m_tx_data_hold_r;  
             s_tx_data_hold_r         <= s_tx_data_hold_r;    
             m_stream_pkt_end_hold_r  <= m_stream_pkt_end_hold_r;
             s_stream_pkt_end_hold_r  <= s_stream_pkt_end_hold_r; 
         end
         
         /*******************************************************************************
           PROPOSE m_data_ack <= sel_master & stream_ready_del & ~pkt_ended;
         ********************************************************************************/
         
         // issue ack (tx request grant) and ready (data throttling) on stream_ready_del.
         // data is valid one clock cycle later (data_ack_cycle) -- do all data evaluations on this cycle. 
         case (tx_state) 
             TX_WAIT_REQ: begin
                 // arbitrate tx request from master and slave
                 // when streaming i/f can accept more data (stream_ready_del):
                 //    - write desc_hi to streaming i/f data bus
                 //    - fetch desc_lo 
                 if (m_req & stream_ready_del & (last_was_slave | ~s_req)) begin
                     // grant master if last req was from slave or 
                     // if slave is not requesting access.
                     sel_master  <= 1'b1;
                     m_ack       <= 1'b1;                                                    
                     m_data_ack  <= 1'b1;            
                     has_payload <= m_tx_desc[126];   
                     if (TL_SELECTION == 0) begin           
                         stream_wrdata_txdata <= m_tx_desc[127:64];  
                         tx_desc_hold_lo      <= m_tx_desc[63:0];  
                         pack_dw              <= 1'b0;
                     end
                     else begin
                         tx_desc_hold_lo      <= m_tx_desc[63:0]; 
                         stream_wrdata_txdata <= {m_tx_desc[95:64], m_tx_desc[127:96]};                  // ADAPTOR DWORD SWAP 
                         pack_dw              <= m_tx_desc[126] & ~m_tx_desc[125] & m_tx_desc[34];       // has payld, 3DW hder, nonaligned address
                     end  
                 end
                 else if (s_req & stream_ready_del) begin
                     // grant slave if master is not requesting access or 
                     // if last request serviced was from master.
                     sel_slave   <= 1'b1;
                     s_ack       <= 1'b1;
                     s_data_ack  <= 1'b1; 
                     has_payload <= s_tx_desc[126];  
                     if (TL_SELECTION == 0) begin
                         stream_wrdata_txdata <= s_tx_desc[127:64];   
                         tx_desc_hold_lo <= s_tx_desc[63:0];
                         pack_dw         <= 1'b0;
                     end
                     else begin
                         stream_wrdata_txdata <= {s_tx_desc[95:64], s_tx_desc[127:96]};                   // ADAPTOR DWORD SWAP
                         tx_desc_hold_lo <= s_tx_desc[63:0];                   
                         pack_dw         <= s_tx_desc[126] & ~s_tx_desc[125] & s_tx_desc[34];         // has payld, 3DW hder, nonaligned address
                     end
                 end      
                 
                 if ((m_req | s_req) & stream_ready_del) 
                     tx_state  <= TX_SEND_DESC2;    
                 
                 stream_wrdata_sop <= (m_req | s_req);                          // write sop flag
                 stream_wr         <= ((m_req | s_req) & stream_ready_del);     // write data and flag to streaming i/f
             end 
             TX_SEND_DESC2: begin 
             
                 // if streaming i/f can accept more data (stream_ready_del):
                 //    - fetch data if streaming i/f can accept more data 
                 last_was_slave <= sel_slave;
                 m_data_ack     <= sel_master & stream_ready_del;
                 s_data_ack     <= sel_slave  & stream_ready_del;
                 // write desc_lo to streaming i/f  
                 // any data that is fetched, is accepted by 
                 // streaming i/f (data is only fetched if 
                 // streaming i/f can accept it).
                 
                  if (TL_SELECTION == 0) begin
                      // no desc/data packing
                      if (data_ack_cycle) begin   
                          if (has_payload)
                              tx_state <= TX_SEND_DATA;
                          else
                               tx_state <= TX_NULL_CYCLE; 
                               
                          stream_wrdata_txdata <= tx_desc_hold_lo; 
                          stream_wrdata_eop    <= ~has_payload;
                          stream_wr            <= 1'b1;  
                      end
                  end 
                  else begin
                     if (pack_dw) begin
                         if (data_ack_cycle) begin   
                             if (has_payload & ((sel_master & ~m_last_datacyc) | (sel_slave &  ~s_last_datacyc)))
                                 tx_state <= TX_SEND_DATA;
                             else 
                                 tx_state <= TX_NULL_CYCLE; 
                                 
                             if (sel_master)   
                                 stream_wrdata_txdata <= {m_tx_data[63:32], tx_desc_hold_lo[63:32]};
                             else
                                 stream_wrdata_txdata <= {s_tx_data[63:32], tx_desc_hold_lo[63:32]};  

                             stream_wrdata_eop <= ~has_payload | (sel_slave & s_last_datacyc) | (sel_master & m_last_datacyc);
                             stream_wr         <= 1'b1;  
                         end
                      end
                      else begin 
                         if (data_ack_cycle) begin   
                             if (has_payload)
                                 tx_state <= TX_SEND_DATA;
                             else 
                                 tx_state <= TX_NULL_CYCLE; 
                                 
                             stream_wrdata_txdata <= {tx_desc_hold_lo[31:0], tx_desc_hold_lo[63:32]}; 
                             stream_wrdata_eop    <= ~has_payload;
                             stream_wr            <= 1'b1;  
                         end
                      end
                  end                
             end    
             TX_SEND_DATA: begin 
                 // whenever streaming i/f can accept more data, do fetch. 
                 // fetch until end of pkt is detected. 
                 m_data_ack   <= sel_master & stream_ready_del & ~pkt_ended;  
                 s_data_ack   <= sel_slave  & stream_ready_del & ~pkt_ended;  
                 
                 if ((sel_master & m_data_ack & m_last_datacyc ) | (sel_slave  & s_data_ack & s_last_datacyc))
                     pkt_ended <= 1'b1;
                 else
                     pkt_ended <= pkt_ended;
                     
                 if (pack_dw) begin
                     // when packing data, data cycles are not deferred
                     if (data_ack_cycle) begin    
                         if ((sel_master & m_last_datacyc) | (sel_slave & s_last_datacyc)) begin
                             tx_state   <= TX_WAIT_REQ; 
                         end
                         
                         if (sel_master) begin
                             stream_wrdata_txdata <= m_tx_data;
                             stream_wrdata_eop    <= m_last_datacyc;
                         end
                         else begin
                             stream_wrdata_txdata <= s_tx_data;
                             stream_wrdata_eop    <= s_last_datacyc;
                         end
                         
                         stream_wr  <= 1'b1;
                        
                         if (m_last_datacyc) 
                             sel_master <= 1'b0;   
                         else
                             sel_master <= sel_master;
                             
                         if (s_last_datacyc) 
                             sel_slave <=  1'b0;
                         else
                             sel_slave <= sel_slave;
                         
                     end                 
                 end
                 else begin
                     // every time input data is advanced, write the previous data to streaming i/f.
                     // because they are multiplexed onto the same bus, data coinciding with the desc 
                     // phase is held and written on the next cycle.  
                     // all data thereafter is also held and written one cycle later.
                     // when end of pkt is written, return to start  
                     if (data_ack_cycle) begin     
                         if ((sel_master & m_stream_pkt_end_hold_r) | (sel_slave  & s_stream_pkt_end_hold_r))
                             tx_state <= TX_WAIT_REQ;
                             
                         if (sel_master) begin    
                             stream_wrdata_txdata <= m_tx_data_hold_r;
                             stream_wrdata_eop    <= m_stream_pkt_end_hold_r;
                         end
                         else begin
                             stream_wrdata_txdata <= s_tx_data_hold_r;
                             stream_wrdata_eop    <= s_stream_pkt_end_hold_r;
                         end
                         
                         stream_wr  <= 1'b1;
                         
                         if (m_stream_pkt_end_hold_r) 
                             sel_master <= 1'b0; 
                         else
                             sel_master <= sel_master;
                             
                         if (s_stream_pkt_end_hold_r) 
                             sel_slave <= 1'b0;
                         else
                             sel_slave <= sel_slave;
                             
                     end
                 end
             end 
             TX_NULL_CYCLE: begin
                // no data phase.  ack being issued on this cycle 
                tx_state   <= TX_WAIT_REQ;
                sel_master <= 1'b0;
                sel_slave  <= 1'b0;
             end     
         endcase
       end
    end  
    
                     
endmodule
