// synthesis verilog_version verilog_2001
`timescale 1 ps / 1 ps 
//-----------------------------------------------------------------------------
// Title         : PCI Express Reference Design Example Application 
// Project       : PCI Express MegaCore function
//-----------------------------------------------------------------------------
// File          : altpcierd_apprxstream_formatter.v
// Author        : Altera Corporation
//-----------------------------------------------------------------------------
// Description :
// This module extracts packet information from the streaming interface and
// routes the packets to the master or slave of the "simple dma" design example.
//-----------------------------------------------------------------------------
// Copyright (c) 2006 Altera Corporation. All rights reserved.  Altera products are
// protected under numerous U.S. and foreign patents, maskwork rights, copyrights and
// other intellectual property laws.  
//
// This reference design file, and your use thereof, is subject to and governed by
// the terms and conditions of the applicable Altera Reference Design License Agreement.
// By using this reference design file, you indicate your acceptance of such terms and
// conditions between you and Altera Corporation.  In the event that you do not agree with
// such terms and conditions, you may not use the reference design file. Please promptly
// destroy any copies you have made.
//
// This reference design file being provided on an "as-is" basis and as an accommodation 
// and therefore all warranties, representations or guarantees of any kind 
// (whether express, implied or statutory) including, without limitation, warranties of 
// merchantability, non-infringement, or fitness for a particular purpose, are 
// specifically disclaimed.  By making this reference design file available, Altera
// expressly does not recommend, suggest or require that this reference design file be
// used in combination with any other product not provided by Altera. 
//----------------------------------------------------------------------------- 
 
 
module altpcierd_apprxstream_formatter #(
                                          parameter TL_SELECTION = 6   // streaming interface format selection.
                                                                       //  0 = ICM format.  6 = 7.2 format.
                                         ) (clk, rstn, fifo_valid, fifo_data, rx_st_data, rx_st_valid,
                                             rx_st_ready,  fifo_ready);
 
   // rx_stream_state 
   // module local parameters
   parameter     SOP_CYCLE              = 3'h0;   // wait for 1st desc cycle of any pkt
   parameter     H3DW_PAYLD_DESC2_CYCLE = 3'h1;   // wait for 2nd desc cycle of 3DW Payload Pkt
   parameter     SEND_DELAYED_DATA      = 3'h2;   // receiving data from 3DW non-aligned pkt (desc/data unpacking required)
   parameter     SEND_NORMAL_DATA_EOP   = 3'h3;   // receiving data from aligned or 4DW pkt (no desc/data unpacking required)
   parameter     SEND_DELAYED_EOP       = 3'h4;   // sends last 'delayed data' for 3DW non-aligned pkts
 
   input         clk;  
   input         rstn;     
   input         fifo_valid;        // from streaming intf.  means streaming data (fifo_data) is valid
   input[107:0]  fifo_data;         // from streaming intf.  data from streaming interface
   input         rx_st_ready;       // from application.  streaming intf throttle control.
   
   output[107:0] rx_st_data;       // to application.  reformatted streaming data.
   output        rx_st_valid;      // to application.  means rx_st_data is valid.
   output        fifo_ready;       // to streaming interface.  streaming intf throttle control.  
   
   reg[107:0]    rx_st_data;     
    
   reg           rx_st_valid;
   reg[107:0]    fifo_data_last;    // previous fifo_data
   reg[7:0]      fifo_be_data_last;      // previous byte enable field of fifo_data
   wire[7:0]     fifo_be_data;         // byte enable field of fifo_data 
   wire          fifo_ready;
   reg           rx_st_ready_del;
   
   reg[2:0]      rx_stream_state;    // state machine - mux controls for reformatting the output streaming data 
   
   reg       rx_sop;
   reg       rx_eop;
   reg[63:0] rx_data;
   reg[7:0]  rx_be;
   reg[7:0]  rx_bar;
   reg[7:0]  rx_bar_r;
   
   wire      fifo_data_eop;
   wire      fifo_data_sop;
   reg       fifo_data_sop_last;

   wire[63:0]  fifo_data_swapped;
   wire[63:0]  fifo_data_shifted;
   
   assign    fifo_data_eop = fifo_data[72];
   assign    fifo_data_sop = fifo_data[73];



   // parse byte enable bits    
   assign fifo_be_data = fifo_data[81:74];
   
   // read from streaming FIFO as long as application can accept 
   // more data.  except when inserting Delayed EOP --
   // last cycle data is sourced from this module rather than from fifo.
    
   // allow to read ahead on fifo_data_eop even tho we need to insert a cycle.  
   // defer hold-off of read to next cycle.

     assign fifo_ready = (rx_st_ready == 1'b1) & (rx_stream_state != SEND_DELAYED_EOP);    
    
   ////////////////
   // REGISTERS  
   ////////////////
   
   always @ (posedge clk or negedge rstn) begin
        if (~rstn) begin
            fifo_data_last     <= 108'h0;    
            fifo_data_sop_last <= 1'b0; 
            fifo_be_data_last  <= 8'h0;
            rx_stream_state    <= SOP_CYCLE; 
            rx_st_ready_del    <= 1'b0;
            rx_bar_r           <= 8'h0;
        end
        else begin 
            if (rx_st_ready) begin
                fifo_data_last     <= fifo_data; 
                fifo_data_sop_last <= fifo_data_sop;  
                fifo_be_data_last  <= fifo_be_data; 
            end 
            else begin
                fifo_data_last     <= fifo_data_last; 
                fifo_data_sop_last <= fifo_data_sop_last;  
                fifo_be_data_last  <= fifo_be_data_last; 
            end
          
            rx_st_ready_del    <= rx_st_ready; 
            rx_bar_r           <= rx_bar;
            
            //////////////////////////////////////////////////////////////
            // 7.2 state machine controls output data mux.
            // reformats streaming data input from 72 format to 71 format
            //////////////////////////////////////////////////////////////
            case (rx_stream_state) 
                SOP_CYCLE: begin
                    // wait for new pkt.
                    // if pkt has payload and is 3DW, wait for 2nd desc phase to decode address,
                    // else, there is no desc/data packing - pass data straight thru   
                    // in 7.2, data is dword swapped.  payload bit = 30.
                    if (rx_st_ready & fifo_data_sop) begin
                        if (fifo_data[30] & ~fifo_data[29])
                            rx_stream_state <= H3DW_PAYLD_DESC2_CYCLE;  
                        else 
                            rx_stream_state <= SEND_NORMAL_DATA_EOP; 
                    end
                end
                H3DW_PAYLD_DESC2_CYCLE: begin
                    if (rx_st_ready) begin
                        // header is 3DW and there is payload
                        if (fifo_data[2] & fifo_data_eop)           // 1DW packed payld
                            rx_stream_state <= SEND_DELAYED_EOP;
                        else if (fifo_data[2])                              // multiple DW packed payload (unaligned address)
                            rx_stream_state <= SEND_DELAYED_DATA;
                        else 
                            rx_stream_state <= SEND_NORMAL_DATA_EOP;      // non-packed payload 
                    end
                end
                SEND_DELAYED_DATA: begin
                    // payload cycles for packed 3DW pkts
                    if (rx_st_ready & fifo_data_eop) begin
                        rx_stream_state <= SEND_DELAYED_EOP;
                    end
                end
                SEND_NORMAL_DATA_EOP: begin
                    // for non-packed data, pass data thru as-is
                    if (rx_st_ready & fifo_data_eop) begin
                        rx_stream_state <= SOP_CYCLE;
                    end
                end
                SEND_DELAYED_EOP: begin  
                    // insert eop cycle when application requests Fifo read
                    // on 2nd to last data cycle
                    if (rx_st_ready)
                        rx_stream_state <= SOP_CYCLE;
                end   
            endcase
        end
   end
   
      
   
   ////////////////////////////////////////////////////////
   // MUXED OUTPUT TO APP
   // REFORMAT STREAMING DATA TO LOOK LIKE 71_ICM FORMAT
   //////////////////////////////////////////////////////// 
   
   wire[31:0] zero32;  assign zero32 = 32'h0; 
   wire[25:0] zero26;  assign zero26 = 26'h0; 
   
   assign fifo_data_swapped =  {fifo_data[31:0], fifo_data[63:32]} ; 
   assign fifo_data_shifted =  {fifo_data[31:0], zero32}; 
   
   always @ (fifo_valid or fifo_data or rx_be or rx_sop or rx_eop or rx_bar or rx_data or
             rx_st_ready_del or rx_stream_state or fifo_data_swapped or fifo_data_sop or 
             fifo_data_shifted or fifo_data_last or fifo_be_data_last or fifo_data_sop_last or 
             fifo_be_data or fifo_data_eop or rx_bar_r) begin  
             
       if (TL_SELECTION == 0) begin
           // streaming data is per ICM MODE
           // pass thru as-is 
           rx_st_valid <= fifo_valid;
           rx_st_data  <= fifo_data; 
           rx_sop      <= 1'b0;           
           rx_eop      <= 1'b0;           
           rx_bar      <= 8'h0;          
           rx_data     <= 64'h0;    
           rx_be       <= 8'h0;
       end  
       else begin 
           // streaming data is 7.2 MODE
           // convert to 7.1 MODE
           rx_st_data <= {zero26, rx_be, rx_sop, rx_eop, rx_bar, rx_data};
           
           // pass valid signal thru.  
           // add 1 cycle at end if data was packed -- generate valid 1 cycle after
           // application requests Fifo read & there is only EOP left to send.
           rx_st_valid <= fifo_valid | ((rx_stream_state == SEND_DELAYED_EOP) & (rx_st_ready_del==1'b1));  
            
         
          
           // reformat streaming data to match 7.1 ICM data  
           case (rx_stream_state)  
                SOP_CYCLE: begin
                    rx_data  <= fifo_data_swapped;  // DWord swap the descriptor
                    rx_be    <= 8'h0; 
                    rx_eop   <= 1'b0;
                    rx_sop   <= fifo_data_sop;
                    rx_bar   <= fifo_data[71:64];
                end
                H3DW_PAYLD_DESC2_CYCLE: begin
                   rx_data   <= fifo_data_shifted;             // Shift the descriptor
                   rx_be     <= 8'h0; 
                   rx_eop    <= 1'b0;
                   rx_sop   <= fifo_data_sop;
                   rx_bar   <= fifo_data[71:64];
                end
                SEND_DELAYED_DATA: begin
                   rx_data  <= fifo_data_last;
                   rx_be    <= fifo_be_data_last; 
                   rx_eop   <= 1'b0;
                   rx_sop   <= fifo_data_sop;
                   rx_bar   <= fifo_data[71:64]; 
                end
                SEND_NORMAL_DATA_EOP: begin 
                   if (fifo_data_sop_last)
                       rx_data  <= fifo_data_swapped;  // swap on descriptor cycle
                   else 
                       rx_data  <= fifo_data;          // no swap on data cycle
                       
                   rx_be    <= fifo_be_data;
                   rx_eop   <= fifo_data_eop; 
                   rx_sop   <= fifo_data_sop;
                   rx_bar   <= fifo_data[71:64];
                end
                SEND_DELAYED_EOP: begin
                   rx_data  <= fifo_data_last;
                   rx_be    <= fifo_be_data_last;
                   rx_eop   <= 1'b1; 
                   rx_sop   <= 1'b0;
                   rx_bar   <= rx_bar_r;
                end
                default: begin                      // Should never reach this
                    rx_data  <= fifo_data_swapped;  
                    rx_be    <= 8'h0; 
                    rx_eop   <= 1'b0;
                    rx_sop   <= fifo_data_sop;
                    rx_bar   <= fifo_data[71:64];
                end
            endcase 
       end
   end 
   


endmodule
