// synthesis verilog_version verilog_2001
`timescale 1 ps / 1 ps 
//-----------------------------------------------------------------------------
// Title         : PCI Express Reference Design Example Application 
// Project       : PCI Express MegaCore function
//-----------------------------------------------------------------------------
// File          : altpcierd_apprxstream_decode.v
// Author        : Altera Corporation
//-----------------------------------------------------------------------------
// Description :
// This module extracts packet information from the streaming interface and
// routes the packets to the master or slave of the "simple dma" design example.
//-----------------------------------------------------------------------------
// Copyright (c) 2006 Altera Corporation. All rights reserved.  Altera products are
// protected under numerous U.S. and foreign patents, maskwork rights, copyrights and
// other intellectual property laws.  
//
// This reference design file, and your use thereof, is subject to and governed by
// the terms and conditions of the applicable Altera Reference Design License Agreement.
// By using this reference design file, you indicate your acceptance of such terms and
// conditions between you and Altera Corporation.  In the event that you do not agree with
// such terms and conditions, you may not use the reference design file. Please promptly
// destroy any copies you have made.
//
// This reference design file being provided on an "as-is" basis and as an accommodation 
// and therefore all warranties, representations or guarantees of any kind 
// (whether express, implied or statutory) including, without limitation, warranties of 
// merchantability, non-infringement, or fitness for a particular purpose, are 
// specifically disclaimed.  By making this reference design file available, Altera
// expressly does not recommend, suggest or require that this reference design file be
// used in combination with any other product not provided by Altera. 
//----------------------------------------------------------------------------- 
 
 
module altpcierd_apprxstream_decode  (clk, rstn, 
                             rx_st_valid, rx_st_data, rx_st_ready, 
                             m_busy, s_busy, m_ack, s_ack,
                             m_rx_sop, s_rx_sop,  
                             rx_eop, rx_data, rx_desc, rx_be, m_rx_dv, s_rx_dv );
 
   input         clk;  
   input         rstn;     
 
   input         rx_st_valid;   // means streaming data (rx_st_data) is valid
   input[135:0]  rx_st_data;    // data from streaming interface
   input         m_busy;        // means master is busy and cannot accept new rx request 
   input         s_busy;        // means slave is busy and cannot accept new rx request 
   input         m_ack;         // master accepts rx request (new pkt)
   input         s_ack;         // slave accepts rx request (new pkt)
   
   output        rx_st_ready;      // throttles streaming data (rx_st_data)
   output        m_rx_sop;         // indicates that there is a new pkt available for the master
   output        s_rx_sop;         // indicates that there is a new pkt available for the slave
   output[63:0]  rx_data;          // packet payload data extracted from streaming interface
   output[135:0] rx_desc;          // packet descriptor extracted from streaming interface
   output[7:0]   rx_be;            // packet byte enable mask extracted from streaming interface
   output        m_rx_dv;          // means rx_data is valid for master
   output        s_rx_dv;          // means rx_data is valid for slave
   output        rx_eop;           // means current cycle is the last of the pkt
 
   reg           stream_sop_del;       
   wire          rx_eop;
   wire          m_rx_sop; 
   wire          s_rx_sop; 
   wire [63:0]   rx_data;
   wire[135:0]   rx_desc;    
   reg[63:0]     rx_desc_hi_r;  // registered version of descriptor high bytes (rx_desc[127:64])
   reg[63:0]     rx_desc_lo_r;  // registered version of descriptor low boytes (rx_desc[63:0]) 
   reg           stream_sop_del2;   
   wire          rx_st_ready;   
   reg[7:0]      rx_bar_r;       // registered veresion of bar field (rx_desc[135:128])
   
   wire          is_mst;         // indicates that current pkt is for master
   wire          is_tgt;         // indicates that current pkt is for slave (target)
 
   
   reg           is_mst_r;       // registered version of is_mst
   reg           is_tgt_r;       // registered version of is_tgt
   wire          has_payload;    // indicates that the current pkt has payload
   reg           has_payload_r;  // registered version of has_payload
   reg           m_rx_dv;          
   reg           s_rx_dv;  
   reg           rx_st_ready_r;  // registered version of rx_st_ready
   reg           m_rx_sop_r;        // registered version of m_rx_sop
   reg           s_rx_sop_r;        // registered version of s_rx_sop
   
   wire          assert_m_rx_sop;
   wire          assert_s_rx_sop;  
   reg           is_mst_or_tgt_r;
   wire          assert_ms_rx_sop;
   wire          ms_rx_sop;
   reg           ms_rx_sop_r;
   wire          is_mst_inter ;
   wire          is_tgt_inter ;   
     
   
   wire          stream_sop;
   wire          stream_eop;
   wire[63:0]    stream_data_bits; 
   wire[7:0]     stream_bar_bits;
   wire[7:0]     stream_be_bits;
    
   assign    stream_be_bits   = rx_st_data[81:74];  
   assign    stream_sop       = rx_st_data[73];
   assign    stream_eop       = rx_st_data[72]; 
   assign    stream_bar_bits  = rx_st_data[71:64];
   assign    stream_data_bits = rx_st_data[63:0];
   
   //-------------------------------------------------------------- 
   //  Generate app-side signals.
   //    - Demux the Descriptor/Data from the streaming interface
   //      and present on non-multiplexed application interface.
   //    - Generate rx request to master/slave when a new pkt is 
   //      received.
   //--------------------------------------------------------------
   
   // decode descriptor phase.
   // determine if current pkt is for master or for slave.

   assign is_mst_inter = (rx_st_data[61:57] == 5'b00101) ? 1'b1 : 1'b0;
   assign is_mst       = stream_sop ? is_mst_inter : is_mst_r;   
      
  
   assign is_tgt_inter = (rx_st_data[60:58] == 3'b000) ? 1'b1 : 1'b0;
   assign is_tgt       = stream_sop ? is_tgt_inter   : is_tgt_r;    
   
   // decode whether packet has payload 
   assign has_payload = stream_sop ? rx_st_data[62] : has_payload_r;
 
   // extract packet fields (data, byte enable bits, 
   // descriptor) from the multiplexed streaming interface bus
   // to desc/data interface to the application
   // descriptor is presented 
   assign rx_data          = stream_data_bits; 
   assign rx_be            = stream_be_bits; 
   assign rx_desc[135:128] = (stream_sop_del & ~stream_sop) ? stream_bar_bits  : rx_bar_r;      // valid on 2nd stream cycle
   assign rx_desc[127:64]  = rx_desc_hi_r ;                                                     // valid on sop cycle, but presented on next cycle
   assign rx_desc[63:0]    = (stream_sop_del & ~stream_sop) ? stream_data_bits : rx_desc_lo_r;  // valid on 2nd stream cycle  
   
   // assert a request for the master or slave when
   // on 2nd cycle (desc2 phase) of new packet.  deassert when acknowledged.  
   // full descriptor (rx_desc) is available to the application when the request is asserted.


   assign assert_m_rx_sop = ((stream_sop_del & ~stream_sop & is_mst_r) == 1'b1) ? 1'b1 : 1'b0; 
   assign m_rx_sop        = m_ack ? 1'b0 : assert_m_rx_sop ? 1'b1 : m_rx_sop_r;   
   
   assign assert_s_rx_sop = ((stream_sop_del & ~stream_sop & is_tgt_r) == 1'b1) ? 1'b1 : 1'b0; 
   assign s_rx_sop        = s_ack ? 1'b0 : assert_s_rx_sop ? 1'b1 : s_rx_sop_r; 
   
   assign assert_ms_rx_sop = ((stream_sop_del & ~stream_sop & is_mst_or_tgt_r) == 1'b1) ? 1'b1 : 1'b0;  
   assign ms_rx_sop        = (m_ack | s_ack) ? 1'b0 : assert_ms_rx_sop ? 1'b1 : ms_rx_sop_r;  
   
   // detect the end of packet
   assign rx_eop = stream_eop;         

   always @ (posedge clk or negedge rstn) begin
       if (~rstn) begin  
           rx_desc_hi_r     <= 64'h0;
           rx_desc_lo_r     <= 64'h0;
           rx_bar_r         <= 8'h0;
           stream_sop_del2  <= 1'b0;  
           is_mst_r         <= 1'b0;
           is_tgt_r         <= 1'b0;
           is_mst_or_tgt_r  <= 1'b0;
           has_payload_r    <= 1'b0;
           m_rx_dv          <= 1'b0;
           s_rx_dv          <= 1'b0;
           stream_sop_del   <= 1'b0; 
           m_rx_sop_r       <= 1'b0; 
           s_rx_sop_r       <= 1'b0;  
           ms_rx_sop_r      <= 1'b0;  
       end
       else begin   
           if (stream_sop == 1'b1) 
               rx_desc_hi_r     <= stream_data_bits;
           else
               rx_desc_hi_r     <= rx_desc_hi_r;
               
           rx_desc_lo_r     <= rx_desc[63:0];
           rx_bar_r         <= rx_desc[135:128];
           stream_sop_del2  <= stream_sop_del;  
           is_mst_r         <= is_mst;
           is_tgt_r         <= is_tgt;
           is_mst_or_tgt_r  <= is_mst | is_tgt;
           has_payload_r    <= has_payload;
           stream_sop_del   <= stream_sop & rx_st_valid;                             // indicates a new pkt is received.                
           m_rx_sop_r       <= m_rx_sop; 
           s_rx_sop_r       <= s_rx_sop;  
           ms_rx_sop_r      <= ms_rx_sop;  
           
           if (stream_sop_del2 & is_mst_r & has_payload)   // assert data valid flag if there is payload.  
               m_rx_dv  <= rx_st_valid;                    
           else if (rx_eop)                                // deassert after the last data cycle.
               m_rx_dv  <= 1'b0;
           else 
               m_rx_dv  <= m_rx_dv;
               
           
           if (stream_sop_del2 & is_tgt_r & has_payload)
               s_rx_dv <= rx_st_valid;
           else if (rx_eop)
               s_rx_dv <= 1'b0;
           else
               s_rx_dv <= s_rx_dv;
                
         

       end
   end   
   
   //------------------------------------------------
   // Generate streaming interface side 
   // 'ready' signal to throttle data based
   // on master/slave throttling.
   //------------------------------------------------
   
   // when a pkt is received for either the master or slave,
   // throttle the data stream until acked by master/slave.
   // simple dma does not throttle dataphase.
   
   assign rx_st_ready = (m_ack | s_ack) ? 1'b1 : (ms_rx_sop) ? 1'b0 : rx_st_ready_r; 
   
   always @ (posedge clk or negedge rstn) begin
       if (~rstn) begin   
           rx_st_ready_r <= 1'b1; 
       end
       else begin   
           rx_st_ready_r <= rx_st_ready; 
       end
   end
        
endmodule
