# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.

"""Support for Intel version format"""
import re
import sys
from distutils.version import StrictVersion
import six
from ._version import __version__


class IntelVersion(StrictVersion, object):
    """
    Intel Version. Mostly follows StrictVersion convention but adds support
    for classification.

    Format: major.minor[.patch][.classification][prerelease]

    Where:

        - major, minor and patch are decimal numbers.
        - if patch is omitted, it will default to 0.
        - classification is limited to 600 (RED), 400 (ORANGE), 200 (WHITE) or
          100 (GREEN).
        - if classification is omitted, it will default to 600 (RED).
        - prerelease consists of the letter 'a' or 'b' followed by a decimal
          number. If the numeric components of two version numbers are equal,
          then one with a prerelease tag will always be deemed earlier (lesser)
          than one without.

    Args:
        vstring (:obj:`str`): Version string representation.
    """
    version = None
    prerelease = None
    classification = None
    version_re = re.compile(r"""
                             ^(?P<major>\d+)  # major
                             \.(?P<minor>\d+)  # minor
                             (\.(?P<patch>\d+))?  # patch (optional)
                             (\.(?P<classification>600|400|200|100))?  # classification (optional)
                             (?P<prerelease>[ab]\d+)?$  # prerelease (optional)
                             """,
                            re.VERBOSE)

    def __init__(self, vstring):
        if not isinstance(vstring, six.string_types):
            raise TypeError("vstring must be a string")
        super(IntelVersion, self).__init__(vstring)

    def parse(self, vstring):
        """
        Convert a string representation to the appropriate internal
        representation for this style of version numbering.

        Args:
            vstring (:obj:`str`): Version string representation.
        """
        match = self.version_re.match(vstring)
        if not match:
            raise ValueError("invalid version number '%s'" % vstring)

        major = match.group("major")
        minor = match.group("minor")
        patch = match.group("patch") or "0"
        classification = match.group("classification") or "600"
        prerelease = match.group("prerelease")

        self.version = tuple([int(x) for x in [major, minor, patch]])
        self.prerelease = (prerelease[0], int(prerelease[1:])) if prerelease else None
        self.classification = int(classification)

    def get_full_version(self):
        """
        Returns the full version string as listed by pip (i.e. including the color classification)
        """
        version = [str(x) for x in list(self.version) + [self.classification]]
        if self.prerelease:
            version[3] = version[3] + self.prerelease[0] + str(self.prerelease[1])
        return ".".join(version)

    def __str__(self):
        version = [str(x) for x in self.version]
        if self.prerelease:
            version[2] = version[2] + self.prerelease[0] + str(self.prerelease[1])
        return ".".join(version)

    def _cmp(self, other):
        if not isinstance(other, IntelVersion):
            try:
                other = IntelVersion(str(other))
            except ValueError:
                raise ValueError("Can't compare against %s" % other)
        if six.PY2:
            return super(IntelVersion, self).__cmp__(other)
        return super(IntelVersion, self)._cmp(other)

    def __cmp__(self, other):
        return self._cmp(other)

    def __eq__(self, other):
        comparison = self._cmp(other)
        if comparison is NotImplemented:
            return comparison
        return comparison == 0

    def __lt__(self, other):
        comparison = self._cmp(other)
        if comparison is NotImplemented:
            return comparison
        return comparison < 0

    def __le__(self, other):
        comparison = self._cmp(other)
        if comparison is NotImplemented:
            return comparison
        return comparison <= 0

    def __gt__(self, other):
        comparison = self._cmp(other)
        if comparison is NotImplemented:
            return comparison
        return comparison > 0

    def __ge__(self, other):
        comparison = self._cmp(other)
        if comparison is NotImplemented:
            return comparison
        return comparison >= 0
