import time
from namednodes.accesses.register import AccessRegister
from namednodes.accesses.general import AccessRegisterStoredValues
class AccessAVMM(AccessRegister):
    """
    Base AVMM Access Class.

    This class implements the base ``requires`` and helper methods for the :py:class:`AccessTap2Iosfsb`, :py:class:`AccessPrimary2Iosfsb`, and other IOSF-SB Access classes.

    """
    defaults = {}
    requires_from_target = []
    log_level = None # default to existing logger level

    @classmethod
    def read(cls, node):
        """
        Perform an AVMM read.

        Args:
            node (NamedNodeValue): The node that we are trying to get the data from.
        """
        return cls._access(node)

    @classmethod
    def write(cls, node, value):
        """
        Perform an AVMM write.

        Args:
            node (NamedNodeValue): The node that we are trying to write the data to.
            value (object): The data to write to the node.
        """
        return cls._access(node, value)

class AccessVJTAG2AVMM(AccessAVMM):
    """
    Performs a VJTAG->AVMM access

    """
    requires = [
            'region_offset',
            'addressOffset',
            'device',
            'size',
            'service_type',
            'addr_left_shift',
            'addr_right_shift',
        ]

    @classmethod
    def _access(cls, node, value=None):
        if not node.definition.info.get('available', True):
            raise Exception('Register %s is not available in vjtag mode, run load_sopcinfo with qsys genertated file to enable vjtag access'%node.name)
        if 'fpga_common' in __file__:
            import svtools.common.baseaccess
            base = svtools.common.baseaccess.getglobalbase()
        else:
            import pysv_fpga.systemconsolebaseaccess as sc
            base = sc.baseaccess()

        access_info = cls.get_access_info(node)
        offset = access_info['addressOffset']
        if access_info['region_offset'] != LookupError:
            offset += access_info['region_offset']

        if access_info['addr_left_shift'] != LookupError:
            offset = offset << access_info['addr_left_shift']
        if access_info['addr_right_shift'] != LookupError:
            offset = offset >> access_info['addr_right_shift']

        device = access_info['device']
        service_type = 'master'
        if access_info['service_type'] != LookupError:
            service_type = access_info['service_type']
        return base.vjtag2avmm(access_info['device'],offset,size=access_info['size'], data=value,
                service_type=service_type)

class AccessVJTAG2AVMM_1BYTE_SHIFT(AccessAVMM):
    """
    Performs a VJTAG->AVMM access a byte at a time after shifting the address left 2

    """
    requires = [
            'region_offset',
            'addressOffset',
            'device',
            'size',
            'service_type',
        ]

    @classmethod
    def _access(cls, node, value=None):
        if not node.definition.info.get('available', True):
            raise Exception('Register %s is not available in vjtag mode, run load_sopcinfo with qsys genertated file to enable vjtag access'%node.name)
        if 'fpga_common' in __file__:
            import svtools.common.baseaccess
            base = svtools.common.baseaccess.getglobalbase()
        else:
            import pysv_fpga.systemconsolebaseaccess as sc
            base = sc.baseaccess()

        access_info = cls.get_access_info(node)
        device = access_info['device']
        address = access_info['addressOffset']
        if access_info['region_offset'] != LookupError:
            address += access_info['region_offset']
        size = access_info['size']
        service_type = 'master'
        if access_info['service_type'] != LookupError:
            service_type = access_info['service_type']

        if value is None:
            value = 0
            for i in range(int(size), 0, -1):
                value = value << 8
                value |= base.vjtag2avmm(device, address=(address+i-1) << 2, service_type=service_type)
            return value
        else:
            for i in range(0, int(size), 1):
                base.vjtag2avmm(device, address=(address+i) << 2, data=value & 0xff, service_type=service_type)
                value = value >> 8

class AccessWHR_SYN_PHY(AccessRegister):
    """
    Performs indexed access using AVMM registers in parent objects
    """
    requires = [
            'addressOffset',
        ]

    @classmethod
    def _access(cls, node, value=None):
        access_info = cls.get_access_info(node)

        # Get the parent unit
        phy_interface = node.parent.parent.parent

        # read indirect registers
        phy_cr_access_reg_0 = phy_interface.phy_cr_access_reg_0.read()
        phy_cr_access_reg_1 = phy_interface.phy_cr_access_reg_1.read()

        # CR parallel register
        phy_cr_access_reg_1.cfg_cr_para_sel.store(1)

        # Write the address
        phy_cr_access_reg_0.cfg_cr_para_addr.store(access_info['addressOffset'])
        # Set the operation
        # If write, set data
        if value != None:
            phy_cr_access_reg_1.cfg_cr_wr_rd_n.store(1)
            phy_cr_access_reg_0.cfg_cr_para_wr_data.store(value)
        else:
            phy_cr_access_reg_1.cfg_cr_wr_rd_n.store(0)

        # Enable the access when register is flushed
        phy_cr_access_reg_1.cfg_cr_access_en.store(1)

        phy_cr_access_reg_0.flush()
        phy_cr_access_reg_1.flush()

        start = time.time()
        while True:
            #If stub mode mode
            if phy_interface.phy_cr_access_reg_0.access_class == AccessRegisterStoredValues:
                # Clear the pending bit
                phy_interface.phy_cr_access_reg_1.cfg_cr_access_en.write(0)
                # Latch / read value in stub mode
                if value != None:
                    super(AccessWHR_SYN_PHY, cls).write(node, value)
                else:
                    phy_interface.phy_cr_access_reg_1.cfg_cr_para_rd_data.write(super(AccessWHR_SYN_PHY, cls).read(node))
                break

            # phy_cr_access_reg_1.cfg_cr_para_ack is not returning 1 but data is good
            # skipping ack check
            break

            if phy_cr_access_reg_1.cfg_cr_para_ack == 1:
                break
            elif time.time() - start > 0.1: # timeout value needs to be decided
                raise Exception('Register access took too long to complete')

        if value != None:
            return
        else:
            return phy_interface.phy_cr_access_reg_1.cfg_cr_para_rd_data.read()


    @classmethod
    def read(cls, node):
        """
        Perform an indirect PHY read.

        Args:
            node (NamedNodeValue): The node that we are trying to get the data from.
        """
        return cls._access(node)

    @classmethod
    def write(cls, node, value):
        """
        Perform an indirect PHY write.

        Args:
            node (NamedNodeValue): The node that we are trying to write the data to.
            value (object): The data to write to the node.
        """
        return cls._access(node, value)


