from pysv_fpga.fpga_tools.quartus.SysconInterface import syscon_interface as _sc_interface

_base = None

def baseaccess(port=None):
    global _base
    if not _base:
        _base = sc_baseaccess(port=port)
    return _base

class sc_baseaccess(object):
    """
    The System Console baseaccess class
    This class bridges the gap between the low level libosv API and the high level
    functions that users use for accessing memory, IO ports, registers and PCI configuration
    space
    It also maintains a standard set of variables that reflect the type of system we are on.
    """
    class AccessTimeout(Exception):
        pass

    global _global_open_claims
    _global_open_claims = {}

    _sc_interface = None
    _keep_claims_open = True
    # ipc compatible device list
    _devicelist = []
    #JCASTELL discovery for CA
    _ca = None
    _bmc = None

    def __init__(self, port=None):
        self._sc_interface = _sc_interface
        self._sc_interface.start_syscon(port=port)
        self.commands = self._sc_interface.commands

    def _get_master_service_path(self, jtag_id):
        if jtag_id not in _sc_interface.Nios2r2_hash.keys():
            _sc_interface.update_properties()

        if jtag_id not in _sc_interface.Nios2r2_hash.keys():
            raise Exception('Unable to find master service path for jtag_id %d'%jtag_id)

        return _sc_interface.Nios2r2_hash[jtag_id]

    def _get_claim_service_path(self, path, range=(None, None), service_type='master'):
        import copy
        global _global_open_claims

        start, length = range
        if service_type in ['master']:
            if length is None or start is None:
                start = 0
                if 'CJTAG' in path:
                    if 'SDM' in path:
                        length = 0xbc00007c
                    elif 'TNM' in path:
                        length = 0x4000d00
                    else:
                        length = 0x80000fc
                else:
                    length = 0xffffffff

        # If already claimed, return it
        if (path, start, length, service_type) in _global_open_claims.keys():
            return (path, start, length, service_type), _global_open_claims[(path, start, length, service_type)]

        # Close other claims if configured
        if not self._keep_claims_open:
            sc_baseaccess.close_all_open_claims()

        # Check to see if similar path has already been claimed, if so it must be closed first
        for (open_path, s, l, t) in copy.deepcopy(_global_open_claims).keys():
            if (open_path.startswith(path) or path.startswith(open_path)) and \
               ('nios2r2' in open_path or 'nios2r2' in path):
                _sc_interface.commands.close_service(t, _global_open_claims[(open_path, s, l, t)])
                _global_open_claims.pop((open_path, s, l, t))

        # Claim service
        if service_type in ['master']:
            _global_open_claims[(path, start, length, service_type)] = self._sc_interface.commands.claim_service(service_type,
                path, 'pythonsv', '{0x%x 0x%x RW}'%(start, length))
        else:
            _global_open_claims[(path, start, length, service_type)] = self._sc_interface.commands.claim_service(service_type,
                path, 'pythonsv')

        return (path, start, length, service_type), _global_open_claims[(path, start, length, service_type)]

    def _close_service_path(self, claim_key):
        if claim_key not in _global_open_claims.keys():
            return

        (open_path, s, l, t) = claim_key
        _sc_interface.commands.close_service(t, _global_open_claims.pop( (open_path, s, l, t) ))

    @staticmethod
    def close_all_open_claims():
        global _global_open_claims
        for (open_path, s, l, t) in _global_open_claims.keys():
            try: # nosec
                _sc_interface.commands.close_service(t, _global_open_claims[(open_path, s, l, t)])
            except:
                pass
        _global_open_claims = {}

    def cjtag2avmm(self, device, address, size=4, data=None, range=(None, None), service_type='master'):
        """
        device: Jtag Path or cjtag_id
        data can be integar of no more than 32 bits or list of integars
        """
        path = device
        if isinstance(device, Device):
            path = device.path # is this opened?
        elif type(device) == int:
            path = self._get_master_service_path( device )

        # Try to find a NIOS to access a non 4-byte sized location
        if path[-8:-2] == 'sld2mm' and size % 4 != 0:
            for device in self._devicelist:
                if device.path.startswith(path) and device.path.endswith('nios2r2'):
                    path = device.path
                    break

        if path[-8:-2] == 'sld2mm' and size % 4 != 0:
            raise Exception('sld2mm devices can only access increments of 4 bytes, size %d given'%size)

        claim_key, claim = self._get_claim_service_path(path, range=range, service_type=service_type)

        try:
            if size % 4 == 0:
                read_func = self._sc_interface.commands.master_read_32
                write_func = self._sc_interface.commands.master_write_32
                func_size = 4
            elif size % 2 == 0:
                read_func = self._sc_interface.commands.master_read_16
                write_func = self._sc_interface.commands.master_write_16
                func_size = 2
            else:
                read_func = self._sc_interface.commands.master_read_8
                write_func = self._sc_interface.commands.master_write_8
                func_size = 1

            if data == None:
                return read_func(claim, address, int(size / func_size))
            else:
                write_func(claim, address, data)
                return

        except: raise

        #Close the service
        finally:
            if not self._keep_claims_open:
                self._close_service_path(claim_key)


    def vjtag2avmm(self, device, address, size=4, data=None, range=(None, None), service_type='master'):
        """
        device: Jtag Path
        """
        return self.cjtag2avmm(device, address, size, data, range, service_type)


class Device:
    def __init__(self, devtype, path, info={}):
        self.devtype = devtype
        self.path = path
        self.info = info
    def __str__(self):
        return "%s: %s"%(self.devtype, self.path)

import atexit
atexit.register(sc_baseaccess.close_all_open_claims)

