

class Pixmap:
    def __init__(self, x_pixels, y_pixels, x0=None, y0=None):
        self._x_pixels = x_pixels
        self._y_pixels = y_pixels
        self.TOP_ROW = 0
        self.BOTTOM_ROW = y_pixels - 1
        self.FIRST_COL = 0
        self.LAST_COL = x_pixels - 1
        self._x_scale = 1.0
        self._y_scale = 1.0
        if x0 == None:
            self._x0 = float(self._x_pixels / 2)
        else:
            self._x0 = float(x0)
        if y0 == None:
            self._y0 = float(self._y_pixels / 2)
        else:
            self._y0 = float(y0)
        self._pixmap = list()
        for r in range(self._y_pixels):
            row = list([0])
            row *= self._x_pixels
            self._pixmap.append(row)
    def setScale(self, x_scale, y_scale):
        self._x_scale = float(x_scale)
        self._y_scale = float(y_scale)
    def setOrigin(self, x0, y0):
        self._x0 = float(x0)
        self._y0 = float(y0)
    def show(self, color=False):
        for r in range(self._y_pixels):
            for c in range(self._x_pixels):
                v_int = self._pixmap[r][c]
                v_txt = ' ' + str(v_int) + ' '
                if color:
                    fgColor = v_int + 30
                    bgColor = v_int + 40
                    print(("\033[1;{};{}m{}{}\033[0m").format(fgColor, bgColor, "\u0008", v_txt), sep='', end='', )
                else:
                    print(v_txt, sep='', end='')
            print('')
    def calcColIdx(self, x):
        col = round(x*self._x_scale - 0.5 + self._x0)
        col_oor = (col < 0) or (col >= self._x_pixels)
        #print("x=", x, "  col=", col, "  col_oor=", col_oor, sep='')
        if col_oor: return None
        else: return col
    def calcRowIdx(self, y):
        row = round(self._y_pixels - 0.5 - (y*self._y_scale) - self._y0)
        row_oor = (row < 0) or (row >= self._y_pixels)
        #print("y=", y, "  row=", row, "  row_oor=", row_oor, sep='')
        if row_oor: return None
        else: return row
    def addPixel(self, c, r, value):
        if (c!=None) and (r!=None):
            self._pixmap[r][c] = value
    def addPixelStack(self, c, r1, v1, r2, v2):
        if c != None:
            if (r1 != None) and (r2 != None):
                if (r2 < r1): step = -1
                else: step = 1
                for r in range(r1,r2, step):
                    self._pixmap[r][c] = v1
                self._pixmap[r2][c] = v2
    def addPixelBlock(self, c1, r1, c2, r2, v):
        if (c1 != None) and (c2 != None) and (r1 != None) and (r2 != None):
            for r in range(r1, r2 + 1):
                row = self._pixmap[r]
                for c in range(c1, c2 + 1):
                    row[c] = v


def xy_to_unsorted_xud(xy_list):
    x_to_ud_dict = dict()
    for i in xy_list:
        x = i[0]
        y = i[1]
        if y>0: edge='up'
        else: edge='dn'
        if x in x_to_ud_dict:
            if edge in x_to_ud_dict[x]:
                print("WARNING: repeated '", edge, "' value", sep='')
            x_to_ud_dict[x][edge] = y
        else:
            x_to_ud_dict[x] = dict()
            x_to_ud_dict[x][edge] = y
    return x_to_ud_dict


def process_xud(x_to_ud_dict):
    k = list(x_to_ud_dict.keys())
    k.sort()
    result = dict()
    # Ranges
    result['y_up_max'] = -30000
    result['y_up_min'] =  30000
    result['y_dn_max'] = -30000
    result['y_dn_min'] =  30000
    # Data to plot
    result['x'] = list()
    result['y_up'] = list()
    result['y_dn'] = list()
    # Average Step Sizes
    result['x_ass'] = 10000
    result['y_up_ass'] = 10000
    result['y_dn_ass'] = 10000
    # Temporary variables
    prevs = dict()
    prevs['x'] =    None
    prevs['y_up'] = None
    prevs['y_dn'] = None
    # Sums of step sizes
    sums = dict()
    sums['x'] =    0
    sums['y_up'] = 0
    sums['y_dn'] = 0
    for i in k:
        #print("x=",i, "  y_u=", x_to_ud_dict[i]['up'], " y_d=", x_to_ud_dict[i]['dn'], sep='')
        # Capture ranges
        result['y_up_max'] = max(result['y_up_max'], x_to_ud_dict[i]['up'])
        result['y_up_min'] = min(result['y_up_min'], x_to_ud_dict[i]['up'])
        result['y_dn_max'] = max(result['y_dn_max'], x_to_ud_dict[i]['dn'])
        result['y_dn_min'] = min(result['y_dn_min'], x_to_ud_dict[i]['dn'])
        # Capture data
        result['x'].append(i)
        result['y_up'].append(x_to_ud_dict[i]['up'])
        result['y_dn'].append(x_to_ud_dict[i]['dn'])
        # Sums
        if prevs['x'] != None:
            sums['x'] = sums['x'] + abs(prevs['x'] - i)
            sums['y_up'] = sums['y_up'] + abs(prevs['y_up'] - x_to_ud_dict[i]['up'])
            sums['y_dn'] = sums['y_dn'] + abs(prevs['y_dn'] - x_to_ud_dict[i]['dn'])
        # Store reference values for the next iteration
        prevs['x'] = i
        prevs['y_up'] = x_to_ud_dict[i]['up']
        prevs['y_dn'] = x_to_ud_dict[i]['dn']
    # Calculate average step sizes
    result['x_ass'] =    sums['x'] / len(k)
    result['y_up_ass'] = sums['y_up'] / len(k)
    result['y_dn_ass'] = sums['y_dn'] / len(k)
    return result


def vertical_stripe(xud, i, p):
    c = p.calcColIdx(xud['x'][i])
    r_up = p.calcRowIdx(xud['y_up'][i])
    r_dn = p.calcRowIdx(xud['y_dn'][i])
    p.addPixelStack(c, p.TOP_ROW,    1, r_up, 2)
    p.addPixelStack(c, p.BOTTOM_ROW, 1, r_dn, 2)
    return (c, r_up, r_dn)

def pixelize(xud):
    x_border_px = 3
    y_border_px = 3
    # get/calculate ranges
    x_range = xud['x'][-1] - xud['x'][0]
    y_range = xud['y_up_max'] - xud['y_dn_min']
    #print("x_range=", x_range, "  y_range=", y_range, sep='')
    # decide on pixmap size
    x_size = x_range + 1
    y_size = y_range//5 + 1
    #print("x_size=", x_size, "  y_size=", y_size, sep='')
    # calculate scales
    x_scale = 1.0 #float(x_size/x_range)
    y_scale = 0.2 #float(y_size/y_range)
    #print("x_scale=", x_scale, "  y_scale=", y_scale, sep='')
    
    # Create pixmap
    p = Pixmap(x_size + 2*x_border_px, y_size + 2*y_border_px)
    p.setScale(x_scale, y_scale)
    # calculate position of (0,0) pixel
    pixel_x0 = x_border_px - x_scale*xud['x'][0]
    pixel_y0 = round(y_border_px - y_scale*xud['y_dn_min'])
    #print("pixel_x0=", pixel_x0, "  pixel_y0=", pixel_y0, sep='')
    p.setOrigin(pixel_x0, pixel_y0)
    
    # Painting the Eye Contour (from left to right, one vertical stripe at a time
    #  * the background
    p.addPixelBlock(p.FIRST_COL, p.TOP_ROW, p.LAST_COL, p.BOTTOM_ROW, 4)
    #  * the first stripe
    (c0, r_up0, r_dn0) = vertical_stripe(xud, 0, p)
    #  * fill-up on left-most side (on the left from the first stripe)
    p.addPixelBlock(p.FIRST_COL, p.TOP_ROW, c0-1, p.BOTTOM_ROW, 1)
    (c_prev, r_up_prev, r_dn_prev) = (c0, r_up0, r_dn0)
    for i in range(1, len(xud['x'])):
        (c, r_up, r_dn) = vertical_stripe(xud, i, p)
        delta_c = c - c_prev
        if delta_c > 0:
            delta_r_up = (r_up - r_up_prev)/delta_c
            delta_r_dn = (r_dn - r_dn_prev)/delta_c
            for c_gap in range(c_prev+1, c):
                r_up_gap = round( r_up_prev + delta_r_up*(c_gap-c_prev))
                r_dn_gap = round( r_dn_prev + delta_r_dn*(c_gap-c_prev))
                p.addPixelStack(c_gap, p.TOP_ROW,    1, r_up_gap, 2)
                p.addPixelStack(c_gap, p.BOTTOM_ROW, 1, r_dn_gap, 2)
        (c_prev, r_up_prev, r_dn_prev) = (c, r_up, r_dn)
    #  * fill-up on the right-most side
    p.addPixelBlock(c_prev+1, p.TOP_ROW, p.LAST_COL, p.BOTTOM_ROW, 1)
    return p


