import logging
import sys

#LINE_FORMAT = '#%(asctime)s|%(levelname)-5s|%(message)s'
#LINE_FORMAT = '#%(asctime)s|%(levelname)-5s|%(process)d|%(filename)s|%(message)s'
LINE_FORMAT = '#%(asctime)s|%(levelname)-5s|%(thread)d|%(filename)s|%(message)s'
# more details:
# LINE_FORMAT = '#%(asctime)s|%(levelname)-5s|%(name)s|%(funcName)s|%(message)s'
DATE_FORMAT = '%H:%M:%S'

_stdout_handler = None


def get_logger(name):
    # Use get/setLoggerClass to safely tell the global python logging that
    # we want to use the original Logger class
    # (ex. PythonSV uses EasyLogger class which messes up global logging)
    prev_logger_cls = logging.getLoggerClass()
    try:
        logging.setLoggerClass(logging.Logger)
        logger = logging.getLogger(name)
        logger.to_stdout = _get_to_stdout(logger)
        logger.to_file = _get_to_file(logger)
        return logger
    finally:
        logging.setLoggerClass(prev_logger_cls)


def _get_to_stdout(logger):
    thelogger = logger  # capture the logger for currying

    def to_stdout(enable=True):
        if enable:
            add_stdout_handler(thelogger)
        else:
            remove_stdout_handler(thelogger)

    return to_stdout


def _get_to_file(logger):
    thelogger = logger  # capture the logger for currying

    def to_file(file_name, enable=True):
        if enable:
            add_file_handler(thelogger, file_name)
        else:
            remove_handler(thelogger, file_name)

    return to_file


def add_stdout_handler(logger, level=None, handler_name=None):
    """sets up logger for stdout logging"""
    if handler_name is None:
        handler_name = logger.name.replace('.', '_') + "_stdoutx"
    global _stdout_handler
    if _stdout_handler is None:
        _stdout_handler = logging.StreamHandler(sys.stdout)
    if _stdout_handler not in logger.handlers:
        add_handler_to_logger(logger, _stdout_handler, handler_name, level)


class LoggerSettings:

    def __init__(self, level, has_stdout_handler, file_handler_names):
        self.level = level
        self.has_stdout_handler = has_stdout_handler
        self.file_handler_names = file_handler_names

    def init_logger(self, logger):
        remove_handlers(logger)
        if self.has_stdout_handler:
            logger.to_stdout()
        for name in self.file_handler_names:
            logger.to_file(name)
        logger.setLevel(self.level)

    @classmethod
    def get_settings_from_logger(cls, logger):
        return LoggerSettings(level=logger.getEffectiveLevel(),
                              has_stdout_handler=cls.has_stdout_handler(logger),
                              file_handler_names=
                                  cls.get_file_handler_names(logger))

    @staticmethod
    def has_stdout_handler(logger):
        return _stdout_handler is not None and\
               _stdout_handler in logger.handlers

    @staticmethod
    def get_file_handler_names(logger):
        names = []
        for h in logger.handlers:
            if isinstance(h, logging.FileHandler):
                names.append(h.name)
        return names


def add_handler_to_logger(logger,
                          handler,
                          handler_name,
                          handler_level=None,
                          line_format=None,
                          date_format=None):
    handler.setLevel(handler_level if handler_level is not None
                     else logging.DEBUG)
    handler.name = handler_name
    line_format = line_format if line_format is not None else LINE_FORMAT
    date_format = date_format if date_format is not None else DATE_FORMAT
    formatter = logging.Formatter(line_format, date_format)
    handler.setFormatter(formatter)
    logger.addHandler(handler)


def add_file_handler(logger, file_name):

    file_handler = logging.FileHandler(file_name)
    add_handler_to_logger(logger, file_handler, file_name)

    # register a clean up method, because Windows won't like it if we exit
    # with the file handler still open
    def clean_up():
        remove_handlers(logger)
    import atexit
    atexit.register(clean_up)


def remove_stdout_handler(logger):
    if logger:
        index = None
        for i, h in enumerate(logger.handlers):
            if h is _stdout_handler:
                h.close()
                index = i
                break
        if index is not None:
            logger.handlers.pop(index)


def remove_handler(logger, name):
    if logger:
        match = None
        for h in logger.handlers:
            if h.name == name:
                h.close()
                match = h
                break
        if match:
            logger.removeHandler(match)


def remove_handlers(logger):
    if logger:
        handlers = logger.handlers[:]
        for handler in handlers:
            handler.close()
            logger.removeHandler(handler)
