"""
SysconCommands.py

This module contains system console command class.
command_parser class translate xml file that contains System Console commands into Python functions.

"""

from xml.etree import ElementTree as _ElementTree  # nosec
import os as _os
import re as _re
import six as _six

class gen_obj():
    def __getattrs__(self):
        attrs = []
        for attr in dir(self):
            if attr[0] != "_":
                attrs.append(attr)
        return attrs

    def __call__(self):
        # This will give us a list of all the members in the class
        for name in self.__gettattrs__():
            print(name)

    def __getitem__(self, index):
        return getattr(self, self.__getattrs__()[index])

class command_class():
    """
    **************************************************
    If you want the help message for this command type, please run <command>.help()
    **************************************************
    """
    def __init__(self, syscon_api):
        self._error_header = "Quartus error : "
        self._syscol_err_strings = [
                "PythonSV_SystemConsoleError_Exception:",
                "Channel closed",
                "Could not claim",
                "does not have correct claims",
                "must be claimed",
                "cannot be found",
                "Error",
                ]
        self._syscon_api = syscon_api

    def help(self):
        print(self.__doc__)

    def type_checkout(self,supported_types,parameter):
        if type(parameter) in supported_types:
            return parameter
        else:
            supp_types_msg = "[  "
            for supp_types in supported_types:
                supp_types_msg = supp_types_msg + " " + supp_types.__name__

            supp_types_msg = supp_types_msg + "  ]"

            message = 'Invalid type. Types supported : ' + \
                        supp_types_msg

            raise TypeError(message)

    #---------------------- Argument parsing -------------------------

    def parse_str(self,parameter):
        """
        Used in arguments to add quotes
        """
        supported_types = [str]
        parameter_verified = self.type_checkout(supported_types,parameter)
        return "{" + parameter_verified.strip() + "}"

    def parse_tcl_obj(self,parameter):
        """
        Used in arguments no quotes added
        """
        supported_types = [str]
        parameter_verified = self.type_checkout(supported_types,parameter)
        return parameter_verified.strip()

    def parse_int(self,parameter):
        """
        Used in arguments
        """
        supported_types = list(_six.integer_types)
        parameter_verified = self.type_checkout(supported_types,parameter)
        #convert into string hex values
        return "0x%x" % parameter_verified

    def parse_list_str_2_tcl_obj(self,parameter):
        """
        Used in arguments
        """
        tcl_string_list = ""
        supported_types = [list]
        parameter_verified = self.type_checkout(supported_types,parameter)
        item_types = [str]
        for item in parameter_verified:
            item_verified = self.type_checkout(item_types,item)
            tcl_string_list = tcl_string_list + item_verified + " "
        return tcl_string_list


    def parse_list_int_2_str(self,parameter):
        """
        Used in arguments
        """
        list_int = list()
        #verify if is a list of integers or just an integer
        if type(parameter) in list(_six.integer_types):
            list_int.append(parameter)
        else:
            list_int = parameter

        tcl_string_list = "{"
        supported_types = [list]
        parameter_verified = self.type_checkout(supported_types,list_int)
        item_types = list(_six.integer_types)
        for item in parameter_verified:
            item_verified = self.type_checkout(item_types,item)
            tcl_string_list = tcl_string_list + ("0x%x " % item_verified)
        tcl_string_list += "}"
        return tcl_string_list

    def parse_list_str_2_str(self,parameter):
        """
        Used in arguments add quotes to each string
        """
        tcl_string_list = ""
        supported_types = [list]
        parameter_verified = self.type_checkout(supported_types,parameter)
        item_types = [str]
        for item in parameter_verified:
            item_verified = self.type_checkout(item_types,item)
            tcl_string_list = tcl_string_list + "\"" +item_verified + "\" "
        return tcl_string_list

    def convert_int_to_bytes(self,parameter,length):
        supported_types = list(_six.integer_types)
        parameter_verified = self.type_checkout(supported_types,parameter)
        bytes = []
        for i in range(length):
            bytes.append( parameter_verified & 0xff )
            parameter_verified >>= 8
        return bytes

    def argument_parsing(self,arg_dict,type,argument):
        import math, ast
        if  (type == "str"):
            return self.parse_str(argument)
        elif(type == "int"):
            return self.parse_int(argument)
        elif(type == "tcl_object"):
            return self.parse_tcl_obj(argument)
        elif(type == "list_int"):
            return self.parse_list_int_2_str(argument)
        elif(type == "list_str"):
            return self.parse_list_str_2_str(argument)
        else:
            try:
                return ast.literal_eval(type)
            except:
                raise TypeError("Unsupported type for an argument: " + type)
    #-----------------------------------------------------------------
    #----------------------- Return parsing --------------------------

    #``````````````````````````````````````````````````````````````````
    #````````````Common functions used in return parsing ``````````````
    #``````````````````````````````````````````````````````````````````
    def attrib_normalization(self,att_name):
        """
        Function used to avoid conflicts in the attribute
        naming generation on the fly
        """
        #Clean all special characters
        name_norm = _re.sub('[^A-Za-z0-9]+', '_',str(att_name))
        #Clean any white space
        name_norm = _re.sub('\s', '_',name_norm)
        #add att at the beginning to avoid starting with numbers
        name_norm = "att_"+name_norm
        return name_norm

    def tlc_list_decoding(self,str_list):
        """
        Function used to decode simple (not double array) tcl list
        """
        #Flag definition
        pos_open = 0x0
        pos_close = 0x0
        string_lock = False
        bracket = False
        space = False
        #variable used to store each individual character for processing
        temp_str = ""
        #Variable that will store the list conversion
        string_list  = list()
        #Iterate the list or processing
        for str_char in str_list:
            ##Beginning of data marker
            #Determine the type of delimiter used for this chuck of data
            if str_char  == "{" and string_lock == False:
                #Bracket
                string_lock = True
                bracket = True
                pos_open += 1
                continue
            else:
                #Element is white Space delimiter
                if string_lock == False:
                    string_lock = True
                    space = True
            ## get the data and determine if we need to move to next item in the list
            #IF the string is locked get the data
            if string_lock:
                #String_lock is true we need to store data
                #determine the type
                if space:
                    ##Space type
                    #Add the character
                    if _re.search("\s",str_char):
                        #white space detected flush the data and unlock
                        #The string will be added only if it is not empy
                        if temp_str != "":
                            string_list.append(temp_str)
                        string_lock = False
                        space = False
                        temp_str = ""
                        continue
                    else:
                        temp_str += str_char
                elif bracket:
                    ##Bracket type
                    #get information regarding the brackets
                    if str_char  == "{":
                        pos_open += 1
                    elif str_char  == "}":
                        pos_close += 1
                    #If number of brackets matched get the items
                    if pos_open == pos_close:
                        #Flush the data
                        string_list.append(temp_str)
                        string_lock = False
                        bracket = False
                        temp_str = ""
                        continue
                    else:
                        temp_str += str_char
        #If last element in the tcl list is segmented by space, the space
        #may or may not be there if the space is there the script is already
        #captured in the previous processing, if not we ned to verify is space
        #is selected and capture the data
        if string_lock and space:
            string_list.append(temp_str)
        #Return the data
        return string_list

    def tcl_2_python_list_depth(self,depth,tcl_str_list):
        """
        Function used to decode square tcl list (depth x depth)
        """
        real_depth = int(depth)
        if real_depth >= 1:
            local_depth = 2
            processing_list = self.tlc_list_decoding(tcl_str_list)
            temp_list= list()
            while local_depth <= real_depth:
                for item in processing_list:
                    temp_list.append(self.tlc_list_decoding(item))
                processing_list = list(temp_list)
                temp_list = list()
                local_depth +=1
            return processing_list
        else:
            msg = "unable to decode a list with depth="+ depth +\
                " please fix the XML schema for quartus commands"
            raise IndexError(msg)
    #``````````````````````````````````````````````````````````````````

    def return_parse_none(self,parameter):
        if parameter == '':
            return
        else:
            raise SysconError(parameter)

    def return_parse_str_2_str(self,parameter):
        if any(x in parameter for x in self._syscol_err_strings):
            raise SysconError(parameter)
        else:
            return parameter

    def return_string_2_int(self,parameter):
        if any(x in parameter for x in self._syscol_err_strings):
            raise SysconError(parameter)
        else:
            return int(parameter,16)

    def return_parse_string_2_string_list(self,depth,parameter):
        if any(x in parameter for x in self._syscol_err_strings):
            raise SysconError(parameter)
        else:
            return self.tcl_2_python_list_depth(depth,parameter)

    def return_parse_string_2_dict(self,depth,parameter):
        if any(x in parameter for x in self._syscol_err_strings):
            raise SysconError(parameter)
        else:
            array = self.tcl_2_python_list_depth(depth,parameter)
            if len(array) % 2 != 0:
                raise SysconError('Can not convert odd numbered string to dict')
            retvalue = {}
            for i in range(0, len(array), 2):
                retvalue[ ' '.join(array[i]) ] = ' '.join(array[i+1])
            return retvalue

    def return_parse_string_2_int_list(self,depth,parameter):
        if any(x in parameter for x in self._syscol_err_strings):
            raise SysconError(parameter)
        else:
            int_list = list()
            list_str = self.tcl_2_python_list_depth(depth,parameter)
            for item in list_str:
                int_list.append(self.return_string_2_int(item))
            return int_list

    def return_parse_string_2_int(self,depth,parameter):
        shifts = {
                'bytes':  8,
                'words':  16,
                'dwords': 32
        }
        if any(x in parameter for x in self._syscol_err_strings):
            raise SysconError(parameter)
        else:
            shift = shifts[depth]
            processing_list = self.return_parse_string_2_int_list(1,parameter)
            retval = 0
            for i in range(len(processing_list)-1, -1, -1):
                retval = (retval << shift) | processing_list[i]
            return retval

    def return_parse_string_2_list_obj(self,depth,parameter):
        if any(x in parameter for x in self._syscol_err_strings):
            raise SysconError(parameter)
        else:
            return self.return_parse_string_2_string_list(depth,parameter)

    def return_parsing(self,type,list_depth,return_value):
        if  (type == "str"):
            return self.return_parse_str_2_str(return_value)
        elif(type == "int"):
            return self.return_string_2_int(return_value)
        elif(type == "none"):
            return self.return_parse_none(return_value)
        elif(type == "list_obj"):
            return self.return_parse_string_2_list_obj(list_depth,return_value)
        elif(type == "list_int"):
            return self.return_parse_string_2_int_list(list_depth,return_value)
        elif(type == "list_int_to_1_int"):
            return self.return_parse_string_2_int(list_depth,return_value)
        elif(type == "list_str"):
            return self.return_parse_string_2_string_list(list_depth,return_value)
        elif(type == "dict_str"):
            return self.return_parse_string_2_dict(list_depth,return_value)
        else:
            raise TypeError("Unsupported type returned: " + type)
    #-----------------------------------------------------------------

    def __call__(self, *args):
        """
        Function used to call Quartus command
        """
        if len(args)-len(self._argListOptional) > len(self._argListMandatory):
            errorMsg = self.name + "() takes at most " + str(len(self._argListMandatory)) + " arguments " +\
                        "(" +str(len(args))+" given)"
            raise Exception(errorMsg )
        else:
            parameters =""
            arg_dict = {}
           #--------------------------------------------------------------------------
           #Create the mandatory parameters
           #--------------------------------------------------------------------------
            for parameter_index in range (0,len(self._argListMandatory)):
                arg_type = getattr(self._argObj, self._argListMandatory[parameter_index])
                try:
                    arg_value = args[parameter_index]
                except:
                    errorMsg = "Parameter : "+self._argListMandatory[parameter_index] + " was not set in " + \
                                self.name + "() function"
                    raise Exception(errorMsg )

                #parameter conversion
                arg_normalized = self.argument_parsing(arg_dict,arg_type,arg_value)
                #Create the parameter string for tcl
                parameters=parameters + " " + arg_normalized
                arg_dict[ self._argListMandatory[parameter_index] ] = arg_value

           #--------------------------------------------------------------------------
           #Add the optional parameters
           #--------------------------------------------------------------------------
            for parameter_index in range (len(self._argListMandatory),len(args)):
                arg_type = getattr(self._argObj, self._argListOptional[parameter_index - len(self._argListMandatory)])
                arg_value = args[parameter_index]
                #parameter conversion
                arg_normalized = self.argument_parsing(arg_dict,arg_type,arg_value)
                #Create the parameter string for tcl
                if self._argListOptional[parameter_index - len(self._argListMandatory)] == 'command_options':
                    parameters=" " + arg_value + parameters
                else:
                    parameters=parameters + " " + arg_normalized
                arg_dict[ self._argListOptional[parameter_index - len(self._argListMandatory)] ] = arg_value

        #print(self.name + parameters)
        return self.return_parsing(self._return_type, self._list_depth,\
            # qcontrol._socket_obj.command_wrapper(self.name + parameters))
            self._syscon_api.command_wrapper(self.name + parameters))

class command_parser:
    """
    Class used to construct the methods base on xml command definition
    """
    def __init__(self, command_list, syscon_api):
        # Check that the command_list exist
        # if command_list is None:
            # errorMsg = 'command_list must be set.'
            # raise Exception(errorMsg)
        # else:
            # self._command_list = command_list

        self._command_list = command_list
        self._syscon_api = syscon_api

        #Parsing profile
        if _os.path.exists(command_list):
            #get the XML information
            self.__parseObj = _ElementTree.parse(self._command_list) # nosec
            self.__commandsRoot = self.__parseObj.getroot()
        else:
            errorMsg = 'Unable to find the list of commands at %s' % command_list
            raise Exception(errorMsg)

    def _parsed_commands(self):
        """
        Class used to generate the object on the fly
        """
        qcommands = gen_obj()
        identation = "    "
        for command in self.__commandsRoot:
            command_obj = command_class( self._syscon_api )
            setattr(command_obj, "name", command.attrib['name'])
            setattr(command_obj, "type", command.attrib['type'])
            setattr(command_obj, "_list_depth", command.attrib['list_depth'])
            setattr(command_obj, "_return_type", command.attrib['return_type'])
            setattr(command_obj, "_argObj", gen_obj())
            setattr(command_obj, "_argListMandatory", list())
            setattr(command_obj, "_argListOptional", list())
            #Set the completed info partially:
            info_msg = identation + command.attrib['info'] + "\n"
            #---------------------- Argument parsing -------------------------
            #complete the information while the parameters are being generated
            arg_index = 0
            arg_msg = ""

            #define the table with the index included
            argument_table = [["Arg. Index","Name", "Mandatory"]]
            #Iterate the parameters add them to the object and create the table
            for argument in command:
                if arg_msg != "":
                    arg_msg = arg_msg + ", "

                if argument.attrib['mandatory'] == "yes":
                    command_obj._argListMandatory.append(argument.attrib['arg_name'])
                    info_msg = info_msg + "\n" + identation + argument.attrib['arg_name']
                    arg_msg = arg_msg + " " + argument.attrib['arg_name']
                else:
                    command_obj._argListOptional.append(argument.attrib['arg_name'])
                    info_msg = info_msg + "\n" + identation + argument.attrib['arg_name'] + "=None"
                    arg_msg = arg_msg + " " + argument.attrib['arg_name'] + "=None"

                if 'info' in argument.attrib:
                    info_msg = info_msg + ": " + argument.attrib['info']

                #Updating the table
                argument_table.append(["%d" % arg_index, argument.attrib['arg_name'],\
                                                                argument.attrib['mandatory']])

                #add the attribute to _argObj
                setattr(command_obj._argObj,argument.attrib['arg_name'], argument.attrib['type'])
                arg_index = arg_index + 1

            info_msg = command.attrib['name'] + "(" + arg_msg + " )\n" + info_msg
            info_msg = info_msg + "\n"

            #adding the table to the string
            # parameter_table = tools.logger.stringTable(argument_table)
            # info_msg = info_msg + parameter_table
            #-----------------------------------------------------------------
            # info_msg = info_msg + "\n\n***************************************\n"
            #Set the info message
            setattr(command_obj, "__doc__", info_msg)
            #set the command
            ##substitute any white space in the name for underscore
            setattr(qcommands,(command.attrib['name']).replace(" ","_"),command_obj)
        return qcommands

#### Quartus exception used to report error in the commands
# class SysconError(Exception):
class SysconError(Exception):
    """
    Quartus connection exception, used to report any
    issue related with quartus interface
    """
    def __init__(self,message):
        #Logging information
        self.msg = message
        self.value = "[Quartus Error]: " + self.msg
    def __str__(self):
        return repr(self.value)

