"""
Defines the general server commands to be extended upon

(includes BogusCommands for testing)
"""
import os
is_pysv_initialized = False


def initialize_pysv():
    import namednodes
    package = os.path.abspath(__file__).split(os.sep)[-3]
    namednodes.settings.PROJECT = package
    namednodes.sv.initialize()
    global is_pysv_initialized
    is_pysv_initialized = True


class Commands:
    """Base class of general commands for this fpga application"""
    def __init__(self, logger=None):
        if logger is None:
            try:
                from pysv_fpga.logs import get_logger
            except ImportError:
                import logging
                get_logger = logging.getLogger
            import inspect  # get the name of the derived class module
            name = inspect.getmodule(self.__class__).__name__
            logger = get_logger(name)
        self.logger = logger
        # this class is inherited, do not put any implementation imports
        # or dependencies in this constructor

    @staticmethod
    def add(a, b):
        """simple 'add' method for testing"""
        return a + b

    @staticmethod
    def register_device(path, devtype, service_type):
        if not is_pysv_initialized:
            initialize_pysv()
        from namednodes import sv
        return sv.fpga.register_device(path, devtype, service_type)

    def set_servermode_port(self, port):
        """Sets system console baseaccess and assigns port to the sc"""
        if not is_pysv_initialized:
            initialize_pysv()
        package = os.path.abspath(__file__).split(os.sep)[-3]
        self.logger.debug("set_servermode_port received: %s", package)
        import importlib
        sc = importlib.import_module('%s.systemconsolebaseaccess'%package)
        base = sc.baseaccess(port)
        return True

    def read_register(self, path):
        from namednodes import sv
        reg = sv.get_by_path(path)
        return int(reg.read())

    def write_register(self, path, data):
        from namednodes import sv
        reg = sv.get_by_path(path)
        reg.write(data)
        return True


class BogusCommands(Commands):
    """Provides a bogus set of general Commands useful for testing"""

    def __init__(self, logger):
        super(BogusCommands, self).__init__(logger)
        self._regcache = {}  # implement simple stub mode register read/write

    @staticmethod
    def register_device(path, devtype, service_type):
        return "bogus device path"

    def set_servermode_port(self, port):
        return True

    def read_register(self, path):
        return self._regcache.get(path, 0)

    def write_register(self, path, data):
        self._regcache[path] = data
        return True
