# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




from __future__ import absolute_import
from __future__ import division
from __future__ import print_function


class SimilarSteppings(object):
    """
    Helper close to create an easy way to look up similar settpings
    """
    def __init__(self, stepping_dict):
        """takes a tuple to map similar steppings to single stepping for collateral. Example:
        {
            ('A0','A1') : 'A0'
            ('B0','B1','Y0') : 'B0'
        }
        """
        # save original mapping
        self.stepping_dict = stepping_dict or {}
        self._mapping_to_base = {}
        self._mappings_from_base = {}
        self._update_mappings()

    def _update_mappings(self):
        self._mapping_to_base = {}
        self._mappings_from_base = {}
        for similar, primary in self.stepping_dict.items():
            # make sure similar is a tuple if not, make it one
            if not isinstance(similar, (tuple,list)):
                similar = [similar]
            # base is key, similar is value
            self._mappings_from_base.setdefault(primary, []).extend(similar)
            self._mappings_from_base[primary].sort()
            # similars are key, base is value
            for s in similar:
                self._mapping_to_base[s] = primary

    def add_similar(self, similar, primary):
        """
        Adds a new look for similar/primary using same formats as constructor
        similar must be a tuple, primary must be a single value

        Args:
            similar: tuple of steppings that are similar
            primary: stepping that other steppings are similar too

        """
        #if primary not in self.stepping_dict.values():
        self.stepping_dict[similar] = primary
        self._update_mappings()

    def get_primary(self, stepping):
        """
        Args:
            stepping (str): actual stepping that we want to find what stepping all the collateral
              was based on

        Returns:
            stepping that collateral was named after. If no match is found, just returns the
            stepping that was passed in
        """
        return self._mapping_to_base.get(stepping, stepping)

    def get_similar(self, stepping):
        """
        Args:
            stepping (str) : stepping that other steppings may be based on
        Returns:
            list of steppings that are derived from the one passed in. If no match is found
            returns a list with just the stepping that was passed in
        """
        return self._mappings_from_base.get(stepping,[stepping])






