

# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ..logging import getLogger

_log = getLogger()


class PluginManager(type):
    # value may later be changeed to a list, so this should not be used directly
    _plugins = {}
    #: Name of the class that is the baseclass for all plugins, that we should not actually add
    #: as a known plugin
    basetype = ""
    def __init__(cls, cls_name, cls_bases, cls_dict):
        # our own, ont the fly created base class, this is needed to provide our get/load/load all functions
        # it also prevents somone from screwing up and forgetting to add _plugins to their class
        # yeah, a bit odd to have within an __init__...dont try this at home
        super(PluginManager,cls).__init__(cls_name, cls_bases, cls_dict)
        if cls.basetype=="":
            raise Exception("Base class for pluging was created incorrectly, must specify basetype")
        if cls_name != cls.basetype:
            if 'name' not in cls_dict:
                # don't print error, assume that this is because we are going to manually register
                #_log.error("Plugin {0} did not specify name".format(cls_name))
                return
            if cls_dict['name'] == None: # don't register if it has None, must be builtin
                _log.debug("Plugin {0} specified 'None' as name".format(cls_name))
                return
            cls._plugins[cls_dict['name']] = cls
            # tweak "name" of __call__ for documentation purposes
            if "__call__" in cls_dict:
                setattr( cls, cls_dict['name'], cls.__call__)

    @classmethod
    def register(cls, plugin_class):
        """
        Manually register a plugin class
        """
        cls._plugins[plugin_class.name] = plugin_class

    @classmethod
    def unregister(cls, plugin_class):
        """
        Manually register a plugin class
        """
        if plugin_class.name in cls._plugins:
            del cls._plugins[plugin_class.name]
        else:
            raise ValueError("Unknown plugin: %s"%plugin_class.name)

    @classmethod
    def get(cls, name, obj, noerror=False):
        """Get the specified plugin object if it supports this object
        Returns:
            Plugin object if it is supported or None if plugin does not support this object
        """
        plugin = cls._plugins.get(name,None)
        if plugin == None:
            if noerror:
                return
            else:
                raise ValueError("no plugin found with the name: {0}".format(name))
        # assumes all plugins have a "create"
        plugin = cls._plugins[name]
        # should return either none or the plugin created
        plug_obj = plugin.create( obj )
        # if plugin object has call method, then that is what gets returned
        # this is case a plugin is designed with __getattr__ (like snapshot) that
        # might query the component...we don't want that during initialization
        if (plug_obj is not None and
            "__call__" in plug_obj.__dict__ or
            "__call__" in dir(type(plug_obj))):
            plug_obj = plug_obj.__call__
            plug_obj.__func__.__name__ = name
        return plug_obj

    @classmethod
    def plugins(cls):
        return list(cls._plugins.keys())

    @classmethod
    def load(cls, name, obj):
        """
        Attempts to create the specified plugin and add to the object

        Args:
            name : name of plugin
            obj : object to see if plugin supports, and add to it

        Returns:
            True / False - True, if plugin was added, otherwise False

        """
        plugin = cls.get(name,obj)
        if plugin is not None:
            setattr( obj, name, plugin )
            return True
        else:
            return False

    @classmethod
    def load_all(cls, obj):
        for p in cls._plugins:
            cls.load( p, obj)


class _BasePlugin(object):
    @classmethod
    def get(cls, name, obj):
        """Get the specified plugin object if it supports this object
        Returns:
            Plugin object if it is supported or None if plugin does not support this object
        """
        plugin = cls._plugins.get(name,None)
        if plugin == None:
            raise ValueError("no plugin found with the name: {0}".format(name))
        # assumes all plugins have a "create"
        plugin = cls._plugins[name]
        # should return either none or the plugin created
        plug_obj = plugin.create( obj )
        return plug_obj

