# Copyright (c) 2009 Raymond Hettinger
#
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge,
# publish, distribute, sublicense, and/or sell copies of the Software,
# and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
#     The above copyright notice and this permission notice shall be
#     included in all copies or substantial portions of the Software.
#
#     THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
#     EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
#     OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
#     NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
#     HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
#     WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
#     FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
#     OTHER DEALINGS IN THE SOFTWARE.
# original version
# http://code.activestate.com/recipes/576693-ordered-dictionary-for-py24/




from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ._py2to3 import *

import pprint as _pprint
from six import PY2,PY3

if PY2:
    from collections import MutableMapping
if PY3:
    from collections.abc import MutableMapping

# may switch to using this...
#''' # NOT A supported class, this is backup code...
#class sodict(dict, MutableMapping):
#    def __init__(self,**kwargs):
#        self.__order = []
#        for k,v in kwargs.items():
#            self[k] = v
#            self.__order.append( k )

#    def __getattr__(self,attr):
#        if attr.startswith("__"):
#            return object.__getattribute__(self,attr)
#        else:
#            return self[attr]

#    def __setattr__(self,attr,value):
#        if attr.startswith("_sodict"):
#            return object.__setattr__(self,attr,value)
#        else:
#            self[attr] = value

#    def __delitem__(self,item):
#        del self[item]
#        self.__order.pop(item)

#    def __setitem__(self, key, value):
#        if key not in self:
#            self.__order.append(key)
#        dict.__setitem__(self,key, value)

#    def __iter__(self):
#        for k in self.__order:
#            yield k

#    def keys(self):
#        return list(self)

#    def __getstate__(self):
#        """for pickle return an ordered list to pass back in to setstate"""
#        state = []
#        for k,v in self.iteritems():
#            state.append( (k,v) )
#        return state

#    def __setstate__(self,state):
#        for k,v in state:
#            self[k] = v

#    setdefault = MutableMapping.setdefault
#    update = MutableMapping.update
#    pop = MutableMapping.pop
#    values = MutableMapping.values
#    items = MutableMapping.items
#    iterkeys = MutableMapping.iterkeys
#    itervalues = MutableMapping.itervalues
#    iteritems = MutableMapping.iteritems
##'''

## we may later tweak odictionary for performance, but for now, the built-in ordered one is
#from collections import OrderedDict
#class odict_odict(OrderedDict):
#    def __init__(self,*args,**kwargs):
#        self.__dict__['_odict__initialized']=False # to avoid setattr call
#        OrderedDict.__init__(self,*args,**kwargs)
#        self.__initialized=True

#    def __getstate__(self):
#        print 'called'
#        return []

#    def __setstate__(self, state):
#        pass

#    # attempt to get it from dict
#    #def __getattr__(self,attr):
#    #    try:
#    #        return self[attr]
#    #    except KeyError:
#    #        return object.__getattribute__(self,attr)

#    #def __setattr__(self,attr,value):
#    #    if not self.__initialized:
#    #        return object.__setattr__(self,attr,value)
#    #    elif attr.startswith("_OrderedDict"):
#    #        return object.__setattr__(self,attr,value)
#    #    else:
    #       self[attr] = value

from copy import copy, deepcopy

# way faster for creation than the default "OrderedDict", deletions may be slower
class odict(MutableMapping):
    __slots__ = ['_data','_order']

    def __init__(self,*args,**kwargs):
        super(MutableMapping,self).__init__()
        # this is only because we have a VERY greedy setattr and we are using slots
        # please, do not use this as a BKM
        object.__setattr__(self,"_data", {})
        object.__setattr__(self,"_order", [])
        len_args = len(args)
        len_kwargs = len(kwargs)
        if not len_args and not len_kwargs:
            return
        elif len_args == 1:
            if isinstance(args[0], list):
                _data = self._data
                _order = self._order
                for k, v in args[0]:
                    _data[k] = v
                    _order.append(k)
            elif isinstance(args[0], odict):
                # if an odict, copy and preserve order
                object.__setattr__(self, "_data", copy(args[0]._data))
                object.__setattr__(self, "_order", copy(args[0]._order))
            elif isinstance(args[0], dict):
                object.__setattr__(self,"_data", copy(args[0]))
                object.__setattr__(self, "_order", list(args[0].keys()))
            else:
                _data = self._data
                _order = self._order
                # not sure how to handle?
                for k, v in args[0]:
                    _data[k] = v
                    _order.append(k)
        elif len_args > 1:
            raise TypeError("expected at most 1 arg")
        elif len_kwargs>0:
            # if kwargs given, nothing we can do about order..
            self._data.update( dict(**kwargs) )
            if PY2:
                object.__setattr__(self,"_order", self._data.keys())
            else:
                object.__setattr__(self,"_order", list(self._data.keys()))
        # else : no args ?

    def __len__(self):
        return len(self._data)

    def __iter__(self):
        return self._order.__iter__()

    def __getitem__(self, item):
        return self._data[item]

    def get(self, key, default=None):
        try:
            return self._data[key]
        except KeyError:
            return default

    def __setitem__(self, item, value):
        if item not in self._data:
            self._order.append(item)
        self._data[item] = value

    def __delitem__(self, item):
        del self._data[item]
        self._order.remove(item)

    def __getstate__(self):
        return (self._data, self._order)

    def __setstate__(self, state):
        object.__setattr__(self,"_data", state[0])
        object.__setattr__(self,"_order", state[1])

    def __copy__(self):
        return type(self)(self)

    def __deepcopy__(self, memo=None):
        n = type(self)()
        object.__setattr__(n, "_data", deepcopy(self._data, memo))
        object.__setattr__(n, "_order", deepcopy(self._order, memo))
        return n

    def __setattr__(self, attr, value):
        # just be sure to always use object.__setattr__ for data and order
        self[attr] = value

    def __getattr__(self,attr):
        try:
            return self[attr]
        except KeyError:
            return object.__getattribute__(self,attr)

    def __getnewargs__(self):
        return ()

    def __dir__(self):
        return list(self)

    def __repr__(self):
        return _pprint.pformat( dict(self.items()) )

def _dict_update(older, newer):
    """recusive update of dictionary"""
    import collections
    for k, v in newer.items():
        if isinstance(v, collections.Mapping):
            older[k] = _dict_update(older.get(k, {}), v)
        else:
            older[k] = v
    return older






def expand_ordered_dict(an_ordered_dict):
    from collections import OrderedDict
    new_ordered_dict = OrderedDict()
    for key, val in an_ordered_dict.items(): 
        if type(key) is tuple:
            new_key = ','.join(key)           
            new_ordered_dict[new_key] = val
        else:
            new_ordered_dict[key]=val
    return new_ordered_dict