
#!/usr/bin/env python
# INTEL CONFIDENTIAL
# Copyright 2014 2017 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.


from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import namednodes
import namednodes.nodes
import namednodes.plugins.lmdb_dict
from namednodes.discoveries import static
import namednodes.utils._py2to3
from optparse import OptionParser
from collections import MutableMapping

import wx.dataview as dv

import os.path
import six
from six.moves import range
try:
    import wx
    import wx.gizmos as gizmos
except ImportError:
    print("-"*10)
    print("nnviewer needs a few additional requirements than namednodes, to run, please make sure wx is pip installed")
    print("-" * 10)
    raise



import sys

_WX4 = wx.__version__.split(".")[0] == "4"


# panel for holding
class DictPanel(wx.Panel):
    def __init__(self, *args, **kwds):
        # this is just setup boilerplate
        kwds["style"] = wx.DEFAULT_FRAME_STYLE
        wx.Panel.__init__(self, *args, **kwds)
        self.tree = gizmos.TreeListCtrl(self, 1, wx.DefaultPosition, (-1,-1),
                                        style = wx.TR_DEFAULT_STYLE
                                        #| wx.TR_HAS_BUTTONS
                                        #| wx.TR_TWIST_BUTTONS
                                        #| wx.TR_ROW_LINES
                                        #| wx.TR_COLUMN_LINES
                                        #| wx.TR_NO_LINES 
                                        | wx.TR_HIDE_ROOT 
                                        | wx.TR_FULL_ROW_HIGHLIGHT
                                   )
        sizer = wx.BoxSizer(wx.VERTICAL)
        sizer.Add( self.tree, 1, wx.EXPAND)
        self.SetSizer(sizer)
        #self.tree.SetItemHasChildren(self.rootID, False)
        self.tree.AddColumn("key",width=200)
        self.tree.SetMainColumn(0)
        self.tree.AddColumn("value",width=100)
        self.rootID = self.tree.AddRoot("root")

    def update(self, nnobj, title=None):
        """given the specified node or component object, create our tree"""
        self._clear()
        ###############
        # access group
        ###############
        if isinstance(nnobj, namednodes.nodes.NamedNodeValue):
            # must be an offline mode
            nnobj.access_class = None
            nnobj.access_class_name = ""

        if hasattr(nnobj, "access_group"):
            access_groupID = self.tree.AppendItem(self.rootID, "access_group")
            self.tree.SetItemText(access_groupID, str(nnobj.access_group), 1)
        ###############
        # accesses
        ###############
        if hasattr(nnobj, "accesses"):
            accessesID = self.tree.AppendItem(self.rootID, "accesses")
            self._add_node(accessesID, nnobj.accesses)
            self.tree.Expand(accessesID)
        if hasattr(nnobj, "item_keys"):
            item_keysID = self.tree.AppendItem(self.rootID, "item_keys")
            self._add_node(item_keysID, nnobj.item_keys)
            self.tree.Expand(item_keysID)
        ###############
        # info
        ###############
        if hasattr(nnobj, "info"):
            infoID = self.tree.AppendItem(self.rootID, "info")
            self._add_node(infoID, nnobj.info)
            if len(nnobj.info) > 0:
                self.tree.SetItemHasChildren(infoID, True)
                self.tree.Expand(infoID)
        if _WX4:
            self.tree.Expand(self.rootID)
        ###############
        # target_info
        ###############
        if hasattr(nnobj, "target_info"):
            target_infoID = self.tree.AppendItem(self.rootID, "target_info")
            self._add_node(target_infoID, nnobj.target_info)
            if len(nnobj.target_info) > 0:
                self.tree.SetItemHasChildren(target_infoID, True)
                self.tree.Expand(target_infoID)
        if _WX4:
            self.tree.Expand(self.rootID) 

    def _clear(self):
        self.tree.CollapseAndReset(self.rootID)
        self.tree.SetItemHasChildren(self.rootID, False)

    def _add_node(self, parentnode, item):
        if isinstance(item,(dict,MutableMapping)):
            #keys=sorted(item.keys())
            keys=list(item.keys())
            if len(keys)==0:
                self.tree.AppendItem(parentnode,"{}")
                return
            for key in keys:
                value = item[key]
                nextnode = self.tree.AppendItem(parentnode, str(key))
                self._add_node(nextnode, value)
        elif isinstance(item,list):
            if len(item)==0:
                self.tree.AppendItem(parentnode,"[]")
                return            
            for i in range(len(item)):
                nextnode = self.tree.AppendItem(parentnode,str(i))
                self._add_node(nextnode, item[i] )
        elif isinstance(item,six.string_types): # if its a string, show the quotes
            self.tree.SetItemText(parentnode, "'%s'"%str(item), 1)
            #self.tree.AppendItem(parentnode,"'%s'"%str(item))
        else: # otherwise, assume string and don't show quotes since we don't know what it is
            #self.tree.AppendItem(parentnode,str(item))
            self.tree.SetItemText(parentnode, str(item), 1)


class NodePanel(wx.Panel):
    def __init__(self, *args, **kwds):
        wx.Panel.__init__(self, *args, **kwds)
        ###########################
        # init tree
        ###########################
        isz = (16,16)
        il = wx.ImageList(isz[0], isz[1])
        self.il = il
        if not _WX4:
            self.i_folder      = il.Add(wx.ArtProvider_GetBitmap(wx.ART_FOLDER,      wx.ART_OTHER, isz))
            self.i_folder_open = il.Add(wx.ArtProvider_GetBitmap(wx.ART_FILE_OPEN,   wx.ART_OTHER, isz))
            self.i_file        = il.Add(wx.ArtProvider_GetBitmap(wx.ART_NORMAL_FILE, wx.ART_OTHER, isz))
        else:
            # assume 4 ?
            self.i_folder      = il.Add(wx.ArtProvider.GetBitmap(wx.ART_FOLDER,      wx.ART_OTHER, isz))
            self.i_folder_open = il.Add(wx.ArtProvider.GetBitmap(wx.ART_FILE_OPEN,   wx.ART_OTHER, isz))
            self.i_file        = il.Add(wx.ArtProvider.GetBitmap(wx.ART_NORMAL_FILE, wx.ART_OTHER, isz))

        # our tree object, self.tree
        # self.tree = wx.TreeCtrl(self, -1, style=wx.TR_HAS_BUTTONS|wx.TR_DEFAULT_STYLE|wx.SUNKEN_BORDER)
        self.tree = wx.TreeCtrl(self, 1, wx.DefaultPosition, (-1,-1),
                                        style = wx.TR_DEFAULT_STYLE
                                        #| wx.TR_HAS_BUTTONS
                                        #| wx.TR_TWIST_BUTTONS
                                        #| wx.TR_ROW_LINES
                                        #| wx.TR_COLUMN_LINES
                                        #| wx.TR_NO_LINES 
                                        | wx.TR_FULL_ROW_HIGHLIGHT
                                   )
        self.tree.SetImageList(il) 

        sizer = wx.BoxSizer(wx.VERTICAL)
        sizer.Add(self.tree, 1, wx.EXPAND, )
        # sizer.Fit(self)
        self.SetSizer( sizer )
        self.tree.Bind(wx.EVT_TREE_ITEM_EXPANDING, self.onExpand)
        self.tree.Bind(wx.EVT_TREE_ITEM_COLLAPSING, self.onCollapse)
        self.rootID = None

    def _clear(self):
        self.tree.CollapseAndReset(self.rootID)
        self.tree.SetItemHasChildren(self.rootID, False)

    def LoadFile(self,file):
        self.filetype = "lmdb"
        if file.endswith(".mdb"): # pass parent dir to get from file
            file = os.path.dirname(file)
        elif file.endswith(".spkx"):
            self.filetype = "spkx"            

        # add tree root
        if self.filetype == "spkx":
            root_comps = static.load_offline(file)
            # simple case
            if len(root_comps) == 1:
                root_comp = root_comps[0]
            elif len(root_comps) > 1:
                # if mutliple we have to create a fake component to add in each one
                root_comp_def = namednodes.CreateComponentDefinition(name="sv")
                root_comp = root_comp_def.create("sv")
                for c in root_comps:
                    root_comp.add_component(c)
            else:
                # nothing found ?
                return

        elif self.filetype == "lmdb":
            root_comp = namednodes.GetComponentFromFile(file)


        if self.rootID is None:
            self.rootID = self.tree.AddRoot(root_comp.name)
        else:
            self._clear()
            self.tree.SetItemText(self.rootID, root_comp.name)
            
        self._images_component(self.rootID)

        if _WX4:
            self.tree.SetItemData(self.rootID, (root_comp,1))
        else:
            self.tree.SetPyData(self.rootID, (root_comp,1))

        if len(root_comp.sub_components) > 0:
            self.tree.SetItemHasChildren( self.rootID )
        elif len(root_comp.nodes) > 0:
            self.tree.SetItemHasChildren( self.rootID )
        else:
            # though this would be bad...
            self.tree.SetItemHasChildren( self.rootID , False)
        # events...
        self.__collapsing = False
        # forse collapse and re-expand
        self.tree.Expand(self.rootID)
        #self.Refresh()

    _last_item = None
    def onCollapse(self, event):
        item = event.GetItem()
        # we have to do some extra stuff to prevent odd recursion error from the collapse and reset
        # tried doing just the reset and event.Skip so that event propigates, but no luck...
        if self._last_item != item.ID.__int__():
            self._last_item = item.ID.__int__()
            self.tree.CollapseAndReset(item)
        event.Skip()
        # just delete all - doesn't work, can re-collapse


    def onExpand(self, event):
        '''onExpand is called when the user expands a node on the tree
        object. It checks whether the node has been previously expanded. If
        not, the extendTree function is called to build out the node, which
        is then marked as expanded.'''
        
        # get the wxID of the entry to expand and check it's validity
        parentID = event.GetItem()
        if not parentID.IsOk():
            parentID = self.tree.GetSelection()
        
        # only build that tree if not previously expanded
        if _WX4:
            parentComp = self.tree.GetItemData(parentID)[0]
        else:
            parentComp = self.tree.GetPyData(parentID)[0]

        if hasattr(parentComp,"sub_components"):
            subkeys = list(parentComp.sub_components.keys())
            subkeys.sort()
            subcomps = [ parentComp.sub_components[s] for s in subkeys ]
            for child in subcomps:
                # add the child to the parent
                childID = self.tree.AppendItem(parentID, child.name)
                self.tree.SetItemHasChildren( childID )
                self._images_component( childID )
                # associate the full child with its tree entry
                if _WX4:
                    self.tree.SetItemData(childID, (child, False))
                else:
                    self.tree.SetPyData(childID, (child, False))

        if hasattr(parentComp,"nodes"):
            nodenames = list(parentComp.nodenames)
            nodenames.sort()
            if hasattr(parentComp,"item_definition"):
                child = parentComp.item_definition
                # add the child to the parent
                childID = self.tree.AppendItem(parentID, child.name)
                # if has children OR has item_definition
                has_children = len(child.nodes)>0
                has_children |= hasattr(child,"item_definition")
                self.tree.SetItemHasChildren( childID, has_children )
                self._images_node( childID )
                # associate the full child with its tree entry
                if _WX4:
                    self.tree.SetItemData(childID, (child, False))
                else:
                    self.tree.SetPyData(childID, (child, False))
            for nodename in nodenames:
                child = parentComp.get_node(nodename)
                # add the child to the parent
                childID = self.tree.AppendItem(parentID, child.name)
                # if has children OR has item_definition
                has_children = len(child.nodes)>0
                if isinstance(child, namednodes.nodes.NamedNodeArrayDefinition):
                    has_children = True
                self.tree.SetItemHasChildren( childID, has_children )
                self._images_node( childID )
                # associate the full child with its tree entry
                if _WX4:
                    self.tree.SetItemData(childID, (child, False))
                else:
                    self.tree.SetPyData(childID, (child, False))

    def _images_component(self, node):
        self.tree.SetItemImage(node, self.i_folder, which = wx.TreeItemIcon_Normal)
        self.tree.SetItemImage(node, self.i_folder_open, which = wx.TreeItemIcon_Expanded)

    def _images_node(self, node):
        self.tree.SetItemImage(node, self.i_file, which = wx.TreeItemIcon_Normal)
        # self.tree.SetItemImage(node, self.i_folder_open, which = wx.TreeItemIcon_Expanded)
        
 
class MyFrame(wx.Frame):
    def __init__(self, *args, **kwds):
        # this is just setup boilerplate
        kwds["style"] = wx.DEFAULT_FRAME_STYLE
        wx.Frame.__init__(self, *args, **kwds)

        # MENU
        menuBar = wx.MenuBar()
        menu = wx.Menu()
        menu.Append(101,"Open...","For opening database file")
        menuBar.Append(menu, "File")
        self.SetMenuBar(menuBar)
        self.Bind(wx.EVT_MENU, self.OpenFileMenu, id=101)

        # SPLITTER & PANELS
        self.splitter = wx.SplitterWindow(self, -1)
        # Left Size
        self.nodePanel = NodePanel(self.splitter, -1)
        # RIGHT SIDE
        self.dictPanel = DictPanel(self.splitter, -1)

        self.splitter.SplitVertically(self.nodePanel, self.dictPanel)
        self.Centre()
        width=600
        height=500
        self.SetSize((width,height))
        self.splitter.SetSashPosition(width/3)
        self.Layout()

        # register the self.onExpand function to be called
        # wx.EVT_TREE_ITEM_EXPANDING(self.tree, self.tree.GetId(), self.onExpand)
        self.nodePanel.tree.Bind(wx.EVT_TREE_SEL_CHANGED, self.onSelect)
        # initialize the tree

    def onSelect(self, event):
        item = event.GetItem()
        if _WX4:
            nnobj = self.nodePanel.tree.GetItemData(item)
            if nnobj is not None:
                nnobj = nnobj[0]
        else:
            nnobj = self.nodePanel.tree.GetPyData(item)[0]
        if nnobj is not None:
            self.dictPanel.update(nnobj, nnobj.path)
       
    def OpenFileMenu(self, event):
        if not _WX4:
            style=wx.OPEN | wx.CHANGE_DIR
        else:
            style=wx.FD_OPEN | wx.FD_CHANGE_DIR
        dlg = wx.FileDialog(
                self, message="Choose a file",
                defaultDir=os.getcwd(), 
                defaultFile="",
                wildcard="NamedNodes LMDB of SPKX (*.mdb;*.spkx)|*.mdb;*.spkx",
                style=style,
            )

        # Show the dialog and retrieve the user response. If it is the OK response, 
        # process the data.
        if dlg.ShowModal() == wx.ID_OK:
            # This returns a Python list of files that were selected.
            path = dlg.GetPath()
            self.nodePanel.LoadFile( path )

        dlg.Destroy()
    
def main(argv=None):
    if argv is None:
        argv = sys.argv

    parser = OptionParser()
    parser.add_option("-f",dest="file",help="Named Nodes file to open",default=None)
    (options, args) = parser.parse_args(argv)

    app = wx.GetApp()
    if app is None:
        app = wx.App()
    #wx.InitAllImageHandlers()
    frame = MyFrame(None, -1, "")
    #frame.OpenDB("
    app.SetTopWindow(frame)
    frame.Show()
    if options.file is not None:
        if not os.path.exists( options.file ):
            raise RuntimeError("File %s not found"%options.file)
        frame.nodePanel.LoadFile( options.file )
    app.MainLoop()

if __name__ == "__main__":
    main()

