

# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



"""
BitMapEntry and BitMap(list) are used as part of AFD flows. The typical
TDEF conversion flow ends up assigning return values to a dictionary. Like so::

    ops = {}
    ops[1] = ipc.irdrscan(0,2,32)


Later we use the a list of BitMapEntries to map the various bits on to fields.
Each field may have a list of BitMapEntries so that we swizzle the bits
appropriatley to create the final field value

"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ._py2to3 import *
from collections import namedtuple
from ipccli import BitData


class BitMap(list):
    #: can be used so that the bitmap gets evaluated later
    datasource = None

    @property
    def totalbits(self):
        """returns the maximum bit we will be writing to once datasource data is given"""
        maxbit = 0
        for entry in self:
            maxbit = max(maxbit, entry.dest_lower+entry.numbits)
        return maxbit

    def get_value(self, datasource_dict):
        """provided the datasource dictionary, determine the value

        Args:
            datasource_dict (dict): dictionary mapping source to data where both
                                    are integer values

        """
        myvalue = 0
        for mapinfo in self:
            if isinstance(mapinfo.datasource, BitData):
                d = long(mapinfo.datasource)
            else:
                d = datasource_dict.get(mapinfo.datasource, None)
                if d is None:
                    raise BaseException("Missing data source {0}".format(mapinfo.datasource))
            if isinstance(d, BitMap):
                d = d.get_value()
            mask = (1 << long(mapinfo.numbits))-1
            d = (long(d) >> long(mapinfo.from_lower)) & long(mask)
            if mapinfo.invert:
                d = ~d
            myvalue |= d << (mapinfo.dest_lower)
        return BitData.CreateFromLong(self.totalbits, myvalue)

    def __long__(self):
        if self.datasource is None:
            raise RuntimeError("Datasource must be specified ahead of time to implicitly turn in to long")
        return long(self.get_value(self.datasource))


BitMapEntryTuple = namedtuple("BitMapEntry", "datasource from_lower numbits dest_lower invert")
# this is pretty much the same thing but pull from a constant instead of an operation dictionary
BitMapConstantEntryTuple = namedtuple("BitMapConstantEntry", "constant from_lower numbits dest_lower invert")

class BitMapEntry(BitMapEntryTuple):
    def __repr__(self):
        datasource = self.datasource if self.datasource is not None else "None"
        return "<bitmap source=%s from=%d numbits=%d dest=%d invert=%s>"%(
                datasource, self.from_lower, self.numbits, self.dest_lower, self.invert)

    @property
    def from_upper(self):
        """
        use from_lower and numbits to calculate the upper bit of where we are getting data from""
        """
        return self.from_lower+self.numbits-1

    @property
    def dest_upper(self):
        """
        use dest_lower and numbits to calculate the upper bit of where we are getting data from""
        """
        return self.from_lower+self.numbits-1

class BitMapConstantEntry(BitMapConstantEntryTuple):
    def __repr__(self):
        return "<bitmap constant=%s from=%d numbits=%d dest=%d invert=%s>"%(
                self.constant, self.from_lower, self.numbits, self.dest_lower, self.invert)

    @property
    def from_upper(self):
        """
        use from_lower and numbits to calculate the upper bit of where we are getting data from""
        """
        return self.from_lower+self.numbits-1

    @property
    def dest_upper(self):
        """
        use dest_lower and numbits to calculate the upper bit of where we are getting data from""
        """
        return self.from_lower+self.numbits-1