
# INTEL CONFIDENTIAL
# Copyright 2017 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.


"""
Module containing functions for sending data to Lantern Rock telemetry database.
"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import atexit
from copy import copy
import sys
import os
import uuid
import socket
import traceback

from . import settings
from ._version import __version__
from .events.discovery import DiscoveryEventTypes
from .logging import getLogger

try:
    from svtools.common import baseaccess
except:
    baseaccess = None

_LOG = getLogger()

# Remote logger might not be installed
LanternRock = None
try:
    if sys.platform == 'win32':
        from lanternrock import LanternRock, LanternRockError
except ImportError:
    pass
except Exception:
    _LOG.debug(traceback.format_exc())


_APP_NAME_PROD = "ITS_namednodes"
_APP_NAME_DEV = "ITS_namednodes_dev"
_TID_PROD = "60b44ee5-e43e-49cc-8119-20f055570802"
_TID_DEV = "576b3f50-5167-444e-ae44-a483ee6d098b"
# _TID_DEV = "4acb70b8-462a-485a-be47-ae2d01561ac3" # original namednodes one...
_TID_DEV = "00000000-1111-2222-3333-444444444444"


_SESSION_NAME = "main"
_APP_VERSION = str(__version__)
_PYSV_SESSION_VARIABLE = "PYSV_SESSION_UUID"


class _LanternRockManager(object):
    """ 
    remote_logger Lantern Rock access wrapper with some namednodes-specific helper methods 
    Don't instantiate this class directly - use get_instance() instead
    """

    _instance = None

    def __init__(self):
        self.api_available = False
        self._base = None
        self._session = None
        self._count_info = {}

        if LanternRock and sys.platform == 'win32':
            # make sure we are not running on team city
            for key in os.environ.keys():
                # dont send ANY events if we are in team city..
                if key.upper().startswith("TEAMCITY"):
                    break
            else:
                try:
                    self._base = LanternRock()
                    self.api_available = True
                    # Make sure session is ended and API de-initialized before Python exits
                    atexit.register(self._end_session)
                except:
                    import traceback
                    traceback.print_exc()
                    print("Telemetry data will not be captured")
                    self.api_available = False

    @classmethod
    def get_instance(cls):
        """ 
        Always gets the same instance of this class so one session handle is maintained 
        If remote_logger is unavailable, then the 'available' attribute of the instance
        is set to False
        """

        if cls._instance is None:
            cls._instance = cls()
        return cls._instance

    def record_event(self, name, data, count=1, sum=1.0):
        """ 
        Send an event to the Lantern Rock API if API is available 
        Returns the return code of RecordEventEx call, or None if API is unavailable
        """

        return_val = None

        if self.api_available:
            try:
                if not self._session:
                    self._start_session()
                return_val = self._base.RecordEventEx(self._session, name, count=count, sum=sum, datadict=data)
            except:
                # survive bad Lanternrock installs
                import traceback
                traceback.print_exc()
                self.api_available = False
                _LOG.error("A lanternrock error occurred, but everything should still work normally")
                return

        return return_val

    def record_event_count(self, name, data):
        # dont bother saving the info if the api is not available
        # admittedly not sure how to do counts for the actual data yet...
        if self.api_available:
            self._count_info.setdefault(name, []).append(data)

    def _flush_events_with_count(self, name, data):
        for k, v in self._count_info:
            self.record_event(k, None, count=len(v))

    def _get_pysv_session(self):
        sessid = os.environ.get(_PYSV_SESSION_VARIABLE, None)
        if sessid is not None:
            return {_PYSV_SESSION_VARIABLE: sessid}
        else:
            sessid = str(uuid.uuid4())
            os.environ[_PYSV_SESSION_VARIABLE] = sessid
            return {_PYSV_SESSION_VARIABLE: sessid}

    def _get_app_session_data(self):
        # fill this in with session data expected for this app
        session_data = {}
        session_data['python_version'] = sys.version
        session_data['python_version_major'] = sys.version_info[0]
        session_data['python_version_minor'] = sys.version_info[1]
        session_data['python_version_major_minor'] = str(sys.version_info[0]) + "." + str(sys.version_info[1])

        # grab all the namednodes settings...
        for k in dir(settings):
            session_data[k] = getattr(settings,k)

        return session_data

    def _start_session(self):
        """ 
        Call Initialize with namednodes parameters and start a session  
        Returns the return code of Initialize call
        """

        # Initialize API
        dev = settings.USE_DEV_LR_DATABASE
        _LOG.debug("initializing Lantern Rock and starting session")
        app_name = _APP_NAME_DEV if dev else _APP_NAME_PROD
        self.telemetry_id = _TID_DEV if dev else _TID_PROD
        self._base.Initialize(app_name, _APP_VERSION, self.telemetry_id)

        # Start session, sending current namednodes settings as session data
        session_data = self._get_app_session_data()
        session_data.update(self._get_pysv_session())
        self._session = self._base.BeginSessionEx(_SESSION_NAME, session_data)

    def _end_session(self):
        """ 
        API/session clean-up method to register with atexit; do not call directly unless debugging
        Returns tuple of return codes from end session and deinitialize calls, or None if there
        already was no session handle
        """
        if self._session:
            _LOG.debug("ending Lantern Rock session and deinitializing")
            try:
                self._base.EndSession(self._session)
            except LanternRockError as e:
                _LOG.error("Error during end session: %d"%e.HRESULT)

            # from LR folks, make sure deinitialize is BEFORE upload
            try:
                self._base.Deinitialize()
            except LanternRockError as e:
                _LOG.error("Error during Lantern Rock deinitialization: %d" % e.HRESULT)

            try:
                self._base.Upload(self.telemetry_id, options={'show': False})
            except LanternRockError as e:
                _LOG.error("Error during end upload: %d" % e.HRESULT)

        return


def _dict_to_key_string(dictionary):
    """ 
    Get a string containing a comma separated list of the keys of given dictionary
    e.g. {'socket0': <...>, 'socket1': <...>} -> "socket0,socket1"
    """
    return ','.join([str(key) for key in dictionary])



def _get_pysv_access_():
    if baseaccess is not None:
        # if common is available, try to find out which access mode we are in
        access = str(baseaccess.getaccess())
        # default is to just use what get access had
        pysv_access = access
        if access == "stub":
            #environment is relatively new for stub mode, can probably remove the getattr check by end of 2017
            base = baseaccess.getglobalbase()
            environment = getattr(base,"environment", "stub")
            return "%s_%s"%(access,environment)
        elif access in ['itpii', 'ipc']:
            base = baseaccess.getglobalbase()
            return "%s_%s"%(access, base.environment)
        else:
            # not a complex environment just return the current access
            # do not get baseobject so that we can not initalize if we dont have to
            return access
    else:
        # specifically: No access being used, due to no pythonsv
        return "None"


def record_discovery_event(event):
    """ Event handler for sending discovery event data to Lantern Rock API """
    mgr = _LanternRockManager.get_instance()

    # If API is not available, do nothing
    if mgr.api_available:
        # First, format event data to suit API (see LanternRockWindowsBaseAccess.dict_to_lr)
        if event.event_type == DiscoveryEventTypes.post_initialize:
            lr_event_name = 'initialize'
            lr_data = {
                'refresh': event.event_data['refresh'],
                'project': event.event_data['project'],
                'items': _dict_to_key_string(event.event_data['items']),
                'environment': _get_pysv_access_(),
            }

        elif event.event_type == DiscoveryEventTypes.post_find_all:
            lr_event_name = 'discovery'
            lr_data = {
                'refresh': event.event_data['refresh'],
                'item': event.event_data['item'],
                'items_found': _dict_to_key_string(event.event_data['items_found']),
            }
        else:
            # Ignore all other event types
            return

        # Then, record the event
        mgr.record_event(lr_event_name, lr_data)

def record_event(name, data):
    mgr = _LanternRockManager.get_instance()
    return mgr.record_event(name, data)

def record_event_count(name, data):
    mgr = _LanternRockManager.get_instance()
    return mgr.record_event_count(name, data)
