
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



"""
Settings
========

Known settings:

    - **PROJECT**
    - **DEBUG** - Used to enable some


"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function
from ._version import __classification__

import os

# Use production Lantern Rock database if namednodes sdist/bdist is created using pylod,
# otherwise use development database
_LR_DEV = False


class Settings(object):


    # LIST FORM
    default_settings = dict(
        CLASSIFICATION = __classification__,
        DEBUG = False,  # turns on more warnings and error messages
        DISPLAY_SHOW_COMP_SORT = "name",
        EVENTS_ON_ACCESSES = False,
        EVENTS_ON_DISCOVERY = True,
        EVENTS_ON_SEARCH = False,
        EVENTS_ON_QUEUING = True,
        HIDE_METHODS = False,
        STATIC_DISCOVERY_COMPRESSION=1,
        ALPHA_FEATURE_STATIC_JSON = True,
        LMDB_DEFINITION_CACHE = True,
        LMDB_FORCE_RO = False,
        NO_FILELOCKS = False,       # Used to turn off file locks for some bad behaving OS's/environments
        PROJECT = "__PYSVCONFIG__",
        UNZIP_LOCK_TIMEOUT = 5*60,  # Timeout used to signal a failure if we cannot get a lock in time
        USE_DEV_LR_DATABASE = _LR_DEV,  # If True, send Lantern Rock data to development database
        STATIC_REQUIRES_DATABASE=False, # If False, then loading static or offline files will not error if database (lmdb) file is missing
        USE_ONLY_RELATIVE_PATHS=False, # True/False as to whether it is safe to use pysv_path when building spks/databases
        GET_ALL_STOP_ON_ERROR=False, #True/False to stop/pdb when discovery code hits an error
    )
    def __setattr__(self,attr,value):
        if attr not in self.default_settings:
            raise Exception("Unknown setting: %s"%attr)
        object.__setattr__(self, attr, value)

    def __getattr__(self,attr):
        if attr not in self.default_settings:
            raise AttributeError("Unknown setting: %s"%attr)
        try:
            # won't get called if setting was set
            env_var = "NN_"+attr.upper()
            if env_var in os.environ:
                value = os.environ[env_var]
                # convert if we can...start with bool
                if value.lower() in [ 'true','false' ]:
                    value = value.lower() == 'true'
                else:
                    try:
                        value = int( value, 0)
                    except ValueError:
                        pass
            else:
                # not in env variable, save default and return it
                value = self.default_settings[attr]
            # save the value we determined
            object.__setattr__(self, attr, value)
            return value
        except AttributeError as e:
            raise RuntimeError(e)

    def __dir__(self):
        return list(self.default_settings.keys())

    def reset(self):
        """reset settings back to default. This does not impact the fact that
        environment variables override the default"""
        for setting in self.default_settings:
            if setting in self.__dict__:
                del self.__dict__[setting]


import sys
# workaround for the fact that we are about to hijack our module
_ref = sys.modules[__name__]
# replace with our settings object
sys.modules[__name__] = Settings()


