
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




"""
==========
Registers
==========

The RegisterDefinition, RegisterValue, FieldDefinition, and FieldValue objects
are subclasses of the NamedNodeDefinition and NamedNodeValue classes. They are
intended to provide functions and attributes that will give backwards
compatibility with PythonSv


.. TODO::
    - Backwards compatible (or new) logging file for logging all named reads/writes

Backwards Compatbility
========================

-------------------
Component
-------------------

=========================   ===========     ==============================================================================================
Attribute                   Implemented     Note
=========================   ===========     ==============================================================================================
addBar                      YES             formula plugin
build_supported_accesses    TBD             will have similar capability (depends collateral conversion)
features                    NO
getBarFormula               YES             formula plugin
getBarNames                 YES             formula plugin
getBarPolicy                YES             formula plugin
getaccess                   YES             :py:meth:`RegisterComponent.getaccess`
getaccesschoices            YES             :py:meth:`RegisterComponent.getaccesschoices`
getbypath                   YES             :py:meth:`RegisterComponent.getbypath`
getfeatureobject            NO
getregisterobject           YES             :py:meth:`RegisterComponent.getregisterobject`
groups                      YES             does as a plugin
hasattr                     NO              (this was 'faster' has attr, shouldn't be needed since we have 'info')
load                        TBD             need more customer feedback
readBar                     YES             formula plugin
register_filter_file        NO              filtering data will be with lmdb file
registers_paths             NO              alternative method will be used
register_paths_file         NO              alternative method will be used
registers                   YES             :py:data:`FieldDefinition.default`
registers_masterlist        NO              unlikely to be ported/needed
save                        TBD
searchaddress               YES             plugin - also known as search_info
setBarPolicy                YES             formula plugin
setName                     NO              -- can revist if needed, but should be used only by discovery
set_supported_accesses      TBD             will have similar capability, API still TBD (collateral conversion tool)
setaccess                   YES             :py:meth:`RegisterComponent.setaccess`
setdid                      NO              -- can revist if needed, but should be used only by discovery
show                        YES             plugin - will add to Rally
showBars                    YES             formula plugin
stepping                    WILL            will be part of info
updateBars                  YES             formula plugin
writeBz2                    NO              different data storage for named nodes
writecode                   NO              different data storage for named nodes
writexml                    NO              different data storage for named nodes
addSubComponent             YES             :py:meth:`RegisterComponent.addSubComponent`
addfeature                  NO              plan to migrate features to use registers
addregister                 YES             :py:meth:`RegisterComponent.addregister`
applyfilter                 WILL            yes - will add to rally == or equivalent
checkdefaults               YES             will add to rally
clearUsedRegisters          NO              dropping '_used' feature
copySubComponent            NO              this was only needed due to lack of definition concept
createalias                 NO              this feature appears to only partially work
getSubComponents            YES             :py:meth:`RegisterComponent.getSubComponents`
getUsedRegisters            NO              dropping '_used' feature
getfeatdef                  NO
getreg                      YES             :py:meth:`RegisterComponent.getreg`
getregdef                   YES             :py:meth:`RegisterComponent.getregdef`
loadregisters               YES             via plugin
logregisters                YES             via plugin
makelinks                   WILL
mergeUsedRegisters          NO              dropping '_used' feature
readfeature                 NO
readregister                YES             :py:meth:`RegisterComponent.writeregister`
regverify                   NO              Flexcon does a much more thorough job here
removeSubComponent          YES             :py:meth:`RegisterComponent.removeSubComponents`
removefeature               NO
removeregister              YES             :py:meth:`RegisterComponent.removeregister`
replaceregister             NO              only used in legacy regex XML parser
search                      YES             search plugin
search_attrs                NO              was not widely used
setStepping                 NO
setregisterobject           NO              this just provided warning anyway
showSubs                    YES             show subs plugin
showsearch                  WILL            display plugin
writedefaults               WILL            need to confirm this as a plugin....
writefeature                NO              plan to migrate features to use registers
writeregister               YES             :py:meth:`RegisterComponent.writeregister`

_bus
=========================   ===========     ==============================================================================================


-------------------
RegisterDefinition
-------------------

:py:class:`~RegisterDefinition`

==============  ===========     ==============================================================================================
Attribute       Implemented     Note
==============  ===========     ==============================================================================================
addattribute    NO              legacy to old ITP
addbitdecode    TBD             need to create solution here
addfield        WILL            should just be an alias to add_node
description     YES             :py:data:`RegisterDefinition.description`
fields          YES             :py:data:`RegisterDefinition.fields`
getaddress()    YES             return "self.info"
getfieldname    WILL            will check for usage to determine priority
name            YES             :py:data:`namednodes.nodes.NamedNodeDefinition.name`
read            NO              should be using the value objects not the definition
readfield       NO              should be using the value objects not the definition
setaddress      NO              doesn't handle old address classes
write           NO              should be using the value objects not the definition
writefield      NO              Should not be done on register definition
IntelRsvd       YES             :py:data:`RegisterDefinition.IntelRsvd`
_getIntelRe..   NO              Users should have been using IntelRsvd attribute
_address        NO              point people to self.info
_used           NO              not widely used anymore
_sortfields     NO              ...
_w1cpresent     NO              this was definitely just an internal variable
==============  ===========     ==============================================================================================

----------------
RegisterValue
----------------

======================  ===========     ===================================================================================
Attribute               Implemented     Note
======================  ===========     ===================================================================================
IntelRsvd               YES             :py:data:`RegisterValue.IntelRsvd`
name                    YES             :py:data:`namednodes.nodes.NamedNodeValue.name`
description             YES             :py:meth:`RegisterValue.description`
fields                  YES             :py:meth:`RegisterValue.fields`
addbitdecode(           NO              will probably be added via definition
bits(                   YES             exists as a plugin
flush()                 YES             available now - :py:meth:`namednodes.nodes.NamedNodeValue.flush`
getdefault()            YES             :py:meth:`GetDefaultPlugin`
getfieldobject(         YES             :py:meth:`RegisterValue.getfieldobject`
getfielddefinition(     YES
getfieldname(           WILL
getspec()               YES             implemented as a plugin
readfield(              YES             Return field object and force a read - :py:meth:`RegisterValue.readfield`
read()                  YES             inherits from NodeValue - :py:meth:`namednodes.nodes.NamedNodeValue.read`
setfieldobject(         NO
show()                  YES             implement as a plugin
showfields()            PLUGIN          implement as a plugin
store(                  YES             inherits from NodeValue: :py:meth:`namednodes.nodes.NamedNodeValue.store`
storefield(             YES             :py:meth:`RegisterValue.storefield`
valuestr()              PLUGIN          implement as a PLUGIN
write(                  YES             available now - :py:meth:`namednodes.nodes.NamedNodeValue.write`
getstorefield(          TBD...
showfieldsstr()         PLUGIN          implement as a plugin
writefield(             YES             :py:meth:`RegisterValue.writefield`
getaddress()            YES             return self.definition.info
showvalues()            PLUGIN          implement as a plugin
_address                TBD             likely no
_description            NO              this is just due to really old backwards compatibility, "description" will work
_regvalue               YES             alias this to "value"
_register               YES             alias to definition
_state                  YES             path available, but for backwards compatibility: :py:data:`RegisterValue._state`
_store(                 NO              store is avaialble in both
_storage_in_use         NO              this should not have been used
_storedvalue            NO              this should not have been directly used
_topysvxml              NO              will use other XML writers
_used                   NO              not widely used by the community
_write                  NO              can use write
_writefield             NO              can use writefield
======================  ===========     ===================================================================================


-----------------
FieldDefinition
-----------------

======================  ===========     =======================================================================================
Attribute               Implemented     Note
======================  ===========     =======================================================================================
IntelRsvd               YES             :py:data:`FieldDefinition.IntelRsvd`
attribute               YES             :py:data:`FieldDefinition.attribute`
description             YES             :py:data:`FieldDefinition.description`
name                    YES             part of NamedNode: :py:data:`namednodes.nodes.NamedNodeDefinition.name`
upperbit                YES             :py:data:`FieldDefinition.upperbit`
addattribute(           NO              this is used by older component code before info objects existed
decodedict              NO              decode infastructure fundamentally different
getbits(                WILL            this will be available as a plugin
resettype               TBD             need to investigate how widespread usage is
adddecode(              NO              decode infastructure fundamentally different
default                 YES             :py:data:`FieldDefinition.default`
lowerbit                YES             :py:data:`FieldDefinition.lowerbit`
setbits(                WILL            will be available as a plugin
_knownattribs           NO              No longer needed, info object replaces
_tocodestr()            NO              old component code specific
_topysvxml()            NO              old component code specific
_used                   NO              likely not, will investigate how widely used
======================  ===========     =======================================================================================

-----------
FieldValue
-----------

======================  ===========     ==========================================================================================
Attribute               Implemented     Note
======================  ===========     ==========================================================================================
attribute               YES             just refence back to definition.attribute: :py:data:`FieldValue.attribute`
default                 YES             just refence back to definition.default: :py:data:`FieldValue.default`
description             YES             just refence back to definition.description: :py:data:`FieldValue.description`
decodedict              NO              decode infastructure fundamentally different
IntelRsvd               YES             just refence back to definition.IntelRsvd: :py:data:`FieldValue.IntelRsvd`
lowerbit                YES             just refence back to definition.lowerbit: :py:data:`FieldValue.lowerbit`
name                    YES             part of any NamedNodeValue: :py:data:`namednodes.nodes.NamedNodeValue.name`
resettype               TBD             need to investigate how widespread usage is
upperbit                YES             just refence back to definition.upperbit: :py:data:`FieldValue.upperbit`
adddecode(              NO              decode infastructure fundamentally different
bits(                   YES             plugin exists
getspec()               YES             plugin exists
flush()                 YES             Part of access
getdecodeval()          YES             plugin, decode infastructure still being worked out
read()                  YES             Part of NamedNodeValue, field access for read is:
                                        :py:meth:`namednodes.accesses.register.AccessField.read`
registerflush(
show()                  PLUGIN
store(                  YES             Part of NamedNodeValue, field access for store is:
                                        :py:meth:`namednodes.accesses.register.AccessField.store`
write(                                  Part of NamedNodeValue, field access for write is:
                                        :py:meth:`namednodes.accesses.register.AccessField.write`
update()                TBD             this is same as the new :py:data:`namednodes.nodes.NamedNodeValue.get_value` capabilities.
                                        Which the AccessField implements: :py:meth:`namednodes.accesses.register.AccessField.get_value`
valuestr()              YES             plugin
_state                  YES             path available, but for backwards compatibility: :py:data:`FieldValue._state`
_read()                 NO              read exists both in new and old infastructure, code can be udpated to use that
_storedvalue            NO              No longer available, stored at register level
                                        This could *potentially* be added back as an attribute
                                        if it is really needed
_field                  YES             would be the same as the definition object
_register               YES             would be the same as parent
_topysvxml()            NO              very pythonsv specified
_fieldvalue             YES
======================  ===========     ==========================================================================================


"""
# to hold code that mimics PythonSv's old RegisterDefinition, FieldDefinition attributes
# this is to only provide backwards compatibility with user scripts NOT for the per project enabling code

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from .nodes import NamedNodeDefinition, NamedNodeValue
from . import nodes
from .comp import NamedComponent
from .utils._py2to3 import *
from . import settings

from .logging import getLogger
_log = getLogger()

class RegisterComponent(NamedComponent):
    """
    Special Component to return RegisterValue instead of NodeValue objects
    all other code is identical to NamedComponent
    """
    ############################
    # need to add all sorts of attributes for backwards compatibilty here
    ############################
    # some simple renaming to give backwards compatibility
    # rename was done only to comply with pep*
    getaccess = NamedComponent.get_access
    setaccess = NamedComponent.set_access
    getaccesschoices = NamedComponent.get_access_choices
    addSubComponent = NamedComponent.add_component
    removeregister = NamedComponent.remove_node
    removeSubComponent = NamedComponent.remove_component

    @property
    def _state(self):
        """Alias from old _state variable to new 'path' variable"""
        return self.path

    @property
    def registers(self):
        """returns nodenames for this component"""
        return self.nodenames

    def getSubComponents(self):
        """Returns a list of strings with the names of the sub components

        Note::
            This is equivalent to comp.sub_components.keys()
        """
        return list(self.sub_components.keys())

    def getreg(self,regname):
        """Returns the register object without doing a read

        Args:
            regname : name of register to return a RegisterValue object for

        """
        return self.get_node( regname )

    def getregisterobject(self,regname):
        """Returns the register object without doing a read

        Args:
            regname : name of register to return a RegisterValue object for

        """
        return self.get_node( regname )

    def getregdef(self,regname):
        """Returns the register object without doing a read

        Args:
            regname : name of register to return the RegisterDefinition for

        """
        return self.definition.get_node( regname )

    def readregister(self, regname):
        """
        This will read the specified register and return the RegisterValue object

        Args:
            regname : name of register to read
        """
        return self.get_node(regname).read()

    def writeregister(self, regname, value):
        """
        This will write value to the specified

        Args:
            regname : name of register to read
            value : value to write to the register
        """
        self.get_node(regname).write(value)


    def addregister(self, register, name=None):
        """
        Adds a register to the component definition that
        goes with this component
        """
        name = register.name if name is None else name
        self.definition.add_node( register )

class RegisterValue(NamedNodeValue):
    """
    Subclass of NamedNodeValue, implemented to provide attributes
    and functions specific to Registers and backwards compatible with
    older PythonSv "corecode"

    **See Also:**
        :py:class:`~namednodes.nodes.NamedNodeValue`
    """
    type = nodes.NodeTypes.Register

    @property
    def fields(self):
        """Return a list of field names that are within this register"""
        return self.definition.fields

    @property
    def _state(self):
        """Alias from old _state variable to new 'path' variable"""
        return self.path

    @property
    def IntelRsvd(self):
        """
        shortcut for definition.IntelRsvd, see RegisterDefinition.IntelRsvd description
        of what is returned here: :py:meth:`RegisterDefinition.IntelRsvd`
        """
        return self.definition.IntelRsvd

    @property
    def description(self):
        """Description that is part of definition.info object, if it does not exists '' is returned """
        return self.definition.info.get("description","")

    @property
    def numbits(self):
        """
        Returns the number of bits in the register
        """
        numbits = self.target_info.get("numbits", None)
        if numbits is None:
            numbits = self.definition.info.get("numbits", None)
        return numbits

    @property
    def _regvalue(self):
        """
        Alias from old _regvalue variable to this register's value.
        """
        return self.value

    @property
    def _register(self):
        """
        Alias from old _register variable to this register's definition.
        """
        return self.definition

    def read(self, *args, **kwargs):
        # clear child nodes BEFORE we do read
        # this should be perhaps a general part of the node read, but I don't know how that
        # would affect AFD
        self._nodes_values = {}
        return super(RegisterValue,self).read(*args, **kwargs)

    def getfieldobject(self, fieldname):
        """Return the FieldValue object for the specified field name

        Args:
            fieldname (str) : name of field to returnt the field object for

        .. Note:: this is the same as calling :py:meth:`~NamedNodeValue.get_node`

        No read's are performed by calling this function
        """
        return self.get_node(fieldname)

    def getfielddefinition(self, fieldname):
        """
        Return the FieldDefinition object for the specified field name.

        Args:
            fieldname (str): name of field to return the field definition for.

        Note:: this is the same as calling :py:meth:`~NamedNodeValue.definition`
        """
        field_object = self.get_node(fieldname)
        return field_object.definition

    def getaddress(self):
        """
        Returns the info object that this register is currently using.
        """
        return self.definition.info

    def readfield(self,fieldname):
        """Get's a FieldValue object with the specified name and forces a read to occur

        Args:
            fieldname (str) : name of field to returnt the field object for

        Use :py:meth:`~NamedNodeValue.get_node` if you do not need the read to occur
        immediately, and you just want a FieldValue object

        """
        field = self.get_node(fieldname)
        field.read()
        return field

    def write(self, value, *args, **kwargs):
        """
        For writing a new value to the register

        Args:
            value : value to given to the register current access.write function

        See Also::
            :py:meth:`namednodes.access.NodeAccess.write`

        """
        if settings.DEBUG:
            _log.caller("RegisterValue.write")
            _log.debugall("RegisterValue.write: ENTER : '{0}', value to write: {1}".format(self.name,value))

        # handle negative values
        # not using 'isinstance' to increase performance
        if (value.__class__ == long or value.__class__ == int) and value < 0:
            mask = (1 << self.numbits) - 1
            value = mask + value + 1

        return super(RegisterValue, self).write(value, *args, **kwargs)

    def writefield(self, fieldname, value):
        """Write the specified value to the field with the given name

        Args:
            fieldname (str) : name of field to write
            value : value to write

        This is the equivalent of: get_node(name).write(value)

        See Also::
            -: py:meth:`~NamedNodeValue.get_node`
            -: py:meth:`~NamedNodeValue.write`

        """
        return self.get_node(fieldname).write(value)

    def storefield(self, fieldname, value):
        """Write the specified value to the field with the given name

        Args:
            fieldname (str) : name of field to store value for future flush operation
            value : value to store off for future flush operation

        This is the equivalent of: get_node(name).store(value)

        See Also::
            -: py:meth:`~NamedNodeValue.get_node`
            -: py:meth:`~NamedNodeValue.store`

        """
        return self.get_node(fieldname).store(value)

    def __getitem__(self, key):
        """get the value for the specified bit in this register"""
        if PY2 or type(key) is int:
            return self.__getslice__(key, key)
        else:
            return self.__getslice__(key.start, key.stop)

    def __setitem__(self, key, value):
        """set the value for the specified bit in this register"""
        if PY2 or type(key) is int:
            if value not in [0, 1]:
                raise ValueError("Invalid value for setting a single bit: %s" % value)
            return self.__setslice__(key, key, value)
        else:
            return self.__setslice__(key.start, key.stop, value)

    def __getslice__(self, i, j):
        """provides the capability for getting certain bits in the register"""
        i = long(i)
        j = long(j)

        register_length = self.numbits
        if register_length != None:
            valid_indices = xrange(register_length)
            if i not in valid_indices or j not in valid_indices:
                raise IndexError("List index out of range")

        hi = i if i > j else j
        lo = j if i > j else i
        mask = (2**(hi - lo + 1)) - 1
        data = self.get_value()
        value = (data >> lo) & mask
        return value

    def __setslice__(self, i, j, value):
        """provides the capability for setting certain bits in the register"""
        i = long(i)
        j = long(j)
        value = long(value)

        register_length = self.numbits
        if register_length != None:
            valid_indices = xrange(register_length)
            if i not in valid_indices or j not in valid_indices:
                raise IndexError("List index out of range")

        hi = i if i > j else j
        lo = j if i > j else i
        mask = (2**(hi - lo + 1)) - 1
        value = (value & mask) << lo
        mask = mask << lo
        self.read_write(mask, value)
        return None

class RegisterDefinition(NamedNodeDefinition):
    """
    Subclass of NamedNodeDefinition, implemented to provide attributes
    and functions specific to Registers and backwards compatible with
    older PythonSv "corecode"

    **See Also:**
        :py:class:`~namednodes.nodes.NamedNodeDefinition`
    """
    type = nodes.NodeTypes.Register
    value_class = RegisterValue

    def add_node(self,node):
        """Add child node, if name is given then it will be used instead of the node.name

        Args:
            child (NamedNodeDefinition) : NamedNode object

        The Register version of this function also checks child for attribute and updates
        the w1c_present in the info if a known w1c attribute is given. Child/Field can specify
        it is write-one-to-clear by having either::

            - w1c=True as one of its info key=value data
            - attribute containing 'w1c' or 'w/1c'

        """
        # we created
        attribute = node.info.get("attribute","")
        # check the info for a boolean that we are adding, because that is the
        # faster than doing the string compare
        if (node.info.get("w1c") or
                attribute.count("w/1c") or
                attribute.count("w1c")):
            # save it off as an info variable
            node.info['w1c'] = True
            self.info['w1c_present'] = True
            # update our w1c mask
            upper = node.upperbit
            lower = node.lowerbit
            mask = (( ((1<<(upper-lower+1)) -1) ) << lower)
            w1c_mask = self.info.get("w1c_mask",0)
            w1c_mask |= mask
            self.info['w1c_mask'] = w1c_mask
        # now call the parent
        super(RegisterDefinition,self).add_node(node)

    @property
    def description(self):
        """Description that is part of info object, if it does not exists '' is returned """
        return self.info.get("description","")

    @property
    def IntelRsvd(self):
        """
        Returns IntelRsvd value in the info object, if it exists in the register. If not,
        then all the child nodes are checked for the IntelRsvd

        """
        # try to get from info, if it is there, it overrides fields
        intelrsvd = self.info.get("IntelRsvd", KeyError)
        if intelrsvd is not KeyError:
            return intelrsvd
        else:
            len_nodes = len(self.nodes)
            # no child nodes AND not in info, assume it is True
            if len_nodes == 0:
                return True
            # otherwise, check our fields
            # assumes all children our fields
            count = 0
            for child in self.nodes:
                # if ANY child has True for intel rsvd, the whole reg is true
                # count how many "Falses" we have, a single True, make
                if child.info.get("IntelRsvd", True)==True:
                    return True
            # must not have found ANY fields with IntelRsvd==True
            return False

    @property
    def fields(self):
        """Return list of children node's names"""
        return self.nodenames

    def getaddress(self):
        """
        Returns the info object that this register definition is
        currently using.
        """
        return self.info


class FieldValue(NamedNodeValue):
    """
    Subclass of NamedNodeValue, implemented to provide attributes
    and functions specific to Fields and backwards compatible with
    older PythonSv "corecode"

    **See Also:**
        :py:class:`~namednodes.nodes.NamedNodeValue`
    """
    type = nodes.NodeTypes.Field

    @property
    def stored_value(self):
        """retrieves the stored value from the parent object"""
        if self.parent.stored_value is None:
            return 0
        # determine our mask
        upper = self.upperbit
        lower = self.lowerbit
        field_mask = ((1<<(upper-lower+1)) -1)
        # and out our value, shift it according to our lower bit
        # and return it
        return ((self.parent.stored_value>>lower) & field_mask)

    @stored_value.setter
    def stored_value(self, value):
        """retrieves the stored value from the parent object"""
        # this is just namedndoes clearing out old stored values
        if value is None and self.parent.stored_value is None:
            return
        # otherwise, go through typical store flow
        self.store(value)

    @property
    def stored_mask(self):
        """calculates the current field mask we are contributing to
        for the register"""
        if self.parent.stored_mask is None:
            return 0
        upper = self.upperbit
        lower = self.lowerbit
        field_mask = ((1 << (upper - lower + 1)) - 1)
        return ((self.parent.stored_mask>> lower) & field_mask)

    @stored_mask.setter
    def stored_mask(self, value):
        """retrieves the stored value from the parent object"""
        # this is just namedndoes clearing out old stored values
        if value is None and self.parent.stored_mask is None:
            return
        else:
            raise ValueError("directly setting the stored mask is not supported on fields")

    @property
    def IntelRsvd(self):
        """
        shortcut for definition.IntelRsvd. See :py:meth:`FieldDefinition.IntelRsvd`
        for what is returned here
        """
        return self.definition.IntelRsvd

    @property
    def attribute(self):
        """
        shortcut for definition.attribute. See :py:meth:`FieldDefinition.attribute`
        for what is returned here
        """
        return self.definition.attribute

    @property
    def description(self):
        """
        shortcut for definition.description. See :py:meth:`FieldDefinition.description`
        for what is returned here
        """
        return self.definition.description

    @property
    def upperbit(self):
        """
        shortcut for definition.upperbit. See :py:meth:`FieldDefinition.upperbit`
        for what is returned here
        """
        upperbit = self.definition.info.get("upperbit", None)
        if upperbit is None:
            # be sure to use property for lowerbit, be sure to NOT use
            # property for numbits
            return self.lowerbit + self.numbits - 1
        else:
            return upperbit

    @property
    def numbits(self):
        """
        shortcut for definition.info['numbits']
        """
        upperbit = self.definition.info.get("upperbit", None)
        if upperbit is None:
            # Get numbits from target info
            numbits = self.target_info.get("numbits", None)
            # Get numbits from definition info
            if numbits is None:
                numbits = self.definition.info.get("numbits", None)
            return numbits
        else:
            return upperbit-self.lowerbit+1

    @property
    def lowerbit(self):
        """
        shortcut for definition.lowerbit. See :py:meth:`FieldDefinition.lowerbit`
        for what is returned here
        """
        # Get lowerbit from target info
        lowerbit = self.target_info.get("lowerbit", None)

        # Get lowerbit from definition info
        if lowerbit is None:
            lowerbit = self.definition.lowerbit

        return lowerbit

    @property
    def default(self):
        """
        shortcut for definition.default. See :py:meth:`FieldDefinition.default`
        for what is returned here
        """
        return self.definition.default

    @property
    def _state(self):
        """Alias from old _state variable to new 'path' variable"""
        return self.path

    @property
    def _field(self):
        """
        Alias from old _field variable to this field's definition.
        """
        return self.definition

    @property
    def _register(self):
        """
        Alias from old _register variable to this field's parent.
        """
        return self.parent

    @property
    def _fieldvalue(self):
        """
        Alias from old _fieldvalue variable to this field's value.
        """
        return self.value

    def __getitem__(self, key):
        """get the value for the specified bit in this field"""
        if PY2 or type(key) is int:
            return self.__getslice__(key, key)
        else:
            return self.__getslice__(key.start, key.stop)

    def __setitem__(self, key, value):
        """set the value for the specified bit in this field"""
        if PY2 or type(key) is int:
            if value not in [0, 1]:
                raise ValueError("Invalid value for setting a single bit: %s" % value)
            return self.__setslice__(key, key, value)
        else:
            return self.__setslice__(key.start, key.stop, value)

    def __getslice__(self, i, j):
        """provides the capability for getting certain bits in the field"""
        i = long(i)
        j = long(j)

        valid_indices = xrange(0, self.numbits)
        if i not in valid_indices or j not in valid_indices:
            raise IndexError("List index out of range")

        hi = i if i > j else j
        lo = j if i > j else i
        mask = (2**(hi - lo + 1)) - 1
        data = self.get_value()
        value = (data >> lo) & mask
        return value

    def __setslice__(self, i, j, value):
        """provides the capability for setting certain bits in the field"""
        i = long(i)
        j = long(j)
        value = long(value)

        valid_indices = xrange(0, self.numbits)
        if i not in valid_indices or j not in valid_indices:
            raise IndexError("List index out of range")

        hi = i if i > j else j
        lo = j if i > j else i
        mask = (2**(hi - lo + 1)) - 1
        value = (value & mask) << lo
        mask = mask << lo
        self.read_write(mask, value)
        return None


class FieldDefinition(NamedNodeDefinition):
    """
    Subclass of NamedNodeDefinition, implemented to provide attributes
    and functions specific to Fields  and backwards compatible with
    older PythonSv "corecode"

    **See Also:**
        :py:class:`~namednodes.nodes.NamedNodeDefinition`
    """
    type = nodes.NodeTypes.Field
    value_class = FieldValue

    def __init__(self,*args,**kwargs):
        NamedNodeDefinition.__init__(self,*args,**kwargs)
        # fields always have the field access method
        self.add_access("AccessField", "AccessField", True)
        # check for upper/lowerbit?
        # some parser are adding upper/lowerbit, later, need to update those before
        # putting in this check:
        #assert 'upperbit' in self.info, "FieldDefinitions must have upperbit and lowerbit"
        #assert 'lowerbit' in self.info,  "FieldDefinitions must have upperbit and lowerbit"

    @property
    def attribute(self):
        """Attribute information provided by design. If it does not exist, '' is returned"""
        return self.info.get("attribute",'')

    @property
    def description(self):
        """Field Description provided by design. If it does not exist, '' is returned"""
        return self.info.get("description",'')

    @property
    def default(self):
        """Default value provided by design. If it does not exist, 0 is returned"""
        return self.info.get("default",0)

    @property
    def upperbit(self):
        """returns value from *self.info.upperbit* """
        upperbit = self.info.get("upperbit", None)
        if upperbit is None:
            return self.info.get("lowerbit") + self.info.get("numbits") - 1
        else:
            return upperbit


    @property
    def lowerbit(self):
        """returns value from *self.info.lowerbit* """
        return self.info.get("lowerbit")

    @property
    def IntelRsvd(self):
        """return IntelRsvd value provided by design. If it does not exist, '' is returned"""
        return self.info.get("IntelRsvd",True)

