
# INTEL CONFIDENTIAL
# Copyright 2017 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



"""Queuing module for queing multiple operations before passing them to the probe."""


from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from .utils._py2to3 import *
from namednodes import settings
from namednodes.events import queuing as queuing_events
from namednodes.events.queuing import QueuingEventTypes

if PY2:
    import cPickle as pickle
else:
    import pickle


_QUEUE_CLASSES = {} # populated at the end of the module
_BUNDLE_QUEUE_EXT = ".ipcqueue"


class QueueBase(object):
    """Base class for queuing."""

    type = None
    sub_type = None

    def __init__(self, *args, **kwargs):
        if settings.EVENTS_ON_QUEUING:
            event_data = {
                "type": self.type,
                "sub_type": self.sub_type,
                "args": list(args),
                "kwargs": dict(kwargs)
                }
            queuing_events.send_event(QueuingEventTypes.create(event_data))

    def __enter__(self):
        if settings.EVENTS_ON_QUEUING:
            event_data = {
                "type": self.type,
                "sub_type": self.sub_type,
                }
            queuing_events.send_event(QueuingEventTypes.enter(event_data))

    def __exit__(self, exc_type, exc_value, traceback):
        if settings.EVENTS_ON_QUEUING:
            event_data = {
                "type": self.type,
                "sub_type": self.sub_type,
                "exc_type": exc_type,
                "exc_value": exc_value,
                "traceback": traceback
                }
            queuing_events.send_event(QueuingEventTypes.exit(event_data))


class StubQueue(QueueBase):
    def __init__(self, *args, **kwargs):
        super(StubQueue, self).__init__(*args, **kwargs)
        return

    def __enter__(self):
        super(StubQueue, self).__enter__()
        return

    def __exit__(self, exc_type, exc_value, traceback):
        super(StubQueue, self).__exit__(exc_type, exc_value, traceback)
        return


class StubImmediateQueue(StubQueue):
    type = "immediate"
    sub_type = "offline"


class DeviceLockerQueue(QueueBase):
    type = "immediate"
    sub_type = "device_locker"

    def __init__(self, device=None, queue=None):
        super(DeviceLockerQueue, self).__init__(device=device, queue=queue)
        import ipccli
        ipc = ipccli.baseaccess()
        self.locker = ipc.device_locker(device, queue)

    def __enter__(self):
        super(DeviceLockerQueue, self).__enter__()
        self.locker.__enter__()

    def __exit__(self, exc_type, exc_value, traceback):
        super(DeviceLockerQueue, self).__exit__(exc_type, exc_value, traceback)
        self.locker.__exit__(exc_type, exc_value, traceback)


class BundleSaveQueue(QueueBase):
    type = "save"
    sub_type = "bundle"

    def __init__(self, filename):
        super(BundleSaveQueue, self).__init__(filename=filename)
        from svtools.probe_bundles.bundle import Bundle, bundle_cli_override_create
        self.bundle = Bundle()
        self.operations = []
        self.override = bundle_cli_override_create(self.bundle, self.operations)
        if filename.lower().endswith(_BUNDLE_QUEUE_EXT):
            self.filename = filename
        else:
            self.filename = filename + _BUNDLE_QUEUE_EXT

    def __enter__(self):
        super(BundleSaveQueue, self).__enter__()
        self.override.__enter__()

    def __exit__(self, exc_type, exc_value, traceback):
        super(BundleSaveQueue, self).__exit__(exc_type, exc_value, traceback)
        self.override.__exit__(exc_type, exc_value, traceback)
        bundle_serialized = bytearray(self.bundle.SerializeBundle())
        with open(self.filename, "wb") as outfile:
            pickle.dump((bundle_serialized, self.operations), outfile, pickle.HIGHEST_PROTOCOL)


class StubLoadQueue(StubQueue):
    type = "load"
    sub_type = "offline"


class BundleLoadQueue(QueueBase):
    type = "load"
    sub_type = "bundle"

    def __init__(self, filename, deviceid):
        super(BundleLoadQueue, self).__init__(filename=filename, deviceid=deviceid)
        from svtools.probe_bundles.bundle import execute, bundle_cli_override_read
        if not filename.lower().endswith(_BUNDLE_QUEUE_EXT):
            filename = filename + _BUNDLE_QUEUE_EXT
        with open(filename, "rb") as infile:
            bundle_serialized, operations = pickle.load(infile)
        bundle_data = execute(deviceid, bundle_serialized)
        self.override = bundle_cli_override_read(bundle_data, operations)

    def __enter__(self):
        super(BundleLoadQueue, self).__enter__()
        self.override.__enter__()

    def __exit__(self, exc_type, exc_value, traceback):
        super(BundleLoadQueue, self).__exit__(exc_type, exc_value, traceback)
        self.override.__exit__(exc_type, exc_value, traceback)


def _find_queue_class(queue_type, sub_type):
    queue_class = _QUEUE_CLASSES.get((queue_type, sub_type), None)
    if queue_class is None:
        # look for a default class
        queue_class = _QUEUE_CLASSES.get((queue_type, "default"), None)
        if queue_class is None:
            raise ValueError("unsupported queue type:%s sub_type:%s" % (queue_type, sub_type))
    return queue_class


def queue(queue_type, sub_type=None, *args, **kwargs):
    if sub_type is None:
        try:
            from svtools.common import baseaccess
        except ImportError:
            pass
        else:
            access = baseaccess.getaccess()
            base = baseaccess.getglobalbase()
            environment = getattr(base, "environment", None)
            if environment == "ipc":
                if queue_type == "immediate":
                    sub_type = "device_locker"
                elif queue_type in  ["save", "load"]:
                    sub_type = "bundle"
    queue_class = _find_queue_class(queue_type, sub_type)
    return queue_class(*args, **kwargs)


_QUEUE_CLASSES = {("immediate", "default") : StubImmediateQueue,
                  ("immediate", "offline") : StubImmediateQueue,
                  ("immediate", "device_locker") : DeviceLockerQueue,
                  ("save", "bundle") : BundleSaveQueue,
                  ("load", "offline") : StubLoadQueue,
                  ("load", "bundle") : BundleLoadQueue}

