# INTEL CONFIDENTIAL
# Copyright 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.

from ..precondition import ComponentPreconditionGen2 as _ComponentPrecondition
from ..precondition import SystemPreconditionGen2 as _SystemPreconditionGen2
from ..precondition import MultipleAccesses as _MultipleAccesses
try:
    from svtools.common import baseaccess
except ImportError:
    baseaccess = None
import ipccli


class IpcComponentDeviceLocker(_ComponentPrecondition):
    """
    This precondition works for Nodes or Components being passed in to the pre-condition
    """
    requires = ['did']

    @classmethod
    def precondition(cls, node_or_component):
        """Using the component's did, this will create and enter a device_locker"""
        component = getattr(node_or_component, "component", node_or_component)
        # CANT use self until we decide whether to change preconditions to not be singletons
        ipc = ipccli.baseaccess()
        did = component.target_info.get("did", None)
        if did == None:
            raise RuntimeError("did needed to use with this device_locker precondition")
        locker = ipc.device_locker(device=did)
        locker.__enter__()
        cls.put_info(component, "locker", locker)

    @classmethod
    def postcondition(cls, node_or_component):
        """exits the device_locker entered by the precondition"""
        component = getattr(node_or_component, "component", node_or_component)
        locker = cls.get_info(component, "locker")
        locker.__exit__(None, None, None)


class IpcSystemDeviceLocker(_SystemPreconditionGen2):
    """
    * This precondition gets called once per discovery item and will do a full device_locker with no DID
    * it will check to make sure the access is "ipc" before running the device locker
    """

    @classmethod
    def precondition(cls, node_or_component):
        """Using the component's did, this will create and enter a device_locker"""
        if baseaccess is None or baseaccess.getaccess() != "ipc":
            return
        ipc = ipccli.baseaccess()
        locker = ipc.device_locker()
        locker.__enter__()
        cls.put_info(None, "locker", locker)

    @classmethod
    def postcondition(cls, node_or_component):
        """exits the device_locker entered by the precondition"""
        # precondition can be None if we weren't inside multiple access
        locker = cls.get_info(None, "locker", None)
        if locker is not None:
            locker.__exit__(None, None, None)


class IpcSystemDeviceLockerMultipleOnly(IpcSystemDeviceLocker):
    """
    * This precondition gets called once per discovery item and will do a full device_locker with no DID
    * it will check to make sure the access is "ipc" before running the device locker
    * it will ONLY run inside a MultipleAccess context statement
    """

    @classmethod
    def precondition(cls, node_or_component):
        """Using the component's did, this will create and enter a device_locker"""
        if not _MultipleAccesses.in_use:
            return
        super().precondition(node_or_component)


class IpcHaltRequired(_ComponentPrecondition):
    """
    This precondition only works for Nodes being passed in to the pre-condition
    """
    requires = [('did','msr_did')]

    @classmethod
    def precondition(cls, node_or_component):
        """Using the component's msr_did or did, this will attempt to halt that device if it is running"""
        ipc = ipccli.baseaccess()
        did = cls.lookup_info(node_or_component, ("msr_did", "did"))
        isrunning = ipc.cmds.isrunning(did)
        cls.put_info(node_or_component, "isrunning", isrunning)
        if isrunning:
            ipc.cmds.halt(device=did)

    @classmethod
    def postcondition(cls, node_or_component):
        """Using the component's msr_did or did, this will restore the runstate found by the precondition"""
        ipc = ipccli.baseaccess()
        did = cls.lookup_info(node_or_component, ("msr_did", "did"))
        was_running = cls.get_info(node_or_component, "isrunning")
        if was_running:
            ipc.cmds.go(device=did)

class BundleCreate(_ComponentPrecondition):
    """
    This precondition class sets up (and tears down) storage for saving the current Extended PortId in use (used to save overhead on repeated access using the same Extended PortId)
    """

    @classmethod
    def precondition(cls, node_or_component):
        """Set up storage for the current Extended PortId"""
        from svtools.probe_bundles import cli_override
        override = cli_override.bundle_cli_override_create_then_execute()
        cls.put_info(node_or_component, "cli_override", override)
        override.__enter__()

    @classmethod
    def postcondition(cls, node_or_component):
        """Tear down storage for the current Extended PortId, and restore the saved_extport if applicable"""
        override = cls.get_info(node_or_component, "cli_override")
        override.__exit__(None, None, None)
