# INTEL CONFIDENTIAL
# Copyright 2019 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.

from ..precondition import ComponentPreconditionGen2 as _ComponentPrecondition
from ..precondition import RunWhenNeededPrecondition as _RunWhenNeededPrecondition

import weakref


class LookupInfoCache(_ComponentPrecondition):
    """
    This precondition can get used to speed-up lookup_info and get_access_info calls
    by caching any component related information during a MultipleAccess window.

    For this precondition to have the most impact:

    * most of the access information should come from the component
    * the get_access_info should be used instead of multiple lookup_info calls

    """

    class NodeOnly(object):
        pass

    # key to store value for whether cache is available for use
    # in component's precondition info
    CACHE_AVAILABLE_KEY = "cache_available"


    @classmethod
    def cache_available(cls, component):
        """
        Determine if cache for component is ready to be used.

        Arguments:
          - component : The component with the cache.

        Returns:
          True if component's cache data is available.
        """
        if cls not in component.target_info.get('preconditions', {}):
            return False
        return cls.get_info(component, cls.CACHE_AVAILABLE_KEY, default=False)


    @classmethod
    def get_all_info(cls, component):
        """
        Get all cached properties for component.

        Arguments:
          - component : The component with the cache.

        Returns: A dictionary of the cached properties.
        """
        # should this return a copy?
        return component.target_info.get('preconditions', {}).get(cls, {})


    @classmethod
    def precondition(cls, node_or_component):
        if not cls.cache_available(node_or_component.component):
            component_info = \
                node_or_component.access_class.get_access_info(node_or_component,
                                                               component_only=True)
            for keys, value in component_info.items():
                if value is not LookupError:
                    cls.put_info(node_or_component, keys, value)
                else:
                    cls.put_info(node_or_component, keys, cls.NodeOnly)
            # component's cache is ready to be used
            cls.put_info(node_or_component, cls.CACHE_AVAILABLE_KEY, True)
        # else: - cache has been filled already


    @classmethod
    def postcondition(cls, node_or_component):
        # we are done, no need to explicitely set component's cache available
        # flag to False due to component's cache is removed by parent class.
        # cls.put_info(node_or_component, cls.CACHE_AVAILABLE_KEY, False)
        pass


# this will run on every single node, but uses (if needed) a system wide cache
# this should NOT be used as the standard precondition flow, it makes
# direct use of runprecondtion / runpostcondition which is atypical

class GetAccessInfoCache(_RunWhenNeededPrecondition):
    save_weakrefs = True
    node_cache = []

    @classmethod
    def precondition(cls, node_or_component):
        if '_get_access_info' not in node_or_component.target_info:
            # assumed to be a node...
            node_or_component.target_info['_get_access_info'] = \
                node_or_component.access_class.get_access_info(node_or_component)
            cls.node_cache.append(weakref.ref(node_or_component))

    @classmethod
    def postcondition(cls, node_or_component):
        # we are done, no need to explicitely set component's cache available
        # flag to False due to component's cache is removed by parent class.
        # cls.put_info(node_or_component, cls.CACHE_AVAILABLE_KEY, False)
        for node in cls.node_cache:
            n = node()
            if n is not None:
                del n.target_info['_get_access_info']
        cls.node_cache = []
