
# INTEL CONFIDENTIAL
# Copyright 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.

"""Namednodes's precondition module

This moudle contains multiple types of precondition abstract base classes. This
is implemented such that any class that inherits from these classes will be a
singleton. To make this work the __new__ method must not be overriden.  A
precondition is designed to run a section of code(the precondition) before
something and another section of code(the postcondition) after something.
Nesting is handled such that precondition code is only ran once at the high
level call and any nested calls will be no ops. For nested postcondition code,
it is similar.

SystemPrecondition        - global precondition to be ran regardless of the component
DiscoveryItemPrecondition - to be used with discovery items for a precondition
                            that must be ran per discovey item
ComponentPrecondition     - to be used with standard components for a precondition
                            that must be ran per component
                            
It is expected you will inherit from one of the base classes above and
implement the precondition and postcondition methods.

Example Usage:

class MyPrecondition(ComponentPrecondition):
    def precondition(self, component):
        # do something that needs to be done before doing something with the componet
    def postcondition(self, component):
        # do something that needs to be done after doing something with the componet

def do_something(component, some_arg):        
    my_precondition = MyPrecondition()
    my_precondition.run_precondition(component)
    # do something with the component like read or write
    my_precondition.run_postcondition(component)

# example with nesting
my_precondition = MyPrecondition()
my_precondition.run_precondition(component)
for i in range(10):
    do_something(component, i) # since it is nested the precondition/postcondition code will not be ran
my_precondition.run_postcondition(component)

"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from abc import ABCMeta, abstractmethod

from six import add_metaclass

# make Gen2 of the code available for import from this same file
# do not remove these....
from ._precondition_gen2 import *

@add_metaclass(ABCMeta)
class _Precondition(object):
    _instance = None
    def __new__(cls, *args, **kwargs):
        if cls._instance is None:
            cls._instance = object.__new__(cls, *args, **kwargs)
        return cls._instance
    @abstractmethod
    def _aquire_lock(self, component):
        raise NotImplementedError
    @abstractmethod
    def _release_lock(self, component):
        raise NotImplementedError
    @abstractmethod
    def precondition(self, component):
        """This abstract method should contain the actual precondition code."""
        raise NotImplementedError
    @abstractmethod
    def postcondition(self, component):
        """This abstract method should contain the actual postcondition code.
        This will run during the last"""
        raise NotImplementedError
    def run_precondition(self, component):
        """This method is to be always called before access code is ran. This
        in turn will call the precondition method only if run_precondition 
        hasn't been previously called without a run_postcondition also being 
        called previously."""
        if self._aquire_lock(component):
            try:
                self.precondition(component)
            except:
                self._release_lock(component)
                raise
    def run_postcondition(self, component):
        """This method is to be always called after access code is ran. This in
        turn will call the postcondition method only if there are no
        outstanding calls to run_postcondition that there hasn't been a call to
        run_postcondition for."""
        if self._release_lock(component):
            try:
                self.postcondition(component)
            except:
                self._aquire_lock(component)
                raise

class SystemPrecondition(_Precondition):
    """Abstract base class of a system level precondition. The precondition and
    postcondition methods must be implemented.
    """
    _lock = 0
    def _aquire_lock(self, component):
        type(self)._lock += 1
        return type(self)._lock == 1
    def _release_lock(self, component):
        if type(self)._lock == 0:
            raise RuntimeError("already unlocked")
        type(self)._lock -= 1
        return type(self)._lock == 0
    
class DiscoveryItemPrecondition(_Precondition):
    """Abstract base class of a discovery item level precondition. The
    precondition and postcondition methods must be implemented.
    """
    def _aquire_lock(self, component):
        name = "precondition_%s_%s" % (self.__class__, id(self))
        target_info = component.origin.target_info
        if name not in target_info:
            target_info[name] = 0
        target_info[name] += 1
        return target_info[name] == 1
    def _release_lock(self, component):
        name = "precondition_%s_%s" % (self.__class__, id(self))  
        target_info = component.origin.target_info      
        if name not in target_info or target_info[name] == 0:
            raise RuntimeError("already unlocked")
        target_info[name] -= 1
        return target_info[name] == 0
    
class ComponentPrecondition(_Precondition):
    """Abstract base class of a component level precondition. The
    precondition and postcondition methods must be implemented.
    """
    def _aquire_lock(self, component):
        name = "precondition_%s_%s" % (self.__class__, id(self))
        target_info = component.target_info
        if name not in target_info:
            target_info[name] = 0
        target_info[name] += 1
        return target_info[name] == 1
    def _release_lock(self, component):
        name = "precondition_%s_%s" % (self.__class__, id(self))   
        target_info = component.target_info     
        if name not in target_info or target_info[name] == 0:
            raise RuntimeError("already unlocked")
        target_info[name] -= 1
        return target_info[name] == 0


