
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.


'''
expression_parser.py
TDEF expression parser (in Python)

Authors: Bill Hodges, Jason Kirschenbaum, Brendan Long

This module holds the parsing engine for TDEF expressions. It includes a BNF of our best knowledge of the grammer for
a TDef expression. It also includes a simple function to parse a string into an AST.
'''

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

# Jason: We really need a parser to handle building up a formula, otherwise it's going to be hard to handle arbitrary things
# using the py2ipc parsing to limit our dependencies
from pyparsing import Literal, Word,Combine,Group,Optional, Forward, nums, alphanums, hexnums, delimitedList, FollowedBy, OneOrMore

parseTally = {}
def updTally(v,i):
    try:
        parseTally[v][i] += 1
    except KeyError:
        parseTally[v] = [0, 0, 0]
        parseTally[v][i] = 1
def tallyTry(n):
    def tally(*args):
        updTally(n,0)
    return tally
def tallyFail(n):
    def tally(*args):
        updTally(n,2)
    return tally
def tallyMatch(n):
    def tally(*args):
        updTally(n,1)
    return tally

grammar_symbols = ""

# Parens
lparen = Literal("(").suppress()
rparen = Literal(")").suppress()
comma = Literal(",").suppress()
lcurly = Literal("{").suppress()
rcurly = Literal("}").suppress()
colon = Literal(":")
lbracket = Literal("[").suppress()
rbracket = Literal("]").suppress()
binary_b = Literal("b").suppress()

grammar_symbols = grammar_symbols + " lparen rparen comma lcurly rcurly colon lbracket rbracket binary_b "

# atomic terms
hex_number = Combine(Literal("0x") + Word(hexnums))# Combine(Word("0",max=1) + Word("x",max=1) + Word(hexnums))
bin_number = Combine(Literal("0y") + Word('01')) #Combine(Word("0",max=1) + Word("y",max=1) + Word('01'))
dec_number = Word(nums)
#bit_data_number = Combine(Literal("[") + dec_number + Literal("b") + Literal("]") + hex_number)
bit_data_number = (lbracket + dec_number + binary_b + rbracket + (hex_number | dec_number)).setParseAction(lambda s, l, t : ["BD", [t[0]], [t[1]]])
number = (FollowedBy("0") + (hex_number | bin_number)) | bit_data_number | dec_number
ident = Word('$',alphanums + "_") #+ ~ FollowedBy("(")
ident_function = Word('$',alphanums + "_")
field_name = Word(alphanums + "_")

grammar_symbols = grammar_symbols + " hex_number bin_number dec_number bit_data_number number ident ident_function field_name "

# Random helpers:
dec_num_list = delimitedList(dec_number)
grammar_symbols = grammar_symbols + " dec_num_list "

# Functions:
OR = Literal("OR") + FollowedBy("(")
XOR = Literal("XOR") + FollowedBy("(")
SHL = Literal("SHL") + FollowedBy("(")
SELV = Literal("SELV") + FollowedBy("(")
JOINV = Literal("JOINV") + FollowedBy("(")
DEP = Literal("DEP") + FollowedBy("(")
AND = Literal("AND") + FollowedBy("(")
SHR = Literal("SHR") + FollowedBy("(")
ADD = Literal("ADD") + FollowedBy("(")
SUB = Literal("SUB") + FollowedBy("(")
MUL = Literal("MUL") + FollowedBy("(")
DIV = Literal("DIV") + FollowedBy("(")
MOD = Literal("MOD") + FollowedBy("(")
CNT = Literal("CNT") + FollowedBy("(")
INV = Literal("INV") + FollowedBy("(")
XTR = Literal("XTR") + FollowedBy("(")
REV = Literal("REV") + FollowedBy("(")

DOT = Literal(".")

grammar_symbols = grammar_symbols + " OR XOR SHL SELV DEP AND SHR ADD SUB MUL DIV MOD CNT INV DOT JOINV REV"

COMM_OP = OR | AND | MUL | ADD | JOINV | SUB
BIN_OP = SHL | SHR | DIV | MOD | XOR
U_OP = CNT | INV

grammar_symbols = grammar_symbols + " COMM_OP BIN_OP U_OP "

expr = Forward()

expr_list = delimitedList(expr)

com_expr = COMM_OP + lparen + expr_list + rparen

bin_exp = BIN_OP + lparen + expr + comma + expr + rparen

index_list_exp = (expr + colon + expr).setParseAction(lambda s, l, t : [t[1], t[0], t[2]])
#index_list_exp = (number + colon + number).setParseAction(lambda s, l, t : [t[1], [t[0]], [t[2]]])

list_exp = Group(lcurly + (index_list_exp | expr_list) + rcurly) # Need to try an '|' here.

sel_exp = SELV + lparen + expr + comma + list_exp + rparen

dep_exp = DEP + lparen + expr + comma + list_exp + comma + expr + rparen

u_exp = U_OP + lparen + expr + rparen

xtr_exp = XTR + lparen + expr + comma + list_exp + rparen

rev_expr = REV + lparen + expr + rparen

dot_list = Group(DOT + delimitedList(field_name,DOT))

ident_exp_list = delimitedList(expr).setParseAction(lambda s, l, t : [':', t[0], t[0]])

#ident_op_slice = ident_function + Optional(dot_list) + Group(lparen + (index_list_exp ^ ident_exp_list) + rparen) # Need to test | here
#ident_op_field_only = ident_function + dot_list
#ident_op = ident_op_slice ^ ident_op_field_only # Need to refactor to reduce the ^ usage
#ident_op = ident_function + ((FollowedBy(DOT) + dot_list) | (Optional(dot_list) + Group(lparen + (index_list_exp ) + rparen))) # | ident_exp_list
lame_ness = Group(lparen + (index_list_exp | ident_exp_list) + rparen)
ident_op = ident_function + (lame_ness | (dot_list + Optional(lame_ness)))

expr <<= Group((FollowedBy("$") + (ident_op | ident)) | number | com_expr | bin_exp | sel_exp | dep_exp | u_exp | xtr_exp | rev_expr)

grammar_symbols = grammar_symbols + "lame_ness expr_list com_expr bin_exp index_list_exp list_exp sel_exp dep_exp u_exp xtr_exp dot_list ident_exp_list ident_op expr rev_expr"

#for g in grammar_symbols.split():
#    try:
#        vars()[g].setName(g).setDebugActions(tallyTry(g),tallyMatch(g),tallyFail(g))
#    except Exception:
#        print g

def debug_stats():
    key_list = sorted(parseTally.keys())
    for symbol in key_list:
        info = parseTally[symbol]
        print("[", symbol, "] Tries:", info[0], ", Matches: ", info[1], ", Fails: ", info[2], ", Percent:", float(info[1])/float(info[0]))

def parse_expression(expression):
    """Parses an expression; returns the resulting Abstract-ish Syntax Tree. """
    try:
        tokens = expr.parseString(expression)
    except Exception:
        #print "Failed on parsing", expression
        raise

    return tokens.asList()[0] # Jason: Let's listify now; we drop the extraneous listification that is required by the grammer

