# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.


# this is not in the generated docs and is more for any developer that happens
# to be ramping on our tdef support:
"""
TDEF Support
=============
Typical TDEF's have one or more State nodes. Since loading a single file does not reference
other files, a new component is created for each TDEF XML that is parsed. If multiple XML files
need to be coalesced in to a single component, then the discovery code will help manage that.

The TDEF's are supported via the standard namednodes "GetComponentFromFile". However, there are many
cases where calling this loader directly may be desired so that finer control can be had
over the creation process.

Since the TDEFs have procedural information, the procedures are turned in to python code. By
default, the python code is generated and the access functions are created an inserted in
to an in memory module::

    access_modname = "namednodes.plugins.tdef.accesses_"+comp.name.

However, the parsing function will take a filename and will write the
accesses to the file at the specified location. If a filename is specified, a fake module
name must be given as well. At that point, it is also recommended to pass in whether to overwrite
or to append to that file if it exists.

"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import string
import random
import re
import sys
import time
import copy
from namednodes import settings
from ..utils.ordereddict import odict
from ..comp import ComponentLoaderPlugin
from ..comp import CreateComponentDefinition
from ..utils._py2to3 import *
from ..utils.bitmap import BitMap, BitMapEntry, BitMapConstantEntry
from .tdef_expression_parser import parse_expression
from .. import nodes
import types

from six import PY2, PY3
if PY2:
    import imp
    from collections import MutableMapping
    from cStringIO import StringIO
if PY3:
    import importlib as imp
    from importlib.machinery import SourceFileLoader
    from collections.abc import MutableMapping
    from io import StringIO
from collections import namedtuple

# need to import here to not break compatibility
from .tdef_accesses import TdefAccessParams

from ..utils.ordereddict import _dict_update
from ..logging import getLogger
from ipccli.bitdata import BitData

_LOG = getLogger()

import xml.etree.cElementTree as ET


class TdefError(Exception):
    pass


class FirstParamType:
    not_added = 0  # don't add a special first paramter
    device = 1  # first parameter should be device object
    node = 2  # first parameter should be node object


TdefCommand = namedtuple("TdefCommand", "xml_tag function_name first_param")


class TdefLoader(ComponentLoaderPlugin):
    file_ext = "xml"

    field_name_re = re.compile(r"(\w+)(?:\((\w+)\)*)?(?:\:(\w+)\))?")
    node_re = re.compile(r"_node\('(.*)'\)")
    bitval_re = re.compile(r"\[\d+b\](?:0x)?(\w+)")  # may or may not have 0x infront of it

    # for tdef..not so much we can make it more readable pretty easily
    _operation_translation = {
        "XOR": "operator.xor",
        "OR": "operator.or_",
        "INV": "operator.invert",
        "AND": "operator.and_",
        "ADD": "operator.add",
        "SUB": "operator.sub",
        "MUL": "operator.mul",
        "SHL": "operator.lshift",
        "SHR": "operator.rshift",
        "MOD": "operator.mod",
        "DIV": "operator.floordiv",
    }

    _max_operation_depth = 100

    def __init__(self, filepath, **kwargs):
        self.filepath = filepath
        self.additional_access_code = kwargs.get("additional_access_code", None)
        # used in case filepath was a file object
        self._seekto = 0
        # add reference to the loader so it can be retreived
        self._access_header = _access_header
        # to track known commands/xml tags
        self._known_commands = {}
        self._add_default_commands()

    @classmethod
    def create(cls, filepath, **kwargs):
        """
        May do additional checking the filepath and confirms that it supports
        the file by returning a instnace of the loader
        """
        loader = cls(filepath, **kwargs)
        if loader._test_for_tdef():
            return loader
        else:
            return None

    def _add_default_commands(self):
        """
        to provide default commands that most tdef's need for scan
        These commands are included with the default the access code
        """
        self.support_command("RegBus.SetRegBus", first_param="not_added")
        self.support_command("RegBus.Write")
        self.support_command("RegBus.Read")
        self.support_command("StatePort.Write")
        self.support_command("StatePort.Read")
        self.support_command("Jtag.Shift")
        self.support_command("Jtag.GoToState")
        self.support_command("LDatSetup", first_param="not_added")
        self.support_command("LDatRead")
        self.support_command("LDatWrite")
        self.support_command("LDatIn")
        self.support_command("LDatOut")
        self.support_command("PreScan.MakeTapAccessible")
        self.support_command("State.PopContextType")

    def support_command(self, xml_tag, function_name=None, first_param='device'):
        """
        Args:
            xml_tag : xml element to parse out
            function_name : name of function to call for this tag
                            (default is to assume the same as xml_tag)
            first_param : is it device/node/not_added
        """
        if isinstance(first_param, basestring):
            first_param = getattr(FirstParamType, first_param)
        # use xml_tag if function_name is not given
        function_name = function_name or xml_tag
        cmd = TdefCommand(xml_tag, function_name, first_param)
        self._known_commands[xml_tag] = cmd

    def _element_template_substitute(self, element, match, replacement):
        for attribute, value in element.attrib.items():
            new_value = value.replace(match, replacement)
            element.attrib[attribute] = new_value
        for child in element.getchildren():
            self._element_template_substitute(child, match, replacement)

    def _get_expanded_loop_element(self, loop_element):
        new_elements = []
        var = "$" + loop_element.attrib["Var"]
        indices = loop_element.attrib["Indices"]
        end, start = indices.split(":")
        for index in range(int(start), int(end) + 1):
            for child in loop_element:
                new_child = copy.deepcopy(child)
                self._element_template_substitute(new_child, var, str(index))
                new_elements.append(new_child)
        return new_elements

    def _expand_loop_elements(self, parent, replacements={}):
        for index, child in enumerate(parent.getchildren()):
            if child.tag == "Loop":
                parent.remove(child)
                new_elements = self._get_expanded_loop_element(child)
                for element in reversed(new_elements):
                    parent.insert(index, element)
            else:
                self._expand_loop_elements(child)

    def parse(self, access_file=None, access_modname=None, overwrite=True):
        """Entry point to parsing a TDEF XML

        Args:
            access_file (str) : name of file to create in the same directory as
                                the XML file that will hold the access code
            access_modname (str) : since the file may or may not be in the python path,
                                   this is used to load the file directly in to some module.
            overwrite (bool) : whether to overwrite any existing access_file

        Returns:
            Returns a component definition that has the following info variables filled in:
                - access_file
                - access_modname

        Note:
            the recommended access_modename is something along the lines of
            namednodes.plugins.tdef.accesses.*

        """
        if settings.DEBUG:
            _LOG.debug("TdefLoader.filepath = %s" % self.filepath)
            _LOG.debug("TdefLoader.parse( access_file={0}, access_modname={1}, ovewrite={2} )". \
                       format(access_file, access_modname, overwrite))

        access_filename = None  # default if no access file is given

        # we got to _parse since that one handle recursion...
        if isinstance(self.filepath, basestring):
            # close if when finished if we opened file...
            close_db_when_done = True
            file = open(self.filepath, "rb")
        else:
            close_db_when_done = False
            file = self.filepath
            file.seek(self._seekto)

        if isinstance(access_file, basestring):
            if access_modname == None:
                raise ValueError("if you specify access file, you must specify modname as well")
            access_filename = access_file
            # either overwrite and read/write , or no overwrite and read write
            access = "w+" if overwrite else 'r+'
            access_file = open(access_filename, access)
            # make sure we are at end of file
            access_file.seek(0, 2)
            # if new file, then we need to write the access_header...
            if overwrite:
                access_file.write(_access_header)
                if self.additional_access_code is not None:
                    access_file.write(self.additional_access_code)
        elif access_file == None:
            access_filename = None
            access_file = StringIO()
        else:
            pass

        try:
            iterator = ET.iterparse(file, events=("start", "end"))

            # Parsing looking for diminsions, instances, or operations and call down to next level
            # only supports single state def currently
            sharedinfo = odict()
            # current comp is none
            comp = firstcomp = None
            if access_filename is not None:
                sharedinfo['access_filename'] = access_filename
            for event, elem in iterator:
                elem_tag_lower = elem.tag.lower()    
                if elem_tag_lower == '_tdef' and event == 'start':
                    for k, v in elem.items():
                        sharedinfo[k] = v
                elif elem_tag_lower == 'statedef' and event == 'start':
                    # returns a ComponentDefinition object, which is the goal of this parser
                    # give each component its own copy of the sharedinfo...usually there is just one comp though...
                    # if comp != None:
                    #    raise Exception("Can only support one StateDef node tag per tdef")
                    comp = self._parse_statedef(elem, iterator, access_file, copy.deepcopy(sharedinfo))
                    # this first one through, so nothing special needed
                    if firstcomp == None:
                        firstcomp = comp
                    else:  # we have seen a comp already...make sure this new one is the same
                        if firstcomp.name != comp.name:
                            raise Exception("Can only support one StateDef type per tdef...found both '%s' and '%s'" % (
                            firstcomp.name, comp.name))
                        # move all the nodes to the first comp
                        for node in comp.nodes:
                            firstcomp.add_node(node)
                        # hopefully this frees up memory...
                        _dict_update(firstcomp.info, comp.info)
                        del comp
        finally:
            if close_db_when_done:
                file.close()

        if firstcomp == None:
            raise TdefError("statedef parsing didn't find a StateDef tag")

        ##########
        ########## Process access information for the component we found
        ##########
        # if a module for the access code was not given, then save it off
        if access_modname is None:
            access_modname = "namednodes.plugins.tdef.accesses_" + firstcomp.name
        firstcomp.info['access_modname'] = access_modname
        import warnings
        # turn off warning when we import to due to our access module being a made up thing
        warnings.filterwarnings("ignore", category=RuntimeWarning, module=access_modname)
        # go back to beggining of access file so that we can load the source in to the module...
        access_file.seek(0)
        # if a filename was passed
        if access_filename is not None:
            import compileall
            compileall.compile_file(access_filename, force=True, quiet=True)
            if PY2:
                sys.modules[access_modname] = imp.load_source(access_modname, access_filename, access_file)
            else:
                class LoadFromFileobj(SourceFileLoader):
                    def get_code(self, fullname):
                        return access_file.read()

                loader = LoadFromFileobj(access_modname, access_filename)
                mod = types.ModuleType(loader.name)
                sys.modules[access_modname] = mod
                loader.exec_module(mod)

        else:
            # at this point we should be at statedef and END
            access_code = access_file.read()
            # create access access_modname if it does not exist...super long name to avoid possible collision
            if access_modname in sys.modules:
                modobj = sys.modules[access_modname]
            else:
                # we have to ignore runtime warning's for this module to
                if access_modname in sys.modules:
                    modobj = sys.modules[access_modname]
                else:
                    modobj = sys.modules[access_modname] = types.ModuleType(access_modname)
                    # if module is new, make sure it starts with our imports
                    modobj.__dict__['__source__'] = _access_header
                    exec (_access_header, modobj.__dict__)
                    if self.additional_access_code is not None:
                        exec (self.additional_access_code, modobj.__dict__ in modobj.__dict__)
            # done creating module, now add our new source...
            try:
                # save off source for reference...
                modobj.__dict__['__source__'] += "\n" + access_code
                exec (access_code, modobj.__dict__)
            except:
                _LOG.exception(access_code)
                raise
        # if file name passed in close it, when we are finished...
        if access_filename is not None:
            access_file.close()

        return firstcomp

    def _parse_statedef(self, elem, iterator, access_file, sharedinfo):
        """continue parsing statedef object and returns a ComponentDefinition"""
        # create new state
        name = elem.get("Path") or elem.get("path")
        if name == None:
            name = "uarch"
        # any other info to grab ? and add to sharedinfo ?
        comp_def = CreateComponentDefinition(name, sharedinfo)
        # now...on to parsing its states/nodes
        statename = ""
        access_class = None
        lhs = rhs = None
        length = None
        indent_level = 2
        for event, elem in iterator:
            elem_tag_lower = elem.tag.lower()
            if elem_tag_lower == 'state' and event == 'start':
                # need statename without the (...this is duplicated in parse state...but
                # parse state is recursive and last_node might not be the first one...so..
                # we have this code here...
                name = elem.get("_name") or elem.get("name")
                # Note: this is when we have scan
                lhs = name[name.find("(") + 1:name.find(":")] if name.count("(") else None
                rhs = name[name.find(":") + 1:name.find(")")] if name.count(":") else None
                if lhs is not None and rhs is not None:
                    lhs = int(lhs)
                    rhs = int(rhs)
                    if rhs > lhs:
                        t = rhs
                        rhs = lhs
                        lhs = t
                    length = int(lhs) - int(rhs) + 1

                name = name[: name.find("(")] if name.count("(") else name
                statename = name
                # last node is the node where the accesses should go, not necessarily
                # the node that is the immediate child of the component
                last_node = self._parse_state(elem, iterator, comp_def)
                # at this point we should be at STATE and END
                # reset access class so that we watch for new operations
                if access_class is not None:
                    self._write_bundle_access(access_class, access_file)
                access_class = None
            elif elem_tag_lower == 'operation' and event == 'end':
                # parse out operation for this node
                codeblock = self._parse_operation(elem, iterator, statename, length)
                if codeblock:  # ok, we have code we will definitely need an access
                    if access_class is None:
                        # make sure access class names are unique across all afd...
                        randomstr = [random.choice(string.ascii_letters) for i in range(5)]
                        randomstr = "".join(randomstr)
                        access_class = "Access_" + statename + "_" + randomstr
                        codeblock.insert(0, "\nclass {0}(StateNodeImplementedAccess):".format(access_class))
                        # need to move these up to somewhere after line 403 where we are clearing access_class
                        # and starting on the next one...
                    last_node.add_access("state", access_class, default=True)
                    # even in bundled cases we need to use our default access class here
                    # last_node.add_access("bundle", access_class, default=True)
                    last_node.add_access("bundle", "{0}_BundleAccess".format(access_class))
                    codeblock = "\n".join(codeblock)
                    access_file.write(codeblock)
                # reset length in case next state doesn't have this
                length = None
            elif elem_tag_lower == 'generalinfo' and event == 'start':                
                self._parse_general_info(elem, iterator, comp_def)                
            elif elem_tag_lower == 'statedef' and event == 'end':
                # end of this node...
                elem.clear()
                # before returning...see if we need to write a bit more about the bundle access
                if access_class is not None:
                    self._write_bundle_access(access_class, access_file)
                return comp_def
                
        raise Exception("state tag not found")

    def _write_bundle_access(self, access_class, access_file):
        """
        Args:
            classname bundle is based on,
            file object to write to
        """
        access_file.write("\nclass {0}_BundleAccess(StateNodeBundleAccess, {0}):\n".format(access_class))
        access_file.write("   read_no_bundle = {0}.read\n\n".format(access_class))

    def _parse_state(self, elem, iterator, comp_def):
        """Parse state node and returns the deepest (non-field) node node that we found,
        so that we add accesses to the correct component
        """
        # create new state
        info = odict()
        for k, v in elem.items():
            if k in ["_name", "name"]:
                continue
            else:
                info[k] = v
        # make sure we only get name and not dimensions
        name = elem.get("_name") or elem.get("name")
        # put this log as soon as we can before possible error could occur
        if settings.DEBUG:
            _LOG.debugall("TdefLoader._parse_state: name=%s" % name)
        # now convert name
        name = name[: name.find("(")] if name.count("(") else name
        last_node = state_node = nodes.NamedNodeDefinition(name, info)
        # get the name back, just in case this turned out to be a reserved word
        name = last_node.name
        # add default access
        state_node.add_access("state", "StateNodeAccess")
        state_node.add_access("bundle", "StateNodeBundleAccess")
        comp_def.add_node(state_node)
        comp_def.access_groups["default"] = ["state", "bundle"]
        # now look for dimensions/fields/procedures...
        for event, elem in iterator:
            if elem.tag in ['Dimension', 'Instance'] and event == 'start':
                last_node = self._parse_dimension(elem, iterator, state_node)
            elif elem.tag == "Field" and event == "end":
                # add fields to item not array
                self._parse_field(elem, iterator, state_node)
            elif elem.tag == "GeneralInfo" and event == "start":
                self._parse_general_info(elem, iterator, state_node)
            elif elem.tag.lower() == 'state' and event == 'end':
                # end of this node...
                elem.clear()
                return last_node
        else:
            raise Exception("state tag not found")

    def _parse_dimension(self, elem, iterator, parent_node):
        """Parse dimension AND adds to the parent_node
        """
        indices = elem.get("_instances")
        if settings.DEBUG:
            _LOG.debugall("TdefLoader._parse_dimensions: indices=%s" % indices)
        if indices.find(":") < 0:
            imin = imax = int(indices, 0)
        else:
            # sometimes instances are in hex...
            imin, imax = [int(i, 0) for i in indices.split(":")]
            imin, imax = (imin, imax) if imin < imax else (imax, imin)
        name = elem.get("Name") or elem.get("_name") or elem.get("name")
        assert name != None, "No name?"
        array_node = nodes.NamedNodeArrayDefinition(name, item_keys=list(range(imin, imax + 1)))
        array_node.add_access("state", "StateNodeAccess")
        array_node.add_access("bundle", "StateNodeBundleAccess")
        array_node.item_definition.add_access("state", "StateNodeAccess")
        array_node.item_definition.add_access("bundle", "StateNodeBundleAccess")
        last_node = array_node.item_definition
        parent_node.add_node(array_node)
        for event, elem in iterator:
            if elem.tag in ['Dimension', 'Instance'] and event == 'start':
                # recurisve...add next dimension to item, not array
                last_node = self._parse_dimension(elem, iterator, array_node.item_definition)
            elif elem.tag == "Field" and event == "end":
                # add fields to item not array
                self._parse_field(elem, iterator, array_node.item_definition)
            elif elem.tag in ['Dimension', 'Instance'] and event == 'end':
                # end of this node...
                elem.clear()
                return last_node

    def _parse_field(self, elem, iterator, parent_node):
        """Parse field AND adds to the parent_node"""
        known_items = ["_name",'_indices','_map','_valenum','bits','maps','signal']
        name = elem.get("_name")
        if settings.DEBUG:
            _LOG.debugall("TdefLoader._parse_field: field=%s" % name)
        info = odict()
        info['value_type'] = "BitData"
        scannode = False
        if name.startswith("_node"):
            # scan node...get rid of the "_node" part..
            name = self.node_re.match(name).groups()[0]
            info['signal'] = True
            scannode = True
        ##########
        # indices
        ##########
        indices = elem.get("_indices", None)
        # if no indices, assume, then for scan nodes assume 0, else, nothing...
        if indices == None:
            if scannode:
                indices = [0]
        else:  # must have indices, convert to list
            indices = indices.split(',')
            indices = list(map(int, indices))
        if indices is not None:
            info['bits'] = indices
        ######
        # bit map
        ######
        maps = elem.get("_map")
        if maps is not None:
            maps = maps.split(",")
            maps = list(map(int, maps))
            assert len(maps) == len(indices), "Indices and maps must have same length"
            info['maps'] = maps
            # we are going to order these so LSB is first in the list
            # vs. the XML shows it as MSB first for easier display
            indices.reverse()
            maps.reverse()

        # check if we have an upper/lower
        match = self.field_name_re.match(name)
        # if we dont avoid
        if match and not scannode:
            name, upper, lower = match.groups()
            # if only one given, must be single bit
            if upper is not None and lower is None:
                lower = upper
            # if nothing given must be single bit and assume 0
            elif upper is None and lower is None:
                lower = upper = 0
            # TBD - what is the downfall of leaving the lower None here ?
            info['upper'], info['lower'] = int(upper), int(lower)        
        for additional_elem in elem.attrib:
            if additional_elem not in known_items:
                info.update({additional_elem:elem.attrib.get(additional_elem)})
        # now, create our new node
        fieldobj = nodes.NamedNodeDefinition(name, info)
        # if we have a maps, added it as a scanfield
        if maps is not None:
            fieldobj.add_access("state", "ScanFieldAccess")
        else:
            # if no map given, assume it will be filled in by parent...
            fieldobj.add_access("state", "AccessBitMapFromParent")
            # raise Exception("not sure what access to give to this field...no map info, no upper/lower")
        parent_node.add_node(fieldobj)
        # now delete the element since we are done with it...
        enum_elems = elem.findall("_valenum")
        if len(enum_elems):
            enum_dict = {}
            try:
                for enumval in enum_elems:
                    enumstr = enumval.get("enum")
                    enumval = int(enumval.get("value"), 0)  # *seem* to be hex values...
                    enum_dict[enumval] = enumstr
                nextp = parent_node
                # back up till we find a component
                while nextp.component is None:
                    nextp = nextp.parent
                try:
                    enum_name = fieldobj.path
                    nextp.component.enums.create_simple(enum_name, enum_dict)
                except ValueError as e:
                    if str(e).count("has been defined"):
                        # gotta love it, someone defined a field twice...it happens
                        # apparently...
                        time.sleep(0.01)  # make sure time is unique
                        enum_name += str(time.time()).replace(".", "")
                        nextp.component.enums.create_simple(enum_name, enum_dict)
                    else:
                        raise
                fieldobj.info['enum'] = enum_name
            except Exception as e:
                raise e
                _LOG.warning("Error parsing enums for %s" % fieldobj.path)
        elem.clear()

    def _parse_field_orig(self, elem, iterator, parent_node):
        # !!!!!!!!
        # !!!!!!!!
        # !!!!! DELETE ME once _parse_field works
        # !!!!!!!!
        # !!!!!!!!
        """Parse field AND adds to the parent_node"""
        name = elem.get("_name")
        if name.startswith("_node"):
            # scan node...bet rid of the "_node" part..
            name = self.node_re.match(name).groups()[0]
            indices = elem.get("_indices", None)
            # if no indices, assume it is 0
            if indices == None:
                indices = [0]
            else:
                indices = indices.split(',')
                indices = list(map(int, indices))
            maps = elem.get("_map")
            maps = maps.split(",")
            maps = list(map(int, maps))
            assert len(maps) == len(indices)
            # we are going to order these so LSB is first in the list
            # vs. the XML shows it as MSB first for easier display
            indices.reverse()
            maps.reverse()
            info = odict()
            info['nodename'] = name
            info['bits'] = indices
            info['maps'] = maps
            fieldobj = nodes.NamedNodeDefinition(name, info)
            fieldobj.add_access("state", "ScanFieldAccess")
            parent_node.add_node(fieldobj)
        else:
            # assume it is part of an array, therefore no access
            match = self.field_name_re.match(name)
            fieldname, upper, lower = match.groups()
            # upper/lower not used here yet...
            fieldobj = nodes.NamedNodeDefinition(fieldname, dict(upper=upper, lower=lower))
            parent_node.add_node(fieldobj)
            fieldobj.add_access("state", "AccessBitMapFromParent")
        # now delete the element since we are done with it...
        elem.clear()

    def _parse_bitval(self, bitvalstr):
        """turn bitval string in to a tuple of values"""
        bitval = self.bitval_re.findall(bitvalstr)
        bitvals = [int(v, 16) for v in bitval]
        bitvals.reverse()
        return bitvals

    def _parse_set_statement(self, elem, variable_context):
        postset = elem
        var = postset.get("Var")
        expr = postset.get("Expr")
        if var.startswith("_dout"):
            raise Exception("Non-constexpr not allowed in Sets.")
        else:
            # Get the variable parsed
            fieldinfo = var
            fieldname, to_upper, to_lower = self.field_name_re.match(fieldinfo).groups()
            if fieldname in nodes.NamedNodeDefinition.reserved_names:
                fieldname = fieldname + "_"
            # if lower not present, but upper is, then it must be a single
            if to_upper is not None and to_lower is None:
                to_upper = int(to_upper, 0)
                to_lower = to_upper
            # no bits present, but we have to have something, so assume 0
            elif to_upper is None and to_lower is None:
                to_lower = to_upper = None
            else:
                # convert to ints if they are not already
                to_lower = int(to_lower, 0)
                to_upper = int(to_upper, 0)
            # Ok, now let's get the expression!
            tdef_operation = parse_expression(expr)
            # Note: don't include our context here.. we don't want to have complex set chains here
            last_opstr = self._parse_tdef_operation(tdef_operation)
            # Note: we should keep track of the sizes of these variables
            if to_lower is None:
                # value = eval(last_opstr.replace("params","variable_context"))
                # if fieldname in variable_context:
                #    _LOG.debug("Variable named: {0} is defined twice in two set statements.".format(fieldname))
                # variable_context[fieldname] = value #save off the size of this variable
                # return "{0} = BitData({1},{2})".format(fieldname, len(variable_context[fieldname]), hex(variable_context[fieldname]))
                return "params['{0}'] = {1}".format(fieldname, last_opstr)
            elif fieldname in variable_context:
                value = eval(last_opstr.replace("params", "variable_context"))
                variable_context[fieldname][to_lower:to_upper] = value
                return "{0}[{1}:{2}] = {3}".format(fieldname, to_lower, to_upper, hex(int(value)))
            else:
                # assume previous set exists...
                return "params['{0}'][{1}:{2}] = {3}".format(fieldname, to_lower, to_upper, last_opstr)

    def _parse_operation(self, elem, iterator, state_name, length):
        """assumes operation has been completely parsed by etree"""
        # if elem.get("Type") != "Read":
        #     print "Write procedures not supported yet"
        #     return
        # if elem.get("Type") != "Read":
        #    raise Exception("Unsupported operation type: {0}".format(elem.get("Type")))
        istr = "  "
        istrs = [i * istr for i in range(5)]
        # note this can increase/decrease if we hit a loop
        ilevel = 3
        end_loop = "END"  # something we use to know we are at end of a loop
        # save indent level and operation
        operation_list = []
        bit_mapping = odict()
        procedures = elem.findall("Procedure")
        assert len(procedures) == 1, "Not sure what to do with more than one procedure"
        procedure = procedures[0]
        # new way to get children due to deprecation of getchildren
        children = [p for p in procedure if p is not procedure]
        cnum = 0
        # because sometimes we will process more than one child...
        opnum = 0
        regbus = None
        static_variable_context = {}
        queued = False
        if length is not None:
            # for scan
            operation_list.append((ilevel, "_dout=ipc.BitData({0},0,queue=True)".format(length)))
        while cnum < len(children):
            # check for odd "fake" tag that we inserted first
            if children[cnum] == end_loop:
                # end of loop, go back a level on indent
                ilevel -= 1
                cnum += 1
            elif children[cnum].tag in self._known_commands:
                # get the operation
                op_function = self._known_commands[children[cnum].tag]
                # build string for function call, starting with function name
                optstr = []
                optstr.append(op_function.function_name + "(")
                if op_function.first_param == FirstParamType.device:
                    optstr.append("device,")
                elif op_function.first_param == FirstParamType.node:
                    optstr.append("node,")
                # else -- must be not_added
                # now convert parameters as needed, and use key=value
                for key, value in children[cnum].items():
                    if value[:1] == "'":  # not an operation,just a string
                        optstr.append("%s=%s," % (key, value))
                    else:
                        try:
                            # attempt to parse operation, and if that fails,
                            # use value as it is
                            expr = parse_expression(value)
                        except RuntimeError as e:  # attemp
                            if str(e).count("recursion depth"):
                                optstr.append(
                                    "%s=Exception(\"unsupported, too long of operation for one line\")," % key)
                                continue
                            else:
                                raise
                        except Exception as e:  # attemp
                            if value.startswith("bitval"):
                                value = self._parse_bitval(value)
                            optstr.append("%s=%s," % (key, value))
                            continue
                        # parsed it, now turn it to single operation
                        pvalue = self._parse_tdef_operation(expr, static_variable_context)
                        optstr.append("%s=%s," % (key, pvalue))
                optstr.append(")")
                if children[cnum].find("PostSet") is None:
                    operation_list.append((ilevel, "".join(optstr)))
                else:
                    opnum = len(operation_list)
                    operation_list.append(
                        (ilevel, "ops[{opnum}] = din = {opstr}".format(opstr="".join(optstr), opnum=opnum))
                    )
                    queued = self._parse_postset(children[cnum], operation_list, bit_mapping, static_variable_context,
                                                 ilevel) or queued
                cnum += 1
            elif children[cnum].tag == "Param":
                # we don't actually consume the param
                cnum += 1
            elif children[cnum].tag == "Set":
                # We need to update our variable context for static variables to look up later
                var = self._parse_set_statement(children[cnum], static_variable_context)
                operation_list.append((ilevel, var))
                cnum += 1
            elif children[cnum].tag == "Loop":
                var = children[cnum].attrib["Var"]
                lstart, lend = children[cnum].attrib["Indices"].split(":")
                # convert to ints...
                lstart, lend = int(lstart), int(lend)
                # reverse these to make sure start is < end
                lstart, lend = (lstart, lend) if lend > lstart else (lend, lstart)
                step = 1 if lend > lstart else -1
                # create loop based on what we found in the loop tag
                # add/sub step from lend
                operation_list.append((ilevel, "for {var} in range({lstart},{lend},{step}):".format(
                    var=var, lstart=lstart, lend=lend + step, step=step))
                                      )
                ilevel += 1
                # all of variabls need to live in our "params" dict
                # not sure how big to make our loop's bit data ??
                operation_list.append((ilevel, "params['{var}']=BitData(128, {var})".format(var=var)))
                # I dont think the bitdata is ever really used, but we need to register this as a known static value
                # so that later we know it is safe to do bit slices on it...
                # static_variable_context[var] = BitData(128,0)
                next_children = children[cnum].getchildren()
                next_children.append(end_loop)
                next_children.reverse()
                # move to next child AND insert this loops children
                cnum += 1
                for child in next_children:
                    children.insert(cnum, child)


            else:
                raise Exception("Unknown tag/function: %s in: \n\t%s" %
                                (children[cnum].tag, self.filepath))
        # if we did a queue, end with last op:
        if queued:
            if length is not None:
                # this section is for scans, and so scans must define
                # upper/lowerbit in their name in the tdef
                operation_list.append((ilevel, "return _dout"))
            else:
                operation_list.append((ilevel, "return ops[{opnum}]".format(opnum=opnum)))

        # No operations found
        if len(operation_list) == 0:
            return None

        # make sure we return some sort of value...so return our last op
        if operation_list[-1][1].count("return") == 0:
            operation_list.append((ilevel, "return 0"))

        # determine if function is read of write...
        if elem.get("Type") == "Read":
            code = [
                "{i}@classmethod".format(i=istr),
                "{i}def read(cls,node):".format(i=istr),
                "{i}{i}node.target_info['data'] = ops = {{}} ".format(i=istr),
            ]
        elif elem.get("Type") == "Write":
            # for writes, we are actually implementing large flush..
            # except...maybe this should be different for arrays vs. scan?
            code = [
                "{i}@classmethod".format(i=istr),
                "{i}def flush(cls,node):".format(i=istr),
                # "{i}def write(cls,node,value):".format(i=istr),
            ]
        else:
            raise Exception("Unsupported operation type: {0}".format(elem.get("Type")))
        #
        code.extend([
            "{i}{i}params = TdefAccessParams(node)".format(i=istr),
            "{i}{i}ipc = baseaccess()".format(i=istr),
            "{i}{i}cls.clear_child_values(node)".format(i=istr),
        ])
        # now add operations
        if len(bit_mapping):
            #code.append("{i}{i}if 'bitmaps' not in node.definition.info:".format(i=istr))
            #currenti = "{i}{i}{i}".format(i=istr)
            currenti = "{i}{i}".format(i=istr)
            code.append("{currenti}node.definition.info['bitmaps'] = dict()".format(currenti=currenti))
            for fieldname in bit_mapping:
                code.append("{currenti}bitmap = node.definition.info['bitmaps']['{fieldname}'] = BitMap()".
                            format(currenti=currenti, fieldname=fieldname))
                for entry in bit_mapping[fieldname]:
                    if isinstance(entry, BitMapEntry):
                        code.append(
                            "{currenti}bitmap.append(BitMapEntry(datasource={0},from_lower={1},numbits={2},dest_lower={3},invert={4}))". \
                                format(
                                entry.datasource,
                                entry.from_lower,
                                entry.numbits,
                                entry.dest_lower,
                                entry.invert,
                                currenti=currenti))
                    elif isinstance(entry, BitMapConstantEntry):
                        datasourcestr = "ipc.BitData(%d,0x%x)" % (entry.numbits, long(entry.constant))
                        code.append(
                            "{currenti}bitmap.append(BitMapConstantEntry(constant={0},from_lower={1},numbits={2},dest_lower={3},invert={4}))". \
                                format(
                                datasourcestr,
                                entry.from_lower,
                                entry.numbits,
                                entry.dest_lower,
                                entry.invert,
                                currenti=currenti))
                    else:
                        raise RuntimeError("unexpected entry type: %s" % str(type(entry)))

        # we use "device" instead of "node" since that is common to both itp and ipc
        # we didn't use ilevel here, since this should be at tope of the file and what drove the
        # ilevel during our code above
        code.extend([
            "{i}{i}with device_locker(ipc):".format(i=istr),
            "{i}{i}{i}device = node.component.target_info['device']".format(i=istr),
            # convert to a string if we have an object
            "{i}{i}{i}if isinstance(device, six.string_types): device = ipc.devicelist[device]".format(i=istr),

            # "{i}{i}{i}ops = {{}}".format(i=istr),
        ])
        # ALL Operations complete, add indent
        for il, op in operation_list:
            code.append("{indent}{code}".format(indent=il * istr, code=op))
        code.append("\n")
        return code

    def _parse_postset(self, elem, operation_list, bit_mapping, context={}, ilevel=0):
        """
        Args:
            elem:
            operation_list: list of operations we will write to file and need to append to
            bit_mapping:
            context:
            ilevel: indent level

        Returns:

        """
        opnum = len(operation_list) - 1
        # used for any new operations that get injected due to constants, but dont mess up other fields pulling
        # from the original operation
        newopnum = opnum
        ops = "ops[{0}]".format(opnum)
        queued = False
        for postset in elem.findall("PostSet"):
            var = postset.get("Var")
            expr = postset.get("Expr")
            # entire dout is being set, so queue the expression and return that
            if var.startswith("_dout"):
                if var.count("."):  # must be a field
                    # get our field info
                    fieldinfo = var.replace("_dout.", "")
                else:  # just use existing var, and fieldname will be dout
                    fieldinfo = var

                # Ok, let's first resolve what we can statically
                # We actually need to know lengths statically as best we can
                for k in context.keys():
                    fieldinfo = fieldinfo.replace("$" + k, "[{0}b]{1}".format(len(context[k]), hex(context[k])))

                try:
                    fieldname, to_upper, to_lower = self.field_name_re.match(fieldinfo).groups()
                    # if lower not present, but upper is, then it must be a single
                    if to_upper is not None and to_lower is None:
                        to_upper = int(to_upper, 0)
                        to_lower = to_upper
                    # no bits present, but we have to have something, so assume 0
                    elif to_upper is None and to_lower is None:
                        to_lower = to_upper = None
                    else:
                        # convert to ints if they are not already
                        to_lower = int(to_lower, 0)
                        to_upper = int(to_upper, 0)
                except:
                    # ok, this means that we have something other than _dout(#:#)
                    # We need to parse both parts of the list here and then reconstitute it
                    exp = parse_expression("$" + fieldinfo)
                    # now, we have something like: ['$_dout', [: <lhs> <rhs>]]
                    # we need to parse the lhs and rhs and get them into a place where we can get the fieldname, etc. out of it
                    lhs = exp[1][1]
                    rhs = exp[1][2]
                    # note...this breaks down if we hit a field var
                    to_upper = self._parse_tdef_operation(lhs, static_variable_context=context)
                    to_lower = self._parse_tdef_operation(rhs, static_variable_context=context)
                    fieldname = exp[0][1:]

                # fieldname may have gotten remapped during createion if it is a
                # reserved keyword
                if fieldname in nodes.NamedNodeDefinition.reserved_names:
                    fieldname = fieldname + "_"

                # now, are we dealing with a field?
                if var.count('.'):
                    # for fields, we can't have None for lower/upper
                    to_lower = to_lower if to_lower is not None else 0
                    to_upper = to_upper if to_upper is not None else 0

                    if expr.startswith("$_din."):
                        # typical case
                        tdef_operation = parse_expression(expr)
                        from_upper = int(tdef_operation[2][1][0])
                        from_lower = int(tdef_operation[2][2][0])
                        bmap = BitMapEntry(opnum,
                                           from_lower,
                                           from_upper - from_lower + 1,
                                           to_lower,
                                           0)
                        if fieldname not in bit_mapping: bit_mapping[fieldname] = []
                        bit_mapping[fieldname].append(bmap)
                    else:
                        # if this is a SIMPLE number, add it directly to the bitmap and
                        # not in the oepration list
                        try:
                            constant = int(expr, 0)
                        except:
                            # this wasn't the known hack...so...throw exception
                            raise Exception("Cannot handle complex din operation for fields")
                        # we must have converted it, this is just a number where we dont care about the operation
                        # so...add a new operation
                        numbits = to_upper - to_lower + 1  # use field size given for the constant...
                        bmap = BitMapConstantEntry(
                            BitData(numbits, constant),
                            0,  # from is just 0...
                            to_upper - to_lower + 1,  # numbits better be the field size...
                            to_lower,
                            0  # no inversion
                        )
                        if fieldname not in bit_mapping: bit_mapping[fieldname] = []
                        bit_mapping[fieldname].append(bmap)
                else:  # no fields, assumes just _dout or _dout with bits
                    # start a queue if we have not already
                    if not queued:
                        operation_list.append((ilevel, "{0}.Queue()".format(ops)))
                        queued = True
                    # check for upper/lower or can we re-use the below...it might work..
                    # if upper/lower:
                    #    pass
                    # else:
                    # do this seperately than if field is specified..because otherwise
                    # we need to know..
                    tdef_operation = parse_expression(expr)
                    last_opstr = self._parse_tdef_operation(tdef_operation, context)
                    if to_lower is not None:  # if lower is not none, neither is upper
                        operation_list.append((ilevel, "{ops}[{lower}:{upper}] = {expr}".format(
                            ops="_dout", lower=to_lower, upper=to_upper, expr=last_opstr)))
                    else:
                        operation_list.append((ilevel, "_dout = {expr}".format(
                            ops=ops, expr=last_opstr)))
            else:
                # if you find yourself debugging here and wondering why parse_epxression
                # comes back with a very simple variable even though the XML had much more,
                # then use make sure they dont have something like [123] where they should have
                # [123 b] <- the b is needed to know number of bits is what is meant
                # var size ?
                tdef_op = parse_expression(var)
                var_parsed = self._parse_tdef_operation(tdef_op, context)
                expr_parsed = self._parse_tdef_operation(parse_expression(expr), context)
                operation_list.append((ilevel, "%s = %s" % (var_parsed, expr_parsed)))
                # raise Exception("unexpected Var value: %s"%var)
        return queued

    def _parse_tdef_operation(self, operation, static_variable_context={}):
        """
        parse operation tuple returning single string
        """
        # we need to break out recursive call so that we can easily
        # handle error case
        try:
            # import pdb;pdb.set_trace()
            return self._parse_tdef_operation_r(operation, [0], static_variable_context=static_variable_context)
        except TdefError as e:
            if str(e).count("too many levels"):
                return "Exception('too many levels')"

    def _parse_tdef_operation_r(self, operation, depth_count, in_dep=False, static_variable_context={}):
        """
        Args:
            operation : is the tuple for the next parsed tdef operation we should process
            depth_count : used to determine when our recursive calls have gone
                          so deep that python can't convert to code in one line
            in_dep : is used to track when we a BitData.Deposit has been seen, this
                     means we cant use a simple python number for numbers, and instead
                     need to create a bitdata for depositing values in to
            static_variable_context: is used to keep track of all of the set'd variables that we've seen already
        """
        if depth_count is None:
            # we use a list so we can pass by reference
            depth_count = [0]
        if depth_count[0] > self._max_operation_depth:
            sys.stdout.write("-")  # warning to certain users about this depth thing
            raise TdefError("too many levels in tdef")
        # if ADD/SUB, etc...convert this in to a python operation call
        if operation[0] in self._operation_translation.keys():
            if len(operation) == 3:
                depth_count[0] += 2
                return "{op}( {param0}, {param1})".format( \
                    op=self._operation_translation[operation[0]],
                    param0=self._parse_tdef_operation_r(operation[1],
                                                        depth_count,
                                                        static_variable_context=static_variable_context),
                    param1=self._parse_tdef_operation_r(operation[2],
                                                        depth_count,
                                                        static_variable_context=static_variable_context))
            elif len(operation) == 2:
                depth_count[0] += 1
                # yes this code got busted at some point, but it does not seem
                # to be used...leaving it as a breadcrumb in case we ever
                # finally hit this
                return "{op}( {param0} )".format( \
                    op=self._operation_translation[operation[0]],
                    param0=self._parse_tdef_operation_r(operation[1],
                                                        depth_count,
                                                        static_variable_context=static_variable_context))
            else:
                raise ValueError("unexpected operaiton with no parameters: %s" % operation)
        elif operation[0] == "REV":
            depth_count[0] += 1
            op1 = self._parse_tdef_operation_r(operation[1], depth_count,
                                               static_variable_context=static_variable_context)
            opstr = "{0}.Reverse()".format(op1)
            return opstr

        elif operation[0] == "SELV":
            depth_count[0] += 1
            selv_index = operation[1]
            selv_list = operation[2]
            for s in selv_list:
                if len(s) > 1:
                    msg = "SELV has programmatic data in it...not supported yet"
                    print(msg)
                    # import pdb;pdb.set_trace()
                    raise Exception(msg)
            index = self._parse_tdef_operation_r(operation[1], depth_count,
                                                 static_variable_context=static_variable_context)
            # if len(selv_index) > 1:
            #    # we may be exposed if this operation[1] is even more complex with some selve inside it...
            #    # but this seems to work in the instances I've seen so far...
            #    index = self._parse_tdef_operation( operation[1] )
            # else:
            #    # replace variables assuming they are parameters
            #    index = re.sub("\$(\w+)","params['\\1']",selv_index[0])
            # it "seems" from looking at some projects that the selve list
            # starts with lowest entry on the far right...so we have to flip this
            selv_list.reverse()
            slist = [hex(int(x[0], 0)) for x in selv_list]
            # not sure if this will work, but try to get file to have hex for easier to read output
            slist = '[' + ",".join(slist) + "]"
            opstr = "{0}[{1}]".format(slist, index)
            return opstr
        elif operation[0] == "JOINV":
            depth_count[0] += 1
            # import pdb;pdb.set_trace()
            op1 = self._parse_tdef_operation_r(operation[1], depth_count,
                                               static_variable_context=static_variable_context)
            op2 = self._parse_tdef_operation_r(operation[2], depth_count,
                                               static_variable_context=static_variable_context)
            opstr = "{0}.Append({1})".format(op2, op1)
            return opstr

        elif operation[0].startswith("$_din"):
            # we don't care about "din.data vs. din.DR" just call it din...
            # this is probably inefficient, b/c I think most time DIN is used, it is the full
            # data...but that may not ALWAYS be true...
            if len(operation) > 2:
                upper = int(operation[2][1][0])
                lower = int(operation[2][2][0])
                # return last dout instead
                return "din[{upper}:{lower}]".format(upper=upper, lower=lower)
            elif len(operation) == 2:
                # just _din.DR, return "din"
                return "din"
        elif operation[0].startswith("$_dout"):
            raise Exception("not used")
        elif operation[0].startswith("0y"):
            # leave it in the binary for readbility ?
            # return 'int("0b{data}",0)'.format(data=operation[0][2:])
            data = int("0b" + operation[0][2:], 0)
            return '0x{0:x}'.format(data)
        elif operation[0].startswith("0x"):
            # if we are in a dep, we need to create a bitdata to wrap it the number
            # since we will do a deposit in to this number
            if not in_dep:
                return operation[0]
            else:
                return "ipc.BitData(\"{0}\")".format(operation[0])
        elif operation[0] == "$StateIn":  # the state object passed in should be our equivalent of current node
            if len(operation) < 3:  # no upper/lowerbit
                return "params['{0}']".format(operation[1][1])
            else:
                return "getbits(params['{0}'],{upper},{lower})".format(operation[1][1],
                                                                       upper=operation[2][1][0],
                                                                       lower=operation[2][2][0])
        elif type(operation) in [str] and operation.startswith("$"):
            # operation is just a variable for one of the dimensions strip out the $
            if operation[1:] in static_variable_context:
                return operation[1:]
            else:
                return re.sub(r"\$(\w+)", "params['\\1']", operation)
        elif operation[0].startswith("$") and len(operation) == 1:
            # operation is just a variable for one of the dimensions strip out the $
            if operation[0][1:] in static_variable_context:
                return operation[0][1:]
            else:
                return re.sub(r"\$(\w+)", "params['\\1']", operation[0])

        elif operation[0].startswith("$"):
            # assume that this is an sublist access
            if operation[0][1:] in static_variable_context:
                t = static_variable_context[operation[0][1:]]
                lhs = self._parse_tdef_operation_r(operation[1:][0][1], depth_count, in_dep=True,
                                                   static_variable_context=static_variable_context)
                rhs = self._parse_tdef_operation_r(operation[1:][0][2], depth_count, in_dep=True,
                                                   static_variable_context=static_variable_context)
                return "{0}[{1}:{2}]".format(operation[0][1:], lhs, rhs)
            else:
                # old
                # return re.sub("\$(\w+)","params['\\1']",operation[0])
                # why does it matter on that static variable context
                lhs = self._parse_tdef_operation_r(operation[1:][0][1], depth_count, in_dep=True,
                                                   static_variable_context=static_variable_context)
                rhs = self._parse_tdef_operation_r(operation[1:][0][2], depth_count, in_dep=True,
                                                   static_variable_context=static_variable_context)
                return "params['{0}'][{1}:{2}]".format(operation[0][1:], lhs, rhs)

        elif operation[0] == "DEP":
            depth_count[0] += 2
            if len(operation[2]) == 1:
                upper = lower = operation[2][0][0]
            else:
                upper = operation[2][1][0]
                lower = operation[2][2][0]
            return "BitDataDeposit({depto},{lower},{numbits},{nextval})".format(
                depto=self._parse_tdef_operation_r(operation[1],
                                                   depth_count,
                                                   in_dep=True, static_variable_context=static_variable_context),
                lower=lower,
                numbits=int(upper) - int(lower) + 1,
                nextval=self._parse_tdef_operation_r(operation[3],
                                                     depth_count=depth_count,
                                                     in_dep=True, static_variable_context=static_variable_context)
            )
        elif operation[0] == "BD":
            return "ipc.BitData({0},{1})".format(operation[1][0], operation[2][0])
        else:
            # see if it is just a number
            try:
                value = int(operation[0])
                return value
            except:
                try:
                    # maybe it's a <#>b
                    value = int(operation[0:-1])
                    return value
                except:
                    try:
                        value = int(operation[0])
                        return value
                    except:
                        # must not be an int....
                        pass
            raise Exception("unsupported operation right now: %s" % operation[0])

    def _parse_general_info(self, elem, iterator, parent_node):
            """
            """
            info = odict()
            for event, elem in iterator:            
                if elem.tag == "GeneralInfo" and event == 'end':
                    elem.clear()
                    return
                else:
                    if event == 'start':
                        datatype_support = elem.attrib.get('dt', None)
                        if datatype_support:
                            if datatype_support == 'int':
                                try:
                                    xml_data = int(elem.text, 0)
                                except ValueError as e:
                                    if elem.text.count(","):
                                        raise ValueError("please use list[int] for parsing lists of ints") from e
                                    else:
                                        raise e
                            elif datatype_support == 'list[int]':
                                if elem.text is None:
                                    xml_data = []
                                else:
                                    split_elements = elem.text.split(",")
                                    xml_data = []
                                    for ie in split_elements:
                                        xml_data.append(int(ie,0))
                            elif datatype_support == 'list[str]':
                                if elem.text is None:
                                    xml_data = []
                                else:
                                    split_elements = elem.text.split(",")
                                    xml_data =[]
                                    for ie in split_elements:
                                        xml_data.append(ie)
                            elif datatype_support == 'str':
                                xml_data = elem.text
                            else:
                                raise ValueError(f"Unsupported datatype {datatype_support} specified")
                        else:
                            xml_data = elem.text
                        parent_node.info[elem.tag] = xml_data


    def _test_for_tdef(self):
        close_file = True
        if isinstance(self.filepath, basestring):
            try:
                file = open(self.filepath, "rb")
            except IOError:
                _LOG.debug("Unable to open file: {0}".format(self.filepath))
                return False
        else:  # if not a string, assume it is a file
            close_file = False
            file = self.filepath
            # to know where to reset to
            self._seekto = file.tell()
        iterator = ET.iterparse(file, events=("start",))
        # First tag should be if it is tdef
        event, elem = next(iterator)
        if close_file:
            file.close()
        # then yes this looks like a file we support
        if elem.tag == "_tdef":
            return True
        else:
            return False


# this is starting point for all access code,
# it is recommend to override some stateports to improve performance
_access_header = """
import operator
import six
from namednodes.access import NodeAccess
from namednodes.accesses.state import StateNodeAccess, StateNodeImplementedAccess
from namednodes.accesses.bundle import StateNodeBundleAccess
from namednodes.utils.bitmap import BitMap,BitMapEntry, BitMapConstantEntry
from namednodes.plugins.tdef_accesses import *
from namednodes.utils.bitmanip import getbits
"""
