
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




"""
==============
Snapshot
==============

This plugin is for components. When a node or register is read via this snapshot plugin, the value is saved. This
is useful for nodes with very large numbers of child nodes that you don't want to re-read unless you specifically request it

This is a VERY simply plugin and simply caches the results of getattr calls. If we have the attribute, we return it
(which is a NodeValue for nodes, and therefore a new read is not done). If we do not have the attribute, we request
it from the component, and add it to our cache.

"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ..comp import ComponentPlugin


class SnaphshotComponentPlugin(ComponentPlugin):
    """
    The snapshot plugin will keep a copy of each NodeValue that is requested. This means it acts as a sort of cache, and will
    always return the last value read, unless the user explicitly asks the node to do another read.

    It also means that all scripts with a reference to this node share the same object. So updating it, will in turn impact
    any script holding a reference.

    Use this feature with caution, if ALL registers read through this snapshot, it can cause the memory footprint to grow
    since unused register references are not cleared from memory. Also, be careful about calling clear as it will impact
    anyone else using this snapshot plugin.

    Example::

        >>> comp.myregister.write(0xa)
        >>> assert comp.myregister == 0xa
        >>> assert comp.snapshot.myregister == 0xa
        >>> # write to original register will NOT change snapshot value without a read
        >>> comp.myregister.write(0xb)
        >>> assert comp.snapshot.myregister == 0xa
        >>> comp.snapshot.myregister.read() # now the snapshot will be updated...
        >>> assert comp.snapshot.myregister == 0xb

    """
    name = "snapshot"

    def __init__(self,component):
        """Default plugin constructor simply keeps a reference to the component that this plugin is attached to"""
        super(SnaphshotComponentPlugin,self).__init__(component)
        self._cache = {}

    def __getattr__(self,attr):
        if attr in self._cache:
            return self._cache[attr]
        else:
            newval = getattr(self.component,attr)
            self._cache[attr] = newval
            return newval

    def clear(self):
        """
        clear the current cache of nodes
        """
        self._cache = {}
