
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.





"""
"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import re

from ..utils.ordereddict import odict
from ..nodes import NodeValuePlugin, NamedNodeValue

nn_re = re.compile("_node\('(.*)'\)")

class ScanNodeWalk(NodeValuePlugin):
    """
    This plugin provides a "scan_walk" attribute to NodeValues so that if the node value
    has a "_node(____)" information then we can allow tab completion discovery of what
    is within the master node.

    Example Usage::

        >>> uarch.uncore_scanout_cnt_scub.scan_walk.nec.vcu.vcuctld.PcRamRdEnF701H
        nec/vcu/vcuctld/PcRamRdEnF701H [12b] 0x000


    This features is not 100% complete since it cannot support nodes with { or [...It is TBD
    on whether those will be supported since this feature is primarly aimed at older state format
    information        
    """
    name = "scan_walk"
    # children we have already seen
    _attr_cache = None
    # next level of children to walk
    _walk_children = None
    # nodevalue we are providing the walk for
    nodevalue = None
    # whether we should start throughing setattr errors
    _frozen = False
    # mostly just used for debug, but help us know what 'level' we are at
    _thispath = None
    def __init__(self, nodevalue, children=None, thispath=None):
        # once we have done a walk, keep up with our attribute so we don't have to rebuild
        self._attr_cache = {}
        self.nodevalue = nodevalue
        # using ordered dictionary helps later for repr
        self._walk_children = odict()
        self._thispath = thispath
        self._frozen = True
        # if not children, walk through the nodevalues children
        if children is None:
            #children = nodevalue.children
            children = {child.name:child for child in nodevalue.nodes}

        MAX = 9999999999999999999999999999
        sep_find = [
            [".", MAX],
            ["/", MAX],
        ]

        # find all the possible children hiearchies
        for nodename, child in children.items():
            while True:
                # fill in with positions of these seperators
                for index,sep in enumerate(sep_find):
                    # only save off if we found something...dont save -1 b/c it screws up sort later
                    found = nodename.find(sep[0])
                    if found>=0: sep_find[0][1] = found

                # now sort...
                sep_find.sort(key=lambda x:x[1])
                # if first one in the list has 0 then that means first letter is a seperator
                # 0 still means found -1 is not found
                if sep_find[0][1]==0:
                    nodename = nodename[1:]
                    continue
                # either seperator is -1 or it is set up
                else:
                    break

            if sep_find[0][1]==MAX:
                this_level = nodename
                next_level = child
            else:
                # seperator found, use it
                find = sep_find[0][1]
                this_level = nodename[:find]
                next_level = nodename[find+1:]

            # if we are at the end save off just node
            if isinstance(next_level, NamedNodeValue):
                self._walk_children[this_level] = child
            # otherwise save off next child as well
            else:
                # now setup to track our children
                if this_level not in self._walk_children:
                    self._walk_children[this_level] = odict()
                self._walk_children[this_level][next_level] = child

    def __getattr__(self,attr):
        if attr in self._attr_cache:
            return self._attr_cache[attr]

        next_child = self._walk_children.get(attr, None)
        if next_child is not None:
            #if len(self._walk_children[attr]) > 1: # if we have children...keep going:
            # leaving in previous statement, not sure why it was there, and
            # obviously my testing for this feature is lacking
            if not isinstance(next_child, NamedNodeValue):
                # append to path if not none, else, just use attr
                next_path = self._thispath + "." + attr if self._thispath is not None else attr
                scan_walk = ScanNodeWalk(self.nodevalue, next_child, next_path)
                self._attr_cache[attr] = scan_walk
                return scan_walk
            else:
                # otherwise, must be time to give a value
                return next_child
        else:
            raise AttributeError("Unknown attribute {0}".format(attr))

    def __setattr__(self, attr, value):
        """check for simulating write, or throw an error"""
        if not self._frozen:
            return object.__setattr__(self, attr, value)
        # frozen, check for nodes...
        next_child = self._walk_children.get(attr, None)
        # setattr and we hit the node to write to, so write it
        if isinstance(next_child, NamedNodeValue):
            # some nodes actually return values on the write
            return next_child.write(value)
        else:
            raise ValueError("Cannot write %s since it does not exist at this path: %s"%(attr,self._thispath))

    def __dir__(self):
        return list(self._walk_children.keys())

    def __repr__(self):
        # this is just the quick and easy path...we could try to show this as some hiearchy
        msg = []
        for key,value in self._walk_children.items():
            # value is a dictionary where ITS value is the actual child node
            for child in value.values():
                msg.append( repr(child) )
        return "\n".join(msg)





