
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




##########################################
# Named Node XML
# Provides an XML writer and reader for
# the named nodes framework
##########################################
"""
========================
Named Nodes To XML
========================

Provides a capability to write a component to an XML file that can later be loaded in.

Example Usage:

    >>> comp.toxml.write("path_to_output.nnxml")
  

"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import os, re
import xml.etree.cElementTree as ET
from copy import copy
from ..utils.ordereddict import odict

from ..comp import ComponentDefinitionPlugin,ComponentLoaderPlugin, NamedComponent, CreateComponentDefinition
from ..nodes import NodeTypes
from ..nodes import NamedNodeArrayDefinition, NamedNodeDefinition, NodeDefinitionPlugin
from ..registers import RegisterDefinition, RegisterComponent, FieldDefinition
from ..utils._py2to3 import *
from ..logging import getLogger
_LOG = getLogger()


# some common variables across all of these plugins
_indent_text = "    "
_supported_node_definitions = { 'NamedNodeDefinition':NamedNodeDefinition,
                                'NamedNodeArrayDefinition': NamedNodeArrayDefinition,
                                'RegisterDefinition': RegisterDefinition ,
                                'FieldDefinition': FieldDefinition ,
                                }

_re_valid_name = re.compile(r'\w+$')

class NNXml_ComponentPlugin(ComponentDefinitionPlugin):
    """
    This plugin knows how to write a named node in to an XML format that can later be loaded by our Loader plugin

    Note:
        The component plugin works alongside the NodeValue and Loader plugins, so it requires both of those
        to be enabled as well (which should happen by default when you import the file containing this plugin
        
    """
    name = "toxml"

    _max_nodes_per_file = 1000
    _single_comp_file = False
    _single_nodes_file = False

    def write(self,filepath,**kwargs):
        """
        Args:
            path_or_file (str)        : path to write to xml, or file object to write to
            max_nodes_per_file        : (default=1000) number of child nodes to check before breaking it out to a seperate file
            single_comp_file (bool)   : True/(False) - used to force all components in to the same file
            single_nodes_file (bool)  : True/(False) - used to force all nodes to single XML regardless of size/count

        Write the ComponentType to XML file with the nodes it contains. If there are nodes with > max_nodes_per_file, then
        a seperate XML will be generated for that particular node

        Note:
        """
                
        # make sure we are writing to filename that matches the extension we can load
        if not filepath.endswith(NNtoXML_Loader.file_ext):
            filepath = filepath + "." + NNtoXML_Loader.file_ext

        self._max_nodes_per_file = kwargs.pop("max_nodes_per_file",1000)
        self._single_comp_file = kwargs.pop("single_comp_file", False)
        self._single_nodes_file = kwargs.pop("single_nodes_file",False)
        #required_imports = kwargs.pop("required_imports", [] ) # idea phase...not implemented
        # reset our unique number in case we have to build up parts of a component
        # in to seperate files node
        self._unique_num = 0
        if len(kwargs): 
            raise ValueError("Unknown parameters provided: {0}".format(kwargs))
        self._write_component( filepath, self.definition, 0 )


    def _write_component(self, filepath, component, indent_level):

        il = indent_level
        # in case indent level becomes a param due to recursive....
        indents = {}
        for i in range(il,il+5): indents[i] = i*_indent_text

        # at this point should have a file
        if isinstance(filepath,basestring):
            f = open(filepath,"wb")
        else:
            f = filepath

        value_class = component.value_class
        classpath = value_class.__module__ + ":" + value_class.__name__
        f.write("{indent}<NNComponent name='{name}' value_class='{nclass}'>\n".format(
                                                                    indent=indents[il],
                                                                    name=component.name, 
                                                                    nclass=classpath))
        f.write("{indent}<ComponentSharedInfo>\n".format(indent=indents[il+1]))
        for key,value in component.info.items():
                # HAS to be same format as NodeDefinition
                if key == "value_class":
                    # we already have that in the component tag
                    continue
                if isinstance(value,basestring):
                    value = "'''"+value+"'''"
                f.write("{indent}<Item key='{key}'><![CDATA[{value}]]></Item>\n".format(
                        indent=indents[il+2],
                        key=key,
                        value=value))
        f.write("{indent}</ComponentSharedInfo>\n".format(indent=indents[il+1]))

        for node in component.nodenames:
            nodedef = component.get_node(node)
            if (len(nodedef.nodes) > self._max_nodes_per_file) and (self._single_nodes_file==False):
                newp = self._write_one_node(filepath,  nodedef) 
                newp = os.path.basename( newp ) # we only need filename portion
                f.write("{indent}<LargeNode>{filepath}</LargeNode>\n".format(indent=indents[il+1],
                                                                                filepath=newp))
            else:
                # must be 'small enough' write out to this existing xml file                 
                f.write( nodedef.toxml(il+1) + "\n")
                        
        #############################################
        # now write out any sub-components
        #############################################
        subitems = component.sub_components.items()
        subcomps = odict()
        if self._single_comp_file:
            for nextpath,subcomp in subitems:
                self._write_component( f, subcomp, il+1)

        else:           
            for compname,compobj in subitems:
                # we only need filename portion
                dirname,filename = os.path.split(filepath)
                comp_path = os.path.splitext( filename )[0]
                nextcomp = comp_path + "." + compname + "." + NNtoXML_Loader.file_ext
                nextcomppath = os.path.join(dirname,nextcomp)
                subcomps[nextcomppath] = compobj
                f.write("{indent}<NNComponentFile>{filepath}</NNComponentFile>\n".format(indent=indents[il+1],
                                                                            filepath=nextcomp))
            #f.write("{indent}</SubComponents>\n".format(indent=indents[il+1]))
            for nextpath,subcomp in subcomps.items():
                self._write_component( nextpath, subcomp, 0)

        f.write("{indent}</NNComponent>\n".format(indent=indents[il]))

        # if a string was passed in, we should close our file
        if isinstance(filepath,basestring):
            f.close()
                       
    def _write_one_node(self, filepath, node):
        """Write a single node to a file based on filepath
        Args:
            filepath (str) : original name of the file we wrote XML to
            node : node object that we are writing out to its own file

        Returns:
            name of the file that we wrote to

        This writes out a single node to an XML formatted file in the same directory
        as the filepath specified. We append either .<nodename> if is alpha numeric, 
        or some incremenditng unque #
        """
        filedir, filebase = os.path.split( filepath )
        # remove extension...add our name, and add extension back
        base,ext = os.path.splitext( filebase )
        if _re_valid_name.match(node.name): # make sure not spaces
            newp = ".{0}".format(node.name)       
        else:
            newp = ".{0}".format(self._unique_num)
            self._unique_num += 1
        # back to new name
        new_name = os.path.join(filedir,base + newp + ext)       
        toxml = node.toxml # wierd, but save function b/c it has some data when we are done
        with open(new_name,"wb") as nodef:
            nodef.write(toxml())
        return new_name

class NNXml_NodeDefinitionPlugin(NodeDefinitionPlugin):
    """The tomxl plugin is used to convert a NamedNode in to an XML representation of its node

    Args:
        level (int) : level of indention we are at
        indent (str) : spaces or tabs that will be multiple by level for the proper spacing

    This is a recursive function
    """
    name = "toxml" 

    def __call__(self,indent_lvl=0):
        xmlstr = []
        nodedef = self.definition
        # create some local indent strings
        il = indent_lvl
        indents = {}
        for i in range(il,il+5): indents[i] = _indent_text*i
        # order is important here since NamedNodeArrayDefinition inherits from NamedNodeDefinition
        nodetypestr = nodedef.__class__.__name__ # in case it is a sub-class   
        nodetypestr = nodedef.__class__.__name__ # in case it is a sub-class
        if nodetypestr not in _supported_node_definitions:
            raise ValueError("Unsupported node definition {0}".format(nodetypestr))

        xmlstr.append("{indent}<{typestr} name='{name}'>".\
                    format(indent=indents[il], typestr=nodetypestr,name=nodedef.name) )

        ###########################
        # Write Information class - if we have info
        ############################
        if len(nodedef.info)>0:
            xmlstr.append("{indent}<Info>".format(indent=indents[il+1]) )
            for key,value in nodedef.info.items():
                # be sure to put values in a cdata tag
                if isinstance(value,basestring):
                    # need to put quotes around strings, everything else will eval fine
                    value = "'''" + value + "'''"
                xmlstr.append("{indent}<Item key='{key}'><![CDATA[{value}]]></Item>".format(indent=indents[il+2],
                                                  key=key,
                                                  vtype=type(value).__name__,
                                                  value=value)  )
            xmlstr.append("{indent}</Info>".format(indent=indents[il+1]) )
        ###########################
        # put access mechanism
        ###########################
        # streamline this if there is only one:
        if len(nodedef.accesses):
            xmlstr.append("{indent}<Accesses group='{group}'>".format(indent=indents[il+1], 
                                                                     group=nodedef.access_group) )
            for aname,aclass in nodedef.accesses.items():
                xmlstr.append("{indent}<Access name='{name}'>{aclass}</Access>".format(indent=indents[il+2],
                                                                                   name=aname,
                                                                                   aclass=aclass))
            xmlstr.append("{indent}</Accesses>".format(indent=indents[il+1]) )
        ###########################
        # Children nodes
        ###########################
        nodes = nodedef.nodes
        if len(nodes)>0:
            xmlstr.append("{indent}<Children>".format(indent=indents[il+1]))
            for child in nodedef.nodes:
                childxml = child.toxml(il+2)
                # add child accesses to our master list
                xmlstr.append( childxml )
            xmlstr.append("{indent}</Children>".format(indent=indents[il+1]))
        ###########################
        # Array Item
        ###########################
        # if it is an array
        if nodedef.type == NodeTypes.Array:
            # check for shortcut
            item_keys = copy(nodedef.item_keys)
            item_keys.sort()
            # ok, this is 0-x range:
            if isinstance(item_keys[-1],int) and item_keys==list(range(item_keys[-1]+1)):
                item_keys = "0:{0}".format(item_keys[-1])
            else:
                item_keys = map(str,nodedef.item_keys)
                item_keys = ",".join(item_keys)

            xmlstr.append("{indent}<Array indices='{0}'>".\
                            format(item_keys,indent=indents[il+1]))
            # hardcode using first enry? we have to have a value
            # object to get access to the plugins
            itemxml = nodedef.item_definition.toxml(il+2)
            xmlstr.append( itemxml )
            xmlstr.append("{indent}</Array>".format(indent=indents[il+1]))
            
        # Close out this node definition
        xmlstr.append("{indent}</{typestr}>".format(indent=indents[il], typestr=nodetypestr ) )
        return "\n".join(xmlstr)



class NNtoXML_Loader(ComponentLoaderPlugin):
    file_ext = "nnxml"

    def _parse_node_definition(self):
        pass

    def _parse_info(self,info_elem):
        """Parse an Info tag getting the keys and items and returning an odict"""
        info_dict = odict()
        for item_elem in info_elem.findall("Item"):
            key = item_elem.get("key")
            # value_type = info_elem.get("type") # needed?
            # convert the string in to the type
            value = eval(item_elem.text)
            info_dict[key] = value
        return info_dict

    def _parse_accesses(self, node, acc_elem ):
        """Parse the access group information and add the accesses to the node"""
        accesses = odict()
        # group
        group = acc_elem.get("group",None)
        group = None if group=="None" else group
        
        for each_elem in acc_elem.findall("Access"):
            node.add_access( each_elem.get("name"), each_elem.text )
        
        node.access_group = group

    def _parse_large_node( self, node_path ):
        """Open Larget Node XML and add its nodes to this component"""
        normed_filepath = os.path.normpath(self.filepath)
        startdir = os.path.dirname( normed_filepath )
        large_node_path = os.path.join( startdir, node_path )
        if not os.path.exists(large_node_path):
            raise Exception("Missing large node file  : {0}".format(large_node_path) )
        # get iterator...and walk parse node file
        iterator = ET.iterparse( open(large_node_path), events=("start","end") )
        _LOG.debugall("NNtoXML_Loader._parse_large_node : parsing {0}".format(large_node_path))
        # parse that file and it should return a new named node which we will pass back to caller
        return self.parse( iterator, single_node=True)

    def parse(self,filepath=None):
        # get filename
        filepath = filepath or self.filepath
        normed_filepath = os.path.normpath(filepath)
        file = open(normed_filepath)
        iterator = ET.iterparse( file, events=("start","end") )
        # first tab should always be component
        event, elem = next(iterator)
        if event != "start" and elem.tag != "NNComponent":
            raise Exception("Unexepcted first tag")

        component = self._parse_component( elem, iterator )
        return component

    def _parse_component(self, comp_elem, iterator=None,single_node=False):
        """
        Args:
            comp_elem : xml element representing the named component that started this call
            iterator : is the current parsing of events for recursive calls
            single_node : is used to parse large nodes where we should return as soon as we have parsed a single node

        """
        import imp,os.path
        import sys
        import importlib


        # this is a recursive fucntion, so these MUST be local variables, do not add to the self        
        curr_node = None
        curr_component = None
        if comp_elem != None:
            name = comp_elem.get('name')
            node_type = comp_elem.get('type','default')
            if node_type == 'default':
                classname = comp_elem.get('value_class')
                # import file with baseclass
                module,classname = classname.split(":")
                modobj = importlib.import_module(module)
                baseclass = getattr(modobj,classname,None)
                if baseclass == None:
                    raise ValueError("could not fine specified baseclass {0}".format(classpath))
                curr_component = CreateComponentDefinition(name,value_class=baseclass)

        # we have to track this one seperately since it is not a recurisve call..
        array_item = None
        nodelist = []
        try: # for catching parsing errors and displaying what file we were parsing when we failed
            for event, elem in iterator:
                if event == "end" and (elem.tag == "NNComponent"):
                    return curr_component

                elif event == "start" and (elem.tag == "NNComponent"):
                    # new component needs to be created and added to this component
                    next_comp = self._parse_component( elem, iterator )
                    curr_component.add_component( next_comp )

                elif event == "start" and (elem.tag == "NNComponentFile"):
                    # new component needs to be created and added to this component
                    start_dirname = os.path.dirname(self.filepath)
                    next_file = os.path.join( start_dirname ,  elem.text)
                    if not os.path.exists(next_file):
                        raise Exception("Could not find file: %s"%next_file)
                    next_comp = self.parse(next_file)
                    curr_component.add_component( next_comp )

                elif event == "end" and (elem.tag == "ComponentSharedInfo"):
                    # parse info to dictionary add to component
                    info = self._parse_info( elem )
                    curr_component.info = info
                    elem.clear()

                elif event == "start" and (elem.tag in _supported_node_definitions.keys()):              
                    # create new state
                    name = elem.get("name")
                    NodeDefClass =  _supported_node_definitions[elem.tag]
                    curr_node = NodeDefClass(name)
                    # if we have a component, add it there...if we are in some children list
                    # don't add it
                    if curr_component is not None:
                        curr_component.add_node( curr_node )
                    else: # must be in a children tag
                        nodelist.append( curr_node )

                elif event == "end" and (elem.tag in _supported_node_definitions.keys()):
                    elem.clear() # free up xml memory
                    # if parsing some large node file, then return the node we built
                    if single_node: 
                        return curr_node
                    #else:
                    # cannot clear out "curr_node" b/c Array closes after this...

                elif event == "end" and (elem.tag == "LargeNode"):
                    next_node = self._parse_large_node( elem.text )
                    # Currently large nodes must be at Component level
                    if curr_component is not None:
                        curr_component.add_node( next_node )
                    else:
                        raise Exception("Not supported")
                    elem.clear()

                elif event == "start" and (elem.tag == "Array"):
                    # get indices and child, but for child we have to do more recursion...
                    item_keys = elem.get("indices")
                    if item_keys.count(":"):
                        # OUR xml only has complete ranges or none at all
                        kmin,kmax = map(int,item_keys.split(":"))
                        item_keys = list(range( max(kmin,kmax)+1))
                    else:
                        item_keys = item_keys.split(",")
                        for i,ik in enumerate(item_keys):
                            if ik.startswith("'"): item_keys[i] = ik.replace("'","")
                            elif ik.startswith('"'): item_keys[i] = ik.replace('"',"")
                            else: item_keys[i] = int(ik) # exception check?
                    curr_node.item_keys = item_keys
                    # now go get child
                    curr_node.item_defintion = self.parse( iterator )

                elif event == "end" and (elem.tag == "Array"):
                    # done parsing the array item, return the current node we found
                    elem.clear()
                    return curr_node

                elif event == "end" and (elem.tag == "Info"):
                    info = self._parse_info( elem )
                    curr_node.info = info
                    elem.clear()
        
                elif event == "end" and (elem.tag == "Accesses"):
                    access_dict = self._parse_accesses( curr_node, elem )
                    elem.clear()

                elif event == "start" and (elem.tag == "Children"):
                    # this node has children, recrusively call back in to this parser
                    child_nodes = self._parse_component( None, iterator )
                    # we need to add each of these child nodes to the current node
                    list(map( curr_node.add_node, child_nodes ))

                elif event == "end" and (elem.tag == "Children"):
                    # done parsing children, return with the list of nodes we found
                    elem.clear()
                    return nodelist
        except:
            _LOG.exception("Error parsing {0}".format(self.filepath))
            raise
                    
