
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



"""
Walk all subcomponents in a given component.

"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from collections import deque
import copy
import numbers
import weakref
from ..logging import getLogger
from ..comp import (ComponentPlugin,
                    ComponentDefinitionPlugin,
                    NamedComponentDefinition,
                    NamedComponent)

from ..nodes import (NamedNodeArrayDefinition,
                     NamedNodeArrayValue,
                     NodeValuePlugin)

_LOG = getLogger()


class WalkComponents(object):
    """
    Walk all subcomponents in a given component or component definition.
    """

    root = None

    def __init__(self, root):
        """
        Dummy initialization, will be overriden in children
        """
        self.root = weakref.ref(root)

    def _component_iterator(self, algorithm, max_depth):
        """
        Build list of subcomponents that will be iterated following the
        specified algorithm.
        """
        # Level 0 is the component calling walk_components
        queue = deque([(self.root(), 0)])

        # While there are components in our processing queue...
        while len(queue) > 0:
            current_comp, current_level = queue.popleft()
            # Add current component to the end of the list
            yield current_comp
            # Child's level is parent's level + 1
            child_level = current_level + 1

            # Only include children up to max_depth (negative is infinite)
            if max_depth > -1 and child_level > max_depth:
                continue

            if algorithm == "depth-first":
                # Add children at the beginning of the queue
                for subcomp in sorted(current_comp.sub_components.keys(),
                                      reverse=True):
                    queue.appendleft((getattr(current_comp, subcomp),
                                      child_level))
            elif algorithm == "breadth-first":
                # Add children at the end of the queue
                for subcomp in sorted(current_comp.sub_components.keys()):
                    queue.append((getattr(current_comp, subcomp),
                                  child_level))
            else:
                # Algorithm not supported
                raise ValueError("Unknown algorithm (%s)" % algorithm)
        return

    def __call__(self, algorithm="depth-first", max_depth=-1, **kwargs):
        """
        Walk all subcomponents in a given component.

        Example:
            Walk subcomponents using depth-first algorithm:

                >>> for subcomp in my_comp.walk_components():
                >>>     subcomp.<do_something...>

            Walk subcomponents using breadth-first algorithm, up to
            subcomponents 4 levels down:

                >>> for subcomp in my_comp.walk_components("breadth-first", 4):
                >>>     subcomp.<do_something...>

        Args:
            algorithm (Optional[str]): Algorithm used to walk subcomponents:

                - depth-first (Default).
                - breadth-first.

            max_depth (Optional[int]): Walk subcomponents up to max_depth
                levels down. Use max_depth < 0 for infinite. Infinite by
                default.

        Returns:
           Iterable obj.

        """
        valid_algorithms = ["depth-first", "breadth-first"]
        valid_kwargs = ["include_root"]

        # Unpack arguments
        include_root = kwargs.pop("include_root", False)

        # Validate arguments
        invalid_kwargs = list(set(kwargs.keys()) - set(valid_kwargs))
        assert len(invalid_kwargs) == 0, \
            "Invalid arguments (%s). Must be: %s" \
               % (", ".join(invalid_kwargs),
                  ", ".join(valid_kwargs))
        assert algorithm in valid_algorithms, \
            "Invalid algorithm (%s). Must be: %s" \
            % (algorithm, ", ".join(valid_algorithms))
        assert isinstance(max_depth, numbers.Number), \
            "Invalid max_depth (%s). Must be a number" % max_depth

        comp_iterator = self._component_iterator(algorithm, max_depth)
        if not include_root:
            next(comp_iterator)
        for comp in comp_iterator:
            yield comp

class WalkComponentsPlugin(ComponentPlugin, WalkComponents):
    """
    This plugin adds the capability to walk all subcomponents in a given
    component, including the component you are calling this from.
    """
    name = "walk_components"

    def __init__(self, component):
        ComponentPlugin.__init__(self, component)
        WalkComponents.__init__(self, component)

    def __call__(self, *args, **kwargs):
        kwargs['include_root'] = kwargs.get('include_root', True)
        return WalkComponents.__call__(self, *args, **kwargs)
    __call__.__doc__ = WalkComponents.__call__.__doc__


class WalkSubComponentsPlugin(WalkComponentsPlugin):
    """
    This plugin adds the capability to walk all subcomponents in a given
    component, without including the component you are calling this from.
    """
    name = "walk_subcomponents"

    def __init__(self, component):
        super(WalkSubComponentsPlugin, self).__init__(component)

    def __call__(self, *args, **kwargs):
        """
        This is the same as walk_components but does not include the component
        you are calling this from.

        **See Also:**

            :py:meth:`walk_components <namednodes.plugins.nn_walk.WalkComponentsPlugin.walk_components>`

        """
        kwargs['include_root'] = False
        return super(WalkSubComponentsPlugin, self).__call__(*args, **kwargs)

class WalkComponentsDefinitionPlugin(
        ComponentDefinitionPlugin, WalkComponents):
    """
    This plugin adds the capability to walk all subcomponents in a given
    component definition, including the component definition you are calling
    this from.
    """
    name = "walk_components"

    def __init__(self, definition):
        ComponentDefinitionPlugin.__init__(self, definition)
        WalkComponents.__init__(self, definition)

    def __call__(self, *args, **kwargs):
        kwargs['include_root'] = kwargs.get('include_root', True)
        return WalkComponents.__call__(self, *args, **kwargs)
    __call__.__doc__ = WalkComponents.__call__.__doc__


class WalkSubComponentsDefinitionPlugin(WalkComponentsDefinitionPlugin):
    """
    This plugin adds the capability to walk all subcomponents in a given
    component definition, without including the component definition you
    are calling this from.
    """

    name = "walk_subcomponents"

    def __init__(self, definition):
        super(WalkSubComponentsDefinitionPlugin, self).__init__(definition)

    def __call__(self, *args, **kwargs):
        """
        This is the same as walk_components but does not include the component
        definition you are calling this from.

        **See Also:**

            :meth:`walk_components <namednodes.plugins.nn_walk.WalkComponentsDefinitionPlugin.walk_components>`

        """
        kwargs['include_root'] = False
        return super(WalkSubComponentsDefinitionPlugin, self).__call__(
            *args, **kwargs)


##########################
# Walk for nodes
##########################
class WalkNodes(object):
    """
    Walk all nodes in a given component or component definition.
    """

    root = None

    def __init__(self, root):
        """
        Dummy initialization, will be overriden in children
        """
        self.root = weakref.ref(root)

    def _node_iterator(self, algorithm, max_depth):
        """
        Build list of nodes that will be iterated following the
        specified algorithm.
        """
        # Use deque as is more efficient doing appends/pops at position 0
        node_list = deque()
        queue = deque([(self.root(), 0)])

        # While there are nodes in our processing queue...
        while len(queue) > 0:
            current_node, current_level = queue.popleft()
            # Add current node to the end of the list
            # if we are first time through loop,
            # the root 'might' be a component or a node...the calling
            # function should determine that and remove it
            # if current_node is not self.root:
            yield current_node

            # Child's level is parent's level + 1
            child_level = current_level + 1

            # Only include children up to max_depth (negative is infinite)
            if max_depth > -1 and child_level > max_depth:
                continue

            # determine if we have item_definition
            if isinstance(current_node, NamedNodeArrayDefinition):
                more_items = [current_node.item_definition]
            # or it is an array value (yeah..cheating a bit here instead of
            # writing another plugin)
            elif isinstance(current_node, NamedNodeArrayValue):
                more_items = list(current_node)
            else:
                more_items = []

            if algorithm == "depth-first":
                # shallow copy of list
                current_nodes = copy.copy(current_node.nodes)
                current_nodes.reverse()

                more_items.reverse()
                for node in current_nodes:
                    queue.appendleft((node, child_level))
                for item in more_items:
                    queue.appendleft((item, child_level))
            elif algorithm == "breadth-first":
                for item in more_items:
                    queue.append((item, child_level))
                # Add children at the end of the queue
                for node in current_node.nodes:
                    queue.append((node, child_level))
            else:
                # Algorithm not supported
                raise ValueError("Unknown algorithm (%s)" % algorithm)

        return

    def __call__(self, algorithm="depth-first", max_depth=-1, **kwargs):
        """
        Walk all nodes in a given component.

        Example:
            Walk nodes using depth-first algorithm:

                >>> for node in my_comp.walk_nodes():
                >>>     node.<do_something...>

            Walk subcomponents using breadth-first algorithm, up to
            subcomponents 4 levels down:

                >>> for node in \
                            my_comp.walk_nodes("breadth-first", 4):
                >>>     node.<do_something...>

        Args:
            algorithm (Optional[str]): Algorithm used to walk nodes:

                - depth-first (Default).
                - breadth-first.

            max_depth (Optional[int]): Walk nodes up to max_depth
                levels down. Use max_depth < 0 for infinite. Infinite by
                default.

            **kwargs: Arbitrary keyword arguments, it can include any of
                the following arguments:

            include_root (Optional[bool]): Include root node (the one
                calling walk_nodes) when doing the walk. False by
                default.

        Returns:
           Iterable obj.

        """

        valid_algorithms = ["depth-first", "breadth-first"]
        valid_kwargs = ["include_root"]

        # Unpack arguments
        invalid_kwargs = list(set(kwargs.keys()) - set(valid_kwargs))
        include_root = kwargs.pop("include_root", False)
        if len(kwargs) > 0:
            raise ValueError(
                "Invalid arguments (%s). Must be: %s"\
                    % (", ".join(invalid_kwargs),
                       ", ".join(valid_kwargs)))
        assert algorithm in valid_algorithms, \
            "Invalid algorithm (%s). Must be: %s" \
            % (algorithm, ", ".join(valid_algorithms))
        assert isinstance(max_depth, numbers.Number), \
            "Invalid max_depth (%s). Must be a number" % max_depth

        node_iter = self._node_iterator(algorithm, max_depth)

        if isinstance(self.root(), (NamedComponentDefinition, NamedComponent)) or include_root == False:
            next(node_iter)

        for node in node_iter:
            yield node

		
class WalkNodesDefinitionPlugin(ComponentDefinitionPlugin, WalkNodes):
    """
    This plugin adds the capability to walk all nodes in a given
    component definition.
    """
    name = "walk_nodes"

    def __init__(self, root):
        ComponentDefinitionPlugin.__init__(self, root)
        WalkNodes.__init__(self, root)

    def __call__(self, *args, **kwargs):
        return WalkNodes.__call__(self, *args, **kwargs)
    __call__.__doc__ = WalkNodes.__call__.__doc__

class WalkNodesPlugin(ComponentPlugin, WalkNodes):
    """
    This plugin adds the capability to walk all nodes in a given
    component.
    """
    name = "walk_nodes"

    def __init__(self, root):
        ComponentPlugin.__init__(self, root)
        WalkNodes.__init__(self, root)

    def __call__(self, *args, **kwargs):
        return WalkNodes.__call__(self, *args, **kwargs)

    __call__.__doc__ = WalkNodes.__call__.__doc__


class WalkNodesNodeValuePlugin(NodeValuePlugin, WalkNodes):
    """
    This plugin adds the capability to walk all sub nodes of the
    node.
    """
    name = "walk_nodes"

    def __init__(self, nodevalue):
        NodeValuePlugin.__init__(self, nodevalue)
        WalkNodes.__init__(self, nodevalue)

    def __call__(self, *args, **kwargs):
        """walk all the sub nodes of this node"""
        return WalkNodes.__call__(self, *args, **kwargs)

    __call__.__doc__ = WalkNodes.__call__.__doc__