

# INTEL CONFIDENTIAL
# Copyright 2020 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.

from ..logging import getLogger
from ..comp import (
    ComponentPlugin,
    )

_LOG = getLogger()

class BuildSibTree(ComponentPlugin):
    """
    This (hidden) plugin builds the hierarchy of a sib tree    
    """
    name = "_build_sib_tree"
    def __call__(self):
        if 'sib_tree' in self.component.definition.info:
            paths = []
            sib_tree = self.component.definition.info['sib_tree']['sibs']
            for curr_name in sib_tree:
                path = []
                self.build_tree_recursively(sib_tree, curr_name, path)
                path.reverse()
                paths.append(path)
            tree_as_list = self._build_tree_as_list(paths)
            return tree_as_list

    def build_tree_recursively(self, sib_tree, curr_name, path):
        current_sib = sib_tree.get(curr_name, None)
        if current_sib is None:
            return None
        parent_sib = current_sib.get("parent_sib")
        if parent_sib is None:
            path.append(curr_name)
            return None
        else:
            path.append(curr_name)
            self.build_tree_recursively(sib_tree, parent_sib, path)

    def _build_tree_as_list(self, sib_tree):
        final_list = []
        sib_tree.sort()
        for current_list in sib_tree:
            self._transform_recursively(final_list, current_list)
        return final_list

    def _transform_recursively(self, final_list, current_list):
        root = current_list[0]
        if root not in final_list:
            final_list.append(root)
            final_list.append([])
        else:
            tmp_list = final_list[final_list.index(root)+1]
            tmp_list2 = current_list[1:]
            self._transform_recursively(tmp_list, tmp_list2)


class ShowSibTree(ComponentPlugin):
    """
    This plugin displays the sib tree
    on screen    
    """
    name = "show_sib_tree"
    _sib_tree_string = ""

    @classmethod
    def create(cls, component):
        # This plugin is just for components that have sibtrees
        if 'sib_tree' in component.definition.info:
            return cls(component)
        else:
            return None

    def __call__(self, output_format="text", **kwargs):
        if 'sib_tree' in self.component.definition.info:
            tree_as_list = self.component._build_sib_tree()
            self._show_sib(tree_as_list, indentation=0)
            _LOG.result(self._sib_tree_string)
            return self._sib_tree_string

    def _show_sib(self, final_list, indentation=0):
        reg_info_keys = ['numbits']
        field_info_keys = ['bitOffset', 'bitWidth']
        for item in final_list:
            if isinstance(item, str):
                registers = self.component.search_info(sib_name=item)                
                if indentation > 0:
                    self._sib_tree_string = self._sib_tree_string +\
                                            self._create_string_for_tree(string_line=item,\
                                                                         indentation=indentation,
                                                                         adder=0,
                                                                         spacer=r" ",
                                                                         symbol=r"\_",
                                                                         extra=None)
                    for register in registers:
                        definition = self.component.get_node_definition(register)
                        fields = definition.nodes
                        tmp_str = ""
                        reg_size = definition.info.get('numbits', "not_found")
                        tregister = register + " ( Size = {size} )".format(size=reg_size)
                        self._sib_tree_string = self._sib_tree_string +\
                                                self._create_string_for_tree(string_line=tregister,\
                                                                             indentation=indentation,
                                                                             adder=4,
                                                                             spacer=r" ",
                                                                             symbol=r"\_",
                                                                             extra=tmp_str)
                        for field in fields:
                            offset = field.info.get('bitOffset', "not_found")
                            bitwidth = field.info.get('bitWidth', "not_found")
                            the_field = "[{msb},{lsb}]  {name}".format(msb=offset+bitwidth,\
                                                                    lsb=offset, 
                                                                    name=field.name)
                            self._sib_tree_string = self._sib_tree_string +\
                                                    self._create_string_for_tree(string_line=the_field,\
                                                                                 indentation=indentation,
                                                                                 adder=8,
                                                                                 spacer=r" ",
                                                                                 symbol="|.....",
                                                                                 extra="")
                else:
                    self._sib_tree_string = self._sib_tree_string +\
                                            self._create_string_for_tree(string_line=item,\
                                                                         indentation=0,
                                                                         adder=0,
                                                                         spacer=r" ",
                                                                         symbol=r"|",
                                                                         extra=None)
                    for register in registers:
                        definition = self.component.get_node_definition(register)
                        tmp_str = ""
                        for key in reg_info_keys:
                            value = definition.info.get(key, "not_found")
                            tmp_str = tmp_str + "{}:{},".format(key, value)                        
                        self._sib_tree_string = self._sib_tree_string +\
                                                self._create_string_for_tree(string_line=register,\
                                                                             indentation=0,
                                                                             adder=0,
                                                                             spacer=r" ",
                                                                             symbol=r"|",
                                                                             extra=tmp_str)
            elif isinstance(item, list):
                self._show_sib(item, indentation + 4)


    def _create_string_for_tree(self, string_line, indentation, adder, spacer, symbol, extra=None, newline="\n"):
        if extra:
            string_ret = "" + (indentation + adder)*spacer + symbol + string_line +spacer + extra + newline
        else:
            string_ret = "" + (indentation + adder)*spacer + symbol + string_line + newline
        return string_ret

class ShowSibTreeInfo(ComponentPlugin):
    """
    This plugin displays the sib tree
    on screen, it can also display any additional
    field or reg information pass as argument
    Args:
        output_format
        **kwargs
    returns (str) : the string with the full tree and the info.

    """
    name = "show_sib_tree_info"
    _sib_tree_string = ""

    @classmethod
    def create(cls, component):
        # This plugin is just for components that have sibtrees
        if 'sib_tree' in component.definition.info:
            return cls(component)
        else:
            return None

    def __call__(self, output_format="text", **kwargs):
        if 'sib_tree' in self.component.definition.info:
            tree_as_list = self.component._build_sib_tree()
            self._show_sib_info(tree_as_list, indentation=0, **kwargs)
            _LOG.result(self._sib_tree_string)
            return self._sib_tree_string

    def _show_sib_info(self, final_list, indentation=0, **kwargs):
        reg_info_keys = kwargs.pop("reg_info_keys", ['_all_'])
        field_info_keys = kwargs.pop("field_info_keys", ['_all_'])
        next_call_args = {'reg_info_keys':reg_info_keys, 'field_info_keys':field_info_keys}
        for item in final_list:
            if isinstance(item, str):
                registers = self.component.search_info(sib_name=item)                
                if indentation > 0:
                    self._sib_tree_string = self._sib_tree_string +\
                                            self._create_string_for_tree(string_line=item,\
                                                                         indentation=indentation,
                                                                         adder=0,
                                                                         spacer=r" ",
                                                                         symbol=r"\_",
                                                                         extra=None)
                    for register in registers:
                        definition = self.component.get_node_definition(register)
                        fields = definition.nodes
                        self._sib_tree_string = self._sib_tree_string +\
                                                self._create_string_for_tree(string_line=register,\
                                                                             indentation=indentation,
                                                                             adder=4,
                                                                             spacer=r" ",
                                                                             symbol=r"\_",
                                                                             extra="")
                        register_info_keys = None
                        if "_all_" in reg_info_keys:
                            register_info_keys = definition.info
                        else:
                            register_info_keys = reg_info_keys
                        for key in register_info_keys:
                            value = definition.info.get(key, "not_found")
                            str_info = "{key} = {value}".format(key=key, value=value)
                            self._sib_tree_string = self._sib_tree_string +\
                                                    self._create_string_for_tree(string_line=str_info,\
                                                                                    indentation=indentation,
                                                                                    adder=8,
                                                                                    spacer=r" ",
                                                                                    symbol=r"*",
                                                                                    extra="")
                        for field in fields:
                            self._sib_tree_string = self._sib_tree_string +\
                                                    self._create_string_for_tree(string_line=field.name,\
                                                                                 indentation=indentation,
                                                                                 adder=12,
                                                                                 spacer=r" ",
                                                                                 symbol=r"\_",
                                                                                 extra=r"")
                            field_info_keys_m = None
                            if "_all_" in field_info_keys:
                                field_info_keys_m = field.info
                            else:
                                field_info_keys_m = field_info_keys                                                                                 
                            for key in field_info_keys_m:                                
                                value = field.info.get(key, "not_found")
                                str_info = "{key} = {value}".format(key=key, value=value)
                                self._sib_tree_string = self._sib_tree_string +\
                                                        self._create_string_for_tree(string_line=str_info,\
                                                                                     indentation=indentation,
                                                                                     adder=16,
                                                                                     spacer=r" ",
                                                                                     symbol=r"**",
                                                                                     extra=r"")

                else:
                    self._sib_tree_string = self._sib_tree_string +\
                                            self._create_string_for_tree(string_line=item,\
                                                                         indentation=0,
                                                                         adder=0,
                                                                         spacer=r" ",
                                                                         symbol=r"|",
                                                                         extra=None)
                    for register in registers:
                        definition = self.component.get_node_definition(register)
                        tmp_str = ""
                        for key in reg_info_keys:
                            value = definition.info.get(key, "not_found")
                            tmp_str = tmp_str + "{}:{},".format(key, value)                        
                        self._sib_tree_string = self._sib_tree_string +\
                                                self._create_string_for_tree(string_line=register,\
                                                                             indentation=0,
                                                                             adder=0,
                                                                             spacer=r" ",
                                                                             symbol=r"|",
                                                                             extra=tmp_str)
            elif isinstance(item, list):
                next_args = dict(next_call_args)
                self._show_sib_info(item, indentation + 4, **next_args)

    def _create_string_for_tree(self, string_line, indentation, adder, spacer, symbol, extra=None, newline="\n"):
        if extra:
            string_ret = "" + (indentation + adder)*spacer + symbol + string_line +spacer + extra + newline
        else:
            string_ret = "" + (indentation + adder)*spacer + symbol + string_line + newline
        return string_ret

    
                

        

