# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.


from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import re
import six
import collections
import copy

from ..comp import (
    NamedComponent,
    ComponentPlugin,
    ComponentDefinitionPlugin,
    NamedComponentDefinition,
    )
from ..nodes import NodeValuePlugin, NodeTypes, NamedNodeArrayDefinition
from ..registers import RegisterComponent
from ..utils._py2to3 import *
from ..utils.ordereddict import odict
from ..logging import getLogger
from ..telemetry import record_event
from .. import settings
from . import lmdb_dict
import warnings

_LOG = getLogger()


def _expand_arrays(node, parts):
    if len(parts) == 0:
        return []
    pos = parts[0].find("[")
    if pos == -1:
        # not at this part, keep walking
        if len(parts)>1:
            next_node = node.get_node(parts[0])
            next_paths = _expand_arrays(next_node, parts[1:])
            next_paths = [f"{parts[0]}.{p}" for p in next_paths ]
            return next_paths
        else:
            return [ parts[0] ]
    else:
        node_name = parts[0][:pos]
        next_node = node.get_node(node_name)
        if isinstance(next_node, NamedNodeArrayDefinition):
            item_keys = next_node.item_keys
        else:
            raise RuntimeError("unexpected component found")
        # get subpaths, and then add in our key...
        children_paths = _expand_arrays(next_node.item_definition, parts[1:])
        next_paths = []
        for item in item_keys:
            if isinstance(item, str):
                prefix = f"{node_name}[\"{item}\"]"
            else:
                prefix = f"{node_name}[{item}]"
            next_paths.append(prefix)
            for child in children_paths:
                next_paths.append(f"{prefix}.{child}")

        return next_paths


def _search_nodes(component, path, find, **kwargs):
    """Look through nodes matching the requested path

    Args:
        path (str) : regular expression used to match against the
                            paths in this component
        find (str) : what to search for:

                - 'n' / 'nodes' - search for nodes matching this path

        comp_recursive (bool, optional) : whether to search sub components (default: True)
        node_recursive (bool, optional) : whether to search sub nodes (default: True)
        expand_arrays (bool, optional) : whether to expand arrays during search matching
        getobj (bool, optional) : whether to return the object instead
                                    of the paths (default: False)

    Returns:
        list of strings or objects that match the specified regex

    """
    comp_recursive = kwargs.pop("comp_recursive", True)
    node_recursive = kwargs.pop("node_recursive", True)
    expand_arrays = kwargs.pop("expand_arrays", True)
    getobj = kwargs.pop("getobj", False)
    if find not in ['n', 'nodes']:
        raise ValueError("Unrecognized find parameter: {}".format(find))
    if not len(kwargs) == 0:
        raise ValueError("Unrecognized kwargs: {}".format(str(list(kwargs.keys()))))

    if not isinstance(path, six.string_types):
        raise TypeError("Only string support for path for now")

    if isinstance(component, NamedComponent) and getobj and not expand_arrays:
        raise ValueError("You must use expand_arrays=True when searching in order for gteobj to reliably work")

    if isinstance(path, six.string_types):
        # if path is a string we are going to need ALL the paths to check
        # against
        all_paths = []
        startpath = component.path
        if comp_recursive:
            # it is ok we are just returning the iterator here since
            # it is just used in loops below
            comps = component.walk_components()
        else:
            comps = [component]

        array_regex = re.compile(r"\[.*?\]")

        if find in ['n', 'nodes']:
            for comp in comps:
                # need to determinve our 'relative path' from starting point
                if comp.path != startpath:
                    rpath = comp.path.replace(startpath + ".", "") + "."
                else:
                    rpath = ""
                # LMDB knowledge...
                if isinstance(comp, NamedComponentDefinition):
                    definition = comp
                else:
                    definition = comp.definition

                all_node_paths = getattr(definition, "_all_node_paths", [])
                if all_node_paths != []:
                    # if we are not in definition and we should expand arrays, then
                    # do that before doing recusive jazz
                    #if False:
                    temp_paths = []
                    if isinstance(comp, NamedComponent) and expand_arrays:
                        for node_path in all_node_paths:
                            # if any part of path has an array we need to do some walking..
                            if node_path.count("["):
                                array_paths = _expand_arrays(comp.definition, node_path.split("."))                                
                                temp_paths.extend(array_paths)
                            else:
                                temp_paths.append(node_path)
                    else:
                        # no array expansion, just use the paths we have
                        temp_paths = copy.copy(all_node_paths)

                    seen_set = set()
                    temp_paths = [p for p in temp_paths if not (p in seen_set or seen_set.add(p))]
                    del seen_set

                    if comp_recursive and node_recursive:
                        # only add rpath in if we were comp_recursive, otherwise it is empty
                        # if node_recursive, we need full path                        
                        all_paths.extend([rpath + p for p in temp_paths])
                        # these are bit arrays, so we agressively delete them when they are no longer needed
                    elif node_recursive:
                        all_paths.extend(temp_paths)
                    else:
                        # not recursive at all, add only paths with the top level node
                        for p in temp_paths:
                            p = p.split(".")
                            if len(p) > 1:
                                continue
                            else:
                                all_paths.extend(p)
                    del temp_paths
                else:
                    # not starting with any node paths:
                    # have to build node paths
                    if comp_recursive and node_recursive:
                        # only add rpath in if we were comp_recursive, otherwise it is empty
                        next_paths = [rpath + node.nodepath for node in comp.walk_nodes()]
                    elif node_recursive:
                        next_paths = [node.nodepath for node in comp.walk_nodes()]
                    else:
                        # not recursive at all
                        next_paths = [node.name for node in comp.iter_nodes()]
                    if not expand_arrays:
                        next_paths = [array_regex.sub("[#]", p) for p in next_paths]
                        # make sure its unique--this does cost some memory...
                        seen_set = set()
                        next_paths = [p for p in next_paths if not (p in seen_set or seen_set.add(p))]
                        del seen_set
                    all_paths.extend(next_paths)
                    del next_paths

        elif find in ['c', 'comps']:
            raise ValueError("Not supported yet")

        # SEARCH BY STRING
        path_re = re.compile(path)
        matches = [p for p in all_paths if path_re.search(p)]
        if getobj:
            matches = [component.get_by_path(p) for p in matches]
        return matches
    else:
        raise Exception("not supported yet")


#############
# Part of the reason we make sure that SearchNodes and SearchRegisters
# both inherit from ComponentPlugin, is to help us out with our
# methodology for showing the "help" in our documentation. So even
# that is really the only reason we do the inheritance here because
# SearchPlugin will be the *actual* plugin that the user gets
# and it creates the correct plugin based on the component type
#############


search_nodes_docstring = \
    """
    Look through nodes matching the requested path

    Args:
        path (str) : regular expression used to match against the
                            paths in this component
        find (str) : what to search for:

                - 'n' / 'nodes' - search for nodes matching this path

        recursive (bool, optional) : whether to search sub components and nodes
        getobj (bool, optional) : whether to return the object instead
                                    of the paths

    Returns:
        list of strings or objects that match the specified regex
    """


class SearchNodes(ComponentPlugin):
    """
    This plugin adds the capability to search for any
    NodeComponent that is not a RegisterComponent
    """
    name = "search"

    def __call__(self, path, find='nodes', **kwargs):
        # see search nodes for complete docs
        recursive = kwargs.pop("recursive", None)
        if recursive is None:
            # recursive not specified use what they gave or default
            kwargs['comp_recursive'] = kwargs.get('comp_recursive', True)
            kwargs['node_recursive'] = kwargs.get('node_recursive', False)
        else:
            kwargs['comp_recursive'] = kwargs['node_recursive'] = recursive
        return _search_nodes(self.component, path, find, **kwargs)

    __call__.__doc__ = search_nodes_docstring


class SearchDefinitionNodes(ComponentDefinitionPlugin):
    """
    Plugin that leverages node value search to provide search on definitions
    """
    name = "search"

    def __call__(self, path, find='nodes', **kwargs):
        # depends on if this is for registers or general AFD
        recursive = kwargs.pop("recursive", True)
        kwargs['comp_recursive'] = kwargs['node_recursive'] = recursive
        return _search_nodes(self.definition, path, find, **kwargs)

    __call__.__doc__ = _search_nodes.__doc__


class SearchRegisters(ComponentPlugin):
    """
    This plugin adds the capability to search for registers
    and fields in a RegisterComponent.
    """
    name = "search"

    def __call__(self, regexpression, searchType='registers', *,
                 recursive=True, getobj=False, expand_arrays=True, **kwargs):
        """
        Look through the register (or field) list for any registers/fields
        that match the given regular expression and return a list of the
        matching registers.

        Args:
            regexpresion (type): Regular expression to use when searching OR
                offset (number) to use when searching by offset.
            searchType (string, optional):  What to search for, options are:

                    - registers or **r**: Search register names (default).
                    - fields or **f**: Search field names.
                    - description or **d**: Search register and field
                        descriptions.
                    - path or **p**: Search register names, including
                        subcomponent paths.
                    - register_paths or **rp**: Search by path and
                        return only registers
                    - field_paths or **fp**: Search by path and
                        return only fields
                    - component_paths or **cp**: Search by path and
                        return only components, expression can be
                        regexpression or a list of regexpressions that will be
                        processed at each level
                    - comp or **c**: Search subcomponents.
                    - **rf**: Search for [register,field], requires
                        regexpression to be a list.
                    - offset or **o**: Search all registers for some sort
                        of offset that matches the one specified for more
                        accurate address search use the searchaddress function.

            recursive (bool, optional): Set to False to disable the recursive
                search. This is supported to provide some functionality used by
                coders. Defaults to True.
            getobj (bool, optional): Set to True to return path to objects and
                actual register/field objects. This one is supported to provide
                some functionality used by coders. Defaults to False.
            expand_arrays (bool): (default: True) whether to expand arrays when searching for fields
        """
        ### Undocumented parameters to help with search by path that returns
        ### only registers or fields
        ### _start_path = path to remove from current path before doing regex matching against...

        known_args = ["_start_path"]
        for k in kwargs.keys():
            assert k in known_args, "Unknown option: %s" % k
        finds = []
        assert searchType in [
            'registers', 'r',
            'fields', 'f',
            'description', 'd',
            'path', 'p',
            'offset', 'o',
            'rf',
            'comp', 'c',
            'register_paths', 'rp',
            'field_paths', 'fp',
            'component_paths', 'cp',
            ], "Invalid searchType: %s" % searchType

        if isinstance(self.component, NamedComponent) and getobj and not expand_arrays:
            raise ValueError("You must use expand_arrays=True when searching in order for gteobj to reliably work")

        if searchType in ['rf']:
            is_expression_a_list = isinstance(regexpression, list)
            assert (is_expression_a_list and len(regexpression) == 2), \
                (
                    'rf search type requires ["register","field"] as '
                    'regular expression'
                )
            regpattern = re.compile(regexpression[0], re.IGNORECASE)
            fieldpattern = re.compile(regexpression[1], re.IGNORECASE)
        elif searchType in ['o', 'offset']:
            offset = regexpression
            if isinstance(offset, six.string_types):
                raise Exception(
                    "Must specify number when searching by offset"
                )
            warnings.warn("\n\tWARNING: searching by offset is a legacy option and unreliable search_info should be used instead")
        elif isinstance(regexpression, list) and searchType in ['component_paths', 'cp']:
            # expect this to also be for registerpaths/fieldpaths one day
            assert len(regexpression) > 0, "Empty list given for search, not supported"
            # only compile this "level" we'll compile at each level
            regpattern = re.compile(regexpression[0], re.IGNORECASE)
        else:
            # check for string?
            regpattern = re.compile(regexpression, re.IGNORECASE)

        # if path not given, then the default is this path...
        # this does NOT get modified
        start_path = kwargs.get("_start_path", self.component.path)

        registers = []
        # fields dictionary has:
        #  key = register_name
        #  value = array of register's field names
        fields = odict()
        # if we have paths, use them to build our register/fields
        # if the paths are there but empty, then use them...in case they
        # are wrong
        all_node_paths = getattr(self.component.definition, "_all_node_paths", [])
        requires_fields = searchType in ['rf', 'f', 'fields', 'fp']
        if all_node_paths != []:
            # assume we have all registers/fields (?)
            for p in all_node_paths:
                parts = p.split(".")
                sub_parts = parts[0].split("[")
                register_name = sub_parts[0]
                if register_name in self.component._removed_nodes:
                    continue
                # just a register, we should ALWAYS hit this before next hitting
                # the register + field info. We also always hit this before hiting
                # arrays of registers
                if len(parts) == 1 and len(sub_parts) == 1:
                    # we may need to make "removed_nodes" a public interface since
                    # we are using it here...
                    registers.append(register_name)
                    # i experimented with skipping this on register/field search types, but it had little performance impact
                    # on the one or two use cases at least...
                    if requires_fields:
                        fields[register_name] = []  # create list to eventually hold field names
                # register and field (assumes we already added register)
                # we don't need fields unless searching for them, and this simple append is apparently time sensitive
                # also...not an array
                elif len(parts) == 2 and requires_fields and len(sub_parts) == 1:
                    field_name = parts[1]
                    fieldlist = fields.get(register_name)
                    if fieldlist is not None:
                        fields[register_name].append(field_name)
                    # if fieldlist is None, then there's a register with a "." in the name, and
                    # we cant support that OR it means the register was removed....
                elif len(parts) == 2 and requires_fields and len(sub_parts) > 1:
                    field_name = parts[1]
                    # unfortunately we HAVE to get the array to get it's keys
                    array = self.component.definition.get_node(register_name)
                    for key in array.item_keys:
                        if isinstance(key, str):
                            fields.setdefault(f"{register_name}[\"{key}\"]", []).append(field_name)
                        else:
                            fields.setdefault(f"{register_name}[{key}]", []).append(field_name)
                else:
                    # not sure how to handle currently, should have only gotten
                    # registers and fields
                    continue
        else:
            for node in self.component.definition.nodes:
                if node.name in self.component._removed_nodes:
                    continue
                if node.type == NodeTypes.Array:
                    registers.append(node.name)
                    for _index in node.item_keys:
                        if isinstance(_index, str):
                            node_name = node.name+f"[\"{_index}\"]"
                        else:
                            node_name = node.name + f"[{_index}]"
                        if requires_fields:
                            fields[node_name]=node.item_definition.fields
                else:
                    registers.append(node.name)
                    # store both field and register name, this is key
                    # when searching fields and field object is requested
                    if requires_fields:
                        fields[node.name] = node.nodenames

        # if expand arrays was false, remove all registers and fields with nodenams
        if not expand_arrays:
            regex = re.compile(r"\[.*?\]")  # , "\[#\]", )
            registers = [regex.sub("[#]", p) for p in registers]
            fields = {regex.sub("[#]", k):v for k,v in fields.items()}
            # make sure its unique--this does cost some memory...
            seen_set = set()
            registers = [p for p in registers if not (p in seen_set or seen_set.add(p))]
            del seen_set

        if searchType in ['registers', 'r', 'rf']:
            register_object = None
            for register_name in registers:
                match = regpattern.search(register_name)
                if match is not None:
                    # Found a register, now look for fields
                    if searchType in ['rf']:
                        for field_parent, field_names in fields.items():
                            # Skip field if it does not belong to current
                            # register
                            if register_name != field_parent:
                                continue

                            for field_name in field_names:
                                match = fieldpattern.search(field_name)
                                if match is not None:
                                    field_path = \
                                        "%s.%s" % (register_name, field_name)
                                    if not getobj:
                                        finds.append(field_path)
                                    else:
                                        # Get the register object if we haven't
                                        # done so already, or if we are looking
                                        # into fields from a different register
                                        not_valid_object = \
                                            register_object is None
                                        if not_valid_object or \
                                           register_object.name != register_name:
                                                register_object = \
                                                    self.component.get_node(
                                                        register_name
                                                    )
                                        field_object = register_object.get_node(
                                            field_name
                                        )
                                        finds.append([field_path, field_object])
                    else:
                        if not getobj:
                            finds.append(register_name)
                        else:
                            register_object = self.component.get_node(
                                register_name
                            )
                            finds.append([register_name, register_object])

        elif searchType in ['fields', 'f']:
            register_object = None
            for field_parent, field_names in fields.items():
                for field_name in field_names:
                    match = regpattern.search(field_name)
                    if match is not None:
                        field_path = "%s.%s" % (field_parent, field_name)
                        if not getobj:
                            finds.append(field_path)
                        else:
                            # Get the register object if we haven't done so
                            # already, or if we are looking into fields from
                            # a different register
                            not_valid_object = register_object is None
                            if not_valid_object or \
                               register_object.name != field_parent:
                                    register_object = self.component.get_node(
                                        field_parent
                                    )
                            field_object = register_object.get_node(field_name)
                            finds.append([field_path, field_object])

        elif searchType in ['offset', 'o']:
            for register_name in registers:
                register_object = self.component.get_node(register_name)
                register_offset = register_object.definition.info.get(
                    'offset', None
                )
                if offset == register_offset:
                    if not getobj:
                        finds.append(register_name)
                    else:
                        finds.append([register_name, register_object])

        elif searchType in ['path', 'p']:
            # for searching by path, just use our other shared function
            all_paths = _search_nodes(self.component,
                          regexpression,    # pass on original expression
                          'nodes',
                          node_recursive=True,
                          comp_recursive=recursive,
                          expand_arrays=True,
                          getobj=getobj)
            if getobj:
                remove = len(self.component.path)
                all_paths = [(p.path[remove + 1:], p) for p in all_paths]
                return all_paths
            else:
                return all_paths

        elif searchType in ['register_paths', 'rp']:
            match_path = self.component.path.replace(start_path + ".", "") + "."
            for register_name in registers:
                rpath = match_path + register_name
                match = regpattern.search(rpath)
                if match is not None:
                    # only append register name, recursive part will add in the path...
                    if getobj:
                        finds.append([register_name, self.component.get_node(register_name)])
                    else:
                        finds.append(register_name)

        elif searchType in ['field_paths', 'fp']:
            if start_path is not None:
                # removes leading ".", adds trailing "."
                match_path = self.component.path.replace(start_path + ".", "") + "."
            else:
                match_path = ""
            for register_name, field_names in fields.items():
                for fname in field_names:
                    fpath = register_name + "." + fname
                    tpath = match_path + fpath  # tpath = total path
                    match = regpattern.search(tpath)
                    if match is not None:
                        # only append register name, recursive part will add in the path...
                        if getobj:
                            finds.append([fpath, self.component.get_node(register_name).get_node(fname)])
                        else:
                            finds.append(fpath)

        elif (searchType in ['description', 'd']):
            for register_name in registers:
                # First look for registers
                register_object = self.component.get_node(register_name)
                description = register_object.definition.info.get(
                    'description', None
                )
                # this will make sure its not None AND that it is not a number or something else not-compatible with re
                if isinstance(description, six.string_types):
                    match = regpattern.search(description)
                    if match is not None:
                        if getobj:
                            finds.append([register_name, register_object])
                        else:
                            start_pos = match.start() - 15
                            if start_pos < 0:
                                start_pos = 0
                            end_pos = match.end() + 15
                            description_string = description[start_pos:end_pos]
                            finds.append(
                                (
                                    register_name + " \"...%s...\""
                                ) % description_string
                            )
                # And now look in the register's fields
                for field_data in register_object.nodes:
                    description = field_data.definition.info.get(
                        'description', None
                    )
                    # this will make sure its not None AND that it is not a number or something else not-compatible with re
                    if isinstance(description, six.string_types):
                        match = regpattern.search(description)
                        if match is not None:
                            if getobj:
                                path = "%s.%s" % (
                                    register_name, field_data.name
                                )
                                finds.append([path, field_data])
                            else:
                                start_pos = match.start() - 15
                                if start_pos < 0:
                                    start_pos = 0
                                end_pos = match.end() + 15
                                description_string = \
                                    description[start_pos:end_pos]
                                finds.append(
                                    (
                                        register_name + "." +
                                        field_data.name +
                                        " \"...%s...\""
                                    ) % description_string
                                )

        # search on subcomponents, if requested
        if recursive or searchType in ['comp', 'c', 'components_paths', 'cp']:
            # if in component path searching, setup for what we will will search for next, as being a bit differnt
            if searchType in ['cp', 'component_paths'] and isinstance(regexpression, list):
                next_regexpression = regexpression[1:]
            else:
                next_regexpression = regexpression

            for subcomp_name, subcomp in self.component.sub_components.items():
                if searchType != 'p' and not isinstance(subcomp, RegisterComponent):
                    _LOG.error("SKIPPING '%s' since it is not a RegisterComponent and may not have registers..."%subcomp.path)
                    continue

                # if search type simple components OR we are walking through a list of regexpressions
                # either way compare this component against the current regex
                if searchType in ['comp', 'c']:
                    match = regpattern.search(subcomp_name)
                    if match is not None:
                        if not getobj:
                            finds.append(subcomp_name)
                        else:
                            finds.append([subcomp_name, subcomp])

                elif searchType in ['component_paths', 'cp'] and isinstance(regexpression, list):
                    match = regpattern.search(subcomp_name)
                    # if we don't match, make sure we dont recurse in to this component
                    if match is None:
                        continue
                    # match must be there, so ok to recurse...but..if this is the "end of the list"
                    # then we just need to add to the finds, no need to recurse....
                    if len(regexpression) == 1:
                        if not getobj:
                            finds.append(subcomp_name)
                        else:
                            finds.append([subcomp_name, subcomp])
                        # no recurursion left...
                        continue

                elif searchType in ['component_paths', 'cp']:  # and therefore NOT a list of regexpressions
                    if start_path is not None:
                        # removes leading ".", adds trailing "."
                        match_path = self.component.path.replace(start_path + ".", "") + "."
                    else:
                        match_path = ""
                    match = regpattern.search(match_path + subcomp_name)
                    if match is not None:
                        if not getobj:
                            finds.append(subcomp_name)
                        else:
                            finds.append([subcomp_name, subcomp])

                # probably rare/unlikely we get called with recursive=False, and searchType component
                # related, but just in case
                if recursive:
                    subcomponent_finds = subcomp.search(
                        next_regexpression,
                        searchType,
                        recursive=recursive,
                        getobj=getobj,
                        _start_path=start_path,
                    )
                    # pre-pend the component name to the results
                    if not getobj:
                        finds.extend(["%s.%s" % (subcomp_name, f) for f in subcomponent_finds])
                    else:
                        # finds has object and path...
                        finds.extend(
                            [("%s.%s" % (subcomp_name, f[0]), f[1]) for f in subcomponent_finds]
                        )
        # Doing this to log a single event
        # and not one per find
        if len(finds) and start_path == self.component.path:
            _telemetry_match_wrapper("search_register_component", kwargs)
        return finds


def _telemetry_match_wrapper(lr_event_name, kwargs):
    """ A simple wrapper to send events (search matches parameters)
    to lanternrock"""
    if settings.EVENTS_ON_SEARCH:
        black_list = ["_start_path", "regexpression"]
        telemetry_dict = {}
        # For some reason, the logger fails if the keys
        # of the dict start with _
        for search_arg in kwargs:
            if search_arg not in black_list:
                if search_arg.startswith("_"):
                    key_strip = search_arg.replace("_", "")
                else:
                    key_strip = search_arg
                telemetry_dict[key_strip] = kwargs[search_arg]
        record_event(lr_event_name, telemetry_dict)


# this needs to be after the SearchRegisters and SearchNodes
# so that it is registers as THE search class
class SearchPlugin(ComponentPlugin):
    name = "search"

    @classmethod
    def create(cls, component):
        # This plugin should not be used for RegisterComponent
        # leave that for the backwards compatible search
        if isinstance(component, RegisterComponent):
            return SearchRegisters(component)
        else:
            return SearchNodes(component)


class SearchComponentInfoPlugin(ComponentPlugin):
    """
    This plugin adds the capability to search for info attributes in all
    of a component  and its subcomponents.
    """

    name = "search_component_info"

    def __call__(self, **kwargs):
        """
        search for info attributes in all of a component and its subcomponents.

        If you want to search all components that met the following criteria :
        type = 'mem'  and bar = '.*ubox\.ncdecs.*" you would call it like this
        component.search_component_info(bar = ".*ubox\.ncdecs.*", type = 'mem')

        If you want to get a list of components whose bus equals 0, and only that
        then you would :component.search_component_info(bus = 0)

        If you want it to be a little more flexible, like getting a list of
        components whose buses are either 0, 3 and 30, then you can :
        component.search_component_info(bus = [0,3,30])

        Args:
            getobj (bool, optional): Used to determine whether to return
                the object instead of the name/path. Defaults to False.
            getpath (bool, optional): Used to get the entire path to the node.
                Defaults to False.
            startpath (str, optional): Used to keep up with initial path so we
                returned a relative path. Defaults to empty string.
            firstfind (bool, optional): Used to stop the search after the first
                find. Defaults to False.
            recursive (bool, optional): Used to determine whether we search
                through subcomponents. Defaults to True.
            isregex (bool, optional): Used to determine whether we should
                attempt to perform a regex-based search on the arguments.
                Defaults to True.
        """
        getobj = kwargs.pop('getobj', False)
        getpath = kwargs.pop('getpath', False)
        startpath = kwargs.pop('startpath', "")
        firstfind = kwargs.pop('firstfind', False)
        recursive = kwargs.pop('recursive', True)
        isregex = kwargs.pop('isregex', True)
        # Intentionally avoid performing a check on empty kwargs since all
        # other args that remain there are actually used for performing the
        # search.

        # ok now we have tested for valid args, we have to build it back for
        # our recursive calls
        nextkwargs = {
            "getobj": getobj,
            "getpath": getpath,
            "startpath": startpath,
            "firstfind": firstfind
        }
        if startpath == "":
            startpath = self.component.path + "."
        matchlist = []
        testkeys = kwargs.keys()
        check_dict = odict()
        for tkey in testkeys:  # This is a scoreboard. Needed to make the search
            check_dict[tkey] = 0  # work like an 'AND' of conditions.
        for tkey in testkeys:
            found = [False]
            val_of_dict = kwargs[tkey]
            if isinstance(val_of_dict, list):
                self._bruteforce_info_dict(self.component.definition.info, tkey, val_of_dict, found, "list")
            elif isinstance(val_of_dict, str):
                regkey = re.compile(tkey, re.IGNORECASE)
                regval = re.compile(kwargs[tkey], re.IGNORECASE)
                self._bruteforce_info_dict(self.component.definition.info, regkey, regval, found, "regexp")
            else:  # using the values 'as is'. No regexp, no nothing.
                self._bruteforce_info_dict(self.component.definition.info, tkey, val_of_dict, found, "asis")
            if found[0]:
                check_dict[tkey] = 1  # Scoreboard set to 1 for this specific key
        # Now let's do the AND stuff
        found_all = True  # Assuming we found all the items
        for tkey, val in check_dict.items():
            if val == 0:  # Using the scoreboard undo the previous asumption
                found_all = False
        if found_all:
            matchlist += self._add_component_to_match(getobj, getpath, startpath)

        if recursive is False:
            return matchlist
        # Search in the subcomponents as well(this is actually the most common case)
        subcomponents_matches = []
        sub_components = self.component.sub_components
        for component_name, component in sub_components.items():
            allargs = dict(nextkwargs)
            allargs.update(kwargs)
            subcomponents_matches = component.search_component_info(**allargs)
            matchlist.extend(subcomponents_matches)
        return matchlist

    def _bruteforce_info_dict(self, dict_to_traverse, regkey, regval, found, search_type):
        """
        dict_to_traverse : dictionary (with probably more dictionaries in it)
        regkey : one of the search/match parameters, it should match to a dictionary key
        regval : The other search/match parameter, it should match the content of the dictionary
        found : Boolean, to indicate if the item was found or not
        search_type : regexp, list or as-is.
        *Regexp* means that regkey and regval will be interpreted as regular expressions.
        *List* means that the items in the list will be used as a match criteria against the contents
        of each individual entry in the dictionary.
        *As-is* means that the value given by the user will be used as is and will be matched directly
        with a comparisson against the contents of the dict

        """
        matchkey = False
        matchval = False
        for key, val in dict_to_traverse.items():
            # ignore 'priviate' keys
            if isinstance(key, six.string_types) and key.startswith("_"):
                continue
            if search_type == "regexp" and isinstance(key, six.string_types):
                matchkey = regkey.search(key)
                matchval = regval.search(str(val))
            elif search_type == "list":
                if regkey == key:
                    for or_key in regval:  # any match will do
                        if or_key == val:
                              matchkey = True
                              matchval = True
            else:
                if regkey == key and val == regval:
                    matchkey = True
                    matchval = True

            if matchkey and matchval:
                found[0] = True
            if isinstance(val, dict) or isinstance(val, odict):
                self._bruteforce_info_dict(val, regkey, regval, found, search_type)

    def _add_component_to_match(self, getobj, getpath, startpath):
        """handles how the component is added to the found list
        it can be as string(trimmed or not) or as an object"""
        matchlist = []
        if getobj:
            matchlist.append(self.component)
        elif getpath:
            matchlist.append(self.component.path)
        else:  # default is relative path
            matchlist.append(self.component.path.replace(startpath, ""))
        return matchlist


class SearchInfoPlugin(ComponentPlugin):
    """
    This plugin adds the capability to search for info attributes in all
    of a component's nodes.
    """
    name = "search_info"

    def __call__(self, **kwargs):
        """
        This function provides the "search_info" command to NamedComponent
        nodes.

        This script searches all the component's nodes and looks for the info
        that has attributes matching what is passed in to the kwargs.
        if a list, is given for an attribute then the info will be considered
        matched if it is in that list.

        Some examples::

            # matches if node's info has device=2, offset=0x4, assumes other
            # parameters do not matter
            search_info(device=2,offset=0x4)
            # matches to device 2, considers a match if offset is 0x4 or 0x8
            search_info(device=2,offset=[0x4,0x8])
            # matches to all nodes where device is 2
            search_info(device=2)
            # matches all nodes with offset 0x4
            search_info(offset=0x4)
            # matches only nodes with constant access and offset 0x4
            search_info(access="constant",offset=0x4)

        Currently the kwords must exactly match the attributes of the info
        object, so "offset" will NOT match "msr_offset" or "cr_offset"

        The value for the key can also be a function that should return True/False. It
        will be passed the value of the key found on the node OR the value will be
        the LookupError class if the node did not have that key. As an exmple, you can find
        all registers with an msr_offset using::

            comp.search_info(msr_offset=lambda x: x is not LookupError)

        Other special keyword arguments (processed in this order--dont specify
        multiple):

        Args:
            getobj (bool, optional): Used to determine whether to return
                the object instead of the name/path. Defaults to False.
            getpath (bool, optional): Used to get the entire path to the node.
                Defaults to False.
            startpath (str, optional): Used to keep up with initial path so we
                returned a relative path. Defaults to empty string.
            firstfind (bool, optional): Used to stop the search after the first
                find. Defaults to False.
            recursive (bool, optional): Used to determine whether we search
                through subcomponents. Defaults to True.
            isregex (bool, optional): Used to determine whether we should
                attempt to perform a regex-based search on the arguments.
                Defaults to True.
            skipfields (bool, optional): Skip fields when searching. Defaults to
                True.
            db_in_memory(bool, optional): If True it uses the CacheControl functionality from the DB.
            Defaults to False (to reduce the memory consumption)
        """
        getobj = kwargs.pop('getobj', False)
        getpath = kwargs.pop('getpath', False)
        startpath = kwargs.pop('startpath', "")
        firstfind = kwargs.pop('firstfind', False)
        recursive = kwargs.pop('recursive', True)
        isregex = kwargs.pop('isregex', True)
        skipfields = kwargs.pop('skipfields', True)
        info_only = kwargs.pop('info_only', False)
        db_in_memory = kwargs.pop('db_in_memory', False)
        # Intentionally avoid performing a check on empty kwargs since all
        # other args that remain there are actually used for performing the
        # search.

        if startpath == "":
            startpath = self.component.path + "."

        # ok now we have tested for valid args, we have to build it back for
        # our recursive calls
        nextkwargs = {
            "getobj": getobj,
            "getpath": getpath,
            "startpath": startpath,
            "firstfind": firstfind,
            "info_only": info_only,
            # put ALL the kwargs back in
            "isregex": isregex,
            "skipfields": skipfields,
            'db_in_memory': db_in_memory,
        }
        # whats left of kwargs becomes our search criteria
        search_criteria = kwargs

        matchlist = []
        # check search cache first....
        with lmdb_dict.CacheContext(db_in_memory):
            matches = self._search_info_by_cache(search_criteria, nextkwargs)
            # watch out for special case, if first find, return just one entry
            if matches and firstfind:
                return matches[0:1]
            elif matches:  # if matches is not None and not empty
                matchlist.extend(matches)
            else:
                # used for storing the regex compiled version of the input values if
                # they are a string
                regular_expressions = {}
                for key in search_criteria:
                    regular_expressions[key] = None
                    if isregex and isinstance(search_criteria[key], six.string_types):
                        regular_expressions[key] = re.compile(search_criteria[key])

                for child in self.component.definition.walk_nodes():
                    if skipfields and child.type == NodeTypes.Field:
                        continue
                    number_of_key_matches = 0
                    for key in search_criteria:
                        value = child.info.get(key, LookupError)
                        if value is LookupError:
                            if info_only is False:
                                value = getattr(child, key, LookupError)

                        # if a function is passed in call that passing in the value to check
                        if callable(search_criteria[key]):
                            try:
                                number_of_key_matches += int(search_criteria[key](value))
                            except Exception as e:
                                message_error = "Exception {} occured while checking {} against {}".format(str(value), key, self.component.path)
                                if message_error not in matchlist:
                                    matchlist.append(message_error)
                                    print(message_error)
                        elif value is LookupError:
                            # key is not in the node's definition info
                            # but maybe we are looking for accesses
                            if key in ['type', 'access']:
                                if search_criteria[key] in child.accesses:
                                    # this attribute matched
                                    number_of_key_matches += 1
                            else:
                                # skipping since we could not find the key
                                continue
                        else:
                            if (value == search_criteria[key] or
                                    (isinstance(search_criteria[key], list) and value in search_criteria[key]) or
                                    (regular_expressions[key] and value is not None and regular_expressions[key].match(value))):
                                # this attribute matched
                                number_of_key_matches += 1

                    # all attributes matched...
                    if number_of_key_matches == len(search_criteria):
                        # Need to handle fields in register arrays differently
                        if child.type == NodeTypes.Field and child.parent.parent is not None:
                            array = child.parent.parent
                            paths = []
                            for item_key in array.item_keys:
                                if isinstance(item_key, six.string_types):
                                    paths.append(array.name + "[\"%s\"]" % item_key + ".%s" % child.name)
                                else:
                                    paths.append(array.name + "[%s]" % str(item_key) + ".%s" % child.name)
                        else:
                            paths = [child.nodepath if child.type == NodeTypes.Field else child.name]
                        for path in paths:
                            if getobj:
                                matchlist.append(self.component.getbypath(path))
                            elif getpath:
                                path = ".".join([self.component.path, path])
                                matchlist.append(path)
                            else:  # default is relative path
                                path = ".".join([self.component.path, path])
                                path = path.replace(startpath, "")
                                matchlist.append(path)
                            if len(matchlist) == 1 and firstfind:
                                # caught our first find, we are done here
                                return matchlist

            # if not recursive, we are finished
            if not recursive:
                return matchlist

            sub_components = self.component.sub_components
            for component_name, component in sub_components.items():
                # add back in our parameters
                allargs = dict(nextkwargs)
                allargs.update(kwargs)
                sub_component_matchlist = component.search_info(**allargs)
                # if getobj=True, set matchlist to the objects returned
                if allargs['getobj'] == True:
                    matchlist.extend(sub_component_matchlist)
                else:
                    for match in sub_component_matchlist:
                        # Need to make this check, otherwise we end up
                        # having some weird path with the subcomponent
                        # name preceding the parent component name.
                        # if not getpath:
                        #    matchlist.append(component_name + "." + match)
                        # else:
                        matchlist.append(match)
                # check whether we should stop on first find
                if len(matchlist) == 1 and firstfind:
                    return matchlist
            return matchlist

    def _search_info_by_cache(self, search_criteria, search_kwargs):
        """
        Returns
            - list of keys left
            - list of matches
        """
        # import pdb;pdb.set_trace()
        # quickly check if we even have a cache...
        _search_info_cache = self.component.definition.info.get("_search_info_cache", None)
        if _search_info_cache is None:
            return None
        # make sure all the requested keys are present...
        for key in search_criteria:
            if key not in _search_info_cache:
                return None

        # used for storing the regex compiled version of the input values if
        # they are a string
        regular_expressions = {}
        for key in search_criteria.keys():
            regular_expressions[key] = None
            if search_kwargs['isregex'] and isinstance(search_criteria[key], six.string_types):
                regular_expressions[key] = re.compile(search_criteria[key])

        # keep a list of all the matches, later we'll go back
        # and use set logic to make sure we have a unique list...
        criteria_matches = []
        keys_found = set()
        for key in search_criteria.keys():
            matches = []  # default is to append empty list if we dont find anything
            if key in _search_info_cache:
                keys_found.add(key)
                for value in _search_info_cache[key]:
                    # if a function...
                    if callable(search_criteria[key]):
                        matched = search_criteria[key](value)
                        if matched:
                            matches = _search_info_cache[key][value]
                    elif (value is LookupError):
                            continue                            
                        # if a regular expression
                    elif (value == search_criteria[key] or
                            (regular_expressions[key] and value is not None and regular_expressions[key].match(value))):
                            matches = _search_info_cache[key][value]

                    # else , accesses ?
                    # pass
            criteria_matches.append(matches)

        # turn matches in to the correct object/return type
        matchlist = []
        if len(criteria_matches):
            matched_all = set(criteria_matches[0])
            for m in criteria_matches[1:]:
                matched_all = matched_all.intersection(set(m))
            pathlist = sorted(matched_all)
            for path in pathlist:
                if search_kwargs['getobj']:
                    matchlist.append(self.component.getbypath(path))
                elif search_kwargs['getpath']:
                    matchlist.append(".".join([self.component.path, path]))
                else:  # default is relative path
                    path = ".".join([self.component.path, path])
                    path = path.replace(search_kwargs['startpath'], "")
                    matchlist.append(path)
        return matchlist


class SearchInfoOptimize(ComponentPlugin):
    # this plugin will add search information to each component definition, so that future search
    # calls for the specified key will be faster
    name = "search_info_optimize"

    def __call__(self, info_keys, **kwargs):
        """
        Args:
            info_keys : the key (or list of keys) to build an index off of for named accesses
        """
        if isinstance(info_keys, six.string_types) or not isinstance(info_keys, collections.Iterable):
            info_keys = [info_keys]
        warned = False
        skipfields = kwargs.pop('skipfields', False)
        for component in self.component.walk_components():
            search_info_cache = component.definition.info.setdefault("_search_info_cache", {})
            for info_key in info_keys:
                key_cache = search_info_cache[info_key] = {}
                for child in component.definition.walk_nodes():
                    if skipfields and child.type == NodeTypes.Field:
                        continue
                    info_value = child.info.get(info_key, LookupError)
                    if isinstance(info_value, collections.Iterable) and not True:
                        if not warned:
                            _LOG.warning("Cannot use search_info_optimize on keys that have lists for values")
                            warned = True
                    key_cache.setdefault(info_value, []).append(child.nodepath)


class SearchAddressLegacyPlugin(SearchInfoPlugin):
    """
    This plugin is used to rename the parent SearchInfoPlugin and provide
    backwards compatibility with the "searchaddress" command from PythonSv.

    All search functionality is implemented in SearchInfoPlugin parent class.
    """
    name = "searchaddress"


class NodeSearch(NodeValuePlugin):
    """
    This plugin provides the 'search' command to registers. It will display the current field names that match
    the pattern.

    The plug in will try to match first as regular expression, if no match it'll try a simple find.
    """
    name = "search"

    def __call__(self, pattern, **kwargs):
        known_args = ['getobj','searchType']
        for k in kwargs.keys():
            assert k in known_args, "Unknown option: %s" % k        
        finds = []
        getobj = kwargs.pop('getobj', False)
        searchType = kwargs.pop('searchType','fields')
        try:
            compiled_regexp = re.compile(pattern, re.IGNORECASE)
        except:
            compiled_regexp = None
        # Fields names should be used for registers
        # but something like a General Node may rely
        # on nodenames
        search_list = self.nodevalue.nodenames
        for field in search_list:
            if searchType in ['description', 'd']:
                field_object = self.nodevalue.get_node(field)
                description = field_object.definition.info.get(
                    'description', None
                )
                if isinstance(description, str):
                    match = compiled_regexp.search(description)
                    if match:
                        field_path = "{register}.{field}".format(register = self.nodevalue.name, field = field)
                        if getobj:
                            finds.append([field_path, field_object])
                        else:
                            start_pos = match.start() - 15
                            if start_pos < 0:
                                start_pos = 0
                            end_pos = match.end() + 15
                            description_string = description[start_pos:end_pos]
                            finds.append(
                                (
                                    field_path + " \"...{}...\"".format(description_string)
                                )
                            )                
            else:
                if compiled_regexp is not None:
                    match_regex = compiled_regexp.match(field)
                else:
                    match_regex = None
                match_find = field.find(pattern)
                if match_regex or match_find > -1:
                    self._add_to_finds(getobj, finds, field)
        return finds

    def _add_to_finds(self, getobj, array, field_name):
        """
        this method handles how to add a match to the array that stores the results
        Args:
            getobj: Either True or False (if True, it means we return the object as well)
            array: just a simple array to add matches
            field_name: string

        Returns:
            None
        """
        if getobj is False:
            array.append(field_name)
        else:
            field_obj = self.nodevalue.get_node(field_name)
            array.append([field_name, field_obj])


class SearchAccessInfoPlugin(ComponentPlugin):
    """
    This function provides the "search_info" command to NamedComponent
    nodes.

    This script searches all the component's nodes and looks for the info
    that has attributes matching what is passed in to the kwargs.
    if a list, is given for an attribute then the info will be considered
    matched if it is in that list.

    Some examples::

        # matches if node's info has device=2, offset=0x4, assumes other
        # parameters do not matter
        search_access_info(device=2,offset=0x4)
        # matches to device 2, considers a match if offset is 0x4 or 0x8
        search_access_info(device=2,offset=[0x4,0x8])
        # matches to all nodes where device is 2
        search_access_info(device=2)
        # matches all nodes with offset 0x4
        search_access_info(offset=0x4)
        # matches only nodes with constant access and offset 0x4
        search_access_info(access="constant",offset=0x4)

    Currently the kwords must exactly match the attributes of the info
    object, so "offset" will NOT match "msr_offset" or "cr_offset"

    Other special keyword arguments (processed in this order--dont specify
    multiple):

    The value for the key can also be a function that should return True/False. It
    will be passed the value of the key found on the node OR the value will be
    the LookupError class if the node did not have that key. As an exmple, you can find
    all registers with an msr_offset using::

        comp.search_info(portid=lambda x: x is not LookupError)


    Args:
        getobj (bool, optional): Used to determine whether to return
            the object instead of the name/path. Defaults to False.
        getpath (bool, optional): Used to get the entire path to the node.
            Defaults to False.
        firstfind (bool, optional): Used to stop the search after the first
            find. Defaults to False.
        isregex (bool, optional): Used to determine whether we should
            attempt to perform a regex-based search on the arguments.
            Defaults to True.
    """
    name = "search_access_info"

    def __call__(self, **kwargs):
        getobj = kwargs.pop('getobj', False)
        getpath = kwargs.pop('getpath', False)
        firstfind = kwargs.pop('firstfind', False)
        recursive = kwargs.pop('recursive', True)
        isregex = kwargs.pop('isregex', False)
        nextkwargs = {
            "getobj": getobj,
            "getpath": getpath,
            "firstfind": firstfind,
            "isregex": isregex,
        }
        matchlist = []
        searchcriteria = kwargs
        if hasattr(self.component, 'nodenames'):
            if self.component.nodenames:
                if isregex:
                    regular_expressions = {}
                    for key in searchcriteria:
                        regular_expressions[key] = None
                        if isinstance(searchcriteria[key], str):
                            regular_expressions[key] = re.compile(searchcriteria[key])
                for node in self.component.nodes:
                    try:
                        access_info = node.get_access_info()
                    except:
                        pass
                    else:
                        if access_info:
                            number_of_key_matches = 0
                            for arg in searchcriteria:
                                v = access_info.get(arg, None)
                                if v is not None:
                                    try:
                                        if callable(searchcriteria[arg]):
                                            try:
                                                number_of_key_matches += int(searchcriteria[arg](v))
                                            except Exception as e:
                                                message_error = "Exception {} occured while checking {} against {}".format(str(v), arg, self.component.path)
                                                if message_error not in matchlist:
                                                    matchlist.append(message_error)
                                                    print(message_error)
                                        elif v == searchcriteria[arg]:
                                            number_of_key_matches += 1
                                        if isinstance(searchcriteria[arg], list):
                                            if v in searchcriteria[arg]:
                                                number_of_key_matches += 1
                                        elif isregex:
                                            if arg in regular_expressions and regular_expressions[arg].match(v):
                                                number_of_key_matches += 1
                                    except Exception as e:
                                        print(str(e))
                            if number_of_key_matches == len(searchcriteria):
                                matchlist.append(self._add_match(nextkwargs, node))
                                if len(matchlist) == 1 and firstfind:
                                    return matchlist
                return matchlist
        sub_components = self.component.sub_components
        for component_name, component in sub_components.items():
            allargs = dict(nextkwargs)
            allargs.update(kwargs)
            sub_component_matchlist = component.search_access_info(**allargs)
            if sub_component_matchlist != []:
                matchlist = matchlist + sub_component_matchlist
                if firstfind and len(matchlist) == 1:
                    return matchlist
        return matchlist

    def _add_match(self, arguments, node):
        getobj = arguments.get('getobj', False)
        getpath = arguments.get('getpath', False)
        if getobj:
            return node
        elif getpath:
            return node.path
        else:
            return node.name

