
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ..nodes import NodeTypes, NodeValuePlugin
from ..utils._py2to3 import *


class WriteFieldsPlugin(NodeValuePlugin):
    name = "writefields"

    @classmethod
    def create(cls, nodevalue):
        if nodevalue.type != NodeTypes.Array:
            return cls(nodevalue)
        else:
            return None

    def __call__(self, **kwargs):
        """
        Write the specified values to the fields with the given name.
        use it like writefields(field1=value, field2=value,field3=value)
        Returns:

        """
        if PY2:
            iteritems = kwargs.iteritems()
        else:
            iteritems = kwargs.items()
        for field_name, field_value in iteritems:
            self.nodevalue.get_node(field_name).store(field_value)
        self.nodevalue.flush()


class SetDynamicFieldsPlugin(NodeValuePlugin):

    name = "set_dynamic_fields"

    @classmethod
    def create(cls, nodevalue):
        # Just add to nodes with more than one children
        if len(nodevalue.nodes) < 1:
            return None

        # Make sure children are fields
        for node in nodevalue.iter_nodes():
            if node.type != NodeTypes.Field:
                return None

        return cls(nodevalue)

    def __call__(self, fieldname_list):
        """
        Removes all but the specified fields from node.

        Args:
            fieldname_list (list): List of fields that will remain active.
        """
        # Make sure all fields exist in the node
        missing_fields = set(fieldname_list) - set(self.nodevalue.definition.nodenames)
        if len(missing_fields) > 0:
            raise ValueError("Unknown fields: " + ", ".join(missing_fields))

        # Remove fields
        remove_list = set(self.nodevalue.definition.nodenames) - set(fieldname_list)
        self.nodevalue._removed_nodes = set()  # Clear removed nodes list
        for fieldname in remove_list:
            self.nodevalue.remove_node(fieldname)

        # Update each field's lowerbit
        numbits = 0
        for field in sorted(self.nodevalue.nodes, key=lambda k: k.lowerbit):
            # lowerbit is the total number of bits used so far
            field.target_info["lowerbit"] = numbits
            numbits += field.numbits

        # Update number of bits
        self.nodevalue.target_info["numbits"] = numbits


