
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




"""
Plugin for register grouping support.
"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ..comp import ComponentGroup
from ..comp import ComponentPlugin
from ..utils._py2to3 import *
from ..utils.ordereddict import odict

import re


class NodeGroupsPlugin(ComponentPlugin):
    """
    A node group contains nodes lists (where each attribute is a type of list)
    that allows us to read/write to multiple nodes at one time. It uses a regex
    to guess at which nodes may be similar based on having the same name. It
    has no smarts other than based on the node name.
    """

    name = "groups"
    _sub_node_groups = None
    _searches = None
    _frozen = False

    def __init__(self, component):
        super(NodeGroupsPlugin, self).__init__(component)
        self._sub_node_groups = {}
        # Specify regular expression with one group in it, specify delimiter
        # to use when regular expression matches
        # Be VERY careful when adding regular expression
        self._searches = odict()
        self._searches[r'^[a-zA-Z_]+?(\d+)_.*'] = 'X'
        self._searches[r'^[a-zA-Z_]+?(d\d+f\d+)_.*'] = 'X'
        self._frozen = True

    def add_group_type(self, regexpression, replacement):
        """
        Adds a new regular expression to match a nodes' group type.

        Args:
            regexpression (regex): The regular expression for the new node
                group type.
            replacement (string): The expression to identify the group of
                nodes.
        """
        self._searches[regexpression] = replacement

    # adding this for backwards compatibility with pythonsv
    addGroupType = add_group_type

    def make_register_groups(self, force=False):
        """
        Using all the nodes we know of, try to turn them into groups.

        Args:
            force (bool, optional): Used to force recreating node groups, this
                means drop the existing groups and create new ones. Defaults to
                False.
        """    
        # Expected to follow the same format as _searches:
        # 'regular_expression': 'groupID',         
        register_group_regexp =  self.component.definition.info.get('register_group_regexp', None)        
        searches = None
        if force:
            self._sub_node_groups.clear()

        # check if groups have been added already, if so bail...
        if len(self._sub_node_groups) != 0:
            return

        threshold = 2
        compiled_regex = odict()
     
        if register_group_regexp:
            searches = register_group_regexp
        else:
            searches = self._searches

        for regex in searches:
            try:
                compiled_regex[regex] = re.compile(regex)
            except TypeError:
                print("Not a valid regular expression: {}".format(regex))



        for child in self.component.nodes:
            replacement = False
            for exp, exp_replacement in reversed(list(searches.items())):
                match = compiled_regex[exp].search(child.name)
                if match:
                    replacement = exp_replacement
                    break
            if not replacement:
                # we didn't find a regex that matches this child's name
                continue
            # We keep a reference to the node definitions in the groups
            # that we create but for __getattr__ and __setattr__ we'll be
            # needing the node values.
            child_definition = self.component.definition.get_node(child.name)
            group_name = (
                child.name[:match.start(1)] +
                replacement +
                child.name[match.end(1):]
            )
            # easy... matches an existing group
            if group_name in self._sub_node_groups:
                self._sub_node_groups[group_name].append(child_definition)
            else:
                # need new group for this thing, may have groups with one
                # entry, but that way we consistently have these lists
                # even if there's only one.

                # It is important to notice that what we store in the
                # component group is the node's definition which we
                # will need for creating node value objects when requested
                # to read/write to nodes in the group
                new_group = ComponentGroup(
                    [child_definition],
                    group_name,
                    grouppath=(
                        self.component.path + "." +
                        self.name + "." +
                        group_name
                    )
                )
                self._sub_node_groups[group_name] = new_group
        if PY2:
            groups = self._sub_node_groups.keys()
        else:
            groups = list(self._sub_node_groups.keys())
        for group in groups:
            if len(self._sub_node_groups[group]) < threshold:
                del self._sub_node_groups[group]
                continue

    # adding this for backwards compatibility with pythonsv
    makeRegGroups = make_register_groups

    def __getattr__(self, attr):
        group = self.get_group(attr, default=AttributeError)
        if group is not AttributeError:
            return group
        else:
            raise AttributeError("Unknown attribute: %s" % attr)

    def __setattr__(self, attr, value):
        # assume that the attribute is ours if not frozen
        if not self._frozen:
            self.__dict__[attr] = value
        elif attr in self._sub_node_groups:
            for node_definition in self._sub_node_groups[attr]:
                node_value = self.component.get_node(node_definition.name)
                node_value.write(value)

    def get_group(self, groupname, **kwargs):
        default = kwargs.get('default', LookupError)
        if groupname in self._sub_node_groups:
            # create a "new group" of node values using the nodes
            # definitions that we know of from when the group was created.
            # It is from/to the nodes values that we read/write.
            known_group = self._sub_node_groups[groupname]
            new_group = ComponentGroup(
                name=groupname,
                grouppath=known_group.grouppath
            )
            for node_definition in known_group:
                node_value = self.component.get_node(node_definition.name)
                new_group.append(node_value)
            return new_group
        if default is not LookupError:
            return default
        else:
            raise LookupError("Unknown register grouping: %"%groupname)


    def __dir__(self):
        dirlist = list(self.__dict__.keys())
        dirlist.extend(dir(self.__class__))
        dirlist.extend(list(self._sub_node_groups.keys()))
        return dirlist

class GetGroupPlugin(ComponentPlugin):
    """
    This plugin adds the capability of logging all registers and
    fields in a RegisterComponent.
    """

    name = "get_group"

    def __call__(self, groupname):
        return self.component._sub_groups.get(groupname, None)