
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.

import re
import six
import collections
import json
from ..comp import (
    ComponentPlugin,
    ComponentDefinitionPlugin,
    NamedComponentDefinition,
    )
from ..nodes import NodeValuePlugin, NodeTypes
from ..registers import RegisterComponent
from ..utils._py2to3 import *
from ..utils.ordereddict import odict
from ..logging import getLogger
from ..telemetry import record_event
from .. import settings
from . import lmdb_dict
import warnings
_LOG = getLogger()

#ComponentPlugin
class GetAddressMap(ComponentPlugin):
    """ This plugin will find and return the address map information
    for a given component and return it as a dict."""

    name = "get_address_map"
    @classmethod
    def create(cls, component):
        if 'osx_address_maps' in component.target_info:
            return cls(component)
        else:
            return None

    def __call__(self, map_name=None, interface_name=None, **kwargs):
        """
        map_name (string, optional)
        interface_name (string,optional)
        If map_name and interface_name are NOT provided, the plugin will
        return all address maps found.

        Providing one or the other ( or both) reduces the size of the return
        dictionary
        """
        ret_dict= {}
        if map_name is None and interface_name is None:
            ret_dict = self.component.target_info['osx_address_maps']
        else:
            tmp_dict = self.component.target_info['osx_address_maps']
            ret_dict = self._search_am_dict(tmp_dict, map_name, interface_name)
        return ret_dict

    def _search_am_dict(self, am_dict, map_name, interface_name):
        search_string = ""
        search_type = ""
        map_and_interface = False
        if map_name and not interface_name:
            search_string = map_name
            search_type = self._is_map
        elif interface_name and not map_name:
            search_string = interface_name
            search_type = self._is_interface
        elif interface_name and map_name:
            #search maps first, and then refine search
            #in the smaller dict matching for interfaces
            map_and_interface=True
            search_string = map_name
            search_type = self._is_map
        tmp_dict = self._search_dict_recursive(am_dict, search_string, search_type)
        if map_and_interface:
            search_string = interface_name
            search_type = self._is_interface
            ret_dict = self._search_dict_recursive(tmp_dict, search_string, search_type)
        else:
            ret_dict = tmp_dict
        return ret_dict

    def _search_dict_recursive(self, am_dict, search_string, search_type):
        found = {}
        if search_string in am_dict:
            if search_type(am_dict[search_string]):
                found.update(am_dict[search_string])
        else:
            for key, dictval in am_dict.items():
                if isinstance(dictval, dict):
                    found.update(self._search_dict_recursive(dictval, search_string, search_type))
        return found

    def  _is_interface(self, int_dict):
        if 'type' in int_dict:
            if int_dict['type']=='interface':
                return True
        return False

    def  _is_map(self, map_dict):
        if 'type' in map_dict:
            if map_dict['type']=='map':
                return True
        return False

class SetAddressMap(ComponentPlugin):
    """This plugin facilitates overriding/adding new info
    to the addressmaps within  a given component"""
    name = "set_address_map_key"
    @classmethod
    def create(cls, component):
        if 'osx_address_maps' in component.target_info:
            return cls(component)
        else:
            return None
    def __call__(self, map_name, interface_name, **kwargs):
        """
        map_name(string)
        interface_name(string)
        key=val with the overrides
        Example:
        component.set_address_map_key('cadb_sub_MEM','cadb_p', membar=0xfa00000, length=128)
        """
        overrides_dict = {}
        for key in kwargs:
            overrides_dict[key]=kwargs[key]
        comp_am_dict = self.component.target_info['osx_address_maps']
        self._modifier(comp_am_dict, map_name, interface_name, overrides_dict)

    def _modifier(self, am_dict, map_name, interface_name, overrides_dict):
        if map_name in am_dict:
            map_dict = am_dict[map_name]
            if interface_name in am_dict[map_name]:
                target_dict= map_dict[interface_name]
                for or_key, or_val in overrides_dict.items():
                    if or_key in  target_dict:
                        _LOG.warning("Overridding Addressmap info. map_name {}, interface_name {}".format(map_name, interface_name))
                        _LOG.warning("Old data {}={}  ... New data {}={}".format(or_key, target_dict[or_key], or_key, overrides_dict[or_key]))
                    else:
                        _LOG.warning("Adding info to Addressmap. map_name {}, interface_name {}".format(map_name, interface_name))
                        _LOG.warning("New data {}={}".format(or_key, overrides_dict[or_key]))
                    target_dict[or_key] = or_val
            else:
                self._search_nested_interface(map_dict, interface_name, overrides_dict)
        else:
            for keyname, keyval in am_dict.items():
                if isinstance(keyval, dict):
                    self._modifier(keyval, map_name, interface_name, overrides_dict)

    def _search_nested_interface(self, am_dict, interface_name, overrides_dict):
        if interface_name in am_dict:
            print("do overrides")
            target_dict= am_dict[interface_name]
            for or_key, or_val in overrides_dict.items():
                if or_key in  target_dict:
                        _LOG.warning("Overridding Addressmap info. interface_name {}".format( interface_name))
                        _LOG.warning("Old data {}={}  ... New data {}={}".format(or_key, target_dict[or_key], or_key, overrides_dict[or_key]))
                else:
                    _LOG.warning("Adding info to Addressmap. interface_name {}".format(interface_name))
                    _LOG.warning("New data {}={}".format(or_key, overrides_dict[or_key]))
                target_dict[or_key] = or_val
        else:
            for keyname, keyval in am_dict.items():
                if isinstance(keyval, dict):
                    self._search_nested_interface(keyval, interface_name, overrides_dict)



class GetAddressMapInfo(NodeValuePlugin):
    """
    Gets the address map info from the register and displays is
    """
    name = "get_address_map_info"

    @classmethod
    def create(cls, nodevalue):
        if 'osx_address_maps' in nodevalue.definition.info:
            return cls(nodevalue)
        else:
            return None

    def __call__(self, **kwargs):
        """
        Args
            None
        Returns:
            Dictionary
        """
        node = self.nodevalue
        ret_dict = {}
        if 'osx_address_maps' in node.definition.info:
            parent_osx_address_maps_info = node.lookup_info('osx_address_maps')
            for address_map_name, address_map_local_info in node.definition.info['osx_address_maps'].items():
                if address_map_name not in ret_dict:
                    ret_dict[address_map_name] = {}
                if isinstance(address_map_local_info, dict):
                    ret_dict[address_map_name].update(address_map_local_info)
                elif isinstance(address_map_local_info, str):                    
                    ret_dict[address_map_name].update({address_map_name:address_map_local_info})              
                address_map_parent_info = parent_osx_address_maps_info.get(address_map_name, None)
                if address_map_parent_info:
                    ret_dict[address_map_name].update(address_map_parent_info)
                    # remove "type" if it happens to be floating around:
                    ret_dict[address_map_name].pop("type", None) 
            return ret_dict
        

            


class ShowAddressMapInfo(NodeValuePlugin):
    """
    Gets the address map info from the register and displays is
    """
    name = "show_address_map_info"

    @classmethod
    def create(cls, nodevalue):
        if 'osx_address_maps' in nodevalue.definition.info:
            return cls(nodevalue)
        else:
            return None

    def __call__(self, **kwargs):
        """
        Args
            None
        Returns:
            None
        """        
        node = self.nodevalue
        if 'osx_address_maps' in node.definition.info:
            ret_dict = node.get_address_map_info()
            print(json.dumps(ret_dict, indent=4))


