
# INTEL CONFIDENTIAL
# Copyright 2014 2019 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.


"""
Logging all registers and fields in a RegisterComponent

"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from collections import OrderedDict
from ..logging import getLogger

from ..comp import ComponentPlugin
from ..nodes import NodeTypes
from ..registers import RegisterComponent
from ..utils._py2to3 import *
from . import nn_display
from . import nn_walk
from .. import telemetry
from ..precondition import MultipleAccesses
from ..access import AvailabilityOptions
from .lmdb_dict import CacheContext

import json
import os
import re
import sys
import time

_LOG = getLogger()

_MAX_FAIL_COUNT = 10

class LogRegistersPlugin(ComponentPlugin):
    """
    This plugin adds the capability of logging all registers and
    fields in a RegisterComponent.
    """

    name = "logregisters"

    @classmethod
    def create(cls, component):
        # This plugin just adds support for RegisterComponent
        if isinstance(component, RegisterComponent):
            return cls(component)
        else:
            return None

    def __call__(self, file, subComponents=None, filter=None, **kwargs):
        """
        Writes the values of all the registers and fields in a component (and
        also dumps the subcomponents, by default).

        Args:
            file (type): filename to write to, can be a string of a file
                handle in the case of a file handle, the file will be
                left open.
            subComponents (list, optional): List of component objects to
                log registers from. Iterates over subComponents list instead
                of this component's default subComponent list. Defaults to
                None.
            filter (str, optional): Only logs registers that match this
                regular expression, case is ignored. Defaults to None.
            **kwargs: Arbitrary keyword arguments, it can include any of
                the following arguments.
            prefix (str, optional): Path to use as a prefix to the
                registers/field dump if its not specified then no prefix
                will be used. Defaults to empty string (""). Be sure to
                set showpath="none" if you want JUST the prefix to be used
            showpath (str): Prefix the register with the path
                that it took to get to register. "full", "relative" (default),
                or "none"
            silent (bool, optional): Used so that logregisters won't print
                its status. Defaults to False.
            recursive (bool, optional): Used to prevent logging subcomponents
                (overrides subComponents parameter). Defaults to True.
            stop_on_error (bool, optional): Whether to stop logging on the
                first exception we hit. Defaults to False.
            appendlog (bool, optional): Whether to open log file for appending
                or overwrite. Defaults to False.
            binary_output (bool, optional): Set to True to dump register/field
                values in binary format.

        Note:
            showpath and relative path cannot both be set to true. if either is set
            to specifically to False, then no path is show in the log file

        """
        _logregisters(
            file=file, \
            component=self.component, \
            subComponents=subComponents, \
            filter=filter,
            output_format=None,
            **kwargs)


class JsonLogRegistersPlugin(ComponentPlugin):
    """
    This plugin adds the capability of logging all registers and
    fields in a RegisterComponent in a JSON formatted file.
    """

    name = "jsonlogregisters"

    @classmethod
    def create(cls, component):
        # This plugin just adds support for RegisterComponent
        if isinstance(component, RegisterComponent):
            return cls(component)
        else:
            return None

    def __call__(self, file, subComponents=None, filter=None, **kwargs):
        """
        Writes the values of all the registers and fields in a component (and
        also dumps the subcomponents, by default).

        Args:
            file (type): filename to write to, can be a string of a file
                handle in the case of a file handle, the file will be
                left open.
            subComponents (list, optional): List of component objects to
                log registers from. Iterates over subComponents list instead
                of this component's default subComponent list. Defaults to
                None.
            filter (str, optional): Only logs registers that match this
                regular expression, case is ignored. Defaults to None.
            prefix (str, optional): Path to use as a prefix to the
                registers/field dump if its not specified then no prefix
                will be used. Defaults to empty string (""). Be sure to
                set showpath="none" if you want JUST the prefix to be used
            silent (bool, optional): Used so that jsonlogregisters won't print
                its status. Defaults to False.
            recursive (bool, optional): Used to prevent logging subcomponents
                (overrides subComponents parameter). Defaults to True.
            stop_on_error (bool, optional): Whether to stop logging on the
                first exception we hit. Defaults to False.
            appendlog (bool, optional): Whether to open log file for appending
                or overwrite. Defaults to False.
            binary_output (bool, optional): Set to True to dump register/field
                values in binary format.
        """
        _logregisters(
            file=file, \
            component=self.component, \
            subComponents=subComponents, \
            filter=filter,
            showpath=True,
            relative_path=None, # force this so that showpath is not overwritten
            output_format='json',
            **kwargs)


def _logregisters(file, component, subComponents=None, filter=None, **kwargs):
    """
    Writes the values of all the registers and fields in a component (and
    also dumps the subcomponents, by default).

    Args:
        file (type): filename to write to, can be a string of a file
            handle in the case of a file handle, the file will be
            left open.
        subComponents (list, optional): List of component objects to
            log registers from. Iterates over subComponents list instead
            of this component's default subComponent list. Defaults to
            None.
        filter (str, optional): Only logs registers that match this
            regular expression, case is ignored. Defaults to None.
        prefix (str, optional): Path to use as a prefix to the
            registers/field dump if its not specified then no prefix
            will be used. Defaults to empty string (""). Be sure to
            set showpath="none" if you want JUST the prefix to be used
        showpath (str): Prefix the register with the path
            that it took to get to register. "full", "relative" (default),
            or "none"
        silent (bool, optional): Used so that logregisters won't print
            its status. Defaults to False.
        recursive (bool, optional): Used to prevent logging subcomponents
            (overrides subComponents parameter). Defaults to True.
        stop_on_error (bool, optional): Whether to stop logging on the
            first exception we hit. Defaults to False.
        appendlog (bool, optional): Whether to open log file for appending
            or overwrite. Defaults to False.
        binary_output (bool, optional): Set to True to dump register/field
            values in binary format.
        output_format (str, optional): Used to specify the output format, set it
            to "json" for a json-formatted output.

    Note:
        showpath and relative path cannot both be set to true. if either is set
        to specifically to False, then no path is shown in the log file

        if an access fails more than a certain number of times on the same component
        then it will be assumed that the access is broken (resets on the next component)

    """
    # Comment from PythonSv:
    # I can definitely see logregisters having a lot of different options,
    # so i think it would be best to handle these through a kwargs so that
    # there is no mistake in someone assuming a certain parameter order
    # for all of these extra options being added on.
    known_args = ['recursive', 'prefix', 'silent', 'showpath', 'appendlog',
                  'stop_on_error', 'relative_path', 'binary_output',
                  'output_format']
    for k in kwargs:
        assert k in known_args, "Unknown option: %s" % k
    recursive = kwargs.get('recursive', True)
    prefix = kwargs.get('prefix', "")
    silent = kwargs.get('silent', False)
    showpath = kwargs.get("showpath", "relative")
    if showpath is True:
        showpath = "full"
    elif showpath is False:
        showpath = "none"
    assert showpath in ["full", "none", "relative"], "Invalid value for showpath"
    relative_path = kwargs.get('relative_path', None)
    if relative_path: # let this specified keyword also set showpath for backwards compatibility:
        showpath = "relative"
        # warning? for this behavior going away?
    appendlog = kwargs.get("appendlog", False)
    stop_on_error = kwargs.get('stop_on_error', False)
    binary_output = kwargs.get('binary_output', False)
    output_format = kwargs.get('output_format', None)

    ### for some stats gathering
    skipped_register_count = 0
    failed_register_count = 0
    read_register_count = 0

    # this should just be a total of the above, but this sanity checks the data
    total_register_count = 0

    total_component_count = 0
    start_time = time.time()

    # will only be populated if json output was requested
    output_dict = OrderedDict()
    if isinstance(file, basestring):
        if appendlog:
            fout = None
            if output_format and output_format == 'json':
                if os.path.exists(file):
                    with open(file) as json_input:
                        nn_simple = json.load(json_input, object_pairs_hook=OrderedDict)
                        output_dict = nn_simple.get('namednodes_simple', OrderedDict())
            else:
                fout = open(file, 'a')
        else:
            fout = open(file, 'w')
    else:
        # if "file" argument is not a string then assume
        # it's a file handle
        fout = file

    if subComponents is not None:
        assert isinstance(subComponents, (list, tuple)), \
            "Must be list or tuple for subComponents"
        walk_paths = tuple([component.path + "." + p for p in subComponents])

    if not recursive:
        dumpcomponents = [component]
    else:
        dumpcomponents = component.walk_components()

    skiplist = []
    multiple = MultipleAccesses()
    with multiple:
        for subcomponent in dumpcomponents:
            if subComponents is not None:
                if not subcomponent.path.startswith(walk_paths):
                    continue
            # unless explicitly told, dont dump those that start with "_"
            if subcomponent.name.startswith("_"):
                continue

            # skip the component if it has available flag set to False
            comp_available = subcomponent.definition.info.get("available", AvailabilityOptions.Available)
            # it may be that comp available overrides the definition one
            comp_available = subcomponent.target_info.get("available", comp_available)
            # comp not available...AND it was not explicitly called on this component
            # -- FYI: cant check == for the actual object since it seems that we get a proxy
            #    back for the subcomponent
            if not comp_available and subcomponent.path != component.path:
                skiplist.append(subcomponent.path)
                continue
            elif comp_available == AvailabilityOptions.ChildrenOnly:
                # dont add to skip list, but dont log any nodes either...
                continue
            else:
                skip = False
                for path in skiplist:
                    if subcomponent.path.startswith(path+"."):
                        skip = True
                if skip:
                    continue

            # use nodenames so that we don't create nodevalue objects for everything
            regtotal = len(subcomponent.nodenames) - 1  # include last one in calc
            if not silent:
                _LOG.result("Dumping config registers for %s, " +
                            "please be patient", subcomponent.path)

            # cache to use to use by "is_available" to allow accesss classes to be smarter
            # per component with keeping track whether groups of registers are available
            subcomponent.target_info['is_available_cache'] = {}
            if output_format and output_format == 'json':
                # for logging progress purposes only
                nodenum = 0
                last_nodenum = nodenum + 1
                # we walk ALL nodes including fields this way...
                last_skip = None
                for node in subcomponent.walk_nodes():
                    # if we hit a child node where we skipped the parent, then skip the children..
                    if last_skip and node.nodepath.startswith(last_skip):
                        continue
                    if not silent:
                        # to avoid printing the same progress more than once
                        if nodenum != last_nodenum:
                            last_nodenum = nodenum
                            if nodenum % 10 == 0 and regtotal > 0:
                                sys.stdout.write("\r%02.1f%% complete" %
                                                (nodenum * 100 / float(regtotal)))
                        if nodenum == regtotal:  # wipe away our status line
                            sys.stdout.write("\r"+" "*40+"\r")

                    # check first if this node should be filtered out
                    nodename = node.name
                    if filter is not None:
                        node_object = re.compile(filter, re.IGNORECASE)
                        if node_object.search(nodename) is None:
                            skipped_register_count += 1
                            continue

                    total_register_count += 1
                    success = _read_if_available(node, stop_on_error)
                    if success is False:
                        # means it wasn't available
                        skipped_register_count += 1
                        last_skip = node.nodepath
                        continue
                    elif success is None:
                        # we attempted read but failed
                        failed_register_count += 1
                        continue
                    # else...read was successful
                    read_register_count += 1
                    # clear last skip
                    last_skip = None

                    node_entries = None
                    if node.type == NodeTypes.Register:
                        # increment number of nodes only when Register
                        # so that we report progress correctly
                        nodenum += 1

                        if not node.nodenames:
                            node_entries = _register_entry(node, prefix, binary_output)
                    if node.type == NodeTypes.Field:
                        node_entries = _field_entry(node, prefix, binary_output)

                    if node_entries:
                        output_dict.update(node_entries)
            else:
                for regnum, registername in enumerate(subcomponent.nodenames):
                    if not silent:
                        if regnum % 10 == 0 and regtotal > 0:
                            sys.stdout.write("\r%02.1f%% complete" %
                                            (regnum * 100 / float(regtotal)))
                        if regnum == regtotal:  # wipe away our status line
                            sys.stdout.write("\r"+" "*40+"\r")

                    # check first if this register should be filtered out
                    if filter is not None:
                        register_object = re.compile(filter, re.IGNORECASE)
                        if register_object.search(registername) is None:
                            skipped_register_count += 1
                            continue

                    total_register_count += 1

                    register_object = subcomponent.get_node(registername)
                    # silently skip if we hit a node type not supported by value str
                    if register_object.type not in [ NodeTypes.Array, NodeTypes.Register, NodeTypes.Field]:
                        continue
                    success = _read_if_available(register_object, stop_on_error)
                    if success is False:
                        # means it wasn't available
                        skipped_register_count += 1
                        continue
                    elif success is None:
                        # we attempted read but failed
                        failed_register_count += 1
                        continue
                    # else...read was successful
                    read_register_count += 1

                    # alias objects have different name, so we need to pass name in
                    if showpath == "relative":
                        # calculate path relative to calling component
                        rel_state = register_object.path
                        rel_state = rel_state[len(component.path)+1:]
                        name_str = rel_state
                    elif showpath == "full":
                        name_str = register_object.path
                    elif showpath == "none":
                        name_str = registername
                    else:
                        raise RuntimeError("invalid showpath, assert missed something: %s"%showpath)
                    try:
                        valuestr = register_object.valuestr(prefix=prefix,
                                                    name=name_str,
                                                    binary_output=binary_output)
                    except LookupError:
                        # skip cases were enum was missing
                        if not stop_on_error:
                            continue
                        else:
                            raise

                    fout.write("%s\n" % valuestr)

            # remove is available cache when finished
            del subcomponent.target_info['is_available_cache']
            subcomponent.definition.tolmdb.clear_cache(recursive=False, silent=True)

    if output_format and output_format == 'json':
        json_output = {'namednodes_simple': output_dict}
        if not fout:
            # file should be a string otherwise we missed something above
            fout = open(file, 'w')
        json.dump(json_output, fout, indent=4)

    # if input was a filename, then close the one we opened
    if isinstance(file, basestring):
        fout.close()

    data = {
        # parameters
        'prefix_used': prefix != "",
        'silent': silent,
        'filter_used': filter is not None,
        'stop_on_error': stop_on_error,
        'binary_output': binary_output,
        'recursive': recursive,
        'appendlog': appendlog,
        'format': output_format,
        # statistics
        'runtime': time.time()-start_time,
        'failed_register_count': failed_register_count,
        'skipped_register_count': skipped_register_count,
        'read_register_count': read_register_count,
        'total_register_count': total_register_count,
        'total_component_count': total_component_count,
    }
    event_type = "logregisters" if not output_format else "jsonlogregisters"
    telemetry.record_event(event_type, data)


def _register_entry(register, prefix, binary_output):
    """
    Get register data as a dictionary with value info.

    Arguments:
      - register      : Register node to get entry for.
      - prefix        : A prefix to add to register's path.
      - binary_output : Set to True format register's value in binary format.

    Returns a dictionary with register's data as:

    {
        "register_path" : { "value": "0x0"}
    }
    """
    pathstr = register.path
    value = register.value or register.get_value()
    entry_key = "%s%s" % (prefix, pathstr)

    if binary_output:
        numbits = register.numbits
        if numbits is None:
            upperbit = register.definition.info.get("upperbit", None)
            lowerbit = register.definition.info.get("lowerbit", None)
            if upperbit is not None and lowerbit is not None:
                numbits = upperbit - lowerbit + 1
            else:
                numbits = 0
        value = ('{:0%ib}' % numbits).format(value)
        entry_value = {"value": "0b%s" % (value)}
    else:
        entry_value = {"value": "0x%x" % (value)}

    return {entry_key: entry_value}


def _field_entry(field, prefix, binary_output):
    """
    Get field data as a dictionary with value and, if available, decode info.

    Arguments:
      - field         : Field node to get entry for.
      - prefix        : A prefix to add to field's path.
      - binary_output : Set to True format field's value in binary format.

    Returns a dictionary with field's data as:

    {
        "field_path" : { "value": "0x0", "decode": "decode_value"}
    }
    """
    pathstr = field.path
    value = field.value or field.get_value()
    if binary_output:
        numbits = field.numbits
        value = ('{:0%ib}' % numbits).format(value)
        entry_value = "0b%s" % (value)
    else:
        entry_value = {"value": "0x%x" % (value)}
    # see if there is a decode we need to get
    decode = field.definition.info.get("enum", None)
    if decode is not None:
        decode = field.decode()
        entry_value['decode'] = str(decode)
    entry_key = "%s%s" % (prefix, pathstr)
    return {entry_key: entry_value}


# ALSO used by the static discovery for creating offline files
def _read_if_available(node, raise_exceptions=False):
    """
    read the node and update its value if possible

    Args:
        node (obj) : node to get the value for
        access_cache (dict) : dictionary to use/update based on whether that access is available
        raise_exceptions (bool) : whether to raise exception when an error is encountered
                                  reading a node

    Returns:
        True/False - whether node was updated
    """
    # check for access availability, cache it for later use
    try:
        access_cache = node.is_available_cache()
        # add to access cache the available and fail count info
        if 'available' not in access_cache:
            # set 'fail_count' first, since we might hit an exception during
            # is_access_available_by_node(), and 'fail_count' is required by
            # the exception handler:
            access_cache['fail_count'] = 0
            access_cache['available'] = node.is_access_available_by_node()

        accessavailable = access_cache['available']

        # skip the register if access is not available
        if accessavailable == AvailabilityOptions.NotAvailable:
            _LOG.debug("Skipping %s b/c access is not available",
                       node.name)
            return False
        elif accessavailable == AvailabilityOptions.NotAvailableTooManyFailures:
            _LOG.debug("Skipping %s b/c access is not available due to too many failures",
                       node.name)
            return False

        # again, if this register isn't available, then skip it
        if node.is_available() == AvailabilityOptions.NotAvailable:
            _LOG.debug("Skipping %s b/c it is flagged as " +
                       "not available", node.name)
            return False

        # register must be available for this to work
        node.get_value()
        # we successfully read register
        access_cache['fail_count'] = 0
        return True
    except KeyboardInterrupt:
        # if got a keyboard interrupt then stop logging registers
        raise
    except Exception:
        _LOG.warning("Could not read node: %s", node.name)
        # update all this info regardless of whether we throw an exception
        access_cache['fail_count'] += 1
        if 'max_fail_count' in access_cache:
            # if it exists, use the cache
            max_fail_count = access_cache.get("max_fail_count")
        else:
            #if not: find it. If nout found, use default
            #store it in cache.
            try:
                tmp_max_fail = node.lookup_info("max_fail_count")
            except LookupError:
                max_fail_count = _MAX_FAIL_COUNT
            else:
                max_fail_count = tmp_max_fail
            access_cache['max_fail_count'] = max_fail_count

        if access_cache['fail_count'] > max_fail_count:
            access_cache['available'] = AvailabilityOptions.NotAvailableTooManyFailures
        if raise_exceptions:
            raise
        else:
            return None
