
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.





"""
Load values from files generated by logregisters.

"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ..logging import getLogger
from ..comp import ComponentPlugin
from ..registers import RegisterComponent
from ..utils._py2to3 import *
from .. import nodes
import re
import sys
import string

_LOG = getLogger()
_use_new_line = False


class LoadRegistersPlugin(ComponentPlugin):
    """
    This plugin adds the capability of loading values logging all registers and
    fields in a RegisterComponent.
    """

    name = "loadregisters"
    _usewhat = ""
    _errout = "exception"
    _silent = False
    _binary_input = False

    @classmethod
    def create(cls, component):
        # This plugin just adds support for RegisterComponent
        if isinstance(component, RegisterComponent):
            return cls(component)
        else:
            return None

    def _display_progress(self, lnum, total_lines):
        """
        Display progress in terms of % lines walked
        """
        global _use_new_line
        if not self._silent:
            if lnum % 10 == 0 and total_lines > 0:
                progress = lnum * 100 / float(total_lines)
                sys.stdout.write("\r%02.1f%% complete" % progress)
                _use_new_line = True
            # Wipe away our status line
            if lnum == total_lines:
                sys.stdout.write("\r" + " " * 40 + "\r")

    def _report_error(self, msg, exception=None, traceback=None):
        """
        Report error per errout configuration
        """
        global _use_new_line
        if self._errout == "exception":
            if _use_new_line:
                _LOG.error("")
            _LOG.error(msg)
            if exception is not None:
                if PY2:
                    exec("raise Exception, None, traceback")  # Hack so Python 3 won't fail
                raise exception.with_traceback(traceback)
            else:
                raise ValueError(msg)
        elif self._errout == "warn":
            if _use_new_line:
                _LOG.warning("")
            _LOG.warning(msg)
        _use_new_line = False

    def _set_value(self, path, value):
        """
        Set register of field value
        """
        try:
            # Try to get register or field, assuming it exists
            node = self.component.get_by_path(path)
        except ValueError:
            try:
                # Maybe we are dealing with an absoulte path, strip first node
                node = self.component.get_by_path(path.split(".", 1)[1])
            except (ValueError, IndexError):
                # The register or field doesn't exist
                self._report_error("Couldn't find '%s'" % path)
                return

        # Make sure we are dealing with a register or field
        if hasattr(node, "type"):
            # We are dealing with a register
            if node.type == nodes.NodeTypes.Register:
                # Set value if usewhat == register
                if self._usewhat in ["registers", "r"]:
                    try:
                        if self._binary_input:
                            node.write(int(value, 2))
                        else:
                            node.write(int(value, 16))
                    except Exception as e:
                        tb = sys.exc_info()[2]
                        self._report_error("Failed to write %s" % path, e, tb)
                return
            # We are dealing with a field
            elif node.type == nodes.NodeTypes.Field:
                # Set value if usewhat == field
                if self._usewhat in ["fields", "f"]:
                    try:
                        if self._binary_input:
                            node.write(int(value, 2))
                        else:
                            node.write(int(value, 16))
                    except Exception as e:
                        tb = sys.exc_info()[2]
                        self._report_error("Failed to write %s" % path, e, tb)
                return

        # We found something, but it is not a register or field
        self._report_error("'%s' is not a register or field" % path)

    def _process_lines(self, lines):
        """
        Process lines read from inputfile.
        """
        # Calculate total lines
        total_lines = len(lines) - 1

        # Line format to look for
        node_re = r"([A-Za-z_0-9]+)"
        array_re = r"(\[[A-Za-z_0-9\"]+\])"
        path_re = r"(?P<path>{node_re}(\.{node_re})*{array_re}?(\.{node_re})?)".format(**locals())
        if self._binary_input:
            value_re = r"(?P<value>0[bB][0-1]+)"
        else:
            value_re = r"(?P<value>0[xX][A-Fa-f0-9]+)"
        regex = re.compile(r" *{path_re} *= *{value_re}\s*(-.*?)*$".format(**locals()))

        for lnum, line in enumerate(lines):
            self._display_progress(lnum, total_lines)

            # If line is empty skip it
            if len(line.strip()) == 0:
                continue
            # If it is a comment, skip it
            if line[0] in [';', '#']:
                continue
            # Skip the info about doing a config dump
            if line.count("Config Dump"):
                continue

            match = regex.match(line)
            if not match:
                # We don't recognize the format of this line
                self._report_error("Don't know how to handle line %d: %s" % (lnum, line.strip()))
            else:
                # Get register or field path from matched expression
                path = match.group("path")
                # Get register or field value from matched expression
                value = match.group("value")
                self._set_value(path, value)

    def __call__(self, inputfile, usewhat, **kwargs):
        """
        Uses a format similar to that spit out by logregisters for loading
        values from a config dump into the register space.

        Args:
            inputfile (str | file obj): File to load the registers/fields from.
            usewhat (str): this specifies what part of a logregisters to load.
                If registers are chosen those values will be used instead of
                the field values:

                    - 'fields' or 'f'.
                    - 'registers or 'r'.

            errout (Optional[str]): How to report any errors when writing
                the registers/fields:

                    - 'exception' (Default).
                    - 'warn'.
                    - 'silent'.

            silent (Optional[bool]): Skip displaying the % complete message.
                False by default.

            binary_input (Optional[bool]): Set to True to load register/field
                values in binary format.
        """

        valid_kwargs = ["errout", "silent", "binary_input"]
        valid_usewhat = ["registers", "r", "fields", "f"]
        valid_errout = ["exception", "warn", "silent"]
        valid_silent = [True, False]
        valid_binary_input = [True, False]

        # Unpack arguments
        self._usewhat = usewhat
        self._errout = kwargs.pop("errout", "exception")
        self._silent = kwargs.pop("silent", False)
        self._binary_input = kwargs.pop("binary_input", False)

        # Validate arguments
        invalid_kwargs = list(set(kwargs.keys()) - set(valid_kwargs))
        assert len(invalid_kwargs) == 0, \
            "Invalid arguments (%s). Must be: %s" \
            % (", ".join(invalid_kwargs),
               ", ".join(valid_kwargs))
        assert self._usewhat in valid_usewhat, \
            "Invalid usewhat (%s). Must be: %s" \
            % (self._usewhat, ", ".join(valid_usewhat))
        assert self._errout in valid_errout, \
            "Invalid errout (%s). Must be: %s" \
            % (self._errout, ", ".join(valid_errout))
        assert self._silent in valid_silent, \
            "Invalid silent (%s). Must be: %s" \
            % (self._silent, valid_silent)
        assert self._binary_input in valid_binary_input, \
            "Invalid binary_input (%s). Must be: %s" \
            % (self._binary_input, valid_binary_input)

        # Get lines from file
        if hasattr(inputfile, 'readlines'):
            # We got a file object
            lines = inputfile.readlines()
        else:
            # We got a file path
            inputfile = open(inputfile)
            lines = inputfile.readlines()
            inputfile.close()

        # Process lines
        self._process_lines(lines)
