
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.



"""
Store bar formulas and values for use by various address/access methods.

"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from copy import copy
from ..logging import getLogger
from ..comp import ComponentPlugin
from ..utils._py2to3 import *
from dataclasses import dataclass
from . import nn_walk

_LOG = getLogger()


class Formula(object):
    """
    Abstract class for formulas. This formula is a simple raw string
    """
    def __init__(self, value):
        self._value = value

    @property
    def value(self):
        """
        Perform any calculations on the internal parts of this formula and return
        a string that can be evaluated.
        """
        return self._value

    def __str__(self):
        return self.value


class FormulaPlugin(ComponentPlugin):
    """
    This plugin adds the capability of adding and evaluating bar formulas.
    """

    name = "formulas"
    _known_policies = ['once', 'always']

    def __init__(self, component):
        """
        Overriden plugin constructor.

        Args:
            component (NamedComponent): The component this plugin will
                be attached to.
        """
        super(FormulaPlugin, self).__init__(component)
        # set default policy as 'always', this is different to PythonSv
        # where the default policy is None.
        self._bar_policy = 'always'
        # _formulas_values stores the current evaluated values
        # for the stored formulas.
        self._formulas_values = {}
        # initialize formulas from the component definition
        if self.component.definition.info.get('formulas', None) is None:
            self.component.definition.info['formulas'] = {}
        formulas = self.component.definition.info['formulas']
        for formula_object in formulas:
            formula_string = str(formula_object)
            self._formulas_values[formula_string] = None

    def add_formula(self, name, formula):
        """
        Add a bar formula for MMIO registers to use. These are added to the definition.info

        Args:
            name (string): The name of the formula for others to reference
                when getting bar value.
            formula ([Formula | string]): Formula to execute in order to
                determine bar's base address.

        Note:

            in most current cases, formula_name is referenced by a node/register
            in a key like "pcibar_name"

        Examples::

            # Raw string to evaluate with hardcoded address:
            component.add_formula("formula_name", "0xFE000000)
            # Raw string based on a register value:
            component.add_formula("formula_name", "self.sub_component.register & 0xFFFF0")
            # Use actual component paths, and mover utility will update these if it
            # is used to move component locations
            part1 = FormulaPart("sub_comp1.sub_comp2", "low_register", shift=0, mask=0xFFFF0)
            part2 = FormulaPart("sub_comp1.sub_comp2", "high_register", shift=32, mask=0xFFFF0)
            component.add_formula("formula_name", FormulaParts([part1,part2]))


        """
        if isinstance(formula, basestring):
            formula = Formula(formula)
        elif isinstance(formula, Formula):
            pass
        else:
            raise ValueError("Unknown formula type")

        self.component.definition.info['formulas'][name] = formula
        self._formulas_values[str(formula)] = None

    def lookup_formula(self, name, **kwargs):
        """
        Get the formula that will be used when asked to determine the
        value of a bar. If the formula is not found in the current
        component then a search up the parent component will be performed.

        Args:
            name (string): The name of the formula to get.
        """
        recursive = kwargs.pop('recursive', True)
        assert len(kwargs) == 0,\
            "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        formula = self.component.definition.info['formulas'].get(name, None)
        formula = self.component.target_info.get('formulas',{}).get(name, formula)
        if formula is not None:
            return formula
        elif recursive and self.component.parent is not None:
            return self.component.parent.formulas.lookup_formula(
                name, recursive=recursive)
        else:
            raise ValueError("Formula '%s' not defined" % name)

    def get_formula(self, name):
        """
        Get the formula that will be used when asked to determine the
        value of a bar.

        Args:
            name (string): The name of the formula to get.
        """
        return self.lookup_formula(name, recursive=False)

    def set_formula_policy(self, policy, recursive=True):
        """
        For recursively setting the formula policy on this component and all subomponents
        """
        if policy not in self._known_policies:
            raise ValueError(
                "Unknown formula policy '%s'. Valid policies are '%s'" %
                (policy, ", ".join(self._known_policies))
            )
        if recursive:
            for comp in self.component.walk_components():
                comp.formulas.formula_policy = policy
        else:
            self.formula_policy = policy

    @property
    def formula_policy(self):
        """
        Get the current policy for formula evaluation.
        """
        return self._bar_policy

    @formula_policy.setter
    def formula_policy(self, policy):
        """
        Set the current policy for formula evaluation.

        Args:
            policy (string): The bar policy to set. Valid policy options
                are "once" and "always".
        """
        if policy not in self._known_policies:
            raise ValueError(
                "Unknown formula policy '%s'. Valid policies are '%s'" %
                (policy, ", ".join(self._known_policies))
            )

        self._bar_policy = policy

    def lookup_value(self, name, **kwargs):
        """
        Returns the base address/calculated value for the bar that is being
        requested. Uses the current bar policy to determine whether it should
        re-evaluate the bar's number, or just return the last evaluated value.
        If the calculated value is not found in the current component then a
        search up the parent component will be performed.

        Args:
            name (string): The name of the formula to get the value from.
        """
        recursive = kwargs.pop('recursive', True)
        assert len(kwargs) == 0,\
            "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        formula = self.component.definition.info['formulas'].get(name, None)
        formula = self.component.target_info.get('formulas',{}).get(name, formula)
        if formula is None:
            if recursive and self.component.parent is not None:
                return self.component.parent.formulas.lookup_value(
                    name, recursive=recursive
                )
            else:
                raise ValueError("Formula '%s' not defined" % name)

        if self._bar_policy not in self._known_policies:
            raise ValueError("Unknown formula policy '%s'" % self._bar_policy)

        # must check against None because this could contain a registervalue object
        # and that object's read may/may not work
        formula_str = str(formula)
        if self._formulas_values.get(formula_str, None) is None or \
                self._bar_policy in ['always']:
            self._formulas_values[formula_str] = self._eval(formula_str, name)

        return self._formulas_values[formula_str]

    def get_value(self, name):
        """
        Returns the base address/calculated value for the bar that is being
        requested. Uses the current bar policy to determine whether it should
        re-evaluate the bar's number, or just return the last evaluated value.

        Args:
            name (string): The name of the formula to get the value from.
        """
        return self.lookup_value(name, recursive=False)

    def get_names(self):
        """
        Returns a list of all the bar names stored and used by various
        address/access functions.
        """
        formula_names = self.component.definition.info.get('formulas', {})
        formula_names = self.component.target_info.get('formulas',formula_names)
        return list(formula_names.keys())

    def show(self, recursive=True):
        """
        Display the known formula information.

        Args:
            recursive (boolean, optional): Set to False to disable the
                recursive query on the sub components. Defaults to True.

        Example output:

            >>> sv.socket0.uncore0.formulas.show()
            dmirc = 0x0000000000000000
                Formula = self.pxpd00f0_dmircbar & ~0xF
            cb_bar4 = 0x0000000000000000
                Formula = (self.cb4_cb_bar_1<<32) | (self.cb4_cb_bar_0 & ~0xF)
            cb_bar5 = 0x0000000000000000
                Formula = (self.cb5_cb_bar_1<<32) | (self.cb5_cb_bar_0 & ~0xF)
            cb_bar6 = 0x0000000000000000
                Formula = (self.cb6_cb_bar_1<<32) | (self.cb6_cb_bar_0 & ~0xF)
            cb_bar7 = 0x0000000000000000
        """
        if recursive:
            dump_components = self.component.walk_components()
        else:
            dump_components = [self.component]

        for component in dump_components:
            formula_names = component.formulas.get_names()
            if len(formula_names):
                _LOG.result("-" * 50)
                _LOG.result("Bars for %s" % component.path)
                _LOG.result("-" * 50)
                for name in formula_names:
                    try:
                        _LOG.result(
                            "  %s" % name + " = 0x%016x" %
                            component.formulas.get_value(name)
                        )
                    except Exception as e:
                        self._log_formula_exception(e, name)
                        continue
                    formula = component.formulas.get_formula(name)
                    _LOG.result(
                        "      Formula = %s" % str(formula)
                    )


    def update(self):
        """
        Update our currently stored base address value for all the bars.
        """
        formula_names = self.get_names()
        for name in formula_names:
            formula = self.get_formula(name)
            formula_str = str(formula)
            self._formulas_values[formula_str] = self._eval(formula_str, name)

    def _eval(self, formula, name):
        """
        Evaluate a formula and return the evaluated value.

        Args:
            formula (string): The formula to evaluate.
            name (string): The name of the formula.
        """
        try:
            value = eval(formula, {'self': self.component})
        except:
            import traceback
            original_error = traceback.format_exc()
            raise ValueError(
                "Error occurred evaluating formula:\n\t"
                "Name: %s, "
                "Formula: %s" % (name, formula) +
                "\n------ original error:\n" + 
                original_error
            ) 
        return value

    @property
    def formula_dict(self):
        """
        Get a formulas dictionary.

        Dictionary keys: The names of the stored formulas.
        Dictionary values: The actual formulas.
        """
        formula_dictionary = {}
        formula_names = self.get_names()
        for name in formula_names:
            formula = self.get_formula(name)
            formula_dictionary[name] = formula
        return formula_dictionary

    @property
    def value_dict(self):
        """
        Get a values dictionary.

        Dictionary keys: The names of the stored formulas.
        Dictionary values: The last evaluated formula values.
        """
        return copy(self._formulas_values)

    def _log_formula_exception(self, exception, formula_name):
        """ Handles known exceptions from Formula.get_value
        Valid exceptions is ValueError ( formula not defined and/or incorrect policy)
        This is needed to be able to call formula.show without errors even if formula.get_value
        fails.
        Args:
            exception (except obj): Error object given by the exception
            formula_name (str): Name of the formula that failed.
        """
        if  type(exception) == ValueError:
            if str(exception).find("defined") >= 0:
                _LOG.result("The formula '%s' is not properly defined"%(formula_name))
            elif str(exception).find("policy") >= 0:
                valid_policies_astr = ""
                for policy in self._known_policies:
                    valid_policies_astr = valid_policies_astr + "%s,"%(policy)
                _LOG.result("Formula %s with Incorrect policy '%s' [Valid ones: %s]"%\
                    (formula_name, self._bar_policy, valid_policies_astr))
            else:
                _LOG.result("Formula '%s' Raised an Unknown ValueError exception %s"%(formula_name, str(exception)))
        else:
            _LOG.result("Unknown exception type raised by formula: '%s', error:\n\t %s"%(formula_name,str(exception)))


class HiLoMaskFormula(Formula):
        """
        Bar formula abstraction having as components the name of the register that represents the low-portion of
        bar, a register (optional) that represent the high-portion of the bar, and a mask ( optional) that indicates
        the amount of bits to shift to create the final bar
        """
        _low_r_named = None
        _high_r_named = None
        _mask = None
        _shift_h = 32
        _shift_l = 0
        """
        Bar formula abstraction.
        """
        def __init__(self, low_r_named=None, high_r_named=None, mask=None):
            self._low_r_named = low_r_named
            self._high_r_named = high_r_named
            self._mask = mask
            self._update_formula()

        @property
        def register_low(self):
            return self._low_r_named

        @register_low.setter
        def register_low(self, value):
            self._low_r_named = value
            self._update_formula()

        @property
        def register_high(self):
            return self._high_r_named

        @register_high.setter
        def register_high(self, value):
            self._high_r_named = value
            self._update_formula()

        @property
        def mask(self):
            return self._mask

        @mask.setter
        def mask(self, value):
            self._mask = value
            self._update_formula()

        @property
        def shift_h(self):
            return self._shift_h

        @shift_h.setter
        def shift_h(self, value):
            self._shift_h = value
            self._update_formula()

        @property
        def shift_l(self):
            return self._shift_h

        @shift_l.setter
        def shift_l(self, value):
            self._shift_l = value
            self._update_formula()

        def _update_formula(self):
            """
            internal method that build the register string. It uses the register names: low and high and combines
            them with the mask to create a final register string that looks like
            high<<mask | low
            Returns:

            """
            if self._high_r_named is not None and self._mask is not None and self._shift_l == 0:
                self._value = "((%s << %d)|%s)& 0x%x" % (self._high_r_named, self._shift_h, self._low_r_named, self._mask)
            elif self._high_r_named is not None and self._mask is not None and self._shift_l != 0:
                self._value = "((%s << %d)|(%s << %d))& 0x%x" % (self._high_r_named, self._shift_h, self._low_r_named,self._shift_l, self._mask)
            else:
                if self._mask is not None:
                    self._value = "(%s)& 0x%x" % (self._low_r_named, self._mask)
                else:
                    self._value = self._low_r_named



class FormulaPartStart:
    """supported types for specifying where the path starts from in a FormulaParts formula"""
    ORIGIN = "origin"
    CURRENT = "current"
    _valid_values = [ORIGIN, CURRENT]

@dataclass
class FormulaPart:
    """Multiple of these objects are passed in to the FormulaParts formula to help build up a formula string"""
    #: path to the component with the register
    component_path: str = ""
    #: register off of the specified component
    register_name: str = ""
    #: number of bits that the register should be shifted left as part of the final formula computation
    shift: int = 0
    #: mask that should be applied before shifting the register
    mask: int = ((1 << 32)-1)


class FormulaParts(Formula):

    def __init__(self, parts, path_start=FormulaPartStart.CURRENT):
        """Must be a list of formula parts. The path in each formula part is assumed
        to be relative to the components.origin

        Args:
            parts (list) : list of FormulaPart objects. the path is relative to whatever is given in path_start
            path_start : determines whether the paths are relative to the current component or the origin (current
                         component is the recommended way)

        """
        args_check_error = "must provide one or more FormulaPart for this formula"
        if not isinstance(parts, list):
            raise TypeError(args_check_error)
        for p in parts:
            if not isinstance(p, FormulaPart):
                raise TypeError(args_check_error)
        self.parts = parts
        self.path_start = path_start

    @property
    def value(self):
        formula_string = []
        if self.path_start == FormulaPartStart.ORIGIN:
            start = "self.origin"
        else:
            start = "self"
        for part in self.parts:
            formula_string.append(f"(({start}.{part.component_path}.{part.register_name} & 0x{part.mask:X}) << {part.shift})")
        return "|".join(formula_string)

