
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.





"""

"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import pprint
import re
import six

from ..logging import getLogger

# some of these functions require the walk plugin
from . import nn_walk as _nn_walk

from ..comp import ComponentPlugin
from ..nodes import NodeTypes
from ..nodes import NodeValuePlugin
from ..nodes import NamedNodeArrayValue
from ..registers import RegisterValue
from ..registers import FieldValue
from ..registers import NamedNodeValue
from ..precondition import MultipleAccesses

from ..utils._py2to3 import *
from ..utils.stringcmp import alpha_numeric_cmp
from ..utils._naturalsort import _natural_keys
from namednodes import settings
from functools import cmp_to_key
import string
import sys
import traceback
import operator
import logging

_log = getLogger()


### TODO:MOVE THIS TO UTILITIES FILE
# code from here:
# http://code.activestate.com/recipes/440694-determine-size-of-console-window-on-windows/
def getConsoleSize():
    from ctypes import windll, create_string_buffer

    # stdin handle is -10
    # stdout handle is -11
    # stderr handle is -12

    h = windll.kernel32.GetStdHandle(-12) #@UndefinedVariable - used by pydev in eclipse
    csbi = create_string_buffer(22)
    res = windll.kernel32.GetConsoleScreenBufferInfo(h, csbi) #@UndefinedVariable

    if res:
        import struct
        (bufx, bufy, curx, cury, wattr, #@UnusedVariable
         left, top, right, bottom, maxx, maxy) = struct.unpack("hhhhHhhhhhh", csbi.raw) #@UnusedVariable
        sizex = right - left + 1
        sizey = bottom - top + 1
    else:
        sizex, sizey = 80, 25 # can't determine actual size - return default values

    return sizex, sizey


class GetSpecPlugin(NodeValuePlugin):
    """
    This plugin provides the "getspec" command to nodes. It will print
    out register attributes, address, description and field description
    information.
    """

    name = "getspec"

    @classmethod
    def create(cls, nodevalue):
        # For now, this plugin adds support just for
        # RegisterValue and FieldValue nodes
        if nodevalue.type in [NodeTypes.Register, NodeTypes.Field]:
            return cls(nodevalue)
        else:
            return None

    def _breakline(self, line, maxlength, prefix=""):
        """
        Breaks a line of text in as many lines as necessary where each
        new line is no longer than the specified maximum length.

        Args:
            line (str): The line to break.
            maxlength (int): The maximum length for each new line
            prefix (str, optional): The prefix to each new line. Defaults
                to empty string ("").

        Returns:
            list: Contains all lines of text that resulted from breaking
                the original line.
        """
        lines = []
        cursorposition = 0
        while cursorposition < len(line):
            # if we have newline just add cursorposition up to the newline
            nextline = line[cursorposition:(cursorposition + maxlength)]
            newline = nextline.find("\n")
            if newline != -1:
                lines.append("%s %s" % (prefix, nextline[:newline]))
                cursorposition = cursorposition + newline + 1 # start after the new line
            elif len(nextline) < maxlength: # less than maxlength, this will be the last line
                lines.append("%s %s" % (prefix, nextline))
                cursorposition = cursorposition + len(nextline) + 1 # start after the newline, should end this loop
            else: # line is longer than maxlength
                space = nextline.rfind(" ")
                if space != -1: # good, we found a space
                    lines.append("%s %s" % (prefix, nextline[:space]))
                    cursorposition = cursorposition + space + 1 # start after the newline
                else:
                    lines.append("%s %s-" % (prefix, nextline[:maxlength-1])) # show dash to indicate incomplete
                    cursorposition = cursorposition + maxlength # start after maxlength, no +1 since we used a - on the print

        return lines

    def _getspec_register(self, quiet=False, noread=False):
        """
        Print out register attributes, description and field description
        information.

        Args:
            quiet (bool, optional): Used to return spec string instead of
                displaying it. Defaults to False.
            noread (bool, optional): Used to indicate whether to perform
                a read or not. Defaults to False.
        """
        if sys.platform != 'win32':
            numcols = 79
        else:
            numcols, numrow = getConsoleSize()

        prefix_length = 3
        numcols -= (1 + prefix_length)
        specstr = []
        specstr.append("#" * (numcols + prefix_length))
        specstr.append("## Register name: %s" % self.nodevalue.name)

        # get node's info
        specstr.append("## %s" % ("-"*10))
        description = ""
        for k, v in self.nodevalue.definition.info.items():
            if k == 'description':
                description = v

            if isinstance(v, float):
                value_display = "%s = %f" % (k, v)
            elif isinstance(v, (long, int)):
                value_display = "%s = 0x%08x" % (k, v)
            else:
                value_display = "%s = %s " % (k, v)
            multiline_value = self._breakline(value_display, numcols, "##")
            specstr.extend(multiline_value)

        specstr.append("## %s" % ("-"*10))
        specstr.append("## Description:")
        description = description.replace("<br>", "\n") if description else ""
        multiline_description = self._breakline(description, numcols, "##")
        specstr.extend(multiline_description)

        for child in self.nodevalue.nodes:
            # the True is to get the spec info instead of displaying it
            child_spec = self._getspec_field(child, True, noread)
            specstr.append(child_spec)

        display = "\n".join(specstr)

        if quiet:
            return display
        else:
            _log.result(display)

    def _getspec_field(self, node, quiet=False, noread=False):
        """
        Print out field attributes and field description information.

        Args:
            node (FieldValue): The field for which to get the spec.
            quiet (bool, optional): Used to return spec string instead of
                displaying it. Defaults to False.
            noread (bool, optional): Used to indicate whether to perform
                a read or not. Defaults to False.
        """
        if sys.platform != 'win32':
            numcols = 79
        else:
            numcols, numrow = getConsoleSize()

        prefix_length = 2 # this is for the "#" prefix added later
        numcols -= (1 + prefix_length) # this is -1 for to prevent wrap nd prefix
        specstr = []
        specstr.append("#" * (numcols + prefix_length))
        specstr.append("# FIELD: %s" % node.name)
        attribute = node.definition.info.get('attribute', None)
        if attribute is not None:
            specstr.append("# Attribute: %s" % attribute)

        # For fields, make sure to print the lowerbit and upperbit
        # in the right place [after 'Attribute' for backwards compatiblity],
        # but if the field does not have upperbit and lowerbit attributes
        # then don't print a default, just don't print them
        upper = node.upperbit
        lower = node.lowerbit
        if upper != None and lower != None:
            specstr.append("# UpperBit: %d" % upper)
            specstr.append("# LowerBit: %d" % lower)

        if not noread:
            value = node.value or node.get_value()
            if value is not None:
                specstr.append("# CurrentValue: %s" % node.value)

        specstr.append("# Description: ")
        description = node.definition.info.get('description', None)
        description = description.replace("<br>", "\n") if description else ""
        multiline_description = self._breakline(description, numcols, "#")
        specstr.extend(multiline_description)

        display = "\n".join(specstr)
        if quiet:
            return display
        else:
            _log.result(display)

    def __call__(self, quiet=False, noread=False):
        """
        Print out register attributes, address, description and field
        description information

        Args:
            quiet (bool, optional): Used to return spec string instead of
                displaying it. Defaults to False.
            noread (bool, optional): Used to indicate whether to perform
                a read or not. Defaults to False.
        """
        node = self.nodevalue
        if node.type == NodeTypes.Register:
            try:
                return self._getspec_register(quiet, noread)
            except ValueError:
                return self._getspec_register(quiet, noread=True)
            else:
                raise
        elif node.type == NodeTypes.Field:
            try:
                return self._getspec_field(node, quiet, noread)
            except ValueError:
                return self._getspec_field(node, quiet, noread=True)
            else:
                raise
        elif node.type == NodeTypes.Array:
            # TODO Define how getspec() will behave for NamedNodeArrayValue
            pass
        elif node.type == NodeTypes.General:
            # TODO Define how getspec() will behave for NamedNodeValue
            pass


class ShowNodePlugin(NodeValuePlugin):
    """
    This plugin provides the "show" command to nodes. It will display the
    current nodes value, and the upperbit/lowerbit if those attributes exist.

    If the encode/decode plugin is present and there is a value, then that
    will be displayed.

    Some optional arguments are available to tweak the output

    Args:
        recursive : (not supported yet) - show children node values
    """

    __display__ = True
    name = "show"

    @classmethod
    def create(cls, nodevalue):
        # For now, this plugin does not add support for NamedNodeArrayValue
        if nodevalue.type != NodeTypes.Array:
            return cls(nodevalue)
        else:
            return None

    def _show_register(self, prefix="", non_zero_only=False, **kwargs):
        """
        Displays all register's fields and their values in the format of:

            fieldval : fieldname (upperbit:lowerbit)

            0x00000000 : vendor_id (15:0)

        Args:
            prefix (str, optional): String to prefix to the display format.
                Defaults to empty string ("").
            non_zero_only (bool, optional): Used to skip displaying fields
                with 0 value. Defaults to False.
            **kwargs: Arbitrary keyword arguments, it can include any of
                the following arguments.
            returnstr (bool, optional): Used to return string instead of
                displaying it. Defaults to False.
            logger (Logger, optional): Used to pass in a different logger
                than the default one. Defaults to the default logger.
            show_path (bool, optional): Used to show the node's path instead
                of it's name. Defaults to False
            show_description (bool) : Optional, used to not show description in output (default is true)
            show_decode (bool) : Optional, used to not show decode in output (default is true)
        """    
        returnstr = kwargs.pop('returnstr', False)
        log = kwargs.pop('logger', _log)
        indent = kwargs.pop('indent', "")
        show_path = kwargs.pop('show_path', False)
        show_description = kwargs.pop('show_description', True)
        show_decode = kwargs.pop('show_decode', True)
        assert len(kwargs) == 0, "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        display = ''
        if indent != "":
            prefix = indent

        if len(self.nodevalue.nodes) > 0:
            for child in self.nodevalue.nodes:
                if non_zero_only and child.get_value() == 0:
                    continue

                # PythonSv had a bug where "indent" was not passed in to the
                # show() call for register fields, we are fixing that here.
                display += "%s\n" % self._show_node(child, prefix,
                                                    returnstr=True,
                                                    show_path=show_path,
                                                    show_description=show_description,
                                                    show_decode=show_decode,
                                                    )
        else:  # this is for when the node doesn't have children
            display = "%s\n" % self._show_node(self.nodevalue, prefix,
                                               returnstr=True,
                                               show_path=show_path,
                                               show_description=show_description,
                                               show_decode=show_decode,
                                               )
        if returnstr:
            return display
        else:
            log.result(display)

    def _show_node(self, node, indent="", **kwargs):
        """
        Displays a field and its values in the format of:

            fieldval : fieldname (upperbit:lowerbit)

            0x00000000 : vendor_id (15:0)

        Args:
            node (NamedNodeValue, FieldValue): The node for which to display
                its values.
            indent (str, optional): String to prefix to the display format.
                Defaults to empty string ("").
            **kwargs: Arbitrary keyword arguments, it can include any of
                the following arguments.
            returnstr (bool, optional): Used to return string instead of
                displaying it. Defaults to False.
            logger (Logger, optional): Used to pass in a different logger
                than the default one. Defaults to the default logger.
            show_path (bool, optional): Used to show the node's path instead
                of it's name. Defaults to False
            show_description (bool) : Optional, used to not show description in output (default is true)
        """ 
        returnstr = kwargs.pop('returnstr', False)
        log = kwargs.pop('logger', _log)
        show_path = kwargs.pop('show_path', False)
        show_description = kwargs.pop('show_description', True)
        show_decode = kwargs.pop('show_decode', True)
        assert len(kwargs) == 0, "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        if sys.platform != 'win32':
            numcols = 79
        else:
            numcols, numrow = getConsoleSize()

        name = node.name if show_path == False else node.path
        #### lower
        # lowerbit might be set in target_info
        lower = node.target_info.get("lowerbit", None)
        if lower is None:
            lower = node.definition.info.get('lowerbit', None)
        #### upper
        upper = None
        numbits = getattr(node,"numbits", None)
        if numbits is None:
            # this is still used for AFD)
            upper = node.definition.info.get("upperbit")
        if upper is None and numbits is not None and lower is not None:
            upper = lower + numbits -1
        value = node.value or node.get_value()
        nodetype = node.value_type
        # if we get upper and lower bit, assume it is field, and that hex is fine
        if upper != None and lower != None:
            node_display = "%s0x%08x : %s (%02d:%02d)" % (indent, value, name, upper, lower)
        else:
            if nodetype == float:
                node_display = "%s%f : %s" % (indent, value, name)
            elif nodetype in [long, int]:
                node_display = "%s0x%08x : %s" % (indent, value, name)
            else:
                node_display = "%s%s : %s" % (indent, value, name)

        try:
            decode = node.decode()
        except LookupError as e:
            # give the user the error we hit so they can see what enum was missing
            decode = str(e)
        else:
            if decode is not None:
                decode = str(decode)
        description = node.definition.info.get('description', None)

        # shown node description is formed by the decode, if there is one,
        # plus the actual description
        node_description = ""

        # add node decode
        if decode is not None and show_decode:
            node_description = " -- " + decode

        # add node description
        if description is not None and show_description:
            node_description += " -- " + description
            # replace any new lines in the description with a space
            node_description = node_description.replace("\n", " ")
            node_description = node_description.replace("<br>", " ")
            # remove any carriage returns in the description
            node_description = node_description.replace("\r", "")

            line_length = len(node_display) + len(node_description)
            if line_length >= numcols:
                max_line_length = numcols-3-len(node_display)-1
                node_description = node_description[0:max_line_length] + "..."

        display = "%s%s" % (node_display, node_description)
        if returnstr:
            return display
        else:
            log.result(display)

    # Reminder, when this is created we have
    # self.component and self.nodevalue available to us
    def __call__(self, **kwargs):
        """
        Displays all register's fields (or just fields) and the values in the
        format of:

            fieldval : fieldname (upperbit:lowerbit)

            0x00000000 : vendor_id (15:0)

        Args:
            indent (str, optional): String to prefix to the display format.
                Defaults to empty string ("").
            non_zero_only (bool, optional): Used to skip displaying fields
                with 0 value. Defaults to False.
            **kwargs: Arbitrary keyword arguments, it can include any of
                the following arguments.
            returnstr (bool, optional): Used to return string instead of
                displaying it. Defaults to False.
            logger (Logger, optional): Used to pass in a different logger
                than the default one. Defaults to the default logger.
            show_path (bool, optional): Used to show the node's path instead
                of it's name. Defaults to False
        """
        node = self.nodevalue
        if node.type == NodeTypes.Register:
            return self._show_register(**kwargs)
        elif node.type == NodeTypes.Field:
            return self._show_node(node, **kwargs)
        elif node.type == NodeTypes.Array:
            # TODO Define how show() will behave for NamedNodeArrayValue
            pass
        elif node.type == NodeTypes.General:
            return self._show_register(**kwargs)


class ShowFieldsNodePlugin(ShowNodePlugin):
    """
    This plugin provides the "showfields" command to nodes. It will display the
    current nodes value, and the upperbit/lowerbit if those attributes exist.

    It leverages the existing "show" plugin but does not show the
    Some optional arguments are available to tweak the output

    Args:
        recursive : (not supported yet) - show children node values
    """

    __display__ = True
    name = "showfields"

    # Reminder, when this is created we have
    # self.component and self.nodevalue available to us
    def __call__(self, **kwargs):
        """
        Displays all register's fields (or just fields) and the values in the
        format of:

            fieldval : fieldname (upperbit:lowerbit)

            0x00000000 : vendor_id (15:0)

        Args:
            indent (str, optional): String to prefix to the display format.
                Defaults to empty string ("").
            non_zero_only (bool, optional): Used to skip displaying fields
                with 0 value. Defaults to False.
            **kwargs: Arbitrary keyword arguments, it can include any of
                the following arguments.
            returnstr (bool, optional): Used to return string instead of
                displaying it. Defaults to False.
            logger (Logger, optional): Used to pass in a different logger
                than the default one. Defaults to the default logger.
            show_path (bool, optional): Used to show the node's path instead
                of it's name. Defaults to False
        """
        node = self.nodevalue
        kwargs['show_description']=False
        kwargs['show_decode'] = True
        if node.type == NodeTypes.Register:
            return self._show_register(**kwargs)
        elif node.type == NodeTypes.Field:
            return self._show_node(node, **kwargs)
        elif node.type == NodeTypes.Array:
            # TODO Define how show() will behave for NamedNodeArrayValue
            pass
        elif node.type == NodeTypes.General:
            return self._show_register(**kwargs)





class ValueStr(object):
    """
    This class implements the behavior of the "valuestr" and "showvalues"
    commands.
    It is meant to be used by the ValueStrPlugin and ShowValuesPlugin
    plugins only.
    """
    def __init__(self, nodevalue):
        self.nodevalue = nodevalue

    def _valuestr_register(self, **kwargs):
        """
        Returns a string with the register and field values. Used by the
        logregisters to build string for showvalues. The format is::

            register = 0x0
            register.field0 = 0 - decode value
            register.field1 = 0 - decode value

        Args:
            prefix (str, optional): Used to add indentation or component
                name as prefix. Defaults to empty string ("").
            name (str, optional): Name to display in place of the register's
                name. Defaults to the register's name.
            binary_output (bool, optional): Set to True to dump register/field
                values in binary format.
            skip_fields (bool, optional): Skip field values. Default is False.
            include_path (bool, optional): include the path when display the register
        """

        prefix = kwargs.pop('prefix', "")
        name = kwargs.pop('name', None)
        non_zero_only = kwargs.pop('non_zero_only', False)
        binary_output = kwargs.pop('binary_output', False)
        skip_fields = kwargs.pop("skip_fields", False)
        include_path = kwargs.pop("include_path", False)
        assert len(kwargs) == 0, "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        output_string = []
        value = self.nodevalue.value or self.nodevalue.get_value()

        if non_zero_only and value == 0:
            return '\n'.join(output_string)

        # make sure item_key is included in array items name
        if name is None:
            name = self.nodevalue.name
        else:
            item_key = self.nodevalue.target_info.get("item_key", None)
            if item_key is not None:
                if isinstance(item_key, six.string_types):
                    name  = name + "[\"%s\"]" % item_key
                else:
                    name = name + "[%s]" % str(item_key)
            # else: Don't need to do anything as node is not an array

        pathstr = self.nodevalue.parent.path+"." if include_path else ""

        if binary_output:
            numbits = self.nodevalue.numbits
            if numbits is None:
                upperbit = self.nodevalue.definition.info.get("upperbit", None)
                lowerbit = self.nodevalue.definition.info.get("lowerbit", None)
                if upperbit is not None and lowerbit is not None:
                    numbits = upperbit - lowerbit + 1
                else:
                    numbits = 0

            value = ('{:0%ib}' % numbits).format(value)
            output_string.append("%s%s%s=0b%s" % (prefix, pathstr, name, value))
        else:
            output_string.append("%s%s%s=0x%x" % (prefix, pathstr, name, value))

        if skip_fields:
            return '\n'.join(output_string)

        # combine the field values
        for child in self.nodevalue.nodes:
            if non_zero_only and child.get_value() == 0:
                continue
            output_string.append("    %s%s.%s" % (prefix, name,
                                                  child.valuestr(binary_output=binary_output)))
        return '\n'.join(output_string)

    def _valuestr_field(self, **kwargs):
        """
        Returns a string with the field, and its value. If the decode for
        this field is known, then the definition of the field value will
        be appended also.

        Args:
            prefix (str, optional): Text to prepend to the field=value
                string. Defaults to empty string ("").
            binary_output (bool, optional): Set to True to dump field
                value in binary format.
        """
        prefix = kwargs.pop('prefix', "")
        non_zero_only = kwargs.pop('non_zero_only', False)
        binary_output = kwargs.pop('binary_output', False)

        assert len(kwargs) == 0, "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        value = self.nodevalue.value or self.nodevalue.get_value()

        decode = self.nodevalue.definition.info.get("enum", None)
        if decode is not None:
            decode = self.nodevalue.decode()
            if decode is not None:
                decode = str(decode)

        if non_zero_only and value == 0:
            output_string = ""
            return output_string

        if decode is None:
            if binary_output:
                numbits = self.nodevalue.numbits
                value = ('{:0%ib}' % numbits).format(value)
                output_string = "%s%s=0b%s" % (prefix, self.nodevalue.name, value)
            else:
                output_string = "%s%s=0x%x" % (prefix, self.nodevalue.name, value)
        else:
            if binary_output:
                numbits = self.nodevalue.numbits
                value = ('{:0%ib}' % numbits).format(value)
                output_string = "%s%s=0b%s - %s" % \
                    (prefix, self.nodevalue.name, value, decode)
            else:
                output_string = "%s%s=0x%x - %s" % \
                    (prefix, self.nodevalue.name, value, decode)

        return output_string

    def _valuestr(self, **kwargs):
        if self.nodevalue.type == NodeTypes.Register:
            return self._valuestr_register(**kwargs)
        elif self.nodevalue.type == NodeTypes.Field:
            return self._valuestr_field(**kwargs)
        elif self.nodevalue.type == NodeTypes.Array:
            valuestr = []
            for item in self.nodevalue:
                if item.type == NodeTypes.Register:
                    valuestr.append(item.valuestr(**kwargs))
            return "\n".join(valuestr) or None

    def _showvalues(self, **kwargs):
        """
        Displays the result of calling valuestr, takes an optional logger,
        for logging to a file.

        Args:
            logger (Logger, optional): Used to pass in a different logger
                than the default one. Defaults to the default logger.
            log_level (str, optional): specify a logging level to use
                for when sending the output to the logger

        """
        logger = kwargs.pop("logger", None)
        log_level = kwargs.pop("log_level", None)
        if logger is None:
            logger = _log
        if log_level is None:
            # we know result exists because we know CLI adds it/supports it
            log_level_number = logging.RESULT
        else:
            log_level_number = getattr(logging,log_level.upper(), None)
            assert isinstance(log_level_number, int), "Could not find log level %s"%log_level
        value_str = self._valuestr(**kwargs)
        if value_str != "":
            # need to call logger per line to not mess up log files
            for line in value_str.split("\n"):
                logger.log(log_level_number, line)

class ValueStrPlugin(NodeValuePlugin, ValueStr):
    """
    This plugin provides the "valuestr" command to nodes. It will return
    a string with the node's name and value information.
    """

    name = "valuestr"

    @classmethod
    def create(cls, nodevalue):
        # This plugin adds support for RegisterValue
        # and FieldValue nodes
        if nodevalue.type in [NodeTypes.Register, NodeTypes.Field, NodeTypes.Array]:
            return cls(nodevalue)
        else:
            return None

    def __call__(self, **kwargs):
        """
        Returns a string with the register and field values. Used by the
        logregisters to build string for showvalues. The format is::

            register = 0x0
            register.field0 = 0 - decode value
            register.field1 = 0 - decode value

        Args:
            prefix (str, optional): Used to add indentation or component
                name as prefix. Defaults to empty string ("").
            name (str, optional): Name to display in place of the register's
                name. Defaults to the register's name.
            binary_output (bool, optional): Set to True to dump register/field
                value in binary format.
        """
        return self._valuestr(**kwargs)


class ShowValuesPlugin(NodeValuePlugin, ValueStr):
    """
    This plugin provides the "showvalues" command to nodes. It will print,
    or log to a file, the result of calling the "valuestr" command on a node.
    """

    name = "showvalues"

    @classmethod
    def create(cls, nodevalue):
        # Plugin that adds support for RegisterValue
        if nodevalue.type == NodeTypes.Register:
            return cls(nodevalue)
        else:
            return None

    def __call__(self, **kwargs):
        """
        Displays the result of calling valuestr, takes an optional logger,
        for logging to a file.

        Args:
            logger (Logger, optional): Used to pass in a different logger
                than the default one. Defaults to the default logger.
        """
        self._showvalues(**kwargs)


class ShowSubsPlugin(ComponentPlugin):
    """
    This plugin provides the "show_subs" command to NamedComponent nodes.
    It will display the node's sub component nodes recursively in a tree
    like list.
    """

    name = "show_subs"

    def __call__(self, deepness=-1, level=0, **kwargs):
        """
        Prints the names of the sub components of this NamedComponent.

        Args:
            deepness (long): Used to specify the depth of the sub components
                that we want to get in the result. Defaults to -1.
            level (long): Used to keep record of the depth level we are
                currently exploring. Defaults to zero.
            logger (Logger, optional): Used to pass in a different logger
                than the default one. Defaults to the default logger.
        """
        log = kwargs.pop('logger', _log)
        assert len(kwargs) == 0, \
            "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        output = ""
        for i in range(level):  # @UnusedVariable
            output += "  "
        if level > 0:
            output += "\\"
        output += self.component.path.rsplit(".", 1).pop()
        log.result(output)

        if level == deepness:
            return ""

        sub_components = self.component.sub_components
        for component in sorted(sub_components.keys(), key=cmp_to_key(alpha_numeric_cmp)):
            # Intentionally removed the check for the component having
            # show_subs attribute because it seemed redundant.
            sub_components[component].show_subs(deepness, level+1, **kwargs)


class ShowSubsLegacyPlugin(ShowSubsPlugin):
    """
    This plugin is used to rename the parent ShowSubsPlugin and provide
    backwards compatibility with the "showSubs" command from PythonSv.

    All functionality is implemented in ShowSubsPlugin parent class.
    """

    name = "showSubs"


class GetDefaultPlugin(NodeValuePlugin):
    """
    This plugin provides the "getdefault" command to RegisterValue nodes.

    It returns the default value for a RegisterValue node. Note that
    the default value is obtained from the value's definition.
    """

    name = "getdefault"

    @classmethod
    def create(cls, nodevalue):
        # This plugin just adds support for RegisterValue nodes
        if nodevalue.type == NodeTypes.Register:
            return cls(nodevalue)
        else:
            return None

    def __call__(self):
        """
        Returns the default value for a RegisterValue.

        The default value is obtained from the definition, not the value.
        """
        # If no fields are present try to get default from node
        if len(self.nodevalue.definition.nodes) == 0:
            return self.nodevalue.definition.info.get("default", 0)

        # Calculate default value from fields
        defaultvalue = 0
        for node in self.nodevalue.definition.nodes:
            info = node.info
            if 'default' in info and 'lowerbit' in info:
                defaultvalue |= node.default << node.lowerbit
        return defaultvalue


class WriteRegisterDefaultPlugin(ComponentPlugin):
    """Write default value to all registers."""
    name = "writedefaults"

    def _write_defaults(self, component):
        for node in component.walk_nodes():
            if isinstance(node, RegisterValue):
                node.write(node.getdefault())

    def __call__(self, recursive=True):
        """
        Writes the default value to all register nodes.

        Args:
            recursive (bool, optional): write defaults for sub-components
        """
        if recursive:
            for sub in self.component.walk_components():
                self._write_defaults(sub)
        else:
            self._write_defaults(self.component)


class ShowComponentPlugin(ComponentPlugin):
    """
    This plugin provides the "show" command to components. It will display the
    current nodes value and name if the component has nodes otherwise
    subcomponents name will be displayed.
    """

    name = "show"

    def __call__(self, sort_by=None, reverse=False, **kwargs):
        stop_on_error = kwargs.pop("stop_on_error", False)
        if len(kwargs)>0:
            raise TypeError("Unknown arguments: %s"%str(list(kwargs.keys())))
        if sort_by is None:
            sort_by = settings.DISPLAY_SHOW_COMP_SORT
        if not len(self.component.nodes) and len(self.component.sub_components):
            try:
                sorted_subcomponents = sorted(self.component.sub_components.values(), key=operator.attrgetter(sort_by),
                                              reverse=reverse)
            except AttributeError:
                _log.debug("%s not valid as a sort parameter" % sort_by)
                _log.debug("will default to sort by name")
                sorted_subcomponents = sorted(self.component.sub_components.values(),
                                              key=operator.attrgetter(settings.DISPLAY_SHOW_COMP_SORT),
                                              reverse=reverse)
            for subcomponent_name in sorted_subcomponents:
                _log.result("<subcomponent> : %s" % subcomponent_name.name)
        else:
            with MultipleAccesses(silent=True):
                # Expand arrays, if any
                nodes = []                
                for node in self.component.nodes:
                    if node.type == NodeTypes.Array:                        
                        for child in node:
                            nodes.append(child)
                            try:
                                # fill nodes value cache
                                child.get_value()
                            except Exception as e:
                                if stop_on_error:
                                    raise
                                child.value = None
                                _log.debug("Failed to read %s during show, received: %s" % (node.path,
                                                                                            traceback.format_exc()))


                    else:
                        nodes.append(node)
                        try:
                            # fill nodes value cache
                            node.get_value()
                        except Exception as e:
                            if stop_on_error:
                                raise
                            node.value = None
                            _log.debug("Failed to read %s during show, received: %s" % (node.path,
                                                                                        traceback.format_exc()))
            # Sort nodes
            try:
                if sort_by in nodes[0].definition.info:
                    sorted_nodes = sorted(nodes,
                                          key=lambda node: node.definition.info[sort_by],
                                          reverse=reverse)
                elif sort_by == settings.DISPLAY_SHOW_COMP_SORT:
                    sorted_nodes = sorted(nodes,                
                                           key=_natural_keys, reverse=reverse)
                else:
                    sorted_nodes = sorted(nodes,
                                        key=lambda node: getattr(node, sort_by),
                                        reverse=reverse)                    
            except:
                _log.debug(
                    "%s can't be used to sort this list as it is not a member, will default to sort by name" %
                    sort_by)
                sorted_nodes = sorted(nodes,                
                                        key=_natural_keys, reverse=reverse)    

            for node in sorted_nodes:
                value = node.value
                nodetype = node.value_type
                try:
                    if value is not None:
                        if nodetype == float:
                            node_display = "%f : %f" % (value, node.name)
                        elif nodetype in [long, int]:
                            node_display = "0x%016x : %s" % (nodetype(value), node.name)
                        else:
                            node_display = "%s : %s" % (value, node.name)
                    else:
                        node_display = "<could_not_read> : %s" % node.name
                except:
                    if stop_on_error:
                        raise
                    node_display = "<could_not_read> : %s" % node.name
                _log.result(node_display)


class ComponetShowSearch(ComponentPlugin):
    """This plugin provides the showsearch command at the component level
    It accepts the following search types: 'registers' or 'r'  to search registers only
    and 'fields' or 'f' to search for fields within the registers"""
    name = "showsearch"

    def __call__(self, pattern, search_type='registers', **kwargs):
        """
        Displays and logs the register.field or register = value for a field or fields within a register
        Args:
            pattern:The pattern or regex to find
            search_type: defaults to 'registers', but supports the same search types as the underlying 'search' command
            logger: optional, an svtools.loggging.toolbox logger that overrides the local one
            so the user can log to their custom files
        Returns:
            Nothing

        """
        logger = kwargs.pop("logger", _log)
        stop_on_error = kwargs.pop("stop_on_error", False)
        if search_type in ['d','description']:
            raise ValueError("Description not supported for search type")
        try:
            search_result = self.component.search(pattern, search_type, **kwargs)
        except:
            logger.exception("underlying search failed")
            raise
        # get node values inside the multiple, then display outside of it
        nodes = []
        with MultipleAccesses(silent=True):
            for node_name in search_result:
                node_obj = None
                try:
                    node_obj = self.component.get_by_path(node_name)
                    node_obj.get_value()
                except:
                    if stop_on_error:
                        raise
                nodes.append((node_name, node_obj))
        for node_name, node in nodes:
            # make sure to compare node to None (b/c zero is valid)
            try:
                if node is not None and node.value is not None:
                    logger.result("%s = %s" % (node_name, node))
                elif node is not None and node.value is None:
                    logger.result("%s = Failed to get value"%node_name)
                elif node is None:
                    logger.result("%s = Was missing somehow"%node_name)
                else:
                    raise RuntimeError("should be unreachable code...")
            except:
                if stop_on_error:
                    raise
                logger.result("%s = Failed to get value" % node_name)


class RegisterShowSearch(NodeValuePlugin):
    """
    This plugin provides the 'showsearch' command to registers. It will display the current nodes values and names
    The plug in will try to match first as regular expression, if no match it'll try a simple find.
    """
    name = "showsearch"

    def __call__(self, pattern, **kwargs):
        """
        Displays and logs the register.field = value for a field or fields within a register
        Args:
            pattern: The pattern or regex to find
            logger: optional, an svtools.loggging.toolbox logger that overrides the local one
            so the user can log to their custom files
        Returns:
            Nothing
        """
        logger = kwargs.pop("logger", _log)
        try:
            search_result = self.nodevalue.search(pattern, getobj=True)
        except:
            logger.warning("%s not found with showsearch"%pattern)

        for  field_group in search_result:
            field_name = field_group[0]
            field_obj = field_group[1]
            logger.result("%s.%s = %x"%(self.nodevalue.name, field_name, field_obj))

class ShowAccessComponentPlugin(ComponentPlugin):
    """
    This plugin provides the "show_access" command to components. It will
    display the access settings for the current component and any subcomponents
    with access settings that differ.
    """

    name = "show_access"

    def __call__(self, deepness=-1, **kwargs):
        """
        Prints the current access settings.

        Args:
            deepness (long): Used to specify the depth of the sub components
                that we want to get in the result. Defaults to -1.
            logger (Logger, optional): Used to pass in a different logger
                than the default one. Defaults to the default logger.
        """
        log = kwargs.pop('logger', _log)
        assert len(kwargs) == 0, \
            "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        log.result(self._format_accesses(self.component, deepness))

    def _add_indent(self, text, indent):
        lines = [" " * indent + line for line in text.splitlines()] # pull apart and add indents
        text = "\n".join(lines) # put it back together
        return text

    def _format_accesses(self, component, deepness, parent_access=None, level=0):
        formatted = ""
        if level != deepness:
            component_access = pprint.pformat(component.target_info["access_settings"])
            if parent_access is None:
                formatted = "%s\n" % component.path
            if component_access != parent_access:
                formatted += "%s\n" % component_access
            children_accesses = {}
            for sub_component in component.sub_components.values():
                child_formatted = self._format_accesses(sub_component, deepness, component_access, level + 1)
                if child_formatted:
                    if child_formatted in children_accesses:
                        children_accesses[child_formatted].append(sub_component.name)
                    else:
                        children_accesses[child_formatted] = [sub_component.name]
            for child_access, names in children_accesses.items():
                for name in names:
                    formatted += "  \\%s\n" % name
                formatted += "%s\n" % self._add_indent(child_access, 3)
        return formatted

class ShowAccessChoiceComponentPlugin(ComponentPlugin):
    """
    This plugin provides the "show_access_choices" command to components. It will
    display the access settings for the current component and any subcomponents
    with access settings that differ.
    """

    name = "show_access_choices"

    def __call__(self, deepness=-1, **kwargs):
        """
        Prints the available access choices.

        Args:
            deepness (long): Used to specify the depth of the sub components
                that we want to get in the result. Defaults to -1.
            recursive (bool): By default True. If set to False, deepness is set to '1'                
            logger (Logger, optional): Used to pass in a different logger
                than the default one. Defaults to the default logger.
        """
        log = kwargs.pop('logger', _log)
        recursive = kwargs.pop('recursive', True)
        if recursive is False:
            deepness = 1
        assert len(kwargs) == 0, \
            "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        log.result(self._format_access_choices(self.component, deepness))

    def _normalize(self, choices):
        normalized = {}
        for group in sorted(choices):
            normalized[group] = []
            for access in sorted(choices[group]):
                normalized[group].append(access)
        return normalized

    def _pretty_formatter(self, choices):
        choices = self._normalize(choices)
        formatted = pprint.pformat(choices)
        return formatted

    def _add_indent(self, text, indent):
        lines = [" " * indent + line for line in text.splitlines()] # pull apart and add indents
        text = "\n".join(lines) # put it back together
        return text

    def _format_access_choices(self, component, deepness, parent_choices=None, level=0):
        formatted = ""
        if level != deepness:
            component_choices = self._pretty_formatter(component.get_access_choices())
            if parent_choices is None:
                formatted = "%s\n" % component.path
            if component_choices != parent_choices:
                formatted += "%s\n" % component_choices
            children_choices = {}
            for sub_component in component.sub_components.values():
                child_formatted = self._format_access_choices(sub_component, deepness, component_choices, level + 1)
                if child_formatted:
                    if child_formatted in children_choices:
                        children_choices[child_formatted].append(sub_component.name)
                    else:
                        children_choices[child_formatted] = [sub_component.name]
            for child_choice, names in children_choices.items():
                for name in names:
                    formatted += "  \\%s\n" % name
                formatted += "%s\n" % self._add_indent(child_choice, 3)
        return formatted

