
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ..nodes import NodeTypes, NodeValuePlugin
from ..utils.bitmanip import bits


class BitsPlugin(NodeValuePlugin):
    """
    This plugin provides the "bits" command to nodes of type Register
    and Field. It allows to get/set the value of certain bits.
    """

    name = "bits"

    @classmethod
    def create(cls, nodevalue):
        if nodevalue.type != NodeTypes.Array:
            return cls(nodevalue)
        else:
            return None

    def _bits_register(self, lowerbit, numberbits, newvalue=None):
        """
        Used to get/set the value of certain bits within the
        register (in case fields do not exist for these bits).

        Args:
            lowerbit (int): Least-significant bit to read/write.
            numberbits (int): Number of bits to read/write.
            newvalue (long, optional): If specified, then a modify
                the specified bits and write new value back out to.
        """
        if numberbits <= 0:
            raise ValueError(
                "numberbits cannot be zero or negative value (%d)" % numberbits
            )

        node = self.nodevalue
        register_length = node.numbits
        if register_length is not None:
            if (lowerbit + numberbits) > register_length:
                raise ValueError(
                    "bits command told given values (%d,%d) that extend "
                    " past this node (%s)" %
                    (lowerbit, numberbits, node.name)
                )

        if newvalue is not None:
            hi = lowerbit + numberbits - 1
            lo = lowerbit
            mask = (2**(hi - lo + 1)) - 1
            newvalue = (newvalue & mask) << lo
            mask = mask << lo
            node.read_write(mask, newvalue)
        else:
            value = node.get_value()
            return bits(value, lowerbit, numberbits)

    def _bits_field(self, lowerbit, numberbits, newvalue=None):
        """
        Used to get/set the value of certain bits within the field.

        Args:
            lowerbit (int): Least-significant bit to read/write.
            numberbits (int): Number of bits to read/write.
            newvalue (long, optional): If specified, then a modify
                the specified bits and write new value back out to.
        """
        if numberbits <= 0:
            raise ValueError(
                "numberbits cannot be zero or negative value (%d)" % numberbits
            )

        node = self.nodevalue
        if (node.lowerbit + lowerbit + numberbits - 1) > node.upperbit:
            raise ValueError(
                "bits command told given values (%d,%d) that extend past this"
                " field (%s)" %
                (lowerbit, numberbits, node.name)
            )
        bits = node.parent.bits(node.lowerbit+lowerbit, numberbits, newvalue)
        return bits

    def __call__(self, lowerbit, numberbits, newvalue=None):
        """bits(lowerbit,numberbits,newvalue=None)

        Used to get/set the value of certain bits within the node

        Args:
            lowerbit (int): Least-significant bit to read/write.
            numberbits (int): Number of bits to read/write.
            newvalue (long, optional): If specified, then a modify
                the specified bits and write new value back out to.
        """
        node = self.nodevalue
        if node.type == NodeTypes.Register:
            return self._bits_register(
                lowerbit=lowerbit, numberbits=numberbits, newvalue=newvalue
            )
        elif node.type == NodeTypes.Field:
            return self._bits_field(
                lowerbit=lowerbit, numberbits=numberbits, newvalue=newvalue
            )
        if node.type == NodeTypes.General:
            return self._bits_register(
                lowerbit=lowerbit, numberbits=numberbits, newvalue=newvalue
            )
