
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




"""
Plugin for providing local scope to enums.

========================
Enums - Encode/Decode
========================

This standard plugin provides enum capabilities to the named nodes that allows
for decoding a node's value to a string (or encoding the string to a value).

Basic Usage
------------

User usage example::

    >>> comp.register.field = 0
    >>> comp.register.field.decode()
    "zero"
    >>> comp.register.field = 2
    >>> comp.register.field.decode()
    "not zero"

Sometimes the register has en enum that can also encode and write to the node::

    >>> comp.register.field = 2
    >>> comp.register.field.encode("zero")
    >>> comp.register.field
    0x0

What if you want to see what the encoded value is without reading or writing to
the register? ::

    >>> comp.register.field = 2
    >>> comp.register.field.decode(lookup_value=0)
    "zero"

    >>> comp.register.field = 2
    >>> comp.register.field.encode("zero",lookup_only=True)
    0
    >>> comp.register.field
    0x2

For Developers
--------------------------------------
*You do not need to read this unless you are developing adding component or
project collateral*

In order to add an enum to a node, you add it to the node's component
definition via the enums plugin's methods create_simple() and
create_functional() for simple and functional enums respectively. The enum is
then added to the definition, therefore ensuring that all value objects share
the same enum.

Here's an example on how to add a SimpleEnum that has a dictionary look up::

    component.definition.enums.create_simple('zero_test', {0:'zero'},default='not zero')

The object does not have to be saved, this registers the enum with a plugin
object that will be able to handle the enum later when the decoded or encoded
value is requested.

Enum Types
++++++++++

SimpleEnum,
FunctionalEnum


Other Info
++++++++++

The NodeValue.encode and NodeValue.decode functions are implemented as
NodeValuePlugins

There is a show_enums method off of the NodeValue that will output the enum to
the log if it is a SimpleEnum; otherwise, log the encode and decode function names
if it is a FunctionalEnum.

In addition, there is also a getdecodeval method off of the NodeValue that will return
the enum string for the value; otherwise, None if one doesn't exist. Optionally,
getdecodeval(True) can be called in which the enum decsription would be returned instead
of None.
"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import xml.etree.cElementTree as xml
from os import path
from ..logging import getLogger
from ..comp import ComponentDefinitionPlugin
from ..nodes import NodeValuePlugin
from ..nodes import NodeTypes
from ..utils._py2to3 import *
from ..utils.ordereddict import odict
from ..utils.xml import prettyprint
import importlib
import os
import sys
from copy import copy

_LOG = getLogger()


def _str_to_function(definition):
    """
    Get a function object from an string definition with
    format path.to.module:function

    Args:
        definition (str): The definition of the function.

    Returns the function object or None when a problem is found.
    """
    # this seems, at the moment, as something that could be shared
    # by xml parsers implementations and that's why it's here.

    # assume we will always have module + function info
    # in string ??
    function_info = definition.split(":")
    module = function_info[0]
    function = function_info[1]
    path_to_module = module.split('.')
    if (
            '__main__' in path_to_module or
            '__builtin__' in path_to_module or
            function == '<lambda>'):
        # don't bail, just log warning and skip to next enum ??
        _LOG.warning(
            "WARNING: Unsupported function type '%s:%s'" %
            (module, function))
        return None
    try:
        modobj = importlib.import_module(module)
    except:
        import traceback;
        traceback.print_exc()
        raise Exception("Error trying to import module:function '%s'"%definition)

    function = getattr(modobj, function)
    return function

def _function_to_str(function):
    return "{0}:{1}".format(function.__module__,function.__name__)


class EnumDefinitionPlugin(ComponentDefinitionPlugin):
    """
    Plugin for managing simple and functional enums.
    """

    name = "enums"
    #: used to store a mapping of enum name to enum objects
    _enums = None
    # used to store the supported xml storage formats for enums.
    _xml_formats = ['nanoscope']

    def __init__(self, definition):
        super(EnumDefinitionPlugin,self).__init__(definition)
        self._enums = definition.info.get('enums', None)
        if self._enums is None:
            self._enums = definition.info['enums'] = odict()

    def create_simple(
            self,
            name,
            decode_lookup,
            encode_lookup=None,
            default="Unknown",
            override=False):
        """
        Create a simple enum.

        Default is to use reverse of decode lookup if encode is not provided.

        Do not use a simple enum for anything more complex than a simple
        key=value dictionary.

        Args:
            name (string): The name of the enum.
            decode_lookup (dict): Dictionary used for look up of values.
            encode_lookup (dict): Dictionary used for lookup of keys.
            default (object): Default value for when decode keys don't exist.
            override (bool): Override enum if it already exists. Default is
                False.
        """
        existing_enum = self._enums.get(name, None)
        if existing_enum is not None and not override:
            raise ValueError("enum '%s' has been defined already" % name)

        simple_enum = SimpleEnum(name, decode_lookup, encode_lookup, default)
        self._enums[name] = simple_enum

    def create_functional(
            self,
            name,
            decode_function,
            encode_function=None,
            default="Unknown",
            override=False):
        """
        Create a functional enum.

        Args:
            name (string): The name of the enum.
            decode_function (function): Function used for decoding.
            encode_function (function): Function used for encoding.
            default (object): Default value for when decode keys don't exist.
            override (bool): Override enum if it already exists. Default is
                False.
        """
        existing_enum = self._enums.get(name, None)
        if existing_enum is not None and not override:
            raise ValueError("enum '%s' has been defined already" % name)

        functional_enum = FunctionalEnum(
            name, decode_function, encode_function, default)
        self._enums[name] = functional_enum

    def lookup(self, name, **kwargs):
        """
        Search for an enum with the specified name. If enum is not found
        in this plugin then the search will continue up the parent definition
        until enum is found or the search has reached the top level definition.

        Args:
            name (string): The name of the enum that we are looking for.

        Returns:
            The enum associated with the specified name when found, otherwise
            it raises LookupError.
        """
        recursive = kwargs.pop('recursive', True)
        assert len(kwargs) == 0,\
            "Unknown arguments: '%s'" % ",".join(kwargs.keys())
        enum = self._enums.get(name, None)
        if enum is not None:
            return enum
        else:
            if recursive and self.definition.parent is not None:
                return self.definition.parent.enums.lookup(
                    name, recursive=recursive)
            else:
                raise LookupError("enum '%s' was not found" % name)

    def get_enum(self, name):
        """
        Get the enum with the specified name in this plugin.

        Args:
            name (string): The name of the enum that we want to get.
        """
        return self.lookup(name, recursive=False)

    def write_xml(self, filename, format):
        """
        Write definition enums dictionary to a file.

        Args:
            filename (string): Filename for the generated xml.
            format (string): XML format for the generated content.
        """
        if format not in self._xml_formats:
            raise ValueError("Unknown '%s' xml format specified." % format)

        parser = FactoryXMLEnumParser.get_parser(format)
        parser.write_xml(self._enums, filename)

    def remove_all(self, **kwargs):
        """
        Remove enums from the info of all nodes (and the component definitions)

        Args:
            recursive (bool) : remove enums from all components and nodes (default is True)
        """
        recursive = kwargs.pop('recursive', True)
        if not recursive:
            components = [self.definition]
        else:
            components = self.definition.walk_components()
        for component in components:
            enums = component.info.get('enums', None)
            if enums is not None:
                enums.clear()
            for node in component.walk_nodes():
                node.info.pop('enum', None)

    def load_xml(self, filename, format, silent=False, override=False):
        """
        Loads enums from an XML file into this definition.

        Args:
            filename (string): Filename for the xml file to be read.
            format (string): XML format of the read content.
            override (bool) : whether to override existing enums
        """
        if format not in self._xml_formats:
            raise ValueError("Unknown '%s' xml format specified." % format)

        parser = FactoryXMLEnumParser.get_parser(format)
        loaded_enums = parser.load_xml(filename)

        # process simple enums first
        simple_enums = loaded_enums['simple']
        for enum_name in simple_enums:
            decode_lookup = simple_enums[enum_name]
            try:
                self.create_simple(enum_name, decode_lookup, override=override)
            except ValueError:
                _LOG.warning(
                    "WARNING: Duplicated entry for enum '%s'" % enum_name)

        # and then the functional enums
        functional_enums = loaded_enums['functional']
        for enum_name in functional_enums:
            encode_function = functional_enums[enum_name].get(
                'encode_function', None)
            decode_function = functional_enums[enum_name].get(
                'decode_function', None)
            try:
                self.create_functional(
                    enum_name,
                    decode_function=decode_function,
                    encode_function=encode_function,
                    override=override
                )
            except ValueError:
                _LOG.warning(
                    "WARNING: Duplicated entry for enum '%s'" % enum_name)
        for assignment in loaded_enums['assignments']:
            # not a regular expression, just use get_by_path
            if not assignment['re']:
                node = self.definition.get_by_path( assignment['path'] )
                node.info['enum'] = assignment['enum']
            else:
                results = self.definition.search( assignment['path'], getobj=True, recursive=True)
                if not silent and len(results) == 0:
                    _LOG.warning("Could not find path '%s' for enum: '%s'" %(assignment['path'], assignment['enum']))
                for r in results:
                    r.info['enum'] = assignment['enum']

    def as_dict(self):
        """returns a shallow COPY of the underlying dictionary

        Note:
            - adding directly to this dictionary has no effect on actual enum availability
            - modifying enums in this dict DOES have an effect on actual enum functionality
        """
        return copy(self._enums)

    def clear(self):
        """remove all known enums"""
        self._enums.clear()

class Enum(object):
    """
    Base class for enums implementation.
    """
    def __init__(self, name):
        """
        Constructor.

        Args:
            name (string): The name of the enum.
        """
        self.name = name

    def encode(self, value, node):
        """
        Returns the value that should be written to the node.

        Args:
            value (object): Value to be encoded into new value.
            node (object): NodeValue object that we will write encoded value to

        In a typical scenario this is the string that we need the
        want the integer for before we write to the node.
        """
        raise Exception("Must be implemented")

    def decode(self, value):
        """
        Returns an interpreted value based on the value from a node.

        Args:
            value (object): Value to be decoded into new value.

        In a typical scenario this is the value to be turned in to
        a string for better display.
        """
        raise Exception("Must be implemented")


class SimpleEnum(Enum):
    """
    Implements a simple enum type.

    Default is to use reverse of decode lookup if encode is
    not provided.

    Do not use this for anything more complex than a simple
    key=value dictionary.
    """
    encode_lookup = None
    decode_lookup = None
    default = None

    def __init__(
            self,
            name,
            decode_lookup,
            encode_lookup=None,
            default="Unknown"):
        """
        Initialize simple enum.

        Args:
            name (string): The name of the enum.
            decode_lookup (dict): Dictionary used for look up of values.
            encode_lookup (dict): Dictionary used for lookup of keys.
            default (object): Default value for when decode keys don't exist.
        """
        Enum.__init__(self, name)
        self.decode_lookup = decode_lookup
        if encode_lookup is None:
            self.encode_lookup = {v: k for k, v in decode_lookup.items()}
        else:
            self.encode_lookup = encode_lookup
        self.default = default

    def encode(self, value, node):
        """
        Returns the value that should be written to the node.

        Args:
            value (object): Value to be encoded into new value.
            node (object) : not used here, but needed to match enum interface

        In a typical scenario this is the string that we need the
        want the integer for before we write to the node.
        """
        if value not in self.encode_lookup:
            raise ValueError("Unknown encoding for value: {0}".format(value))
        return self.encode_lookup.get(value, self.default)

    def decode(self, value):
        """
        Returns an interpreted value based on the value from a node.

        Args:
            value (object): Value to be decoded into new value.

        In a typical scenario this is the value to be turned in to
        a string for better display.
        """
        return self.decode_lookup.get(value, self.default)


class FunctionalEnum(Enum):
    """
    Implements a functional enum type.
    """
    _decode_function = None
    _encode_function = None
    # we primarily store the string
    # so that pickle won't load the function
    _decode_function_str = None
    _encode_function_str = None

    def __init__(
            self,
            name,
            decode_function,
            encode_function=None,
            default="Unknown"):
        """
        Initialize functional enum.

        Args:
            name (string): The name of the enum.
            decode_function (dict): Function to call for decoding node value to string/decode value
            encode_function (dict): Function to call for encoding to a a value before writing to node
            default (object): Default value that will be used if decode function returns None
        """
        Enum.__init__(self, name)
        if decode_function is not None and (
                not isinstance(decode_function,basestring)):
            self._decode_function = decode_function
            self._decode_function_str = _function_to_str(decode_function)
        if encode_function is not None and (
                not isinstance(encode_function,basestring)):
            self._encode_function = encode_function
            self._encode_function_str = _function_to_str(encode_function)
        self.default = default

    @property
    def encode_function(self):
        if self._encode_function is None:
            if self._encode_function_str is not None:
                self._encode_function = _str_to_function( self._encode_function_str )
        return self._encode_function

    @property
    def decode_function(self):
        if self._decode_function is None:
            if self._decode_function_str is not None:
                self._decode_function = _str_to_function( self._decode_function_str )
        return self._decode_function

    def encode(self, value, node):
        """
        Returns the value that should be written to the node.

        Args:
            value (object): Value to be encoded into new value.
            node (object): NodeValue object that we will write decoded value to

        In a typical scenario this is the string that we need the
        want the integer for before we write to the node.
        """
        if self.encode_function is None:
            raise Exception("Encode function not supported for this enum: '{0}'".format(self.name))
        try:
            from inspect import getfullargspec as getargspec
        except ImportError:
            from inspect import getargspec
        args = getargspec(self.encode_function).args
        num_args = len(args)
        if(num_args == 2):
            return self.encode_function(value, node)
        else:
            return self.encode_function(value)

    def decode(self, value):
        """
        Returns an interpreted value based on the value from a node.

        Args:
            value (object): NamedNodeValue to be decoded into new value.

        In a typical scenario this is the value to be turned in to
        a string for better display.
        """
        decode_val = self.decode_function(value)
        return decode_val or self.default

    def __getstate__(self):
        return (self._decode_function_str, self._encode_function_str, self.default)

    def __setstate__(self, state):
        self._decode_function = None
        self._encode_function = None
        self._decode_function_str = state[0]
        self._encode_function_str = state[1]
        self.default = state[2]


class EncodePlugin(NodeValuePlugin):
    """
    This plugin provides the encode function to nodes.
    """
    name = "encode"

    def __init__(self, nodevalue):
        super(EncodePlugin, self).__init__(nodevalue)
        self.enum_name = self.nodevalue.definition.info.get(
            "enum", None)
        if self.enum_name is not None:
            self.enum = self.nodevalue.component.definition.enums.lookup(
                self.enum_name)
        else:
            self.enum = None

    def __call__(self, value, lookup_only=False):
        if self.enum is None and lookup_only is False:
            raise ValueError("Cannot write value due to unknown encode")
        encode_value = self.enum.encode(value, self.nodevalue)
        if lookup_only:
            return encode_value
        # we should write the encodedvalue
        self.nodevalue.write(encode_value)


class DecodePlugin(NodeValuePlugin):
    """
    This plugin provides the decode function to nodes.
    """
    name = "decode"

    def __init__(self, nodevalue):
        super(DecodePlugin, self).__init__(nodevalue)
        self.enum_name = self.nodevalue.definition.info.get(
            "enum", None)
        if self.enum_name is not None:
            self.enum = self.nodevalue.component.definition.enums.lookup(
                self.enum_name)
        else:
            self.enum = None

    def __call__(self, lookup_value=None):
        if self.enum_name is None:  # no enum was specified
            return None
        if self.enum is None:
            raise Exception("Enum '{0}' was not found".format(self.enum_name))
        if lookup_value is None:
            return self.enum.decode(self.nodevalue)
        else:
            return self.enum.decode(lookup_value)


class ShowEnums(NodeValuePlugin):
    """
    This plugin provides the show_enum function to nodes.
    """
    name = "show_enums"

    def __init__(self, nodevalue):
        super(ShowEnums, self).__init__(nodevalue)
        self.enum_name = self.nodevalue.definition.info.get(
            "enum", None)
        if self.enum_name is not None:
            self.enum = self.nodevalue.component.definition.enums.lookup(
                self.enum_name)
        else:
            self.enum = None

    def __call__(self):
        if isinstance(self.enum, SimpleEnum):
            for key, value in self.enum.decode_lookup.iteritems():
                _LOG.result("%s: %s", key, value)
        elif isinstance(self.enum, FunctionalEnum):
            if self.enum.decode_function is None:
                _LOG.result("decode_function: None")
            else:
                _LOG.result("decode_function: %s", self.enum.decode_function.func_name)
            if self.enum.encode_function is None:
                _LOG.result("encode_function: None")
            else:
                _LOG.result("encode_function: %s", self.enum.encode_function.func_name)


class GetDecodeValPlugin(NodeValuePlugin):
    """
    This plugin provides the "getdecodeval" command to nodes.
    """
    name = "getdecodeval"

    @classmethod
    def create(cls, nodevalue):
        if nodevalue.type == NodeTypes.Field:
            return cls(nodevalue)
        else:
            return None

    def __call__(self, desc=False):
        """
        If a decode is present, then return the entry for this node's value.

        Returns None if no decode exists.

        Args:
            desc (boolean): If set to True, return node description instead
            of None when the decode does not exist. Defaults to False.
        """
        decode = self.nodevalue.decode()
        if decode is None:
            if desc:
                return self.nodevalue.description
            else:
                return None

        return decode


class XMLEnumParser(object):
    """
    Base class for xml enums parsers.
    """
    def __init__(self, parse_format):
        """
        Constructor.

        Args:
            parse_format (string): The format this object knows how to parse.
        """
        self.parse_format = parse_format

    def write_xml(self, filename, enums):
        """
        Write definition enums dictionary to a file.

        Args:
            filename (str): Filename for the generated xml.
            enums (dict): The enums to write to an xml file.
        """
        raise Exception("Must be implemented")

    def load_xml(self, filename):
        """
        Loads enums from an xml file.

        Args:
            filename (string): Filename for the xml file to be read.

        Returns a dictionary with the enums that were loaded from file.
        The format of the dictionary is as follows:

            {
                'simple': {
                    'enum_name': {
                        'decode_lookup': {...}
                        }
                    ...
                    }
                'functional': {
                    'enum_name': {
                        'encode_function': encode_function,
                        'decode_function': decode_function
                        }
                    ...
                    }
                'assignments': [
                        'enum':enum_name
                        'path':path,
                        're': boolean, # whether path is a regular expression
                    ]
                }
            }

        Notice that None is the default value for keys when no valid values are
        available, take for example a functional enum with a decode function
        but no encode function, its representation in the dictionary would be:

            'func_enum_name': {
                'encode_function': None,
                'decode_function: decode_function
                }

        """
        raise Exception("Must be implemented")



class NanoscopeXMLEnumParser(XMLEnumParser):
    """
    Enums parser implementation for nanoscope format
    """
    def __init__(self):
        """
        Constructor.
        """
        XMLEnumParser.__init__(self, 'nanoscope')

    def write_xml(self, enums, filename):
        """
        Write definition enums dictionary to a file.

        Args:
            enums (dict): The enums to write to an xml file.
            filename (string): Filename for the generated xml.
        """
        # not sure this is a good or bad idea... just making sure
        # that the filename we receive is a string
        if not isinstance(filename, basestring):
            raise ValueError("Invalid filename")

        root = xml.Element("enums")
        for enum in enums.values():
            enum_elem = xml.SubElement(root, 'enum')
            enum_elem.set('name', enum.name)
            if isinstance(enum, SimpleEnum):
                enum_elem.set('type', 'simple')
                for key, value in enum.encode_lookup.items():
                    key_elem = xml.SubElement(enum_elem, 'key')
                    key_elem.set('text', str(key))
                    key_elem.set('value', "0x%x" % value)
            elif isinstance(enum, FunctionalEnum):
                enum_elem.set('type', 'functional')
                if enum.encode_function is not None:
                    module = enum.encode_function.__module__
                    function = enum.encode_function.__name__
                    if module in ['__main__', '__builtin__'] or function == '<lambda>':
                        # don't bail, just log warning and skip to next enum ??
                        _LOG.warning(
                            "WARNING: Unsupported encode function type '%s:%s'" %
                            (module, function))
                        root.remove(enum_elem)
                        continue
                    encode_elem = xml.SubElement(enum_elem, 'encode')
                    encode_elem.text = "{0}:{1}".format(module, function)

                if enum.decode_function is not None:
                    module = enum.decode_function.__module__
                    function = enum.decode_function.__name__
                    if module in ['__main__', '__builtin__'] or function == '<lambda>':
                        # don't bail, just log warning and skip to next enum ??
                        _LOG.warning(
                            "WARNING: Unsupported decode function type '%s:%s'" %
                            (module, function))
                        root.remove(enum_elem)
                        continue
                    decode_elem = xml.SubElement(enum_elem, 'decode')
                    decode_elem.text = "{0}:{1}".format(module, function)
            else:
                raise Exception("Unknown enum type for enum '%s'" % enum.name)

        prettyprint(root)
        tree = xml.ElementTree(root)
        tree.write(filename)

    def load_xml(self, filename):
        """
        Loads enums from an xml file.

        Args:
            filename (string): Filename for the xml file to be read.

        Returns a dictionary with the enums that were loaded from file.
        The format of the dictionary is as follows:

            {
                'simple': {
                    'enum_name': {
                        'decode_lookup': {...}
                        }
                    ...
                    }
                'functional': {
                    'enum_name': {
                        'encode_function': encode_function,
                        'decode_function': decode_function
                        }
                    ...
                    }
            }

        Notice that None is the default value for keys when no valid values are
        available, take for example a functional enum with a decode function
        but no encode function, its representation in the dictionary would be:

            'func_enum_name': {
                'encode_function': None,
                'decode_function: decode_function
                }

        """
        if not path.exists(filename):
            raise ValueError("File '%s' does not exist." % filename)

        tag_with_error = self._parse_validate_xml(filename)
        if tag_with_error:
            raise Exception("Found malformed XML content '%s' while parsing." %
                            tag_with_error)

        tree = xml.iterparse(filename, events=("end",))

        # must used ordered dictionaries so that we get consistent results
        loaded_enums = {'simple': odict(),
                        'functional': odict(),
                        'assignments': [],
                        }
        #for enum in enums.findall("enum"):
        for event, elem in tree:
            if elem.tag=="enum":
                enum = elem
                enum_type = enum.attrib['type']
                enum_name = enum.attrib['name']
                if enum_type == 'simple':
                    decode_lookup = {}
                    for enum_elem in enum.iter('key'):
                        # here we are making a couple of assumptions:
                        #  1. we can perform long(value, 0)
                        #  2. floats are not used on lookup dictionary, if any they
                        #     will be used on functional enums instead
                        key = long(enum_elem.get('value'), 0)
                        value = enum_elem.get('text')
                        decode_lookup[key] = value
                    loaded_enums['simple'][enum_name] = decode_lookup
                elif enum_type == 'functional':
                    encode_function = None
                    decode_function = None

                    # get encode function
                    encode_definition = enum.find("encode")
                    if encode_definition is not None:
                        encode_function = _str_to_function(
                            encode_definition.text)
                        if not encode_function:
                            continue

                    # get decode function
                    decode_definition = enum.find("decode")
                    if decode_definition is not None:
                        decode_function = _str_to_function(
                            decode_definition.text)
                        if not decode_function:
                            continue
                    loaded_enums['functional'][enum_name] = {
                        'encode_function': encode_function,
                        'decode_function': decode_function
                        }
                # remove element when we are finished
                elem.clear()
            elif elem.tag == "assign_enum":
                enum_name = elem.get("name")
                assert enum_name is not None,"enum_name missing from assign_enum"
                for node in elem.findall("path"):
                    is_re = node.get("re","true").lower() == "true"
                    nodepath = node.text.strip()
                    loaded_enums['assignments'].append(
                        dict(re=is_re,
                             path=nodepath,
                             enum=enum_name))
                elem.clear()
            elif elem.tag == "include":
                next_filename = elem.text
                # filename must not be full path, try to find it using relative path
                if not os.path.exists(next_filename):
                    if next_filename.count("{pysvpath}"):
                        import svtools.common.path
                        next_filename = next_filename.format(pysvpath=svtools.common.path.getpysvpath())
                    elif next_filename.count("{path}"):
                        # try to find the path in the sys.path
                        for sysp in sys.path:
                            test_p = os.path.join(sysp, next_filename.format(path=sysp))
                            if os.path.exists(test_p):
                                next_filename = test_p
                                break
                    else:
                        # assume it was supposed to be relative path
                        next_filename = os.path.join(
                            os.path.dirname(filename),
                            next_filename)
                if not os.path.exists(next_filename):
                    raise RuntimeError("Could not find file: %s"%elem.text)
                results = self.load_xml(next_filename)
                loaded_enums['simple'].update(results['simple'])
                loaded_enums['functional'].update(results['functional'])
                loaded_enums['assignments'].extend(results['assignments'])
            # else:
            # sub sub element, dont clear it, our other elements likely need it
        return loaded_enums

    def _parse_validate_xml(self, filename):
        """
       Parse and perform very basic validation on input xml file.

        Args:
            filename (string): Filename for the xml file to be read.

        Returns the name of a tag with error, if any, otherwise None.

        cElementTree raises an exception on cases like if any tags are left
        open or start and end tags have different names.
        """
        events = ('start', 'end')
        context = xml.iterparse(filename, events=events)
        context = iter(context)

        valid_tags = ['enums', 'enum', 'key', 'decode', 'encode', 'include', 'assign_enum', 'path']
        # used for storing the parent tag names so that when a valid tag
        # is found we can check if its expected parent has been found
        # already, otherwise valid tag it's not where it's supposed to
        parents = []
        for event, elem in context:
            if event == 'start':
                if elem.tag not in valid_tags:
                    return elem.tag
                parents.append(elem.tag)
            if event == 'end':
                if elem.tag == 'enums':
                    if len(parents) > 1:
                        return elem.tag
                elif elem.tag == 'enum':
                    if 'enums' not in parents:
                        return elem.tag
                elif elem.tag == 'key':
                    if (
                            'enum' not in parents or
                            len(elem.attrib) > 2 or
                            not elem.attrib.get('text', None) or
                            not elem.attrib.get('value', None)):
                        return elem.tag
                elif elem.tag == 'decode' or elem.tag == 'decode':
                    if 'enum' not in parents:
                        return elem.tag
                parents.pop()
        return None


class FactoryXMLEnumParser(object):
    """
    Factory class for XML parsers.
    """
    @staticmethod
    def get_parser(parse_format):
        """
        Factory method for getting a parser object.

        Here's where new parsers will be added.
        """
        # maybe instead of adding an instance of the parser we might
        # store a string with the class name and instanstiate only
        # when necessary ??
        parsers = {'nanoscope': NanoscopeXMLEnumParser()}
        return parsers[parse_format]
