# INTEL TOP SECRET
# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.


from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import re
import six
from copy import copy, deepcopy
from . import settings
from .telemetry import record_event
from .errors import PluginError
from .utils._py2to3 import *
from .utils.ordereddict import odict
from .utils.wrappedvalue import WrappedValue
from .utils import PluginManager
from namednodes.events import accesses as access_events
from namednodes.events.accesses import AccessEventTypes, AccessEvent
from .utils.cached_property import slim_cached_property
import weakref
# need to import this for valid node types
from ipccli import BitData
from six import PY2, PY3

if PY2:
    from collections import MutableMapping
if PY3:
    from collections.abc import MutableMapping


_re_valid_name = re.compile(r'\w+$')

from .logging import getLogger
_LOG = getLogger()

# basically an enum...
class NodeTypes(object):
    General = 1
    Array = 2
    Register = 3
    Field = 4
    Alias = 5
    ArrayItem = 6  # ideally this should NOT be used going forward as an item_definition shouldn't
                   # need to be a certain type


# special access class name that means to search for a default access
# using a priority list
_DEFAULT_ACCESS = 'DefaultAccess'

#######################
# Feature Lists
#######################
# enum - optional "nodeenum" class to assign to a definition
#  - encode, for looking up value and attempting a write
#  - decode, for returning string value
#  - lookup, for lookuping value but no write (needed ?)

"""

Summary of Challenges
------------------------

- Display/Dump Requirements
    - need to return string and display at one node point down to all the parents
        - needs to return a string
        - needs to be able to get configured at node and/or component level
    - needs to be able provide hooks similar to 'save' and 'load'

- UI Requests - need ability to view just "fields" or children nodes without also see the functions


only use "_" if we are reserving something to that class..."_" means a customer will NEVER use it or want it
TRY not to use any "_" except within that class....even though this is one big architecture, don't
    use _ attributes from other classse (even if it is slower!)

STILL TBD: do we really need to keep up with Array's as a separate type of class

"""



class NamedNodeDefinitionMetaclass(type):
    """
    Dummy class for achieving the cheat on the isinstance check
    aiming at making  NamedNodeDefinitionAlias objects look like
    instances of NamedNodeDefinition.
    """
    def __instancecheck__(cls, instance):
        return (
            # simple check: either the instance is a subclass of ours
            # (where a class is a subclass of itself) or it is an
            # instance of our special NamedNodeDefinitionAlias and
            # we are checking against NamedNodeDefinition.
            issubclass(instance.__class__, cls) or
            (
                isinstance(instance, NamedNodeDefinitionAlias) and
                cls is NamedNodeDefinition
            )
        )


@with_metaclass(NamedNodeDefinitionMetaclass)
class NamedNodeDefinition(object):
    """
    An instance of this class represents some named "node" that will belong to a class that
    inherits from a NamedComponent.subclass of NamedComponent. This *node* has all the
    same information and characteristics no matter one which *instance* of the NamedComponent it is attached.

    The more specific version of the NamedNodeDefinition is the NamedNodeValue.
    The NamedNodeValue knows both the Definition (that describes the node and how to access it)
    and the NamedComponent instance (which provides the last bit of information needed
    by the definition, to find the node in the system.

    See Also:
        - :py:meth:`NamedNodeValue`
    """

    #: list of reserved names that nodes cannot be used for child nodes, but they sometimes are
    reserved_names = set((
                      'accesses',
                      'name',
                      'info',
                      'type',
                      'read',
                      'write',
                      'store',
                      'flush',
                      'value',
                      'free',
                      'parent',
                      'component',
                      'info',
                      'target_info',
                      ))
    #: which group we should look to asking the NamedComponent what the current
    #: access method is to be used
    access_group = "default"
    #: dictionary to map of access types to class
    accesses = None
    #: children node definitions, this is an ordered dictionary for quick lookup
    #: there will be a childrens propery to return the *values* of this dictionary
    _nodes = None
    #: name for this node
    name = ""
    #: general set of information regarding this node
    info = None
    type = NodeTypes.General
    #: component definition that this nodedefinition belongs to
    component = None
    #: parent node - set only if this is a child node of another node
    parent = None

    # NOTES:
    #  - we do not have a parent attribute here on purpose. For traversing the tree of nodes
    #    only the value can guarenteed to have a consistent line back through all the nodes to the
    #    component

    @property
    def value_class(self):
        """Value class that should be used when creating a Value to represent this node"""
        return NamedNodeValue

    def __init__(self, name, info=None,**kwargs):
        """
        Args:
            name : Name to assign to definitions
            info : Information about
            parent : parent definition node that we may have (weakreference is kept)
            children : list (or dict) of children definitions for this node
            accesses : access dictionary to map of access types to class
            access_group : group that we should associate our accesses with (default: "default")
        """
        self.component = None
        if name in self.reserved_names:
            self.name = name + "_"
        else:
            self.name = name
        self.info = info or odict()
        if not isinstance(self.info, odict): # if not a
            self.info = odict()
            self.info.update( info )
        children = kwargs.pop("children" , None)
        if children == None:
            self._nodes = odict()
        elif isinstance(children,list):
            self._nodes = odict()
            for c in children:
                self.add_node(c)
        elif isinstance(children,(dict,MutableMapping)):
            # must update the children to have this as parent
            self._nodes = children
        else:
            raise TypeError("Unsupported type for 'children' parameter")
        self.accesses = {}
        accesses = kwargs.pop("accesses",{})
        self.access_group = kwargs.pop("access_group","default")
        for access_name,access_class in accesses.items():
            self.add_access( access_name, access_class )

        parent = kwargs.pop("parent",None)
        if parent is not None:
            # all children should have only weakreferences back to their parents
            # this will set self.parent
            parent.add_node(self)

        if len(kwargs)>0:
            raise ValueError("Unknown keyword arguments passed in {0:s}".format(kwargs))

    @property
    def nodes(self):
        """Return a list of child NamedNodeDefinition objects"""
        return list(self.iter_nodes())


    @property
    def nodenames(self):
        """Return a list of the names of all of our children nodes"""
        return list(self.iter_nodenames())

    def iter_nodes(self):
        """Return an iterator for child NamedNodeDefinition objects"""
        for n in self.iter_nodenames():
            yield self.get_node(n)

    def iter_nodenames(self):
        """Return an iterator for the names of all of our children nodes"""
        return iter(self._nodes.keys())

    def __contains__(self,nodename):
        return nodename in self._nodes

    def __getitem__(self,nodename):
        return self._nodes[nodename]

    def add_access(self,access_name,access_classname,default=False):
        """
        Args:
            access_name (str)       : name of access to look for when checking
                        the component
            access_classs_name (str) : name of a class that will have the
                        read/write/available class methods
            default (bool)          : whether this is default access for when
                        the group has an access not supported by this
                        register
        """
        if access_name.count(","):
            raise Exception("sorry, comma's not allowed in access name")

        # we won't know till run time if this is invalid or not
        self.accesses[access_name] = access_classname
        if default or ('default' not in self.accesses):
            self.accesses['default'] = access_classname

        return self

    def _log_access_error(self, component):
        """
        When an error occurs retrieving an access function, then this is used
        to display the best error message possible
        """
        if len(self.accesses)==0:
            raise Exception("This node ({0}) has no access methods".format(self.name))
        msg = []
        if self.access_group != None:
            acc = component.get_access(self.access_group)
        else:
            acc = None
        msg.append(" Access group %s\n"%self.access_group)
        msg.append(" Parent access for that group: %s\n"%acc)
        if acc is not None:
            acc_class = self.accesses.get(acc,None)
            msg.append("Access class is: %s\n"%acc_class)
        else:
            acc_class = self.accesses.get("default",None)
            msg.append(" Missing access, using default: %s"%str(acc_class))
            msg.append(" Access class is: %s\n"%acc_class)
        raise Exception("could not find access:\n%s"%"".join(msg))

    def get_node(self, name, default=ValueError):
        """Retrieve the child NamedNodeDefinition with the name we know it"""
        try:
            child = self._nodes[name]
        except KeyError:
            if default is not ValueError:
                return default
            raise ValueError("Unknown node '%s'"%name)
        return child

    def add_node(self, node):
        """Add child node, if name is given then it will be used instead of the node.name

        Args:
            node (NamedNodeDefinition) : NamedNode object

        Returns:
            returns the node added in case chaining add_nodes is desired

        """
        if node.parent is not None:
            _LOG.warning("This node already belongs to another component, doing a deep copy before adding it to this node.\n",)
        # change name if it is a reserved name
        if node.name in self.reserved_names:
            node.name = node.name + "_"
        self._nodes[node.name] = node
        # set parent node when it is added
        node.parent = weakref.proxy( self )
        # return node for chaining if desired
        return node

    def remove_node(self, nodename):
        """remove a child node from our nodes

        Args:
            node (str) : string name or child object to remove (name recommend usage), if object is
                    given it has to be the original object since there's no built in compare
                    for node objects
        """
        if isinstance(nodename,basestring):
            if nodename not in self._nodes:
                raise ValueError("Child '{0}' not found".format(nodename))
            oldnode = self._nodes.pop(nodename)
            # reset parent in case someone is moving it to somewhere else
            oldnode.parent = None
        else: # assume it is a node
            if nodename.name not in self._nodes:
                raise ValueError("Child '{0}' not found".format(nodename.name))
            oldnode = self._nodes.pop(nodename.name)
            # reset parent in case someone is moving it to somewhere else
            oldnode.parent = None
        return

    @property
    def path(self):
        """path to this node from the topmost component"""
        if self.parent is not None and self.parent.type == NodeTypes.Array:
            mypath = self.name + "[#]"
            if self.parent.parent is None:
                if self.parent.component is None:
                    # this has gotta be rare too...or we just haven't been added yet
                    return mypath
                else:
                    return self.parent.component.path + "." + mypath
            else:
                return self.parent.parent.path + "." + mypath
        elif self.parent is not None:
            return self.parent.path + "." + self.name
        elif self.component is not None:
            return self.component.path + "." + self.name
        else:
            # only if we haven't been added to a component yet
            return self.name

    @property
    def nodepath(self):
        """path to this node from the component it belongs to"""
        # better name for this property?
        if self.parent is not None:
            if self.parent.type == NodeTypes.Array:
                if self.parent.parent is None:
                    return self.name + "[#]"
                else:
                    return self.parent.parent.nodepath + "." + self.name + "[#]"
            else:
                return self.parent.nodepath + "." + self.name
        else:
            return self.name

    def __getattr__(self, attr):
        # try to get plugin and create it if it exists
        if attr in NodeDefinitionPlugins.plugins():
            plugin = NodeDefinitionPlugins.get(attr,self,noerror=True)
            if plugin:
                return plugin
            else:
                raise Exception("The plugin for %s does not support this node type"%attr)
        try:
            # attempt to get node...if that failes, getattribute
            node = self.get_node(attr)
            return node
        except ValueError:
            return object.__getattribute__(self,attr)

    def __copy__(self):
        raise RuntimeError("shallow copy not supported, must use deepcopy")

    def __deepcopy__(self, memodict={}):
        # creaate new object our same type
        new_node = type(self)(self.name,
                              info=deepcopy(self.info),
                              accesses=copy(self.accesses),
                              access_group=self.access_group,
                              )
        for child in self.nodes:
            new_node.add_node(deepcopy(child))
        return new_node

# inherits from "user long" and uses node definition to determine value type
# WrappedValue basically is a User value, but calls get_value if the current value is None
class NamedNodeValue(WrappedValue):
    """The NamedNodeValue is mainly a 'middleman' for calling access functions
    in the definition and providing the component. It ties those two objects
    together to help create accessibility to this particular node in the system.

    The NamedNodeValue can act like various number types, but the default is a long.

    .. todo:
        need to determine how to support value being long OR bitdata

    """
    type = NodeTypes.General
    #: name that the node will reference itself by
    name = None
    #: contains information specific to the **value** not the definition
    target_info = None
    #: NamedNodeComponent that this Value will reference when using our definition's access methods
    component = None
    #: NamedNodeDefinition object that we will reference for access methods
    definition = None
    #: parent is another NamedNodeValue object that this NamedNodeValue is a child of
    parent = None
    # --- I have confirmed with nanoscope that the limitation of not being able to support:
    # --- array.set[0].way[0].set[0] is OK....dont bring it up again
    #: This is a dictionary of the arrays encountered so far, and the keys in to those arrays that were used
    array_info = None
    #: used to hold the data that we will write if flush is called
    stored_value = None
    #: used to show which bits will get updated when flush is called
    stored_mask = None
    #: Dictionary of children to keep up with values
    _nodes_values = None
    #: value type we should use when casting to a number, doing compares, etc...
    _value_type = long
    #: legal values for value type
    _value_types = {
            "long":long,
            "str":str,
            "basestring":basestring,
            "float":float,
            "bool":bool,
            "BitData":BitData}
    #: creating a value object will cause us to determine what the current access class is
    _access_class = None
    #: 
    _access_class_name = None
    #: Used to prevent accidental additions to the lass, this is set by constructor
    _frozen = False
    #: Used to keep up with previous plugins that we may have already created
    _plugins = None
    #: Used to hold the value that this object represents
    _value = None
    #: Used to override the "normal" path for when values are added
    # to component alias.
    _path = None
    #: Internal only - used to track children nodes that are not in this node
    #: value but may be in the node definition
    _removed_nodes = None

    def __init__(self, name, component, definition, parent_nodevalue):
        """
        Args:
            name : name of node value
            component : Component that the definition should be associated with
            definition : contains the access and information needed to perform functions
            parent_nodevalue : for referencing what nodevalue we came from in case data
                               flows down to it
        """
        # directly calling setattr is annoying here but since this object defines a setattr
        # this important in order reclaim some of the performance
        # name passed in because it may not actually be the same as the definition in cases of aliasing...
        object.__setattr__(self, 'name', name)
        object.__setattr__(self, 'component', component)
        object.__setattr__(self, 'definition', definition)
        # only weakrefs should be kept of parents to prevent circular references that
        # hamper the garbage collection
        if parent_nodevalue is not None:  #  must NOT use != that will cause us to do a compare
            object.__setattr__(self, 'parent', parent_nodevalue)
        else:
            # no parent node, then use reference to the component
            object.__setattr__(self, 'parent', component)
        # for keeping up with immediate children's node value objects
        # so that we do not need to constantly recreate
        object.__setattr__(self, '_nodes_values', {})
        # go ahead and create mapping to maintain order
        object.__setattr__(self, '_value', None)
        # it may be that we end up needing to store information local to the value objects
        # if so, this where that information will be stored
        # there's a non-trivial performance hit if anything other than the built-in dictionary is used
        # since this is the "value's" information instead of the definition, then that should be fine
        # it is rare/unlikely that we write the value info out to disk
        object.__setattr__(self, 'target_info', {})
        object.__setattr__(self, 'array_info', {})
        # for tracking what children nodes exist in the definition, but NOT in
        # this instance of the node
        object.__setattr__(self, '_removed_nodes', set())
        # if our parent is not our component, then it is a node
        if self.parent is not self.component:
             self.array_info.update( self.parent.array_info )
        # this is a special piece of 'information' that may exist to tell us
        # what are value type is... value type used to determine if we
        # are an "int"/"float"/"bool"/etc...
        _value_type = self.definition.info.get('value_type', None)
        if _value_type is not None: # only set if info
            # convert from string to a type
            self._value_type = self._value_types.get(_value_type,None)
            if self._value_type is None:
                raise ValueError("Value type '{0}' not supported".format(_value_type))
        # No more attributes past this point
        object.__setattr__(self, '_frozen', True)

    @property
    def accesses(self):
        """short-cut to definition.accesses"""
        if self.definition is not None:
            return self.definition.accesses
        else:
            return {}

    @property
    def info(self):
        """short-cut to definition.info"""
        return self.definition.info

    @property
    def nodenames(self):
        """return a list of the names of all of our children nodes"""
        return list(self.iter_nodenames())

    @property
    def nodes(self):
        """return a list of NodeValue children objects"""
        return list(self.iter_nodes())

    def iter_nodes(self):
        """returns an interator that walks through child NodeValue objects"""
        if self.definition is None:
            return
        for name in self.definition.nodenames:
            if name not in self._removed_nodes:
                yield self.get_node(name)

    def iter_nodenames(self):
        """returns an iterator that walks child node names"""
        if self.definition is None:
            return
        for name in self.definition.iter_nodenames():
            if name not in self._removed_nodes:
                yield name

    def __contains__(self, nodename):
        if nodename in self._removed_nodes:
            return False
        else:
            return nodename in self.definition

    @property
    def path(self):
        """return path from component to this node value"""
        # First check if there's a path override and use it if it's
        # available, this would mean that the node was added to a
        # component alias and that we need to report the aliased path.
        if self._path is not None:
            return self._path

        if self.component is not None:
            return self.component.path + "." + self.nodepath
        else:
            return "<detached>."+self.nodepath

    @path.setter
    def path(self, path):
        """
        Set path override, this is meant to be used when
        node's added to a component alias.
        """
        self._path = path

    @property
    def nodepath(self):
        """path to this node from the component it belongs to"""
        # if our parent is a component, then we our path is just name
        if self.parent is self.component:
            npath = self.name
        elif self.array_info:
            for array_name in self.array_info:
                if self.parent.path.endswith(array_name):
                    # we are in an array, but that array is off a component
                    if self.parent.parent is self.component:
                        npath = self.name
                        break
                    else:
                        # array is off another node
                        npath = self.parent.parent.nodepath + "." + self.name
                        break
            else:
                # none of the array info matched this one we must be below the array entry
                return self.parent.nodepath + "." + self.name
        else:
            # not in an array, much simpler path...
            npath = self.parent.nodepath + "." + self.name
        return npath

    def _load_all_plugins(self):
        """Loads all the supported plugins for this node"""
        for plugin_name in NodeValuePlugins.plugins():
            if plugin_name not in self.__dict__:
                plugin = NodeValuePlugins.get(plugin_name, self)
                if plugin is not None:
                    self.__dict__[plugin_name] = plugin

    def __dir__(self):
        mydir = []
        if settings.HIDE_METHODS is False:
            self._load_all_plugins()  # must be first
            mydir.extend(dir(type(self)))
            mydir.extend(self.__dict__.keys())
            mydir.extend(self.access_class.methods)
        # only return alpha numeric names as visible
        for child_name in self.definition.nodenames:
            if _re_valid_name.match(child_name):
                mydir.append(child_name)
        # uniqueify the values
        mydir = list(set(mydir))
        return mydir

    @property
    def value(self):
        """
        This can be used to retrieve the underlying value object. Be aware that this can
        be None, if the read or get_value has not been called. It is recommended
        to cast the Value object as needed vs. using this method

        Note:
            This does not touch any target or do any read, it simply changes the displayed value for this node.

        Note:
            This does not flush any children nodes values

        """
        return self._value

    @value.setter
    def value(self, val):
        """
        This can be used to force the underlying value object. This does not do anythin
        except change the locally cached value.

        Note:
            This does not touch any target or do any read, it simply changes the displayed value for this node.

        Note:
            This does not flush any children nodes values

        """
        self._value = val

    def get_value(self):
        """Get the value using the access class available when this Value object was created

        Returns:
            this RegisterValue object

        get_value will attempt to get a stored value if possible

        See Also::
            :py:meth:`namednodes.access.NodeAccess.update_value`

        """
        # called due to our value == None
        if settings.DEBUG:
            _LOG.caller("NodeValue.get_value")
            _LOG.debugall("NodeValue.get_value: ENTER : '{0}', previous value was: {1}".format(self. name,self._value))
        #self._value = self.definition.get_access_method(self.component,"get_value",self)()
        if self._value is None:

            # if events are enabled, log this
            if settings.EVENTS_ON_ACCESSES:
                access_event = AccessEvent(
                    AccessEventTypes.pre_update_value, self, [], {}
                )
                access_events.send_event(access_event)
            # in rare pickling events our access class might end up with None
            access_class = self.access_class
            if access_class is not None:
                self._value = self.access_class.update_value(self)

        if settings.EVENTS_ON_ACCESSES:
            access_event = AccessEvent(
                AccessEventTypes.post_update_value, self, [], {}
            )
            access_events.send_event(access_event)

        if settings.DEBUG:
            _LOG.debugall("NodeValue.get_value: EXIT : '{0}', new value is: {1}".format(self.name, self._value))
        return self._value

    def read(self, *args, **kwargs):
        """Read the value using the access class available when this Value object was created

        Returns:
            this RegisterValue object

        See Also::
            :py:meth:`namednodes.access.NodeAccess.read`

        """
        if settings.DEBUG:
            _LOG.caller("NodeValue.read")
            _LOG.debugall("NodeValue.read: ENTER : '{0}', previous value was: {1}".format(self.name,self._value))

        if settings.EVENTS_ON_ACCESSES:
            access_event = AccessEvent(
                AccessEventTypes.pre_read, self, args, kwargs)
            access_events.send_event(access_event)
        # use "is" for special case "as fast as possible" check
        if self.access_class._logged_read is False:
            # we will only record a single event for an access so that we prevent any possible
            # performance impacts
            self.access_class._logged_read = True
            record_event("NodeValue.read",{
                                    # note the class name string may be different than the final/actual
                                    # name on the class we used
                                    'access_class_name_string':self.access_class_name,
                                    'access_class_name':self.access_class.__name__,
                                    'access_class_module':self.access_class.__module__
                                     })

        self._value = self.access_class.read(self, *args, **kwargs)

        if settings.EVENTS_ON_ACCESSES:
            access_event = AccessEvent(
                AccessEventTypes.post_read, self, args, kwargs)
            access_events.send_event(access_event)


        if settings.DEBUG: _LOG.debugall("NodeValue.read: EXIT : '{0}', new value is: {1}".format(self.name,self._value))
        return self

    def write(self, value, *args, **kwargs):
        """
        For writing a new value to the node

        Args:
            value : value to given to the nodes current access.write function

        See Also::
            :py:meth:`namednodes.access.NodeAccess.write`

        """
        if settings.DEBUG:
            _LOG.caller("NodeValue.write")
            _LOG.debugall("NodeValue.write: ENTER : '{0}', value to write: {1}".format(self.name,value))

        if settings.EVENTS_ON_ACCESSES:
            access_event = AccessEvent(
                AccessEventTypes.pre_write, self, (value,)+args, kwargs)
            access_events.send_event(access_event)

        # use "is" for special case "as fast as possible" check
        if self.access_class._logged_write is False:
            # we will only record a single event for an access so that we prevent any possible performance
            # impacts
            self.access_class._logged_write = True
            record_event("NodeValue.write",{
                                    # note the class name string may be different than the final/actual
                                    # name on the class we used
                                    'access_class_name_string':self.access_class_name,
                                    'access_class_name':self.access_class.__name__,
                                    'access_class_module':self.access_class.__module__
                                     })
        # doing a write will always clear our cached value, but this MUST be befre
        # the access class write, because some writes store off a new cached value
        self._value = None

        retcode = self.access_class.write(self, value, *args, **kwargs)

        # now...clear off previous children node values since they may have stale cached values
        self._nodes_values = {}

        if settings.EVENTS_ON_ACCESSES:
            access_event = AccessEvent(
                AccessEventTypes.post_write, self, (value,)+args, kwargs)
            access_events.send_event(access_event)

        return retcode

    def read_write(self, mask, value, *args, **kwargs):
        """
        For performing a read_write to the node

        Args:
            mask: mask of what data should be removed before we or in the new value
            value : value to given to the nodes current access.read_write function

        See Also::
            :py:meth:`namednodes.access.NodeAccess.read_write`

        """
        if settings.DEBUG:
            _LOG.caller("NodeValue.read_write")
            _LOG.debugall("NodeValue.read_write: ENTER : '{0}', mask: {1}, value: {2}".format(self.name,mask,value))
        # doing a write will always clear our cached value
        self._value = None

        if settings.EVENTS_ON_ACCESSES:
            access_event = AccessEvent(
                AccessEventTypes.pre_read_write, self, (mask, value,)+args, kwargs)
            access_events.send_event(access_event)

        # use "is" for special case "as fast as possible" check
        if self.access_class._logged_read_write is False:
            # we will only record a single event for an access so that we prevent any possible
            # performance impacts
            self.access_class._logged_read_write = True
            record_event("NodeValue.read_write",{
                                    # note the class name string may be different than the final/actual
                                    # name on the class we used
                                    'access_class_name_string':self.access_class_name,
                                    'access_class_name':self.access_class.__name__,
                                    'access_class_module':self.access_class.__module__
                                     })

        retcode = self.access_class.read_write(self, mask, value, *args, **kwargs)

        # now...clear off previous children node values since they may have stale cached values
        self._nodes_values = {}

        if settings.EVENTS_ON_ACCESSES:
            access_event = AccessEvent(
                AccessEventTypes.post_read_write, self, (mask, value,)+args, kwargs)
            access_events.send_event(access_event)

        return retcode

    def store(self, value, *args, **kwargs):
        """Store this value for a future flush command

        See Also::
            :py:meth:`namednodes.access.NodeAccess.store`

        """
        if settings.DEBUG:
            _LOG.caller("NodeValue.store")
            _LOG.debugall("NodeValue.store: ENTER : '{0}'".format(self.name))

        if settings.EVENTS_ON_ACCESSES:
            access_event = AccessEvent(
                AccessEventTypes.pre_store, self, (value,)+args, kwargs)
            access_events.send_event(access_event)

        if self.access_class._logged_store is False:
            # we will only record a single event for an access so that we prevent any possible
            # performance impacts
            self.access_class._logged_store = True
            record_event("NodeValue.store",{
                                    # note the class name string may be different than the final/actual
                                    # name on the class we used
                                    'access_class_name_string':self.access_class_name,
                                    'access_class_name':self.access_class.__name__,
                                    'access_class_module':self.access_class.__module__
                                     })

        # no preconditions for store
        self.access_class.store(self, value, *args, **kwargs)

        if settings.EVENTS_ON_ACCESSES:
            access_event = AccessEvent(
                AccessEventTypes.post_store, self, (value,)+args, kwargs)
            access_events.send_event(access_event)

    def flush(self, *args, **kwargs):
        """Flush any data previously provided by a store method

        Args:
            persistent (bool) : whether to keep stored value and mask after
                                the flush is executed (default=False)

        Note:
            each access can add additional args (like skip_read), so be
            sure to check you the specific access class for additional
            argument information

        See Also::
            :py:meth:`namednodes.access.NodeAccess.flush`

        """
        if settings.DEBUG:
            _LOG.caller("NodeValue.flush")
            _LOG.debugall("NodeValue.flush: ENTER : '{0}'".format(self.name))

        if settings.EVENTS_ON_ACCESSES:
            access_event = AccessEvent(
                AccessEventTypes.pre_flush, self, args, kwargs)
            access_events.send_event(access_event)

        if self.access_class._logged_flush is False:
            # we will only record a single event for an access so that we prevent any possible
            # performance impacts
            self.access_class._logged_flush = True
            record_event("NodeValue.flush",{
                                    # note the class name string may be different than the final/actual
                                    # name on the class we used
                                    'access_class_name_string':self.access_class_name,
                                    'access_class_name':self.access_class.__name__,
                                    'access_class_module':self.access_class.__module__
                                     })

        persistent = kwargs.pop("persistent", None)
        retcode = self.access_class.flush(self, *args, **kwargs)

        self._value = None
        if not persistent:
            self.stored_value = None
            self.stored_mask = None
        # now...clear off previous children node values since they may have stale cached values
        self._nodes_values = {}

        if settings.EVENTS_ON_ACCESSES:
            access_event = AccessEvent(
                AccessEventTypes.post_flush, self, args, kwargs)
            access_events.send_event(access_event)
        return retcode

    #########################################################################
    # we purposefully don't 'fast track' the store/flush
    # in case we need to customize the parameters later for some reason
    # that way they can be overwritten by each access
    #########################################################################

    @property
    def value_type(self):
        # the python type that this node will mimic
        return self._value_type

    def get_node(self, nodename, default=ValueError):
        if settings.DEBUG: _LOG.debugall("NodeValue.get_node: '{0}' requesting node '{1}'".format(self.name,nodename))
        # must be a child node that is desired...if the child value node
        # has not been created
        child = self._nodes_values.get(nodename, None)
        if child is not None: # cannot use == or != or that will cause an evaulation
            if settings.DEBUG:
                _LOG.debugall("NodeValue.get_node: nodevalue for node '{0}' already existed".format(nodename))
            return child
        # child value node not created yet
        if nodename in self._removed_nodes:
            if default is not ValueError:
                return default
            raise ValueError("Unknown node: %s on node %s"%(nodename, self.path))
        # need to pass in a specific default so we dont get an exception from get_node
        child_def = self.definition.get_node(nodename, default=RuntimeError)
        if child_def is RuntimeError and default is ValueError:
            raise ValueError("Unknown node: %s on node %s"%(nodename, self.path))
        elif child_def is RuntimeError and default is not ValueError:
            # default was specified, return it...
            return default
        if settings.DEBUG: _LOG.debugall("get_node : creating new nodevalue for '{0}'".format(nodename))
        # get the value class we should use and create one
        newnode = child_def.value_class( nodename, self.component, child_def, self)
        self._nodes_values[nodename] = newnode
        return newnode

    def remove_node(self, nodename):
        """remove the children node with the specified name from this node
        Args:
            nodename (str) : name of node to remove
        """
        if nodename not in self.definition:
            raise ValueError("Unknown node")
        self._removed_nodes.add(nodename)

    def free(self, nodename, error=False):
        """remove cached reference to a child node
        Args:
            nodename (str) : name of node to remove references too
        """
        # theoretically we do not need to do any recursive here
        try:
            del self._nodes_values[nodename]
            # dont do gc.collect here...users should do that..not us
        except KeyError:
            if error:
                raise
            # ignore any key error due to the free not being needed
            pass

    def __getattr__(self,attr):
        if settings.DEBUG:
            _LOG.caller("__getattr__")
            _LOG.debugall("__getattr__ : '{0}' : '{1}' requested".format(self.name,attr))
        # these are purposefully not elifs since each should return if they pass
        # order is still important though...
        if self.definition is not None and attr in self.definition:
            return self.get_node(attr)
        if attr in NodeValuePlugins.plugins():
            plugin = NodeValuePlugins.get(attr, self)
            if plugin is not None:
                self.__dict__[attr] = plugin
                return plugin
            else:
                raise PluginError("The plugin for %s does not support this node type"%attr)        
        if self.access_class is not None and attr in self.access_class.methods:
            # access methods all take node as first paramter, in this case, that
            # is this node...so we are giving the user a function with the node
            # already filled in...that is why we need to create a function first
            access_method = getattr(self.access_class,attr)
            def newf(*args,**kwargs):
                return access_method(self,*args,**kwargs)
            newf.__name__ = access_method.__name__
            newf.__original__ = access_method
            return newf
        else:            
            return object.__getattribute__(self,attr)


    def __setattr__(self,attr,value):
        if settings.DEBUG:
            _LOG.caller("__setattr__")
            # we cant print value here or we could end up in a recursive loop
            _LOG.debugall("__setattr__ : '{0}' : '{1}' requested".format(self.name,attr))
        if (not self._frozen or attr=="_value"): # fast track this one, this has big performance impact
            self.__dict__[attr] = value
        #elif (not self._frozen): # we are still in process of initializing, assume we are not writing a node
        #    self.__dict__[attr] = value
        elif self.definition and (attr in self.definition):
            self.get_node(attr).write(value)
        elif attr in NodeValuePlugins.plugins():
            raise ValueError("Cannot override plugins in this fashion")
        # order is important for this one, it needs to be after the node checks
        elif (not self._frozen) or (attr in self.__dict__) or (attr in dir(self.__class__)):
            object.__setattr__(self,attr,value)
        else:
            raise Exception("cannot add new attributes at this point")

    def __repr__(self):
        # standard debug log
        if settings.DEBUG: _LOG.debugall("__repr__: '{0} : ENTER'".format(self.name))
        # attempt to get value if we do not have one....
        if self._value is None:
            if settings.DEBUG: _LOG.debugall("__repr__: '{0}' : get_value called because _value was None for node'".format(self.name))
            self.get_value()
        # if our access methods has a 'repr' function, use that....
        if self.access_class is not None and 'repr' in self.access_class.methods:
            if settings.DEBUG: _LOG.debugall("__repr__: '{0}' : EXIT : access had repr".format(self.name))
            return self.access_class.repr( self )
        else:
            # no repr...go up to previous level
            if settings.DEBUG: _LOG.debugall("__repr__: '{0}' : EXIT : access did not have repr".format(self.name))
            return super(NamedNodeValue,self).__repr__()

    def _get_access_priority(self, component, key="access_priority"):
        if key in component.target_info:
            return component.target_info[key]
        else:
            if component.parent:
                return self._get_access_priority(component.parent)
            else:
                return None        

    @slim_cached_property
    def access_class_name(self):
        """
        Returns the name of the class we should be using
        """
        #if self._access_class_name is not None:
        #    return self._access_class_name
        # if we don't have an access group, then we can't get from component
        if self.definition.access_group is None:
            acc_class_name = None
        elif self.component is not None:
            # ask the component what the current access is for our access group
            acc = self.component.get_access(self.definition.access_group)
            if settings.DEBUG:
                _LOG.debug("get_access_class: component access: '{0}'"
                           .format(acc))
            # Check our definition to know which class name that access matches
            acc_class_name = self.definition.accesses.get(acc, None)
        else:
            acc_class_name = None

        # if we still don't have an access class, use the default from the node
        if acc_class_name is None:
            acc_class_name = self.definition.accesses.get("default", None)
            if settings.DEBUG:
                _LOG.debug("get_access_class: default access used: '{0}'"
                           .format(acc_class_name))
            # if we still have None, then we have an error
            if acc_class_name is None:
                self.definition._log_access_error(self.component)

        if acc_class_name == _DEFAULT_ACCESS:
            access_priority = self._get_access_priority(self.component)
            if access_priority:
                for nacc in access_priority:
                    class_name = self.accesses.get(nacc, None)
                    if class_name:
                        acc_class_name = class_name
                        break
                else:
                    # we are somewhat broken but return node access and it will break when actual read/write occurs
                    return "DefaultNodeAccess"
            else:
                # we are somewhat broken but return node access and it will break when actual read/write occurs
                return "DefaultNodeAccess"
            
        if settings.DEBUG:
            _LOG.debug('get_access_class: EXIT : acc_class={0}'
                       .format(acc_class_name))

        #self._access_class_name = acc_class_name
        return acc_class_name

    @slim_cached_property
    def access_class(self):
        """
        Returns the access class which we should be using
        """
        # we have left in the code as if this were NOT a cached property ... for now...
        # but in reality this function gets called only ONCE and after that the access_class
        # does not have to flow through this code
        #if self._access_class is not None:
        #    return self._access_class

        if self.component is None:
            # means we were likely from a pickle flow
            return None

        if settings.DEBUG:
            _LOG.debug('access_class: ENTER'.format(self.component.name))

        acc_class_name = self.access_class_name

        # we have the access class name, ask the component for the actual
        # class object
        acc_class = self.component.access_classes.get(acc_class_name, None)
        if acc_class is None:
            # if acc_class is not None, then check for a 'default'
            acc_class = self.component.access_classes.get("default", None)
            if acc_class is None: # still, nothing found:
                raise RuntimeError(
                    "An access class for '%s' was never registered"%acc_class_name)
        # save this for later...so we don't do this over and over...
        #self._access_class = acc_class
        # finished...
        return acc_class

    def __copy__(self):
        raise RuntimeError("shallow copy not supported, must use deepcopy")

    def __deepcopy__(self, memodict={}):
        # creaate new object our same type
        new_node = type(self)(self.name,
                              self.component,
                              self.definition,
                              self.parent)

        new_node.target_info = deepcopy(self.target_info)
        new_node.array_info = deepcopy(self.array_info)
        new_node._removed_nodes = deepcopy(self._removed_nodes)
        new_node._value = self._value

        for cname, child in self._nodes_values.items():
            new_node._nodes_values[cname] = deepcopy(child)

        return new_node

    def __getstate__(self):
        """for pickling just return the value"""
        return {"_value":self._value, "_value_type": self._value_type}

    def __setstate__(self, state):
        self._value = state['_value']
        self._value_type = state['_value_type']
        self.target_info = odict()


class NamedNodeArrayDefinition(NamedNodeDefinition):
    type = NodeTypes.Array
    # same as a typical NNDefinition has one children with
    # where the value is determine by its instance name/index

    #: The keys or indexes that will be used to reference the child node, we keep this private
    #: so that no one accidently changes the order
    _item_keys = None
    #: The NodeDefinition that takes an item_key as an input to its value
    item_definition = None

    @property
    def value_class(self):
        """Value class that should be used when creating a Value to represent this node"""
        return NamedNodeArrayValue

    def __init__(self,name,info=None,item_keys = None,item_definition=None,**kwargs):
        """
        Args:
            item_keys : list of keys to associate with this array
            item_definition : item that will use the item_key as part of its access

        See Also:
            - :py:meth:`NamedNodeArrayValue`
            - :py:meth:`NamedNodeDefinition`
        """
        # use inheritance for most of our initialization
        NamedNodeDefinition.__init__(self,name,info,**kwargs)
        self._item_keys = item_keys or []

        if item_definition is not None:
            # make sure it is a NamedNodeDefinition
            if not isinstance(item_definition, NamedNodeDefinition):
                raise Exception("item_definition must be a NamedNodeDefinition")
        else:
            # item definition was none, create one using our same info
            item_definition = NamedNodeArrayItemDefinition(name=name,
                                                           **kwargs)
        # definitely have an array item def now
        self.item_definition = item_definition
        item_definition.parent = weakref.proxy(self)

    def __getattr__(self,attr):
        # slight attr tweak
        if attr == self.item_definition.name:
            return self.item_definition
        else:
            return super(NamedNodeArrayDefinition, self).__getattr__(attr)

    @property
    def item_keys(self):
        """The keys or indexes that will be used to reference the child node"""
        # so that no one accidently changes our keys
        return copy(self._item_keys)

    @item_keys.setter
    def item_keys(self,ikeys):
        self._item_keys = ikeys


    def __deepcopy__(self, memodict={}):
        # creaate new object our same type
        new_definition = deepcopy(self.item_definition)
        new_node = type(self)(self.name,
                              info=deepcopy(self.info),
                              accesses=copy(self.accesses),
                              access_group=self.access_group,
                              item_keys=self.item_keys,
                              item_definition=new_definition,
                              )
        for child in self.nodes:
            new_node.add_node(copy.deepcopy(child))
        return new_node

    # def __deepcopy__(self, memodict={}):
    #     #the_copy = super(NamedComponentDefinition,self).__deepcopy__(self,memo)
    #     #pdb;pdb.set_trace()
    #     #return the_copy
    #     cls = self.__class__
    #     result = cls.__new__(cls)
    #     memodict[id(self)] = result
    #     for k, v in self.__dict__.items():
    #         # keep up with our parent weakref
    #         if type(v) == weakref.ProxyType:
    #             copied = memodict[id(v.__self__)]
    #             setattr(result, k, weakref.proxy( copied ) )
    #         elif id(v) in memodict:
    #             setattr(result, k, memodict[id(v)])
    #         else:
    #             setattr(result, k, deepcopy(v, memo))
    #     return result

class NamedNodeArrayItemDefinition(NamedNodeDefinition):
    type = NodeTypes.ArrayItem

    @property
    def value_class(self):
        """Value class that should be used when creating a Value to represent this node"""
        return NamedNodeArrayItemValue

    @property
    def path(self):
        """return path from component to this node value"""
        # First check if there's a path override and use it if it's
        # available, this would mean that the node was added to a
        # component alias and that we need to report the aliased path.
        mypath = self.name + "[#]"
        if (self.parent is None): # should be rare that we have this
            return mypath
        # parent has not parent, should have a component
        if self.parent.parent is None:
            if self.parent.component is None:
                # this has gotta be rare too...or we just haven't been added yet
                return mypath
            else:
                return self.parent.component.path + "." + mypath
        else:
            return self.parent.parent.path + "." + mypath

    @property
    def nodepath(self):
        """path to this node from the component it belongs to"""
        # better name for this property?
        mypath = self.name + "[#]"
        if (self.parent is None) or (self.parent.parent is None):
            return mypath
        else:
            return self.parent.parent.nodepath + "." + mypath

# only difference here is that getitem/setitem are added and reference in to the
# _child_node_values using index as a key/value
class NamedNodeArrayValue(NamedNodeValue):
    """
    Value class for representing a NamedNodeArrayDefinition
    """
    type = NodeTypes.Array
    # Named Node Array could have both children and array children
    # this should not be used directly since it may not always be "filled out"
    # the array should be iterated through or should be casted as a list
    ##
    ## TO DO...need to fill out the array capabilities of the NodeArray
    ##
    # this is not ordered for speed, use the definition's keys for ordering
    _array_nodes = {}

    def __init__(self,name,component,definition,parent_nodevalue):
        # call parent class
        NamedNodeValue.__init__(self,name,component,definition,parent_nodevalue)
        # build ordered directionary and key None as default child (no nodevalue created yet)
        self._array_nodes = {k:None for k in self.definition.item_keys}

    def __len__(self):
        """return length as defined by the definition"""
        return len(self.definition.item_keys)

    def __contains__(self, key):
        return key in self.definition.item_keys

    def __iter__(self):
        for i in self.definition.item_keys:
            yield self[i]

    def keys(self):
        for k in self.definition.item_keys:
            yield k

    def values(self):
        for i in self.definition.item_keys:
            yield self[i]

    def items(self):
        for i in self.definition.item_keys:
            yield i, self[i]

    def get(self, item, default=None):
        return self.__getitem__(item, default)

    def __list__(self):
        # return a list of our nodes using the iterator
        return [s for s in self]

    # really we probably have to support slicing
    def __getitem__(self, item, default=LookupError):
        if isinstance(item,slice):
            slice_list = []
            rmin = min(item.start,item.stop)
            rmax = max(item.start,item.stop)
            step = item.step or 1
            for s in range(rmin,rmax+1,step):
                slice_list.append( self.__getitem__(s) )
            return slice_list

        if settings.DEBUG: _LOG.debug("NodeArray.{0} requested".format(self.name))
        nodeval = self._array_nodes.get(item, LookupError)
        # if we dont have this node, return default that was given...
        if nodeval is LookupError and default is not LookupError:
            return default
        elif nodeval is LookupError:  # and default was still LookupError
            raise LookupError("Unknown node: %s"%item)

        # we already have a nodevalue object, return it...
        if nodeval is not None:
            if settings.DEBUG: _LOG.debug("NodeArray.getitem - {name}[{item}] already existed".\
                                  format(name=self.name,item=item))
            return nodeval
        # nodevalue doesn't exist yet...create it
        if settings.DEBUG: _LOG.debug("NodeArray.getitem - {name}[{item}] did not exist, creating new nodevalue".\
                                    format(name=self.name,item=item))
        # new nodeval needs to be created to replace our "None"
        # but tweak the name
        if isinstance(item, six.string_types):
            name  = self.name + "[\"%s\"]" % item
        else:
            name = self.name + "[%s]" % str(item)
        newvalue = self._array_nodes[item] = \
                    self.definition.item_definition.value_class(
                        name,
                        self.component,
                        self.definition.item_definition,
                        self)
        # store off the item key used in case it is needed
        newvalue.target_info['item_key'] = item

        # add target_info if it exists
        nn_instances = self.definition.item_definition.info.get("nn_instances", None)
        if nn_instances is not None:
            if item in nn_instances:
                newvalue.target_info.update(nn_instances[item])
            else:
                _LOG.warning("nn_instances contains no info for item: %s", item)

        # update with the array information we have so far, plus
        # the addtitional array path that we just took
        newvalue.array_info.update(self.array_info)
        newvalue.array_info[self.name] = item
        # pass along any array info that we may have
        newvalue.array_info.update( self.array_info )
        self._array_nodes[ item ] = newvalue
        return newvalue

    def __setitem__(self, item, value):
         self.__getitem__(item).write(value)

    def free(self, item, error=False):
        """remove cached reference to a child node
        Args:
            item (str) : item key for the node to remove the reference to
        """
        # theoretically we do not need to do any recursive here
        try:
            del self._array_nodes[item]
            # dont do gc.collect here...users should do that..not us
        except KeyError:
            if error:
                raise
            # ignore any key error due to the free not being needed
            pass


class NamedNodeArrayItemValue(NamedNodeValue):
    """
    Value class for representing a NamedNodeArrayDefinition
    """
    type = NodeTypes.ArrayItem


class NodeValuePlugins(PluginManager):
    basetype = "NodeValuePlugin"
    _plugins = {} # important to have plugins local to this class

# A NodeValue plugin is a single function
@with_metaclass(NodeValuePlugins)
class NodeValuePlugin(object):
    #: Function name that should be added to nodes
    name = ""

    def __init__(self,nodevalue):
        self.nodevalue = nodevalue

    def __dir__(self):
        return []

    @classmethod
    def __dir__(cls):
        return []

    @classmethod
    def create(cls,nodevalue):
        """
        Create will return an instance of this plugin (which is callable). If this
        plugin does not support the given component or nodevalue object, then it
        should return None

        The actual NodeValue plugin's get created when a user/script attempts to
        get an attribute from the NamedNodeValue. If the NamedNodeValue does not
        already have that attribute, then plugins are checked to see if the attribute
        matches a known plugin. If it does, then the NamedNode plugin is created.
        If the create returns an object, then the user can call that function. However
        if the create retuns None, then the NamedNodeValue will throw an exception

        """
        # can be used to confirm that this plugin works on this nodevalue
        return cls(nodevalue)


class NodeDefinitionPlugins(PluginManager):
    basetype = "NodeDefinitionPlugin"
    _plugins = {} # important to have plugins local to this class


# A NodeDefinition plugin is a single function
@with_metaclass(NodeDefinitionPlugins)
class NodeDefinitionPlugin(object):
    #: Function name that should be added to nodes
    name = ""

    def __init__(self,definition):
        self.definition = definition

    @classmethod
    def create(cls,definition):
        """
        Create wil return an instnace of this plugin (which is callable). If this
        plugin does not support the given component or node object, then it
        should return None

        The actual NodeValue plugin's get created when a user/script attempts to
        get an attribute from the NamedNodeValue. If the NamedNodeValue does not
        already have that attribute, then plugins are checked to see if the attribute
        matches a known plugin. If it does, then the NamedNode plugin is created.
        If the create returns an object, then the user can call that function. However
        if the create retuns None, then the NamedNodeValue will throw an exception

        """
        # can be used to confirm that this plugin works on this nodevalue
        return cls(definition)

    def __call__(self,*args,**kwargs):
        raise Exception("Must be implemented for each plugin")


class NamedNodeDefinitionAlias(object):
    """
    Defines a named node definition alias to help us keep track
    of node definitions and their component values (the named component).
    """
    type = NodeTypes.Alias
    name = None  # different name from the original definition
    _node_definition = None
    _node_component = None

    def __init__(self, name, definition, component):
        self.name = name
        self._node_definition = definition
        self._node_component = component

    def __getattr__(self, attr):
        return object.__getattribute__(self._node_definition, attr)

    def __dir__(self):
        return dir(self._node_definition)

    # value_class definition in NamedNodeDefinition is actually a
    # class method but, for some reason, we can override it here as an
    # instance method with no problem
    @property
    def value_class(self):
        """
        Value class that should be used when creating a value to
        represent the stored node's definition.
        """
        def function(name, component, nodedef, parent):
            # use the original component not the one the alias was tied to
            nodevalue = self._node_definition.value_class(
                name,
                self._node_component,
                self._node_definition,
                parent
            )
            # Override path in node value so that it reports
            # the new aliased path.
            orig_path = self._node_component.path + "." + self._node_definition.nodepath
            path_override = component.path + "." + name
            nodevalue.path = path_override
            nodevalue.target_info['aliases_original_path'] = orig_path
            return nodevalue
        return function

    @property
    def node_definition(self):
        """ Get the node's definition """
        return self._node_definition

    @property
    def node_component(self):
        """ Get the node's component value (the named component) """
        return self._node_component
