# INTEL CONFIDENTIAL
# Copyright 2014 2018 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from ipccli.cli_logging import getLogger as _getLogger
from ipccli.cli_logging import LoggerManager as _LoggerManager
from ipccli.cli_logging import getManager as _getManager
from . import settings
import logging.handlers as _default_handlers
import logging as _logging
from six import PY2
from functools import wraps

_ROOT_LOGGER = "namednodes"

class _DynamicHandler(_logging.Handler):

    def __init__(self, factory, factory_params, formatter=None, loglevel=1):
        """
        Waits till a log event before creating handler requires keyword parameters
        """
        self._factor = factory
        self._params = factory_params
        self._formatter = formatter
        self._handler = None
        self.level = loglevel

    def emit(self, *args, **kwargs):
        if self._handler is None and self._factory is not None:
            try:
                self._handler = self._factor(**self._params)
            except Exception as e:
                import traceback
                traceback.print_exc()
                print("ERROR openning handler")
                self._factory = None
            if self._formatter:
                self._handler.setFormatter(self._formatter)
        # in case we had an exception opening handler due to some read-only current working directory scenario
        if self._handler is not None:
            return self._handler.emit(*args, **kwargs)

    def close(self):
        # special case, if we never opened handler, nothing to do
        if self._handler is None:
            return
        else:
            return self._handler.close()

    def __getattr__(self, attr):
         if self._handler is None:
             # if we got here somehow, then we didn't wrap something we should have, so create handler
             # so we'll create handler and try to survive it..
             self._handler = self._factor(**self._params)
             if self._formatter:
                 self._handler.setFormatter(self._formatter)
             self._handler.setLevel(self.level)
         return getattr(self._handler, attr)

def getLogger(name=_ROOT_LOGGER):
    """for now this routes over to ipccli getLogger"""
    if not name.startswith(_ROOT_LOGGER):
        name = _ROOT_LOGGER + "." + name
    logger = _getLogger(name)
    # make sure we control most logging through namednodes
    if getattr(logger,"_init", None) is None:
        logger.setLevel(1)
        logger._init = True
    return _getLogger(name)


@wraps(_LoggerManager.echo)
def echo(echoOn=None):
    mgr = _getManager()
    return mgr.echo(echoOn)


@wraps(_LoggerManager.level)
def level(loggername=None, loglevel=None):
    # tweaked to only impact the namednodes loggers
    mgr = _getManager()

    names2level = _logging._nameToLevel if not PY2 else _logging._levelNames
    if isinstance(loggername, str) and loggername.upper() in names2level:
        # user got these backwards...just swap them
        loggername, loglevel = loglevel, loggername

    # this is similar code for the ipccli but we need to modify to only do this for our loggers
    if loglevel in ["debugall", "DEBUGALL", 1]:
        settings.DEBUG = True

    if loggername in [None, "all", "ALL"] and loglevel is not None:
        # set for ALL logger's starting with namednodes
        for lname in getLoggerNames():
            mgr.level(lname, loglevel)
    else:
        return mgr.level(loggername, loglevel)


def getLoggerNames():
    """Returns the loggers associated with namednodes"""
    mgr = _getManager()
    loggernames = mgr.getLoggerNames(True)
    ourloggers = [lname for lname in loggernames if lname.startswith(_ROOT_LOGGER)]
    return ourloggers


@wraps(_LoggerManager.setFile)
def setFile(filename, filemode='w', maxBytes=0,backupCount=10,
                logformat='%(asctime)s - %(levelname)s - %(message)s',
            loglevel=1):
    # by default loglevel will recieve anything the loggers send
    # Note we add this to namednodes NOT the root ipccli
    mgr = _getManager()
    our_root_logger = mgr.getLogger(_ROOT_LOGGER)
    # if we already have a handler, remove it, and then add in the new one
    if getattr(our_root_logger, "_root_filehandler", None) is not None:
        closeFile()
    formatter = _logging.Formatter(logformat)
    # turn level in to number if it comes in as a string
    names2level = _logging._nameToLevel if not PY2 else _logging._levelNames
    if isinstance(loglevel, str) and loglevel.upper() in names2level:
        loglevel = names2level[loglevel]

    new_handler = _DynamicHandler(_default_handlers.RotatingFileHandler,
        dict(filename=filename, mode=filemode, maxBytes=maxBytes, backupCount=backupCount),
        formatter=formatter,
        loglevel=loglevel,
    )
    our_root_logger._root_filehandler = new_handler
    our_root_logger.addHandler(new_handler)

def closeFile():
    """Stop Logging from going to file"""
    our_root_logger = _getManager().getLogger(_ROOT_LOGGER)
    if our_root_logger._root_filehandler is not None:
        our_root_logger._root_filehandler.close()
        our_root_logger.removeHandler(our_root_logger._root_filehandler)
        our_root_logger._root_filehandler = None

def reset():
    """resets logging"""
    settings.DEBUG = False
    # reset the default manager
    _getManager().reset()
    closeFile()
    _default_setup()

def _default_setup():
    # during init we will set a dynamic file so that we only get messages on error
    setFile("namednodes.log", loglevel=_logging.INFO)
    getLogger("namednodes").setLevel(_logging.INFO)

# call this once to make sure logger is created before we call default setup
getLogger()
_default_setup()