
# INTEL CONFIDENTIAL
# Copyright 2014 2017 Intel Corporation
#
# The source code  contained or  described herein and  all documents related to
# the source code  ("Material") are owned by Intel Corporation or its suppliers
# or licensors.  Title to the  Material  remains with  Intel Corporation or its
# suppliers  and licensors. The Material contains trade secrets and proprietary
# and  confidential  information  of  Intel  or  its  suppliers  and  licensors.
# The Material  is protected  by worldwide  copyright and trade secret laws and
# treaty provisions.  No part of the Material  may  be used, copied, reproduced,
# modified, published, uploaded, posted, transmitted, distributed, or disclosed
# in any way without Intel's prior express written permission. No license under
# any  patent,  copyright, trade secret or other intellectual property right is
# granted  to  or conferred upon you by disclosure or delivery of the Materials,
# either expressly, by implication, inducement, estoppel or otherwise.
# Any license under such intellectual property rights must be express and
# approved by Intel in writing.




"""
Access logger module for enabling/disabling access events logging.
"""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import traceback
import os
from ...logging import getLogger
from logging.handlers import RotatingFileHandler
from logging import Formatter
from ... import settings
from ...nodes import NodeTypes
from ...events import accesses as access_events
from ...events.accesses import AccessEventTypes
from ...registers import RegisterValue, FieldValue
from ...utils._py2to3 import *

_LOG = getLogger("accesses")


# used to store the supported log formats
_log_formats = ['corecode', 'csv']
# used to store the active log formatter object
# we store a reference to it for when calling disable so that we know
# which handler to unsubscribe from event manager
_log_formatter = None

def _create_file_handler(logfilename):
    handler = RotatingFileHandler(
        logfilename,
        "w",
        maxBytes=0,
        backupCount=3,
    )
    handler.setFormatter(
        Formatter("%(message)s")
    )
    return handler


def enable(format, logfilename=None):
    """
    Enable access events logging.

    Args:
        format (str): The format of the generated log file.
        logfilename (str, optional): The name of the log file to
            write the events to.

    Formats:

    "*corecode" - generates a log file with the same format as the legacy *'corecode'* register
    infastructure

    an corecode format (with some of the leading file information removed)::

        Accessed: cmd=PRE-UPDATE-VALUE; Register=ia32_mcg_cap; IntelRsvd=True; parent=socket0.core0.thread0
        Accessed: cmd=PRE-READ; Register=ia32_mcg_cap; IntelRsvd=True; parent=socket0.core0.thread0
        Accessed: cmd=POST-READ; Register=ia32_mcg_cap; value=0x1; IntelRsvd=True; parent=socket0.core0.thread0
        Accessed: cmd=POST-UPDATE-VALUE; Register=ia32_mcg_cap; value=0x1; IntelRsvd=True; parent=socket0.core0.thread0

    "csv" - generates a comma delimitted file with the first row being column names::

        File, LineNo, CoName, Event, NodeType,NodeName, WriteMask, WriteValue, ReadValue, IntelRSvd,Component
        interactiveshell.py, LineNo: 2828, CoName: run_ast_nodes, Accessed: PRE-STORE,Field,pcie_header.count,None,None,None,True,socket0.uncore
        interactiveshell.py, LineNo: 2828, CoName: run_ast_nodes, Accessed: POST-STORE,Field,pcie_header.count,None,0x1,None,True,socket0.uncore
        interactiveshell.py, LineNo: 2828, CoName: run_ast_nodes, Accessed: PRE-FLUSH,Register,pcie_header,None,None,None,True,socket0.uncore
        interactiveshell.py, LineNo: 2828, CoName: run_ast_nodes, Accessed: POST-FLUSH,Register,pcie_header,None,None,None,True,socket0.uncore
        
    """
    if format not in _log_formats:
        raise Exception("Unknown '%s' log format specified."%format)

    _LOG._previous_level = _LOG.level
    if logfilename is None and format == "csv":
        raise ValueError("logfilename must be provided for csv")

    if logfilename is not None:
        fhandler = _create_file_handler(logfilename)
        _LOG._fhandler = fhandler
        _LOG.addHandler(fhandler)
        # make sure all messages are getting through...
        fhandler.setLevel(1)
    # this is needed regardless of whether we added a log file or not
    _LOG.setLevel(1)

    log_formatter = FactoryLogFormatter.get_log_formatter(format)
    access_events.subscribe(log_formatter.log_event_handler)
    settings.EVENTS_ON_ACCESSES = True


def disable(format):
    """
    Disable access events logging.

    Args:
        format (str): The format of the logger we want to disable.
    """
    settings.EVENTS_ON_ACCESSES = False
    log_formatter = FactoryLogFormatter.get_log_formatter(format)
    if log_formatter is not None:
        access_events.unsubscribe(log_formatter.log_event_handler)
    previous_level = getattr(_LOG, "_previous_level", None)
    if previous_level:
        _LOG.setLevel(_LOG._previous_level)
    fhandler = getattr(_LOG, "_fhandler", None)
    if fhandler is not None:
        _LOG.removeHandler(fhandler)
        del _LOG._fhandler


class AccessLogFormatter(object):
    """
    Base class for access log formatters.

    Implementors of different log formats must inherit from this class.
    """
    def __init__(self, format):
        """
        Constructor.

        Args:
            format (str): The log format.
        """
        self._format = format

    def log_event_handler(self, event):
        """
        Log an access event.

        Args:
            event (AccessEvent): The event to log.
        """
        raise Exception("Must be implemented")


class CorecodeLogFormatter(AccessLogFormatter):
    """
    Log formatter for 'corecode' access log format.
    """
    def __init__(self):
        """
        Constructor.
        """
        super(CorecodeLogFormatter, self).__init__('corecode')

    def log_event_handler(self, event):
        """
        Log an access event.

        Args:
            event (AccessEvent): The event to log.
        """
        if event.event_type in AccessEventTypes.pre_event_types:
            self._log_pre_event(event)
        elif event.event_type in AccessEventTypes.post_event_types:
            self._log_post_event(event)

    def _log_caller(self, caller_info, stacklevel=-1):
        """
        Find the stack frame of the caller so that we can note the source
        file name, line number and function name.

        Args:
            caller_info (str): Text added by to the log entry by whoever is
                calling this.
        """
        stack = traceback.extract_stack()
        laststack=stack.pop() # at least one for previous caller
        while stacklevel<0 and len(stack)>1:
            laststack=stack.pop()
            stacklevel += 1
        _LOG.debugall("File: %s, LineNo: %d, CoName: %s, Accessed: %s" % (
            laststack[0], laststack[1], laststack[2], caller_info))

    def _log_pre_event(self, event):
        """
        Log pre event info.

        Args:
            event (AccessEvent): The pre event.
        """
        if isinstance(event.node, RegisterValue):
            caller = "Register"
        elif isinstance(event.node, FieldValue):
            caller = "Field"
        else:
            caller = "Node"

        # lets check if the node has a parent from
        # which we can get the path
        if event.node.parent is not None:
            parent = event.node.parent.path
        else:
            parent = event.node.path

        self._log_caller("cmd=%s; %s=%s; IntelRsvd=%s; parent=%s" % (
            event.event_type_str.upper().replace("_", "-"),
            caller,
            event.node.name,
            getattr(event.node,"IntelRsvd",None),
            parent),
            # need to back up due to whoever called read
            # unfortunately this may require tweaking from time to time...
            stacklevel=-7)

    def _log_post_event(self, event):
        """
        Log post event info.

        Args:
            event (AccessEvent): The post event.
        """
        if isinstance(event.node, RegisterValue):
            caller = "Register"
        elif isinstance(event.node, FieldValue):
            caller = "Field"
        else:
            caller = "Node"

        # lets check if the node has a parent from
        # which we can get the path
        if event.node.parent is not None:
            parent = event.node.parent.path
        else:
            parent = event.node.path

        # for stores we post the stored value
        if event.event_type in [AccessEventTypes.post_store, AccessEventTypes.post_flush]:
            if event.node.type == NodeTypes.Register:
                stored_value = event.node.stored_value
            elif event.node.type == NodeTypes.Field:
                stored_value = event.node.parent.stored_value
            else:
                stored_value = None
            stored_valuestr = "0x%X"%stored_value if stored_value is not None else "None"
            self._log_caller(
                "cmd=%s; %s=%s; stored_value=%s IntelRsvd=%s; parent=%s" % (
                    event.event_type_str.upper().replace("_", "-"),
                    caller,
                    event.node.name,
                    stored_valuestr,
                    getattr(event.node, "IntelRsvd", None),
                    parent),
                    stacklevel=-7)
        # for writes we want to post the value written
        elif event.event_type not in [AccessEventTypes.pre_read_write, AccessEventTypes.post_read_write]:
            if event.event_type == AccessEventTypes.post_write:
                # write code always passes it through as first argument
                value = event.event_data['args'][0]
            else:
                value = event.node.value  # copy for speed
            valuestr = "0x%X"%value if value is not None else "None"
            self._log_caller(
                "cmd=%s; %s=%s; value=%s; IntelRsvd=%s; parent=%s" % (
                    event.event_type_str.upper().replace("_", "-"),
                    caller,
                    event.node.name,
                    valuestr,
                    getattr(event.node, "IntelRsvd", None),
                    parent),
                    stacklevel=-7)
        else:
            # must be a read/write
            mask, value = event.event_data['args'][0:2]
            if event.event_type == AccessEventTypes.pre_read_write:
                self._log_caller(
                    "cmd=%s; %s=%s; mask=0x%X, value=0x%X; IntelRsvd=%s; parent=%s" % (
                        event.event_type_str.upper().replace("_", "-"),
                        caller,
                        event.node.name,
                        mask,
                        value,
                        getattr(event.node, "IntelRsvd", None),
                        parent),
                        stacklevel=-7)
            else: # must be post
                retvalue = event.node.value
                retvaluestr = "0x%X" % retvalue if retvalue is not None else "None"
                self._log_caller(
                    "cmd=%s; %s=%s; mask=0x%X, value=0x%X; return_value=%s; IntelRsvd=%s; parent=%s" % (
                        event.event_type_str.upper().replace("_", "-"),
                        caller,
                        event.node.name,
                        mask,
                        value,
                        retvaluestr,
                        getattr(event.node, "IntelRsvd", None),
                        parent),
                        stacklevel=-7)



class CommaSeparatedValuesLogFormatter(AccessLogFormatter):
    _header_done = None
    _row_dict = {
        'event':None,
        'caller':None,
        'nodename':None,
        'mask':None,
        'value':None,
        'readvalue':None,
        'intelrsvd':None,
        'parent':None,
    }
    def __init__(self):
        super(CommaSeparatedValuesLogFormatter,self).__init__('csv')
        self._header_done = False
        

    def _get_row_dict(self):
        new_dict = self._row_dict.copy()
        return new_dict

    def _create_row_entry(self, row_dict):
        event = row_dict.get('event', str(None))
        nodetype = row_dict.get('caller', str(None))
        nodename = row_dict.get('nodename', str(None))
        mask = row_dict.get('writemask', str(None))
        value = row_dict.get('writevalue', str(None))
        readvalue = row_dict.get('readvalue', str(None))
        intelrsv = row_dict.get('intelrsvd', str(None))
        parent = row_dict.get('parent', str(None))
        return "{},{},{},{},{},{},{},{}".format(event,
                                                nodetype,
                                                nodename,
                                                mask,
                                                value,
                                                readvalue,
                                                intelrsv,
                                                parent)

    def log_event_handler(self, event):
        """
        Log an access event.

        Args:
            event (AccessEvent): The event to log.
        """
        if self._header_done is False:
            self._header_done = True
            _LOG.debugall("File, LineNo, CoName, Event, NodeType,NodeName, WriteMask, WriteValue, ReadValue, IntelRSvd,Component")
        if event.event_type in AccessEventTypes.pre_event_types:
            self._log_pre_event(event)
        elif event.event_type in AccessEventTypes.post_event_types:
            self._log_post_event(event)

    def _log_caller(self, caller_info, stacklevel=-1):
        """
        Find the stack frame of the caller so that we can note the source
        file name, line number and function name.

        Args:
            caller_info (str): Text added by to the log entry by whoever is
                calling this.
        """
        stack = traceback.extract_stack()
        laststack=stack.pop() # at least one for previous caller
        while stacklevel<0 and len(stack)>1:
            laststack=stack.pop()
            stacklevel += 1
        _LOG.debugall("File: %s, LineNo: %d, CoName: %s, Accessed: %s" % (
            laststack[0], laststack[1], laststack[2], caller_info))

    def _log_pre_event(self, event):
        """
        Log pre event info.

        Args:
            event (AccessEvent): The pre event.
        """
        if isinstance(event.node, RegisterValue):
            caller = "Register"
        elif isinstance(event.node, FieldValue):
            caller = "Field"
        else:
            caller = "Node"

        row_dict = self._get_row_dict()
        row_dict['event'] = event.event_type_str.upper().replace("_", "-")
        row_dict['caller'] = caller
        row_dict['nodename'] = event.node.nodepath
        row_dict['intelrsvd'] = getattr(event.node,"IntelRsvd", None)
        row_dict['parent'] = event.node.component.path
        self._log_caller(self._create_row_entry(row_dict), stacklevel=-7)

    def _log_post_event(self, event):
        """
        Log post event info.

        Args:
            event (AccessEvent): The post event.
        """
        if isinstance(event.node, RegisterValue):
            caller = "Register"
        elif isinstance(event.node, FieldValue):
            caller = "Field"
        else:
            caller = "Node"
        # for stores we post the stored value
        if event.event_type in [AccessEventTypes.post_store, AccessEventTypes.post_flush]:
            if event.node.type == NodeTypes.Register:
                stored_value = event.node.stored_value
            elif event.node.type == NodeTypes.Field:
                stored_value = event.node.parent.stored_value
            else:
                stored_value = None
            stored_valuestr = "0x%X"%stored_value if stored_value is not None else "None"

            row_dict = self._get_row_dict()
            row_dict['event'] = event.event_type_str.upper().replace("_", "-")
            row_dict['caller'] = caller
            row_dict['nodename'] = event.node.nodepath
            row_dict['writevalue'] = stored_valuestr
            row_dict['intelrsvd'] = getattr(event.node, "IntelRsvd", None)
            row_dict['parent'] = event.node.component.path
            self._log_caller(self._create_row_entry(row_dict), stacklevel=-7)
        # for writes we want to post the value written
        elif event.event_type not in [AccessEventTypes.pre_read_write, AccessEventTypes.post_read_write]:
            if event.event_type == AccessEventTypes.post_write:
                # write code always passes it through as first argument
                value = event.event_data['args'][0]
            else:
                value = event.node.value  # copy for speed
            valuestr = "0x%X"%value if value is not None else "None"
            row_dict = self._get_row_dict()
            row_dict['event'] = event.event_type_str.upper().replace("_", "-")
            row_dict['caller'] = caller
            row_dict['nodename'] =  event.node.nodepath
            row_dict['readvalue'] = valuestr
            row_dict['intelrsvd'] = getattr(event.node, "IntelRsvd", None)
            row_dict['parent'] = event.node.component.path
            self._log_caller(self._create_row_entry(row_dict), stacklevel=-7)
        else:
            # must be a read/write
            mask, value = event.event_data['args'][0:2]
            if event.event_type == AccessEventTypes.pre_read_write:
                row_dict = self._get_row_dict()
                row_dict['event'] = event.event_type_str.upper().replace("_", "-")
                row_dict['caller'] = caller
                row_dict['nodename'] = event.node.nodepath
                row_dict['writemask'] = mask
                row_dict['writevalue'] = value
                row_dict['intelrsvd'] = getattr(event.node, "IntelRsvd", None)
                row_dict['parent'] = event.node.component.path      
                self._log_caller(self._create_row_entry(row_dict), stacklevel=-7)          
            else: # must be post
                retvalue = event.node.value
                retvaluestr = "0x%X" % retvalue if retvalue is not None else "None"
                row_dict = self._get_row_dict()
                row_dict['event'] = event.event_type_str.upper().replace("_", "-")
                row_dict['caller'] = caller
                row_dict['nodename'] = event.node.nodepath
                row_dict['writemask'] = mask
                row_dict['writevalue'] = value
                row_dict['readvalue'] = retvaluestr
                row_dict['intelrsvd'] = getattr(event.node, "IntelRsvd", None)
                row_dict['parent'] = event.node.component.path  
                self._log_caller(self._create_row_entry(row_dict), stacklevel=-7)              
 


class FactoryLogFormatter(object):
    """
    Factory class for log formatters.
    """
    # used to store "references" to created log formatters
    _log_formatters = {
        'corecode': CorecodeLogFormatter(),
        'csv': CommaSeparatedValuesLogFormatter(),        
    }

    @classmethod
    def get_log_formatter(cls, format):
        """
        Factory method for getting a log formatter object.

        Here's where new log formatters will be added.
        """
        log_formatter = cls._log_formatters.get(format, None)
        return log_formatter
